-- *************************************************************************
--
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:          axi_sg_pkg.vhd
-- Description: This package contains various constants and functions for
--              AXI SG Engine.
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

package axi_sg_pkg is

-------------------------------------------------------------------------------
-- Function declarations
-------------------------------------------------------------------------------
-- Convert boolean to a std_logic
function bo2int (value : boolean)
            return  integer;

-------------------------------------------------------------------------------
-- Constant Declarations
-------------------------------------------------------------------------------

-- AXI Response Values
constant OKAY_RESP                      : std_logic_vector(1 downto 0)  := "00";
constant EXOKAY_RESP                    : std_logic_vector(1 downto 0)  := "01";
constant SLVERR_RESP                    : std_logic_vector(1 downto 0)  := "10";
constant DECERR_RESP                    : std_logic_vector(1 downto 0)  := "11";

-- Misc Constants
constant CMD_BASE_WIDTH                 : integer := 40;
constant SG_BTT_WIDTH                   : integer := 7;
constant SG_ADDR_LSB                    : integer := 6;

-- Interrupt Coalescing
constant ZERO_THRESHOLD             : std_logic_vector(7 downto 0) := (others => '0');
constant ONE_THRESHOLD              : std_logic_vector(7 downto 0) := "00000001";
constant ZERO_DELAY                 : std_logic_vector(7 downto 0) := (others => '0');

-- Constants Used in Desc Updates
constant DESC_STS_TYPE                  : std_logic := '1';
constant DESC_DATA_TYPE                 : std_logic := '0';

-- DataMover Command / Status Constants
constant DATAMOVER_STS_CMDDONE_BIT      : integer := 7;
constant DATAMOVER_STS_SLVERR_BIT       : integer := 6;
constant DATAMOVER_STS_DECERR_BIT       : integer := 5;
constant DATAMOVER_STS_INTERR_BIT       : integer := 4;
constant DATAMOVER_STS_TAGMSB_BIT       : integer := 3;
constant DATAMOVER_STS_TAGLSB_BIT       : integer := 0;

constant DATAMOVER_CMD_BTTLSB_BIT       : integer := 0;
constant DATAMOVER_CMD_BTTMSB_BIT       : integer := 22;
constant DATAMOVER_CMD_TYPE_BIT         : integer := 23;
constant DATAMOVER_CMD_DSALSB_BIT       : integer := 24;
constant DATAMOVER_CMD_DSAMSB_BIT       : integer := 29;
constant DATAMOVER_CMD_EOF_BIT          : integer := 30;
constant DATAMOVER_CMD_DRR_BIT          : integer := 31;
constant DATAMOVER_CMD_ADDRLSB_BIT      : integer := 32;

-- Note: Bit offset require adding ADDR WIDTH to get to actual bit index
constant DATAMOVER_CMD_ADDRMSB_BOFST    : integer := 31;
constant DATAMOVER_CMD_TAGLSB_BOFST     : integer := 32;
constant DATAMOVER_CMD_TAGMSB_BOFST     : integer := 35;
constant DATAMOVER_CMD_RSVLSB_BOFST     : integer := 36;
constant DATAMOVER_CMD_RSVMSB_BOFST     : integer := 39;

-- Descriptor field bits
constant DESC_STS_INTERR_BIT            : integer := 28;
constant DESC_STS_SLVERR_BIT            : integer := 29;
constant DESC_STS_DECERR_BIT            : integer := 30;
constant DESC_STS_CMPLTD_BIT            : integer := 31;

-- IOC Bit on descriptor update
-- Stored in LSB of TAG field then catinated on status word from primary
-- datamover (i.e. DESCTYPE & IOC & STATUS & Bytes Transferred).
constant DESC_IOC_TAG_BIT               : integer := 32;


end axi_sg_pkg;

-------------------------------------------------------------------------------
-- PACKAGE BODY
-------------------------------------------------------------------------------
package body axi_sg_pkg is
-------------------------------------------------------------------------------
-- Boolean to Integer
-------------------------------------------------------------------------------
function bo2int ( value : boolean)
    return integer  is
variable value_int : integer;
begin
    if(value)then
        value_int := 1;
    else
        value_int := 0;
    end if;
    return value_int;
end function bo2int;

end package body axi_sg_pkg;


-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_sg_reset.vhd
  --
  -- Description:     
  --   This file implements the DataMover Reset module.               
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  
  -------------------------------------------------------------------------------
  
  entity axi_sg_reset is
    generic (
      
      C_STSCMD_IS_ASYNC    : Integer range  0 to  1 :=  0
        -- 0 = Use Synchronous Command/Statys User Interface
        -- 1 = Use Asynchronous Command/Statys User Interface
      );
    port (
      
      -- Primary Clock and Reset Inputs -----------------
                                                       --
      primary_aclk         : in  std_logic;            --
      primary_aresetn      : in  std_logic;            --
      ---------------------------------------------------
      
                                                       
      -- Async operation clock and reset from User ------
      -- Used for Command/Status User interface        --
      -- synchronization when C_STSCMD_IS_ASYNC = 1    --
                                                       --
      secondary_awclk      : in  std_logic;            --
      secondary_aresetn    : in  std_logic;            --
      ---------------------------------------------------
  
                                           
     
      -- Halt request input control -------------------------------
      halt_req             : in  std_logic;                      --
         -- Active high soft shutdown request (can be a pulse)   --
                                                                 --
      -- Halt Complete status flag                               --
      halt_cmplt           : Out std_logic;                      --
         -- Active high soft shutdown complete status            --
      -------------------------------------------------------------
      
       
                                                 
      -- Soft Shutdown internal interface ------------------------------------------------
                                                                                        --
      flush_stop_request   : Out std_logic;                                             --
         -- Active high soft stop request to modules                                    --
                                                                                        --
      data_cntlr_stopped   : in  std_logic;                                             --
         -- Active high flag indicating the data controller is flushed and stopped      --
                                                                                        --
      addr_cntlr_stopped   : in  std_logic;                                             --
         -- Active high flag indicating the address controller is flushed and stopped   --
                                                                                        --
      aux1_stopped         : in  std_logic;                                             --
         -- Active high flag flush complete for auxillary 1 module                      --
         -- Tie high if unused                                                          --
                                                                                        --
      aux2_stopped         : in  std_logic;                                             --
         -- Active high flag flush complete for auxillary 2 module                      --
         -- Tie high if unused                                                          --
      ------------------------------------------------------------------------------------
      
       
          
      -- HW Reset outputs to reset groups  -------------------------------------     
                                                                              --
      cmd_stat_rst_user    : Out std_logic;                                   --
         -- The reset to the Command/Status Module User interface side        --
                                                                              --
      cmd_stat_rst_int     : Out std_logic;                                   --
         -- The reset to the Command/Status Module internal interface side    --
                                                                              --
      mmap_rst             : Out std_logic;                                   --
         -- The reset to the Memory Map interface side                        --
                                                                              --
      stream_rst           : Out std_logic                                    --
         -- The reset to the Stream interface side                            --
      --------------------------------------------------------------------------
      
      );
  
  end entity axi_sg_reset;
  
  
  architecture implementation of axi_sg_reset is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    -- Signals
    
    signal sig_cmd_stat_rst_user_n     : std_logic := '0';
    signal sig_cmd_stat_rst_user_reg_n_cdc_from : std_logic := '0';
    signal sig_cmd_stat_rst_int_reg_n  : std_logic := '0';
    signal sig_mmap_rst_reg_n          : std_logic := '0';
    signal sig_stream_rst_reg_n        : std_logic := '0';
    signal sig_syncd_sec_rst           : std_logic := '0';
    
    -- soft shutdown support
    signal sig_internal_reset          : std_logic := '0';
    signal sig_s_h_halt_reg            : std_logic := '0';
    signal sig_halt_cmplt              : std_logic := '0';
                               
    -- additional CDC synchronization signals
    signal sig_sec_neg_edge_plus_delay : std_logic := '0';
    signal sig_secondary_aresetn_reg   : std_logic := '0';
    signal sig_prim2sec_rst_reg1_n_cdc_to     : std_logic := '0';
    signal sig_prim2sec_rst_reg2_n     : std_logic := '0';
    
                               
                               
  begin --(architecture implementation)
  
  
    -- Assign outputs
   
    cmd_stat_rst_user <=   not(sig_cmd_stat_rst_user_n);
    
    cmd_stat_rst_int  <=   not(sig_cmd_stat_rst_int_reg_n) or
                           sig_syncd_sec_rst;
    
    mmap_rst          <=   not(sig_mmap_rst_reg_n) or
                           sig_syncd_sec_rst;
    
    stream_rst        <=   not(sig_stream_rst_reg_n) or
                           sig_syncd_sec_rst;
    
    
    
    
    
    
    
    -- Internal logic Implmentation
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_SYNC_CMDSTAT_RESET
    --
    -- If Generate Description:
    --  This IfGen assigns the reset for the 
    -- Synchronous Command/Status User interface case
    --
    ------------------------------------------------------------
    GEN_SYNC_CMDSTAT_RESET : if (C_STSCMD_IS_ASYNC = 0) generate
       
       begin
    
          sig_syncd_sec_rst       <= '0';
          
          sig_cmd_stat_rst_user_n <=  not(sig_cmd_stat_rst_user_reg_n_cdc_from);
                    
       end generate GEN_SYNC_CMDSTAT_RESET;
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_ASYNC_CMDSTAT_RESET
    --
    -- If Generate Description:
    --  This IfGen assigns the reset for the 
    -- Asynchronous Command/Status User interface case
    --
    ------------------------------------------------------------
    GEN_ASYNC_CMDSTAT_RESET : if (C_STSCMD_IS_ASYNC = 1) generate
    
      signal sig_sec_reset_in_reg_n      : std_logic := '0';
      
      -- Secondary reset pulse stretcher
      signal sig_secondary_dly1          : std_logic := '0';
      signal sig_secondary_dly2          : std_logic := '0';
      signal sig_neg_edge_detect         : std_logic := '0';
      signal sig_sec2prim_reset          : std_logic := '0';
      signal sig_sec2prim_reset_reg      : std_logic := '0';
      signal sig_sec2prim_reset_reg2     : std_logic := '0';
      signal sig_sec2prim_rst_syncro1    : std_logic := '0';
      signal sig_sec2prim_rst_syncro2    : std_logic := '0';

      
                                 
      begin
         
        -- Generate the reset in the primary clock domain. Use the longer
        -- of the pulse stretched reset or the actual reset.
        sig_syncd_sec_rst <= sig_sec2prim_reset_reg2 or
                             sig_sec2prim_rst_syncro2;
  
        
        -- Check for falling edge of secondary_aresetn input
        sig_neg_edge_detect <=  '1'
          when (sig_sec_reset_in_reg_n = '1' and 
                secondary_aresetn      = '0')
          else '0';
          

        
         
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: IMP_PUSE_STRETCH_FLOPS
        --
        -- Process Description:
        -- This process implements a 3 clock wide pulse whenever the 
        -- secondary reset is asserted
        --
        -------------------------------------------------------------
        IMP_PUSE_STRETCH_FLOPS : process (secondary_awclk)
          begin
            if (secondary_awclk'event and secondary_awclk = '1') then
            
              If (sig_secondary_dly2 = '1') Then
                
                sig_secondary_dly1 <= '0' ;
                sig_secondary_dly2 <= '0' ;
            
              Elsif (sig_neg_edge_detect = '1') Then
                
                sig_secondary_dly1 <= '1';
            
              else

                sig_secondary_dly2 <= sig_secondary_dly1 ;
              
              End if;
            
            
            
            
            end if;       
          end process IMP_PUSE_STRETCH_FLOPS; 
        
        
        
   --  CDC add     
        
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: SYNC_NEG_EDGE
        --
        -- Process Description:
        --  First (source clock) stage synchronizer for CDC of 
        -- negative edge detection,
        --
        -------------------------------------------------------------
        SYNC_NEG_EDGE : process (secondary_awclk)
          begin
            if (secondary_awclk'event and secondary_awclk = '1') then
            
              sig_sec_neg_edge_plus_delay <= sig_neg_edge_detect or
                                             sig_secondary_dly1  or
                                             sig_secondary_dly2;
            
            end if;       
          end process SYNC_NEG_EDGE; 
        
  --      
         
         
         
          
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: SEC2PRIM_RST_SYNCRO
        --
        -- Process Description:
        --    This process registers the secondary reset input to 
        -- the primary clock domain.
        --
        -------------------------------------------------------------
        SEC2PRIM_RST_SYNCRO : process (primary_aclk)
           begin
             if (primary_aclk'event and primary_aclk = '1') then
             
               
               sig_sec2prim_reset_reg   <=  sig_sec_neg_edge_plus_delay ;
               
               sig_sec2prim_reset_reg2  <=  sig_sec2prim_reset_reg;
                            
             end if;       
           end process SEC2PRIM_RST_SYNCRO; 
        
       
       
 
 
 
 
 
 
   --  CDC add     
        
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: REG_SEC_RST
        --
        -- Process Description:
        --  First (source clock) stage synchronizer for CDC of 
        -- secondary reset input,
        --
        -------------------------------------------------------------
        REG_SEC_RST : process (secondary_awclk)
          begin
            if (secondary_awclk'event and secondary_awclk = '1') then
            
              sig_secondary_aresetn_reg <= secondary_aresetn;
            
            end if;       
          end process REG_SEC_RST; 
        
  --      
         
         
         
         
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: SEC2PRIM_RST_SYNCRO_2
        --
        -- Process Description:
        --    Second stage (destination) synchronizers for the secondary
        -- reset CDC to the primary clock.
        --
        -------------------------------------------------------------
        SEC2PRIM_RST_SYNCRO_2 : process (primary_aclk)
           begin
             if (primary_aclk'event and primary_aclk = '1') then
             
               
      -- CDC   sig_sec2prim_rst_syncro1  <= not(secondary_aresetn);
               sig_sec2prim_rst_syncro1  <= not(sig_secondary_aresetn_reg);
               sig_sec2prim_rst_syncro2  <= sig_sec2prim_rst_syncro1;
 
 
             end if;       
           end process SEC2PRIM_RST_SYNCRO_2; 
        
       
         
         
         -- Generate the Command and Status side reset
         sig_cmd_stat_rst_user_n <= sig_sec_reset_in_reg_n and
                                    sig_prim2sec_rst_reg2_n;
    -- CDC                          sig_cmd_stat_rst_user_reg_n_cdc_from;
   
   
    
    
         
         -------------------------------------------------------------
         -- Synchronous Process with Sync Reset
         --
         -- Label: REG_RESET_ASYNC
         --
         -- Process Description:
         --    This process registers the secondary reset input to 
         -- generate the Command/Status User interface reset.
         --
         -------------------------------------------------------------
         REG_RESET_ASYNC : process (secondary_awclk)
            begin
              if (secondary_awclk'event and secondary_awclk = '1') then
              
                 sig_sec_reset_in_reg_n <= secondary_aresetn;
              
              end if;       
            end process REG_RESET_ASYNC; 
         
   
   
   
   
   --  CDC add     
        
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: SYNC_PRIM2SEC_RST
        --
        -- Process Description:
        --  Second (destination clock) stage synchronizers for CDC of 
        -- primary reset input,
        --
        -------------------------------------------------------------
        SYNC_PRIM2SEC_RST : process (secondary_awclk)
          begin
            if (secondary_awclk'event and secondary_awclk = '1') then
            
              sig_prim2sec_rst_reg1_n_cdc_to <= sig_cmd_stat_rst_user_reg_n_cdc_from;
              sig_prim2sec_rst_reg2_n <= sig_prim2sec_rst_reg1_n_cdc_to;
            
            end if;       
          end process SYNC_PRIM2SEC_RST; 
        
  --      
         
         
        
        
      end generate GEN_ASYNC_CMDSTAT_RESET;
 
 
 
  
  
  
  
  
  
  
  
 
 
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: REG_CMDSTAT_PRIM_RESET
    --
    -- Process Description:
    --    This process registers the primary reset input to 
    -- generate the Command/Status User interface reset.
    --
    -------------------------------------------------------------
    REG_CMDSTAT_PRIM_RESET : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
         
            sig_cmd_stat_rst_user_reg_n_cdc_from <= primary_aresetn;
         
         end if;       
       end process REG_CMDSTAT_PRIM_RESET; 
    
   
   
 
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: REG_CMDSTAT_INT_RESET
    --
    -- Process Description:
    --    This process registers the primary reset input to 
    -- generate the Command/Status internal interface reset.
    --
    -------------------------------------------------------------
    REG_CMDSTAT_INT_RESET : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
         
            sig_cmd_stat_rst_int_reg_n <= primary_aresetn;
         
         end if;       
       end process REG_CMDSTAT_INT_RESET; 
    
   
   
 
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: REG_MMAP_RESET
    --
    -- Process Description:
    --    This process registers the primary reset input to 
    -- generate the Memory Map interface reset.
    --
    -------------------------------------------------------------
    REG_MMAP_RESET : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
         
            sig_mmap_rst_reg_n <= primary_aresetn;
         
         end if;       
       end process REG_MMAP_RESET; 
    
   
   
 
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: REG_STREAM_RESET
    --
    -- Process Description:
    --    This process registers the primary reset input to 
    -- generate the Stream interface reset.
    --
    -------------------------------------------------------------
    REG_STREAM_RESET : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
         
            sig_stream_rst_reg_n <= primary_aresetn;
         
         end if;       
       end process REG_STREAM_RESET; 
    
  
  
  
  
  
  
  
  
  -- Soft Shutdown logic ------------------------------------------------------
  
    
    
    sig_internal_reset  <= not(sig_cmd_stat_rst_int_reg_n) or
                           sig_syncd_sec_rst;
    
    
    flush_stop_request  <= sig_s_h_halt_reg;
    
    
    halt_cmplt          <= sig_halt_cmplt;
    
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: REG_HALT_REQ
    --
    -- Process Description:
    --  Implements a sample and hold flop for the halt request 
    -- input. Can only be cleared on a HW reset.
    --
    -------------------------------------------------------------
    REG_HALT_REQ : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (sig_internal_reset = '1') then
              
              sig_s_h_halt_reg <= '0';
            
            elsif (halt_req = '1') then
              
              sig_s_h_halt_reg <= '1';
            
            else
              null;  -- hold current state
            end if; 
         end if;       
       end process REG_HALT_REQ; 
     
     
  
  
  
  
  
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_HALT_CMPLT
    --
    -- Process Description:
    --  Implements a the flop for the halt complete status 
    -- output. Can only be cleared on a HW reset.
    --
    -------------------------------------------------------------
    IMP_HALT_CMPLT : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (sig_internal_reset = '1') then
              
              sig_halt_cmplt <= '0';
            
            elsif (data_cntlr_stopped = '1' and
                   addr_cntlr_stopped = '1' and
                   aux1_stopped       = '1' and
                   aux2_stopped       = '1') then
              
              sig_halt_cmplt <= '1';
            
            else
              null;  -- hold current state
            end if; 
         end if;       
       end process IMP_HALT_CMPLT; 
     
     
  
          
          
  
    
  end implementation;


-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:        axi_sg_sfifo_autord.vhd
-- Version:         initial
-- Description:     
--    This file contains the logic to generate a CoreGen call to create a
-- synchronous FIFO as part of the synthesis process of XST. This eliminates
-- the need for multiple fixed netlists for various sizes and widths of FIFOs. 
-- 
--                  
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-- Structure:   
--                      -- axi_sg_sfifo_autord.vhd
--                          |
--                          |--- sync_fifo_fg (FIFO Generator wrapper) 
--
-------------------------------------------------------------------------------

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

library lib_fifo_v1_0_8;
use lib_fifo_v1_0_8.sync_fifo_fg;


-------------------------------------------------------------------------------

entity axi_sg_sfifo_autord is
  generic (
     C_DWIDTH                : integer := 32;
       -- Sets the width of the FIFO Data
       
     C_DEPTH                 : integer := 128;
       -- Sets the depth of the FIFO
       
     C_DATA_CNT_WIDTH        : integer := 8;
       -- Sets the width of the FIFO Data Count output
       
     C_NEED_ALMOST_EMPTY     : Integer range 0 to 1 := 0;
       -- Indicates the need for an almost empty flag from the internal FIFO
     
     C_NEED_ALMOST_FULL      : Integer range 0 to 1 := 0;
       -- Indicates the need for an almost full flag from the internal FIFO
     
     C_USE_BLKMEM            : Integer range 0 to 1 := 1;
       -- Sets the type of memory to use for the FIFO
       -- 0 = Distributed Logic
       -- 1 = Block Ram
       
     C_FAMILY                : String  := "virtex7"
       -- Specifies the target FPGA Family
       
    );
  port (
    
    -- FIFO Inputs ------------------------------------------------------------------
     SFIFO_Sinit             : In  std_logic;                                      --
     SFIFO_Clk               : In  std_logic;                                      --
     SFIFO_Wr_en             : In  std_logic;                                      --
     SFIFO_Din               : In  std_logic_vector(C_DWIDTH-1 downto 0);          --
     SFIFO_Rd_en             : In  std_logic;                                      --
     SFIFO_Clr_Rd_Data_Valid : In  std_logic;                                      --
     --------------------------------------------------------------------------------
     
    -- FIFO Outputs -----------------------------------------------------------------
     SFIFO_DValid            : Out std_logic;                                      --
     SFIFO_Dout              : Out std_logic_vector(C_DWIDTH-1 downto 0);          --
     SFIFO_Full              : Out std_logic;                                      --
     SFIFO_Empty             : Out std_logic;                                      --
     SFIFO_Almost_full       : Out std_logic;                                      --
     SFIFO_Almost_empty      : Out std_logic;                                      --
     SFIFO_Rd_count          : Out std_logic_vector(C_DATA_CNT_WIDTH-1 downto 0);  --
     SFIFO_Rd_count_minus1   : Out std_logic_vector(C_DATA_CNT_WIDTH-1 downto 0);  --
     SFIFO_Wr_count          : Out std_logic_vector(C_DATA_CNT_WIDTH-1 downto 0);  --
     SFIFO_Rd_ack            : Out std_logic                                       --
     --------------------------------------------------------------------------------
     
    );
end entity axi_sg_sfifo_autord;

-----------------------------------------------------------------------------
-- Architecture section
-----------------------------------------------------------------------------

architecture imp of axi_sg_sfifo_autord is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of imp : architecture is "yes";



-- Constant declarations

   -- none
 
-- Signal declarations

   signal write_data_lil_end         : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');
   signal read_data_lil_end          : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');
   signal raw_data_cnt_lil_end       : std_logic_vector(C_DATA_CNT_WIDTH-1 downto 0) := (others => '0');
   signal raw_data_count_int         : natural := 0;
   signal raw_data_count_corr        : std_logic_vector(C_DATA_CNT_WIDTH-1 downto 0) := (others => '0');
   signal raw_data_count_corr_minus1 : std_logic_vector(C_DATA_CNT_WIDTH-1 downto 0) := (others => '0');
   Signal corrected_empty            : std_logic := '0';
   Signal corrected_almost_empty     : std_logic := '0';
   Signal sig_SFIFO_empty            : std_logic := '0';
  
   -- backend fifo read ack sample and hold
   Signal sig_rddata_valid           : std_logic := '0';
   Signal hold_ff_q                  : std_logic := '0';
   Signal ored_ack_ff_reset          : std_logic := '0';
   Signal autoread                   : std_logic := '0';
   Signal sig_sfifo_rdack            : std_logic := '0';
   Signal fifo_read_enable           : std_logic := '0';
   

 
begin  

 -- Bit ordering translations
       
    write_data_lil_end    <=  SFIFO_Din;  -- translate from Big Endian to little
                                          -- endian.
       
    SFIFO_Dout            <= read_data_lil_end;  -- translate from Little Endian to 
                                                 -- Big endian.                            
 
 
 -- Other port usages and assignments
    SFIFO_Rd_ack          <= sig_sfifo_rdack; 
 
    SFIFO_Almost_empty    <= corrected_almost_empty;
 
    SFIFO_Empty           <= corrected_empty;
 
    SFIFO_Wr_count        <= raw_data_cnt_lil_end;   
                                                    
    
    SFIFO_Rd_count        <= raw_data_count_corr;    
                                                                    
   
    SFIFO_Rd_count_minus1 <= raw_data_count_corr_minus1;    
        
        
        
    SFIFO_DValid          <= sig_rddata_valid; -- Output data valid indicator
   
    fifo_read_enable      <= SFIFO_Rd_en; -- or autoread;
       
    ------------------------------------------------------------
    -- Instance: I_SYNC_FIFOGEN_FIFO 
    --
    -- Description:
    --  Instance for the synchronous fifo from proc common.   
    --
    ------------------------------------------------------------
    I_SYNC_FIFOGEN_FIFO : entity lib_fifo_v1_0_8.sync_fifo_fg 
      generic map(
         C_FAMILY             =>  C_FAMILY,        -- requred for FIFO Gen       
         C_DCOUNT_WIDTH       =>  C_DATA_CNT_WIDTH,     
         C_ENABLE_RLOCS       =>  0,                    
         C_HAS_DCOUNT         =>  1,                    
         C_HAS_RD_ACK         =>  1,                    
         C_HAS_RD_ERR         =>  0,                    
         C_HAS_WR_ACK         =>  1,                    
         C_HAS_WR_ERR         =>  0,                    
         C_MEMORY_TYPE        =>  C_USE_BLKMEM,         
         C_PORTS_DIFFER       =>  0,                    
         C_RD_ACK_LOW         =>  0,                    
         C_READ_DATA_WIDTH    =>  C_DWIDTH,             
         C_READ_DEPTH         =>  C_DEPTH,              
         C_RD_ERR_LOW         =>  0,                    
         C_WR_ACK_LOW         =>  0,                    
         C_WR_ERR_LOW         =>  0,                    
         C_WRITE_DATA_WIDTH   =>  C_DWIDTH,             
         C_WRITE_DEPTH        =>  C_DEPTH,
         C_PRELOAD_REGS       =>  1, -- 1 = first word fall through
         C_PRELOAD_LATENCY    =>  0, -- 0 = first word fall through
         C_USE_EMBEDDED_REG   =>  1  -- 0 ;
         )
      port map(  
         Clk                  =>  SFIFO_Clk,            
         Sinit                =>  SFIFO_Sinit,          
         Din                  =>  write_data_lil_end,   
         Wr_en                =>  SFIFO_Wr_en,          
         Rd_en                =>  fifo_read_enable,     
         Dout                 =>  read_data_lil_end,    
         Almost_full          =>  open,
         Full                 =>  SFIFO_Full,           
         Empty                =>  sig_SFIFO_empty,      
         Rd_ack               =>  sig_sfifo_rdack,      
         Wr_ack               =>  open,                 
         Rd_err               =>  open,                 
         Wr_err               =>  open,                 
         Data_count           =>  raw_data_cnt_lil_end  
        );  
    
    
    
         
    
   -------------------------------------------------------------------------------




    
                                 
   -------------------------------------------------------------------------------
   -- Read Ack assert & hold logic Needed because....
   -------------------------------------------------------------------------------
   --     1) The CoreGen Sync FIFO has to be read once to get valid
   --        data to the read data port. 
   --     2) The Read ack from the fifo is only asserted for 1 clock.
   --     3) A signal is needed that indicates valid data is at the read
   --        port of the FIFO and has not yet been used. This signal needs
   --        to be held until the next read operation occurs or a clear
   --        signal is received.
      
    
    ored_ack_ff_reset  <=  fifo_read_enable or 
                           SFIFO_Sinit or
                           SFIFO_Clr_Rd_Data_Valid;
    
    sig_rddata_valid   <=  hold_ff_q or 
                           sig_sfifo_rdack;
 
 
   
            
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_ACK_HOLD_FLOP
    --
    -- Process Description:
    --  Flop for registering the hold flag
    --
    -------------------------------------------------------------
    IMP_ACK_HOLD_FLOP : process (SFIFO_Clk)
       begin
         if (SFIFO_Clk'event and SFIFO_Clk = '1') then
           if (ored_ack_ff_reset = '1') then
             hold_ff_q  <= '0';
           else
             hold_ff_q  <= sig_rddata_valid;
           end if; 
         end if;       
       end process IMP_ACK_HOLD_FLOP; 
    
    
    
    -- generate auto-read enable. This keeps fresh data at the output
    -- of the FIFO whenever it is available.
    autoread <= '1'                     -- create a read strobe when the 
      when (sig_rddata_valid = '0' and  -- output data is NOT valid
            sig_SFIFO_empty = '0')      -- and the FIFO is not empty
      Else '0';
      
    
    raw_data_count_int <=  CONV_INTEGER(raw_data_cnt_lil_end);
    
    
 
 
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: INCLUDE_ALMOST_EMPTY
    --
    -- If Generate Description:
    --  This IFGen corrects the FIFO Read Count output for the
    --  auto read function and includes the generation of the
    --  Almost_Empty flag.
    --
    ------------------------------------------------------------
    INCLUDE_ALMOST_EMPTY : if (C_NEED_ALMOST_EMPTY = 1) generate
    
       -- local signals
       
          Signal raw_data_count_int_corr        : integer := 0;
          Signal raw_data_count_int_corr_minus1 : integer := 0;
       
       begin
         
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: CORRECT_RD_CNT_IAE
         --
         -- Process Description:
         --  This process corrects the FIFO Read Count output for the
         --  auto read function and includes the generation of the
         --  Almost_Empty flag.
         --
         -------------------------------------------------------------
         CORRECT_RD_CNT_IAE : process (sig_rddata_valid,
                                       sig_SFIFO_empty,
                                       raw_data_count_int)
            begin
         
               
               if (sig_rddata_valid = '0') then

                  raw_data_count_int_corr        <= 0;
                  raw_data_count_int_corr_minus1 <= 0;
                  corrected_empty                <= '1';
                  corrected_almost_empty         <= '0';
                  
               elsif (sig_SFIFO_empty = '1') then   -- rddata valid and fifo empty
                  
                  raw_data_count_int_corr        <= 1;
                  raw_data_count_int_corr_minus1 <= 0;
                  corrected_empty                <= '0';
                  corrected_almost_empty         <= '1';
               
               Elsif (raw_data_count_int = 1) Then  -- rddata valid and fifo almost empty
                  
                  raw_data_count_int_corr        <= 2;
                  raw_data_count_int_corr_minus1 <= 1;
                  corrected_empty                <= '0';
                  corrected_almost_empty         <= '0';
               
               else                                 -- rddata valid and modify rd count from FIFO 
                  
                  raw_data_count_int_corr        <= raw_data_count_int+1;
                  raw_data_count_int_corr_minus1 <= raw_data_count_int;
                  corrected_empty                <= '0';
                  corrected_almost_empty         <= '0';
               
               end if;
         
            end process CORRECT_RD_CNT_IAE; 
      
    
            raw_data_count_corr <= CONV_STD_LOGIC_VECTOR(raw_data_count_int_corr,
                                                         C_DATA_CNT_WIDTH);
        
            raw_data_count_corr_minus1 <= CONV_STD_LOGIC_VECTOR(raw_data_count_int_corr_minus1,
                                                                 C_DATA_CNT_WIDTH);      
             
       end generate INCLUDE_ALMOST_EMPTY;
 
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: OMIT_ALMOST_EMPTY
    --
    -- If Generate Description:
    --    This process corrects the FIFO Read Count output for the
    -- auto read function and omits the generation of the
    -- Almost_Empty flag.
    --
    ------------------------------------------------------------
    OMIT_ALMOST_EMPTY : if (C_NEED_ALMOST_EMPTY = 0) generate
    
       -- local signals
       
          Signal raw_data_count_int_corr : integer := 0;
       
       begin
    
          corrected_almost_empty  <= '0'; -- always low
         
         
          -------------------------------------------------------------
          -- Combinational Process
          --
          -- Label: CORRECT_RD_CNT
          --
          -- Process Description:
          --    This process corrects the FIFO Read Count output for the
          -- auto read function and omits the generation of the
          -- Almost_Empty flag.
          --
          -------------------------------------------------------------
          CORRECT_RD_CNT : process (sig_rddata_valid,
                                    sig_SFIFO_empty,
                                    raw_data_count_int)
             begin
          
              
                if (sig_rddata_valid = '0') then

                   raw_data_count_int_corr <= 0;
                   corrected_empty         <= '1';
                   
                elsif (sig_SFIFO_empty = '1') then   -- rddata valid and fifo empty
                   
                   raw_data_count_int_corr <= 1;
                   corrected_empty         <= '0';
                
                Elsif (raw_data_count_int = 1) Then  -- rddata valid and fifo almost empty
                   
                   raw_data_count_int_corr <= 2;
                   corrected_empty         <= '0';
                
                else                                 -- rddata valid and modify rd count from FIFO 
                   
                   raw_data_count_int_corr <= raw_data_count_int+1;
                   corrected_empty         <= '0';
                
                end if;
          
             end process CORRECT_RD_CNT; 
       
            
             raw_data_count_corr <= CONV_STD_LOGIC_VECTOR(raw_data_count_int_corr,
                                                          C_DATA_CNT_WIDTH);
        
 
         
       end generate OMIT_ALMOST_EMPTY;
  
  
  
       
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: INCLUDE_ALMOST_FULL
    --
    -- If Generate Description:
    --  This IfGen Includes the generation of the Amost_Full flag.
    --
    --
    ------------------------------------------------------------
    INCLUDE_ALMOST_FULL : if (C_NEED_ALMOST_FULL = 1) generate
    
       -- Local Constants
          
         Constant ALMOST_FULL_VALUE : integer := 2**(C_DATA_CNT_WIDTH-1)-1;
       
       begin
    
          SFIFO_Almost_full <= '1'
             When raw_data_count_int = ALMOST_FULL_VALUE
             Else '0';
                
                
       end generate INCLUDE_ALMOST_FULL;

   
   
   
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: OMIT_ALMOST_FULL
    --
    -- If Generate Description:
    --  This IfGen Omits the generation of the Amost_Full flag.
    --
    --
    ------------------------------------------------------------
    OMIT_ALMOST_FULL : if (C_NEED_ALMOST_FULL = 0) generate
    
       begin
    
           SFIFO_Almost_full <= '0';  -- always low   
                
       end generate OMIT_ALMOST_FULL;



end imp;


-- *************************************************************************
--
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:        axi_sg_afifo_autord.vhd
-- Version:         initial
-- Description:
--    This file contains the logic to generate a CoreGen call to create a
-- asynchronous FIFO as part of the synthesis process of XST. This eliminates
-- the need for multiple fixed netlists for various sizes and widths of FIFOs.
--
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

library lib_fifo_v1_0_8;
use lib_fifo_v1_0_8.async_fifo_fg;

-----------------------------------------------------------------------------
-- Entity section
-----------------------------------------------------------------------------

entity axi_sg_afifo_autord is
  generic (
     C_DWIDTH        : integer := 32;
     C_DEPTH         : integer := 16;
     C_CNT_WIDTH     : Integer := 5;
     C_USE_BLKMEM    : Integer := 0 ;
     C_USE_AUTORD    : Integer := 1;
     C_FAMILY        : String  := "virtex7"
    );
  port (
    -- Inputs
     AFIFO_Ainit                : In  std_logic;                                 --
     AFIFO_Wr_clk               : In  std_logic;                                 --
     AFIFO_Wr_en                : In  std_logic;                                 --
     AFIFO_Din                  : In  std_logic_vector(C_DWIDTH-1 downto 0);     --
     AFIFO_Rd_clk               : In  std_logic;                                 --
     AFIFO_Rd_en                : In  std_logic;                                 --
     AFIFO_Clr_Rd_Data_Valid    : In  std_logic;                                 --
                                                                                 --
    -- Outputs                                                                   --
     AFIFO_DValid               : Out std_logic;                                 --
     AFIFO_Dout                 : Out std_logic_vector(C_DWIDTH-1 downto 0);     --
     AFIFO_Full                 : Out std_logic;                                 --
     AFIFO_Empty                : Out std_logic;                                 --
     AFIFO_Almost_full          : Out std_logic;                                 --
     AFIFO_Almost_empty         : Out std_logic;                                 --
     AFIFO_Wr_count             : Out std_logic_vector(C_CNT_WIDTH-1 downto 0);  --
     AFIFO_Rd_count             : Out std_logic_vector(C_CNT_WIDTH-1 downto 0);  --
     AFIFO_Corr_Rd_count        : Out std_logic_vector(C_CNT_WIDTH downto 0);    --
     AFIFO_Corr_Rd_count_minus1 : Out std_logic_vector(C_CNT_WIDTH downto 0);    --
     AFIFO_Rd_ack               : Out std_logic                                  --
    );
end entity axi_sg_afifo_autord;


-----------------------------------------------------------------------------
-- Architecture section
-----------------------------------------------------------------------------

architecture imp of axi_sg_afifo_autord is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of imp : architecture is "yes";

-- Constant declarations


-- Signal declarations
   signal write_data_lil_end       : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');
   signal read_data_lil_end        : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');

   signal wr_count_lil_end         : std_logic_vector(C_CNT_WIDTH-1 downto 0) := (others => '0');
   signal rd_count_lil_end         : std_logic_vector(C_CNT_WIDTH-1 downto 0) := (others => '0');
   signal rd_count_int             : integer range 0 to C_DEPTH+1 := 0;
   signal rd_count_int_corr        : integer range 0 to C_DEPTH+1 := 0;
   signal rd_count_int_corr_minus1 : integer range 0 to C_DEPTH+1 := 0;


   Signal corrected_empty          : std_logic := '0';
   Signal corrected_almost_empty   : std_logic := '0';
   Signal sig_afifo_empty          : std_logic := '0';
   Signal sig_afifo_almost_empty   : std_logic := '0';


 -- backend fifo read ack sample and hold
   Signal sig_rddata_valid         : std_logic := '0';
   Signal hold_ff_q                : std_logic := '0';
   Signal ored_ack_ff_reset        : std_logic := '0';
   Signal autoread                 : std_logic := '0';
   Signal sig_wrfifo_rdack         : std_logic := '0';
   Signal fifo_read_enable         : std_logic := '0';

   Signal first_write              : std_logic := '0';
   Signal first_read               : std_logic := '0';
   Signal first_read1              : std_logic := '0';


-- Component declarations



-----------------------------------------------------------------------------
-- Begin architecture
-----------------------------------------------------------------------------
begin

 -- Bit ordering translations

    write_data_lil_end   <=  AFIFO_Din;  -- translate from Big Endian to little
                                         -- endian.
    AFIFO_Rd_ack         <= sig_wrfifo_rdack;

    AFIFO_Dout           <= read_data_lil_end;  -- translate from Little Endian to
                                                -- Big endian.

    AFIFO_Almost_empty   <= corrected_almost_empty;
GEN_EMPTY : if (C_USE_AUTORD = 1) generate
begin
    AFIFO_Empty          <= corrected_empty;
end generate GEN_EMPTY;
GEN_EMPTY1 : if (C_USE_AUTORD = 0) generate
begin
    AFIFO_Empty          <= sig_afifo_empty;
end generate GEN_EMPTY1;

    AFIFO_Wr_count       <= wr_count_lil_end;

    AFIFO_Rd_count       <= rd_count_lil_end;

    AFIFO_Corr_Rd_count  <= CONV_STD_LOGIC_VECTOR(rd_count_int_corr,
                                                  C_CNT_WIDTH+1);

    AFIFO_Corr_Rd_count_minus1 <= CONV_STD_LOGIC_VECTOR(rd_count_int_corr_minus1,
                                                        C_CNT_WIDTH+1);

    AFIFO_DValid         <= sig_rddata_valid; -- Output data valid indicator


    fifo_read_enable     <= AFIFO_Rd_en or autoread;



   -------------------------------------------------------------------------------
   -- Instantiate the CoreGen FIFO
   --
   -- NOTE:
   -- This instance refers to a wrapper file that interm will use the
   -- CoreGen FIFO Generator Async FIFO utility.
   --
   -------------------------------------------------------------------------------
    I_ASYNC_FIFOGEN_FIFO : entity lib_fifo_v1_0_8.async_fifo_fg
       generic map (
--          C_ALLOW_2N_DEPTH      =>  1,
          C_ALLOW_2N_DEPTH      =>  0,
          C_FAMILY              =>  C_FAMILY,
          C_DATA_WIDTH          =>  C_DWIDTH,
          C_ENABLE_RLOCS        =>  0,
          C_FIFO_DEPTH          =>  C_DEPTH,
          C_HAS_ALMOST_EMPTY    =>  1,
          C_HAS_ALMOST_FULL     =>  1,
          C_HAS_RD_ACK          =>  1,
          C_HAS_RD_COUNT        =>  1,
          C_HAS_RD_ERR          =>  0,
          C_HAS_WR_ACK          =>  0,
          C_HAS_WR_COUNT        =>  1,
          C_HAS_WR_ERR          =>  0,
          C_RD_ACK_LOW          =>  0,
          C_RD_COUNT_WIDTH      =>  C_CNT_WIDTH,
          C_RD_ERR_LOW          =>  0,
          C_USE_BLOCKMEM        =>  C_USE_BLKMEM,
          C_WR_ACK_LOW          =>  0,
          C_WR_COUNT_WIDTH      =>  C_CNT_WIDTH,
          C_WR_ERR_LOW          =>  0
    --      C_USE_EMBEDDED_REG    =>  1, -- 0 ;
    --      C_PRELOAD_REGS        =>  0, -- 0 ;
    --      C_PRELOAD_LATENCY     =>  1  -- 1 ;
         )
      port Map (
         Din                 =>  write_data_lil_end,
         Wr_en               =>  AFIFO_Wr_en,
         Wr_clk              =>  AFIFO_Wr_clk,
         Rd_en               =>  fifo_read_enable,
         Rd_clk              =>  AFIFO_Rd_clk,
         Ainit               =>  AFIFO_Ainit,
         Dout                =>  read_data_lil_end,
         Full                =>  AFIFO_Full,
         Empty               =>  sig_afifo_empty,
         Almost_full         =>  AFIFO_Almost_full,
         Almost_empty        =>  sig_afifo_almost_empty,
         Wr_count            =>  wr_count_lil_end,
         Rd_count            =>  rd_count_lil_end,
         Rd_ack              =>  sig_wrfifo_rdack,
         Rd_err              =>  open,              -- Not used by axi_dma
         Wr_ack              =>  open,              -- Not used by axi_dma
         Wr_err              =>  open               -- Not used by axi_dma
        );


   ----------------------------------------------------------------------------
   -- Read Ack assert & hold logic (needed because:
   --     1) The Async FIFO has to be read once to get valid
   --        data to the read data port (data is discarded).
   --     2) The Read ack from the fifo is only asserted for 1 clock.
   --     3) A signal is needed that indicates valid data is at the read
   --        port of the FIFO and has not yet been read. This signal needs
   --        to be held until the next read operation occurs or a clear
   --        signal is received.


    ored_ack_ff_reset  <=  fifo_read_enable or
                           AFIFO_Ainit or
                           AFIFO_Clr_Rd_Data_Valid;

    sig_rddata_valid   <=  hold_ff_q or
                           sig_wrfifo_rdack;




    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_ACK_HOLD_FLOP
    --
    -- Process Description:
    --  Flop for registering the hold flag
    --
    -------------------------------------------------------------
    IMP_ACK_HOLD_FLOP : process (AFIFO_Rd_clk)
       begin
         if (AFIFO_Rd_clk'event and AFIFO_Rd_clk = '1') then
           if (ored_ack_ff_reset = '1') then
             hold_ff_q  <= '0';
           else
             hold_ff_q  <= sig_rddata_valid;
           end if;
         end if;
       end process IMP_ACK_HOLD_FLOP;



   --  I_ACK_HOLD_FF : FDRE
   --    port map(
   --      Q  =>  hold_ff_q,
   --      C  =>  AFIFO_Rd_clk,
   --      CE =>  '1',
   --      D  =>  sig_rddata_valid,
   --      R  =>  ored_ack_ff_reset
   --    );



  -- generate auto-read enable. This keeps fresh data at the output
  -- of the FIFO whenever it is available.

GEN_AUTORD1 : if C_USE_AUTORD = 1 generate
    autoread <= '1'                     -- create a read strobe when the
      when (sig_rddata_valid = '0' and  -- output data is NOT valid
            sig_afifo_empty = '0')      -- and the FIFO is not empty
      Else '0';
end generate GEN_AUTORD1;


GEN_AUTORD2 : if C_USE_AUTORD = 0 generate
    process (AFIFO_Wr_clk, AFIFO_Ainit)
    begin
           if (AFIFO_Ainit = '0') then
              first_write <= '0';
           elsif (AFIFO_Wr_clk'event and AFIFO_Wr_clk = '1') then
              if (AFIFO_Wr_en = '1') then
                 first_write <= '1';
              end if;
           end if; 
    end process;


    process (AFIFO_Rd_clk, AFIFO_Ainit)
    begin
           if (AFIFO_Ainit = '0') then
              first_read <= '0';
              first_read1 <= '0';
           elsif (AFIFO_Rd_clk'event and AFIFO_Rd_clk = '1') then
              if (sig_afifo_empty = '0') then
                 first_read <= first_write;
                 first_read1 <= first_read;
              end if;
           end if; 
    end process;
    autoread <= first_read xor first_read1;             
end generate GEN_AUTORD2;


    rd_count_int <=  CONV_INTEGER(rd_count_lil_end);


    -------------------------------------------------------------
    -- Combinational Process
    --
    -- Label: CORRECT_RD_CNT
    --
    -- Process Description:
    --  This process corrects the FIFO Read Count output for the
    -- auto read function.
    --
    -------------------------------------------------------------
    CORRECT_RD_CNT : process (sig_rddata_valid,
                              sig_afifo_empty,
                              sig_afifo_almost_empty,
                              rd_count_int)
       begin

          if (sig_rddata_valid = '0') then

             rd_count_int_corr        <= 0;
             rd_count_int_corr_minus1 <= 0;
             corrected_empty          <= '1';
             corrected_almost_empty   <= '0';

          elsif (sig_afifo_empty = '1') then         -- rddata valid and fifo empty

             rd_count_int_corr        <= 1;
             rd_count_int_corr_minus1 <= 0;
             corrected_empty          <= '0';
             corrected_almost_empty   <= '1';

          Elsif (sig_afifo_almost_empty = '1') Then  -- rddata valid and fifo almost empty

             rd_count_int_corr        <= 2;
             rd_count_int_corr_minus1 <= 1;
             corrected_empty          <= '0';
             corrected_almost_empty   <= '0';

          else                                   -- rddata valid and modify rd count from FIFO

             rd_count_int_corr        <= rd_count_int+1;
             rd_count_int_corr_minus1 <= rd_count_int;
             corrected_empty          <= '0';
             corrected_almost_empty   <= '0';

          end if;

       end process CORRECT_RD_CNT;



end imp;



-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:        axi_sg_fifo.vhd
-- Version:         initial
-- Description:     
--    This file is a wrapper file for the Synchronous FIFO used by the DataMover. 
-- 
--                  
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;



library lib_pkg_v1_0_2;
use lib_pkg_v1_0_2.lib_pkg.all;
use lib_pkg_v1_0_2.lib_pkg.clog2;
library lib_srl_fifo_v1_0_2;
use lib_srl_fifo_v1_0_2.srl_fifo_f;


library axi_sg_v4_1_6;
use axi_sg_v4_1_6.axi_sg_sfifo_autord;
use axi_sg_v4_1_6.axi_sg_afifo_autord;


-------------------------------------------------------------------------------

entity axi_sg_fifo is
  generic (
     C_DWIDTH            : integer := 32  ;
       -- Bit width of the FIFO
       
     C_DEPTH             : integer := 4   ;
       -- Depth of the fifo in fifo width words
     
     C_IS_ASYNC          : Integer range 0 to 1 := 0 ;
       -- 0 = Syncronous FIFO
       -- 1 = Asynchronous (2 clock) FIFO
     
     C_PRIM_TYPE         : Integer range 0 to 2 := 2 ;
       -- 0 = Register
       -- 1 = Block Memory
       -- 2 = SRL
     
     C_FAMILY            : String  := "virtex7"
       -- Specifies the Target FPGA device family
     
    );
  port (
     
     
     
     -- Write Clock and reset -----------------
     fifo_wr_reset        : In  std_logic;   --
     fifo_wr_clk          : In  std_logic;   --
     ------------------------------------------
     
     -- Write Side ------------------------------------------------------
     fifo_wr_tvalid       : In  std_logic;                             --
     fifo_wr_tready       : Out std_logic;                             --
     fifo_wr_tdata        : In  std_logic_vector(C_DWIDTH-1 downto 0); --
     fifo_wr_full         : Out std_logic;                             --
     --------------------------------------------------------------------
    
    
     -- Read Clock and reset -----------------------------------------------
     fifo_async_rd_reset  : In  std_logic; -- only used if C_IS_ASYNC = 1 --  
     fifo_async_rd_clk    : In  std_logic; -- only used if C_IS_ASYNC = 1 --
     -----------------------------------------------------------------------
     
     -- Read Side --------------------------------------------------------
     fifo_rd_tvalid       : Out std_logic;                              --
     fifo_rd_tready       : In  std_logic;                              --
     fifo_rd_tdata        : Out std_logic_vector(C_DWIDTH-1 downto 0);  --
     fifo_rd_empty        : Out std_logic                               --
     ---------------------------------------------------------------------
    
    );
end entity axi_sg_fifo;

-----------------------------------------------------------------------------
-- Architecture section
-----------------------------------------------------------------------------

architecture imp of axi_sg_fifo is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of imp : architecture is "yes";


 -- function Declarations
  
  -------------------------------------------------------------------
  -- Function
  --
  -- Function Name: funct_get_prim_type
  --
  -- Function Description:
  --  Sorts out the FIFO Primitive type selection based on fifo
  -- depth and original primitive choice.
  --
  -------------------------------------------------------------------
  -- coverage off
  function funct_get_prim_type (depth            : integer;
                                input_prim_type  : integer) return integer is
  
    Variable temp_prim_type : Integer := 0;
  
  begin
  
    If (depth > 64) Then
    
      temp_prim_type := 1;  -- use BRAM
   
    Elsif (depth <= 64 and 
           input_prim_type = 0) Then
    
      temp_prim_type := 0;  -- use regiaters
    
    else

      temp_prim_type := 1;  -- use BRAM
    
    End if;
    
    
    Return (temp_prim_type);
    
  end function funct_get_prim_type;
  
 -- coverage on
  
  
-- Signal declarations
  
  Signal sig_init_reg          : std_logic := '0';
  Signal sig_init_reg2         : std_logic := '0';
  Signal sig_init_done         : std_logic := '0';
  signal sig_inhibit_rdy_n     : std_logic := '0';
 
   
   
 
-----------------------------------------------------------------------------
-- Begin architecture
-----------------------------------------------------------------------------
begin  


  -------------------------------------------------------------
  -- Synchronous Process with Sync Reset
  --
  -- Label: IMP_INIT_REG
  --
  -- Process Description:
  --  Registers the reset signal input.
  --
  -------------------------------------------------------------
  IMP_INIT_REG : process (fifo_wr_clk)
     begin
       if (fifo_wr_clk'event and fifo_wr_clk = '1') then
          if (fifo_wr_reset = '1') then
            sig_init_reg  <= '1';
            sig_init_reg2 <= '1';
          else
            sig_init_reg <= '0';
            sig_init_reg2 <= sig_init_reg;
          end if; 
       end if;       
     end process IMP_INIT_REG; 
  
  
  -------------------------------------------------------------
  -- Synchronous Process with Sync Reset
  --
  -- Label: IMP_INIT_DONE_REG
  --
  -- Process Description:
  -- Create a 1 clock wide init done pulse. 
  --
  -------------------------------------------------------------
  IMP_INIT_DONE_REG : process (fifo_wr_clk)
     begin
       if (fifo_wr_clk'event and fifo_wr_clk = '1') then
          if (fifo_wr_reset = '1' or
              sig_init_done = '1') then
            
            sig_init_done <= '0';
          
          Elsif (sig_init_reg  = '1' and
                 sig_init_reg2 = '1') Then
          
            sig_init_done <= '1';
          
          else
            null;  -- hold current state
          end if; 
       end if;       
     end process IMP_INIT_DONE_REG; 
  
  
  -------------------------------------------------------------
  -- Synchronous Process with Sync Reset
  --
  -- Label: IMP_RDY_INHIBIT_REG
  --
  -- Process Description:
  --  Implements a ready inhibit flop.
  --
  -------------------------------------------------------------
  IMP_RDY_INHIBIT_REG : process (fifo_wr_clk)
     begin
       if (fifo_wr_clk'event and fifo_wr_clk = '1') then
          if (fifo_wr_reset = '1') then
            
            sig_inhibit_rdy_n  <= '0';
          
          Elsif (sig_init_done = '1') Then
          
            sig_inhibit_rdy_n <= '1';
          
          else
            null;  -- hold current state
          end if; 
       end if;       
     end process IMP_RDY_INHIBIT_REG; 
  
  
  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: USE_SINGLE_REG
  --
  -- If Generate Description:
  --  Implements a 1 deep register FIFO (synchronous mode only)
  --
  --
  ------------------------------------------------------------
  USE_SINGLE_REG : if (C_IS_ASYNC  = 0 and 
                       C_DEPTH    <= 1) generate
  
     -- Local Constants
     
     -- local signals
     signal sig_data_in           : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');
     signal sig_regfifo_dout_reg  : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');
     signal sig_regfifo_full_reg  : std_logic := '0';
     signal sig_regfifo_empty_reg : std_logic := '0';
     signal sig_push_regfifo      : std_logic := '0';
     signal sig_pop_regfifo       : std_logic := '0';
     
     
  
     begin

       -- Internal signals
       
       -- Write signals
       fifo_wr_tready    <=  sig_regfifo_empty_reg;
       
       fifo_wr_full      <=  sig_regfifo_full_reg ;
       
       sig_push_regfifo  <=  fifo_wr_tvalid and
                             sig_regfifo_empty_reg;
       
       sig_data_in       <=  fifo_wr_tdata ; 

       
       -- Read signals
       fifo_rd_tdata     <=  sig_regfifo_dout_reg ;
       
       fifo_rd_tvalid    <=  sig_regfifo_full_reg ;
       
       fifo_rd_empty     <=  sig_regfifo_empty_reg;
       
       sig_pop_regfifo   <=  sig_regfifo_full_reg and
                             fifo_rd_tready;
       
       
       
       
       -------------------------------------------------------------
       -- Synchronous Process with Sync Reset
       --
       -- Label: IMP_REG_FIFO
       --
       -- Process Description:
       --    This process implements the data and full flag for the 
       -- register fifo.
       --
       -------------------------------------------------------------
       IMP_REG_FIFO : process (fifo_wr_clk)
          begin
            if (fifo_wr_clk'event and fifo_wr_clk = '1') then
               if (fifo_wr_reset    = '1' or
                   sig_pop_regfifo  = '1') then
                 
                 sig_regfifo_full_reg  <= '0';
                 
               elsif (sig_push_regfifo = '1') then
                 
                 sig_regfifo_full_reg  <= '1';
                 
               else
                 null;  -- don't change state
               end if; 
            end if;       
          end process IMP_REG_FIFO; 
       
       IMP_REG_FIFO1 : process (fifo_wr_clk)
          begin
            if (fifo_wr_clk'event and fifo_wr_clk = '1') then
               if (fifo_wr_reset    = '1') then
                 
                 
                 sig_regfifo_dout_reg  <= (others => '0');
                 
               elsif (sig_push_regfifo = '1') then
                 
                 sig_regfifo_dout_reg  <= sig_data_in;
                 
               else
                 null;  -- don't change state
               end if; 
            end if;       
          end process IMP_REG_FIFO1; 
      
       
       
       -------------------------------------------------------------
       -- Synchronous Process with Sync Reset
       --
       -- Label: IMP_REG_EMPTY_FLOP
       --
       -- Process Description:
       --    This process implements the empty flag for the 
       -- register fifo.
       --
       -------------------------------------------------------------
       IMP_REG_EMPTY_FLOP : process (fifo_wr_clk)
          begin
            if (fifo_wr_clk'event and fifo_wr_clk = '1') then
               if (fifo_wr_reset    = '1') then
                 
                 sig_regfifo_empty_reg <= '0'; -- since this is used for the ready (invertd)
                                               -- it can't be asserted during reset
                 
               elsif (sig_pop_regfifo  = '1' or
                      sig_init_done    = '1') then
                 
                 sig_regfifo_empty_reg <= '1';
                 
               elsif (sig_push_regfifo = '1') then
                 
                 sig_regfifo_empty_reg <= '0';
                 
               else
                 null;  -- don't change state
               end if; 
            end if;       
          end process IMP_REG_EMPTY_FLOP; 
       

  
     end generate USE_SINGLE_REG;
 
  
  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: USE_SRL_FIFO
  --
  -- If Generate Description:
  --  Generates a fifo implementation usinf SRL based FIFOa
  --
  --
  ------------------------------------------------------------
  USE_SRL_FIFO : if (C_IS_ASYNC  =  0 and
                     C_DEPTH    <= 64 and
                     C_DEPTH     >  1 and
                     C_PRIM_TYPE =  2 ) generate
                     
  
  
    -- Local Constants
    Constant  LOGIC_LOW         : std_logic := '0';
    Constant  NEED_ALMOST_EMPTY : Integer := 0;
    Constant  NEED_ALMOST_FULL  : Integer := 0;
    
    
    -- local signals

    signal sig_wr_full          : std_logic := '0';
    signal sig_wr_fifo          : std_logic := '0';
    signal sig_wr_ready         : std_logic := '0';
    signal sig_rd_fifo          : std_logic := '0';
    signal sig_rd_empty         : std_logic := '0';
    signal sig_rd_valid         : std_logic := '0';
    signal sig_fifo_rd_data     : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');
    signal sig_fifo_wr_data     : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');
      
    begin
  
       

      -- Write side signals
      fifo_wr_tready    <=  sig_wr_ready;
      
      fifo_wr_full      <=  sig_wr_full;
      
      sig_wr_ready      <=  not(sig_wr_full) and 
                            sig_inhibit_rdy_n;
 
      sig_wr_fifo       <=  fifo_wr_tvalid and 
                            sig_wr_ready;
      
      sig_fifo_wr_data  <=  fifo_wr_tdata;
      
      
      
      
      -- Read Side Signals
      fifo_rd_tvalid    <=  sig_rd_valid;
      
      sig_rd_valid      <=  not(sig_rd_empty);
      
      fifo_rd_tdata     <=  sig_fifo_rd_data ;
      
      fifo_rd_empty     <=  not(sig_rd_valid);
      
      sig_rd_fifo       <=  sig_rd_valid and
                            fifo_rd_tready;
       
      
       
      ------------------------------------------------------------
      -- Instance: I_SYNC_FIFO 
      --
      -- Description:
      -- Implement the synchronous FIFO using SRL FIFO elements    
      --
      ------------------------------------------------------------
       I_SYNC_FIFO : entity lib_srl_fifo_v1_0_2.srl_fifo_f
       generic map (

         C_DWIDTH            =>  C_DWIDTH   ,  
         C_DEPTH             =>  C_DEPTH    ,  
         C_FAMILY            =>  C_FAMILY      

         )
       port map (

         Clk           =>  fifo_wr_clk      ,  
         Reset         =>  fifo_wr_reset    ,  
         FIFO_Write    =>  sig_wr_fifo      ,  
         Data_In       =>  sig_fifo_wr_data ,  
         FIFO_Read     =>  sig_rd_fifo      ,  
         Data_Out      =>  sig_fifo_rd_data ,  
         FIFO_Empty    =>  sig_rd_empty     ,  
         FIFO_Full     =>  sig_wr_full      ,  
         Addr          =>  open                
     
         );

       
       
     end generate USE_SRL_FIFO;




 
 
  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: USE_SYNC_FIFO
  --
  -- If Generate Description:
  --  Instantiates a synchronous FIFO design for use in the 
  -- synchronous operating mode.
  --
  ------------------------------------------------------------
  USE_SYNC_FIFO : if (C_IS_ASYNC  =  0 and
                     (C_DEPTH     > 64  or
                     (C_DEPTH     >  1 and C_PRIM_TYPE < 2 ))) generate
  
    -- Local Constants
    Constant  LOGIC_LOW         : std_logic := '0';
    Constant  NEED_ALMOST_EMPTY : Integer   := 0;
    Constant  NEED_ALMOST_FULL  : Integer   := 0;
    Constant  DATA_CNT_WIDTH    : Integer   := clog2(C_DEPTH)+1;
    Constant  PRIM_TYPE         : Integer   := funct_get_prim_type(C_DEPTH, C_PRIM_TYPE);
    
    
    -- local signals
    signal sig_wr_full          : std_logic := '0';
    signal sig_wr_fifo          : std_logic := '0';
    signal sig_wr_ready         : std_logic := '0';
    signal sig_rd_fifo          : std_logic := '0';
    signal sig_rd_valid         : std_logic := '0';
    signal sig_fifo_rd_data     : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');
    signal sig_fifo_wr_data     : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');
    
    begin

      -- Write side signals
      fifo_wr_tready    <=  sig_wr_ready;
      
      fifo_wr_full      <=  sig_wr_full;
      
      sig_wr_ready      <=  not(sig_wr_full) and 
                            sig_inhibit_rdy_n;
 
      sig_wr_fifo       <=  fifo_wr_tvalid and 
                            sig_wr_ready;
      
      sig_fifo_wr_data  <=  fifo_wr_tdata;
      
      
      
      
      -- Read Side Signals
      fifo_rd_tvalid    <=  sig_rd_valid;
      
      fifo_rd_tdata     <=  sig_fifo_rd_data ;
      
      fifo_rd_empty     <=  not(sig_rd_valid);
      
      sig_rd_fifo       <=  sig_rd_valid and
                            fifo_rd_tready;
       
      
      
      
      ------------------------------------------------------------
      -- Instance: I_SYNC_FIFO 
      --
      -- Description:
      -- Implement the synchronous FIFO    
      --
      ------------------------------------------------------------
       I_SYNC_FIFO : entity axi_sg_v4_1_6.axi_sg_sfifo_autord
       generic map (

         C_DWIDTH                =>  C_DWIDTH          ,  
         C_DEPTH                 =>  C_DEPTH           ,  
         C_DATA_CNT_WIDTH        =>  DATA_CNT_WIDTH    ,  
         C_NEED_ALMOST_EMPTY     =>  NEED_ALMOST_EMPTY ,  
         C_NEED_ALMOST_FULL      =>  NEED_ALMOST_FULL  ,  
         C_USE_BLKMEM            =>  PRIM_TYPE         ,  
         C_FAMILY                =>  C_FAMILY             

         )
       port map (

        -- Inputs 
         SFIFO_Sinit             =>  fifo_wr_reset     ,  
         SFIFO_Clk               =>  fifo_wr_clk       ,  
         SFIFO_Wr_en             =>  sig_wr_fifo       ,  
         SFIFO_Din               =>  fifo_wr_tdata     ,  
         SFIFO_Rd_en             =>  sig_rd_fifo       ,  
         SFIFO_Clr_Rd_Data_Valid =>  LOGIC_LOW         ,  
         
        -- Outputs
         SFIFO_DValid            =>  sig_rd_valid      ,  
         SFIFO_Dout              =>  sig_fifo_rd_data  ,  
         SFIFO_Full              =>  sig_wr_full       ,  
         SFIFO_Empty             =>  open              ,  
         SFIFO_Almost_full       =>  open              ,  
         SFIFO_Almost_empty      =>  open              ,  
         SFIFO_Rd_count          =>  open              ,  
         SFIFO_Rd_count_minus1   =>  open              ,  
         SFIFO_Wr_count          =>  open              ,  
         SFIFO_Rd_ack            =>  open                 

         );


      
 
     end generate USE_SYNC_FIFO;
 
 
 
 
 
 
  
  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: USE_ASYNC_FIFO
  --
  -- If Generate Description:
  --  Instantiates an asynchronous FIFO design for use in the 
  -- asynchronous operating mode.
  --
  ------------------------------------------------------------
  USE_ASYNC_FIFO : if (C_IS_ASYNC = 1) generate
  
    -- Local Constants
    Constant  LOGIC_LOW         : std_logic := '0';
    Constant  CNT_WIDTH         : Integer := clog2(C_DEPTH);
    
    
    -- local signals

    signal sig_async_wr_full       : std_logic := '0';
    signal sig_async_wr_fifo       : std_logic := '0';
    signal sig_async_wr_ready      : std_logic := '0';
    signal sig_async_rd_fifo       : std_logic := '0';
    signal sig_async_rd_valid      : std_logic := '0';
    signal sig_afifo_rd_data       : std_logic_vector(C_DWIDTH-1 downto 0);
    signal sig_afifo_wr_data       : std_logic_vector(C_DWIDTH-1 downto 0);
    signal sig_fifo_ainit          : std_logic := '0';
    Signal sig_init_reg            : std_logic := '0';
    
    
    begin

      sig_fifo_ainit  <= fifo_async_rd_reset or fifo_wr_reset;
      
      

      -- Write side signals
      fifo_wr_tready      <=  sig_async_wr_ready;
      
      fifo_wr_full        <=  sig_async_wr_full;
      
      sig_async_wr_ready  <=  not(sig_async_wr_full) and
                              sig_inhibit_rdy_n;
 
      sig_async_wr_fifo   <=  fifo_wr_tvalid and 
                              sig_async_wr_ready;
      
      sig_afifo_wr_data   <=  fifo_wr_tdata;
      
      
      
      
      -- Read Side Signals
      fifo_rd_tvalid    <=  sig_async_rd_valid;
      
      fifo_rd_tdata     <=  sig_afifo_rd_data ;
      
      fifo_rd_empty     <=  not(sig_async_rd_valid);
      
      sig_async_rd_fifo <=  sig_async_rd_valid and
                            fifo_rd_tready;
       
      
 
       
       
       
      ------------------------------------------------------------
      -- Instance: I_ASYNC_FIFO 
      --
      -- Description:
      -- Implement the asynchronous FIFO    
      --
      ------------------------------------------------------------
       I_ASYNC_FIFO : entity axi_sg_v4_1_6.axi_sg_afifo_autord
       generic map (

         C_DWIDTH                   =>  C_DWIDTH          ,  
         C_DEPTH                    =>  C_DEPTH           ,  
         C_CNT_WIDTH                =>  CNT_WIDTH         ,  
         C_USE_BLKMEM               =>  C_PRIM_TYPE       ,  
         C_FAMILY                   =>  C_FAMILY             

         )
       port map (

        -- Inputs 
         AFIFO_Ainit                =>  sig_fifo_ainit    ,  
         AFIFO_Wr_clk               =>  fifo_wr_clk       ,  
         AFIFO_Wr_en                =>  sig_async_wr_fifo ,  
         AFIFO_Din                  =>  sig_afifo_wr_data ,  
         AFIFO_Rd_clk               =>  fifo_async_rd_clk ,  
         AFIFO_Rd_en                =>  sig_async_rd_fifo ,  
         AFIFO_Clr_Rd_Data_Valid    =>  LOGIC_LOW         ,  
         
        -- Outputs
         AFIFO_DValid               =>  sig_async_rd_valid,  
         AFIFO_Dout                 =>  sig_afifo_rd_data ,  
         AFIFO_Full                 =>  sig_async_wr_full ,  
         AFIFO_Empty                =>  open              ,  
         AFIFO_Almost_full          =>  open              ,  
         AFIFO_Almost_empty         =>  open              ,  
         AFIFO_Wr_count             =>  open              ,   
         AFIFO_Rd_count             =>  open              ,  
         AFIFO_Corr_Rd_count        =>  open              ,  
         AFIFO_Corr_Rd_count_minus1 =>  open              ,  
         AFIFO_Rd_ack               =>  open                 

         );


      
 
     end generate USE_ASYNC_FIFO;
 
 

end imp;


-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_sg_cmd_status.vhd
  --
  -- Description:     
  --    This file implements the DataMover Command and Status interfaces.                 
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  library axi_sg_v4_1_6;
  Use axi_sg_v4_1_6.axi_sg_fifo;
  
  -------------------------------------------------------------------------------
  
  entity axi_sg_cmd_status is
    generic (
      
      C_ADDR_WIDTH         : Integer range 32 to 64 := 32;
        -- Indictes the width of the DataMover Address bus
       
      C_INCLUDE_STSFIFO    : Integer range  0 to  1 :=  1;
        -- Indicates if a Stus FIFO is to be included or omitted
        -- 0 = Omit
        -- 1 = Include
       
      C_STSCMD_FIFO_DEPTH  : Integer range  1 to 16 :=  4;
        -- Sets the depth of the Command and Status FIFOs
       
      C_STSCMD_IS_ASYNC    : Integer range  0 to  1 :=  0;
        -- Indicates if the Command and Status Stream Channels are clocked with
        -- a different clock than the Main dataMover Clock
        -- 0 = Same Clock
        -- 1 = Different clocks
       
      C_CMD_WIDTH          : Integer                := 68;
        -- Sets the width of the input command
       
      C_STS_WIDTH          : Integer                :=  8;
        -- Sets the width of the output status
       
      C_FAMILY             : string                 := "virtex7"
        -- Sets the target FPGA family
      
      );
    port (
      
      -- Clock inputs ----------------------------------------------------
      primary_aclk           : in  std_logic;                           --
         -- Primary synchronization clock for the Master side           --
         -- interface and internal logic. It is also used               --
         -- for the User interface synchronization when                 --
         -- C_STSCMD_IS_ASYNC = 0.                                      --
                                                                        --
      secondary_awclk        : in  std_logic;                           --
         -- Clock used for the Command and Status User Interface        --
         --  when the User Command and Status interface is Async        --
         -- to the MMap interface. Async mode is set by the assigned    --
         -- value to C_STSCMD_IS_ASYNC = 1.                             --
      --------------------------------------------------------------------
     
     
      -- Reset inputs ----------------------------------------------------
      user_reset             : in  std_logic;                           --
        -- Reset used for the User Stream interface logic               --
                                                                        --
      internal_reset         : in  std_logic;                           --
        -- Reset used for the internal master interface logic           --
      --------------------------------------------------------------------
      
      
      -- User Command Stream Ports (AXI Stream) -------------------------------
      cmd_wvalid             : in  std_logic;                                --
      cmd_wready             : out std_logic;                                --
      cmd_wdata              : in  std_logic_vector(C_CMD_WIDTH-1 downto 0); --
      cache_data             : in  std_logic_vector(7 downto 0); --
      -------------------------------------------------------------------------
      
      -- User Status Stream Ports (AXI Stream) ------------------------------------
      sts_wvalid             : out std_logic;                                    --
      sts_wready             : in  std_logic;                                    --
      sts_wdata              : out std_logic_vector(C_STS_WIDTH-1 downto 0);     --
      sts_wstrb              : out std_logic_vector((C_STS_WIDTH/8)-1 downto 0); --
      sts_wlast              : out std_logic;                                    --
      -----------------------------------------------------------------------------
      
      
      -- Internal Command Out Interface -----------------------------------------------
      cmd2mstr_command       : Out std_logic_vector(C_CMD_WIDTH-1 downto 0);         --
         -- The next command value available from the Command FIFO/Register          --

      cache2mstr_command       : Out std_logic_vector(7 downto 0);         --
         -- The cache value available from the FIFO/Register          --

                                                                                     --
      mst2cmd_cmd_valid      : Out std_logic;                                        --
         -- Handshake bit indicating the Command FIFO/Register has at least 1 valid  --
         -- command entry                                                            --
                                                                                     --
      cmd2mstr_cmd_ready     : in  std_logic;                                        --
         -- Handshake bit indicating the Command Calculator is ready to accept       --
         -- another command                                                          --
      ---------------------------------------------------------------------------------
      
      
      -- Internal Status In Interface  -----------------------------------------------------
      mstr2stat_status       : in  std_logic_vector(C_STS_WIDTH-1 downto 0);              --
         -- The input for writing the status value to the Status FIFO/Register            --
                                                                                          --
      stat2mstr_status_ready : Out std_logic;                                             --
         -- Handshake bit indicating that the Status FIFO/Register is ready for transfer  --
                                                                                          --
      mst2stst_status_valid  : In  std_logic                                              --
         -- Handshake bit for writing the Status value into the Status FIFO/Register      --
      --------------------------------------------------------------------------------------
     
      );
  
  end entity axi_sg_cmd_status;
  
  
  architecture implementation of axi_sg_cmd_status is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    
    -- Function
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: get_fifo_prim_type
    --
    -- Function Description:
    --  Returns the fifo primitiver type to use for the given input
    -- conditions.
    --
    --  0 = Not used or allowed here
    --  1 = BRAM Primitives (Block Memory)
    --  2 = Distributed memory
    --
    -------------------------------------------------------------------
    function get_fifo_prim_type (is_async : integer;
                                 depth    : integer) return integer is
    
      Variable var_temp_prim_type : Integer := 1;
    
    begin
 
      -- coverage off
    
      if (is_async = 1) then   -- Async FIFOs always use Blk Mem (BRAM)
      
        var_temp_prim_type := 1;

      
      elsif (depth <= 64) then -- (use srls or distrubuted)
      
        var_temp_prim_type := 2; 
      
      
      else  -- depth is too big for SRLs so use Blk Memory (BRAM)
      
        var_temp_prim_type := 1;
      
      end if;
       -- coverage on
      
     Return (var_temp_prim_type);
      
    end function get_fifo_prim_type;
    
   
   
    
    
    -- Constants 
    
    Constant REGISTER_TYPE  : integer := 0; 
    Constant BRAM_TYPE      : integer := 1; 
    --Constant SRL_TYPE       : integer := 2; 
    --Constant FIFO_PRIM_TYPE : integer := SRL_TYPE;
    Constant FIFO_PRIM_TYPE : integer := get_fifo_prim_type(C_STSCMD_IS_ASYNC, 
                                                            C_STSCMD_FIFO_DEPTH);
    
    
    -- Signals
    
    signal sig_cmd_fifo_wr_clk  : std_logic := '0';
    signal sig_cmd_fifo_wr_rst  : std_logic := '0';
    signal sig_cmd_fifo_rd_clk  : std_logic := '0';
    signal sig_cmd_fifo_rd_rst  : std_logic := '0';
    signal sig_sts_fifo_wr_clk  : std_logic := '0';
    signal sig_sts_fifo_wr_rst  : std_logic := '0';
    signal sig_sts_fifo_rd_clk  : std_logic := '0';
    signal sig_sts_fifo_rd_rst  : std_logic := '0';
    signal sig_reset_mstr       : std_logic := '0';
    signal sig_reset_user       : std_logic := '0';
 
 
  
  
  begin --(architecture implementation)
  
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_SYNC_RESET
    --
    -- If Generate Description:
    --  This IfGen assigns the clock and reset signals for the 
    -- synchronous User interface case
    --
    ------------------------------------------------------------
    GEN_SYNC_RESET : if (C_STSCMD_IS_ASYNC = 0) generate
    
       begin
    
          sig_reset_mstr       <= internal_reset  ;
          sig_reset_user       <= internal_reset  ;
        
          sig_cmd_fifo_wr_clk   <=  primary_aclk  ;  
          sig_cmd_fifo_wr_rst   <=  sig_reset_user; 
          sig_cmd_fifo_rd_clk   <=  primary_aclk  ; 
          sig_cmd_fifo_rd_rst   <=  sig_reset_mstr; 
         
          sig_sts_fifo_wr_clk   <=  primary_aclk  ; 
          sig_sts_fifo_wr_rst   <=  sig_reset_mstr; 
          sig_sts_fifo_rd_clk   <=  primary_aclk  ; 
          sig_sts_fifo_rd_rst   <=  sig_reset_user; 
           
         
        
             
         
       end generate GEN_SYNC_RESET;
  
  
    
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_ASYNC_RESET
    --
    -- If Generate Description:
    --  This IfGen assigns the clock and reset signals for the 
    -- Asynchronous User interface case
    --
    ------------------------------------------------------------
    GEN_ASYNC_RESET : if (C_STSCMD_IS_ASYNC = 1) generate
    
       begin
    
         sig_reset_mstr        <= internal_reset  ;
         sig_reset_user        <= user_reset      ;
       
         sig_cmd_fifo_wr_clk   <=  secondary_awclk;  
         sig_cmd_fifo_wr_rst   <=  sig_reset_user ; 
         sig_cmd_fifo_rd_clk   <=  primary_aclk   ; 
         sig_cmd_fifo_rd_rst   <=  sig_reset_mstr ; 
        
         sig_sts_fifo_wr_clk   <=  primary_aclk   ; 
         sig_sts_fifo_wr_rst   <=  sig_reset_mstr ; 
         sig_sts_fifo_rd_clk   <=  secondary_awclk; 
         sig_sts_fifo_rd_rst   <=  sig_reset_user ; 
          
        
            
         
       end generate GEN_ASYNC_RESET;
  
  
  
  
       
    ------------------------------------------------------------
    -- Instance: I_CMD_FIFO 
    --
    -- Description:
    -- Instance for the Command FIFO
    -- The User Interface is the Write Side
    -- The Internal Interface is the Read side    
    --
    ------------------------------------------------------------
     I_CMD_FIFO : entity axi_sg_v4_1_6.axi_sg_fifo
     generic map (
   
       C_DWIDTH            =>  C_CMD_WIDTH          ,  
       C_DEPTH             =>  C_STSCMD_FIFO_DEPTH  ,  
       C_IS_ASYNC          =>  C_STSCMD_IS_ASYNC    ,  
       C_PRIM_TYPE         =>  FIFO_PRIM_TYPE       ,  
       C_FAMILY            =>  C_FAMILY                
      
       )
     port map (
       
       -- Write Clock and reset
       fifo_wr_reset        =>  sig_cmd_fifo_wr_rst ,  
       fifo_wr_clk          =>  sig_cmd_fifo_wr_clk ,  
       
       -- Write Side
       fifo_wr_tvalid       =>  cmd_wvalid          ,  
       fifo_wr_tready       =>  cmd_wready          ,  
       fifo_wr_tdata        =>  cmd_wdata           ,  
       fifo_wr_full         =>  open                ,  
      
      
       -- Read Clock and reset
       fifo_async_rd_reset  =>  sig_cmd_fifo_rd_rst ,     
       fifo_async_rd_clk    =>  sig_cmd_fifo_rd_clk ,   
       
       -- Read Side
       fifo_rd_tvalid       =>  mst2cmd_cmd_valid   ,  
       fifo_rd_tready       =>  cmd2mstr_cmd_ready  ,  
       fifo_rd_tdata        =>  cmd2mstr_command    ,  
       fifo_rd_empty        =>  open                   
      
       );
   
     I_CACHE_FIFO : entity axi_sg_v4_1_6.axi_sg_fifo
     generic map (
   
       C_DWIDTH            =>  8          ,  
       C_DEPTH             =>  C_STSCMD_FIFO_DEPTH  ,  
       C_IS_ASYNC          =>  C_STSCMD_IS_ASYNC    ,  
       C_PRIM_TYPE         =>  FIFO_PRIM_TYPE       ,  
       C_FAMILY            =>  C_FAMILY                
      
       )
     port map (
       
       -- Write Clock and reset
       fifo_wr_reset        =>  sig_cmd_fifo_wr_rst ,  
       fifo_wr_clk          =>  sig_cmd_fifo_wr_clk ,  
       
       -- Write Side
       fifo_wr_tvalid       =>  cmd_wvalid          ,  
       fifo_wr_tready       =>  open ,--cmd_wready          ,  
       fifo_wr_tdata        =>  cache_data           ,  
       fifo_wr_full         =>  open                ,  
      
      
       -- Read Clock and reset
       fifo_async_rd_reset  =>  sig_cmd_fifo_rd_rst ,     
       fifo_async_rd_clk    =>  sig_cmd_fifo_rd_clk ,   
       
       -- Read Side
       fifo_rd_tvalid       =>  open ,--mst2cmd_cmd_valid   ,  
       fifo_rd_tready       =>  cmd2mstr_cmd_ready  ,  
       fifo_rd_tdata        =>  cache2mstr_command  ,  
       fifo_rd_empty        =>  open                   
      
       );
   
    
    
    
    
    
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_INCLUDE_STATUS_FIFO
    --
    -- If Generate Description:
    --  Instantiates a Status FIFO
    --
    --
    ------------------------------------------------------------
    GEN_INCLUDE_STATUS_FIFO : if (C_INCLUDE_STSFIFO = 1) generate
    
       begin
 
  
         -- Set constant outputs for Status Interface
         sts_wstrb             <=  (others => '1');    
         sts_wlast             <=  '1';                
         
         
       
         ------------------------------------------------------------
         -- Instance: I_STS_FIFO 
         --
         -- Description:
         -- Instance for the Status FIFO
         -- The Internal Interface is the Write Side
         -- The User Interface is the Read side    
         --
         ------------------------------------------------------------
         I_STS_FIFO : entity axi_sg_v4_1_6.axi_sg_fifo
         generic map (
       
           C_DWIDTH            =>  C_STS_WIDTH            ,  
           C_DEPTH             =>  C_STSCMD_FIFO_DEPTH    ,  
           C_IS_ASYNC          =>  C_STSCMD_IS_ASYNC      ,  
           C_PRIM_TYPE         =>  FIFO_PRIM_TYPE         ,  
           C_FAMILY            =>  C_FAMILY                  
          
           )
         port map (
           
           -- Write Clock and reset
           fifo_wr_reset        =>  sig_sts_fifo_wr_rst   ,  
           fifo_wr_clk          =>  sig_sts_fifo_wr_clk   ,  
           
           -- Write Side
           fifo_wr_tvalid       =>  mst2stst_status_valid ,  
           fifo_wr_tready       =>  stat2mstr_status_ready,  
           fifo_wr_tdata        =>  mstr2stat_status      ,  
           fifo_wr_full         =>  open                  ,  
          
          
           -- Read Clock and reset
           fifo_async_rd_reset  =>  sig_sts_fifo_rd_rst   ,     
           fifo_async_rd_clk    =>  sig_sts_fifo_rd_clk   ,   
           
           -- Read Side
           fifo_rd_tvalid       =>  sts_wvalid            ,  
           fifo_rd_tready       =>  sts_wready            ,  
           fifo_rd_tdata        =>  sts_wdata             ,  
           fifo_rd_empty        =>  open                     
          
           );
        
    
       end generate GEN_INCLUDE_STATUS_FIFO;
    
    
    
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_OMIT_STATUS_FIFO
    --
    -- If Generate Description:
    --  Omits the Status FIFO
    --
    --
    ------------------------------------------------------------
    GEN_OMIT_STATUS_FIFO : if (C_INCLUDE_STSFIFO = 0) generate
    
       begin
  
         -- Status FIFO User interface housekeeping
         sts_wvalid            <=  '0';
         -- sts_wready         -- ignored
         sts_wdata             <=  (others => '0');
         sts_wstrb             <=  (others => '0');    
         sts_wlast             <=  '0';                
 
         
         
         -- Status FIFO Internal interface housekeeping
         stat2mstr_status_ready <= '1';
         -- mstr2stat_status       -- ignored
         -- mst2stst_status_valid  -- ignored
 
    
       end generate GEN_OMIT_STATUS_FIFO;
    
    
 
  
  end implementation;


-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_sg_rdmux.vhd
  --
  -- Description:     
  --    This file implements the DataMover Master Read Data Multiplexer.                 
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  
  
  -------------------------------------------------------------------------------
  
  entity axi_sg_rdmux is
    generic (
      
      C_SEL_ADDR_WIDTH     : Integer range 1  to   8 :=  5;
        -- Sets the width of the select control bus
      
      C_MMAP_DWIDTH        : Integer range 32 to 1024 := 32;
        -- Indicates the width of the AXI4 Data Channel
      
      C_STREAM_DWIDTH      : Integer range  8 to 1024 := 32
        -- Indicates the width of the AXI Stream Data Channel
      
      );
    port (
      
     
      -- AXI MMap Data Channel Input  -----------------------------------------------
                                                                                   --
      mmap_read_data_in         : In  std_logic_vector(C_MMAP_DWIDTH-1 downto 0);  --
        -- AXI Read data input                                                     --
      -------------------------------------------------------------------------------
      
      
      -- AXI Master Stream  ---------------------------------------------------------
                                                                                   --
      mux_data_out    : Out std_logic_vector(C_STREAM_DWIDTH-1 downto 0);          --
        --Mux data output                                                          --
      -------------------------------------------------------------------------------         
                
                
      -- Command Calculator Interface -----------------------------------------------
                                                                                   --
      mstr2data_saddr_lsb : In std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0)       --
         -- The next command start address LSbs to use for the read data           --
         -- mux (only used if Stream data width is less than the MMap Data         --
         -- Width).                                                                --
      -------------------------------------------------------------------------------
         
      );
  
  end entity axi_sg_rdmux;
  
  
  architecture implementation of axi_sg_rdmux is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    
    -- Function Decalarations -------------------------------------------------
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: func_mux_sel_width
    --
    -- Function Description:
    --   Calculates the number of needed bits for the Mux Select control
    -- based on the number of input channels to the mux.
    --
    -- Note that the number of input mux channels are always a 
    -- power of 2.
    --
    -------------------------------------------------------------------
    function func_mux_sel_width (num_channels : integer) return integer is
    
     Variable var_sel_width : integer := 0;
    
    begin
    
       case num_channels is
         when 2 =>
             var_sel_width := 1;
         when 4 =>
             var_sel_width := 2;
         when 8 =>
             var_sel_width := 3;
         when 16 =>
             var_sel_width := 4;
         when 32 =>
             var_sel_width := 5;
         when 64 =>
             var_sel_width := 6;
         when 128 =>
             var_sel_width := 7;
         
         when others => 
             var_sel_width := 0; 
       end case;
       
       Return (var_sel_width);
        
        
    end function func_mux_sel_width;
    
    
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: func_sel_ls_index
    --
    -- Function Description:
    --   Calculates the LS index of the select field to rip from the
    -- input select bus.
    --
    -- Note that the number of input mux channels are always a 
    -- power of 2.
    --
    -------------------------------------------------------------------
    function func_sel_ls_index (channel_width : integer) return integer is
    
     Variable var_sel_ls_index : integer := 0;
    
    begin
    
       case channel_width is
         when 8 => 
             var_sel_ls_index := 0;
         when 16 =>
             var_sel_ls_index := 1;
         when 32 =>
             var_sel_ls_index := 2;
         when 64 =>
             var_sel_ls_index := 3;
         when 128 =>
             var_sel_ls_index := 4;
         when 256 =>
             var_sel_ls_index := 5;
         when 512 =>
             var_sel_ls_index := 6;
         
         when others => -- 1024-bit channel case
             var_sel_ls_index := 7;
       end case;
       
       Return (var_sel_ls_index);
        
        
    end function func_sel_ls_index;
    
    
    
    
    
    -- Constant Decalarations -------------------------------------------------
    
    Constant CHANNEL_DWIDTH   : integer := C_STREAM_DWIDTH;
    Constant NUM_MUX_CHANNELS : integer := C_MMAP_DWIDTH/CHANNEL_DWIDTH;
    Constant MUX_SEL_WIDTH    : integer := func_mux_sel_width(NUM_MUX_CHANNELS);
    Constant MUX_SEL_LS_INDEX : integer := func_sel_ls_index(CHANNEL_DWIDTH);
    
    
    
    
    -- Signal Declarations  --------------------------------------------
 
    signal sig_rdmux_dout     : std_logic_vector(CHANNEL_DWIDTH-1 downto 0) := (others => '0');



    
  begin --(architecture implementation)
  
  
  
  
   -- Assign the Output data port 
    mux_data_out        <= sig_rdmux_dout;
  
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_STRM_EQ_MMAP
    --
    -- If Generate Description:
    --   This IfGen implements the case where the Stream Data Width is 
    -- the same as the Memory Map read Data width.
    --
    --
    ------------------------------------------------------------
    GEN_STRM_EQ_MMAP : if (NUM_MUX_CHANNELS = 1) generate
        
       begin
    
          sig_rdmux_dout <= mmap_read_data_in;
        
       end generate GEN_STRM_EQ_MMAP;
   
   
    
    
    
     
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_2XN
    --
    -- If Generate Description:
    --  2 channel input mux case
    --
    --
    ------------------------------------------------------------
    GEN_2XN : if (NUM_MUX_CHANNELS = 2) generate
    
       -- local signals
       signal sig_mux_sel_slice     : std_logic_vector(MUX_SEL_WIDTH-1 downto 0)  := (others => '0');
       signal sig_mux_sel_unsgnd    : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_mux_sel_int       : integer := 0;
       signal sig_mux_sel_int_local : integer := 0;
       signal sig_mux_dout          : std_logic_vector(CHANNEL_DWIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_mux_sel_slice   <= mstr2data_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_mux_sel_unsgnd  <=  UNSIGNED(sig_mux_sel_slice);  -- convert to unsigned
        
         sig_mux_sel_int     <=  TO_INTEGER(sig_mux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                 -- with locally static subtype error in each of the
                                                                 -- Mux IfGens
        
         sig_mux_sel_int_local <= sig_mux_sel_int;
         
         sig_rdmux_dout        <= sig_mux_dout;
       
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_2XN_NUX
         --
         -- Process Description:
         --  Implement the 2XN Mux
         --
         -------------------------------------------------------------
         DO_2XN_NUX : process (sig_mux_sel_int_local,
                               mmap_read_data_in)
            begin
              
              case sig_mux_sel_int_local is
                when 0 =>
                    sig_mux_dout <=  mmap_read_data_in(CHANNEL_DWIDTH-1 downto 0);
                
                when others => -- 1 case
                    sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*2)-1 downto CHANNEL_DWIDTH*1);
              end case;
              
            end process DO_2XN_NUX; 
 
         
       end generate GEN_2XN;
  
 
 
 
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_4XN
    --
    -- If Generate Description:
    --  4 channel input mux case
    --
    --
    ------------------------------------------------------------
    GEN_4XN : if (NUM_MUX_CHANNELS = 4) generate
    
       -- local signals
       signal sig_mux_sel_slice     : std_logic_vector(MUX_SEL_WIDTH-1 downto 0)  := (others => '0');
       signal sig_mux_sel_unsgnd    : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_mux_sel_int       : integer := 0;
       signal sig_mux_sel_int_local : integer := 0;
       signal sig_mux_dout          : std_logic_vector(CHANNEL_DWIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_mux_sel_slice     <= mstr2data_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_mux_sel_unsgnd    <= UNSIGNED(sig_mux_sel_slice);    -- convert to unsigned
        
         sig_mux_sel_int       <= TO_INTEGER(sig_mux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                  -- with locally static subtype error in each of the
                                                                  -- Mux IfGens
        
         sig_mux_sel_int_local <= sig_mux_sel_int;
         
         sig_rdmux_dout        <= sig_mux_dout;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_4XN_NUX
         --
         -- Process Description:
         --  Implement the 4XN Mux
         --
         -------------------------------------------------------------
         DO_4XN_NUX : process (sig_mux_sel_int_local,
                               mmap_read_data_in)
           begin
             
             case sig_mux_sel_int_local is
               when 0 =>
                   sig_mux_dout <=  mmap_read_data_in(CHANNEL_DWIDTH-1 downto 0);
               when 1 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*2)-1 downto CHANNEL_DWIDTH*1);
               when 2 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*3)-1 downto CHANNEL_DWIDTH*2);
               
               when others => -- 3 case
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*4)-1 downto CHANNEL_DWIDTH*3);
             end case;
             
           end process DO_4XN_NUX; 
  
         
       end generate GEN_4XN;
  
 
 
 
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_8XN
    --
    -- If Generate Description:
    --  8 channel input mux case
    --
    --
    ------------------------------------------------------------
    GEN_8XN : if (NUM_MUX_CHANNELS = 8) generate
    
       -- local signals
       signal sig_mux_sel_slice     : std_logic_vector(MUX_SEL_WIDTH-1 downto 0)  := (others => '0');
       signal sig_mux_sel_unsgnd    : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_mux_sel_int       : integer := 0;
       signal sig_mux_sel_int_local : integer := 0;
       signal sig_mux_dout          : std_logic_vector(CHANNEL_DWIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_mux_sel_slice     <= mstr2data_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_mux_sel_unsgnd    <= UNSIGNED(sig_mux_sel_slice);    -- convert to unsigned
        
         sig_mux_sel_int       <= TO_INTEGER(sig_mux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                  -- with locally static subtype error in each of the
                                                                  -- Mux IfGens
        
         sig_mux_sel_int_local <= sig_mux_sel_int;
         
         sig_rdmux_dout        <= sig_mux_dout;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_8XN_NUX
         --
         -- Process Description:
         --  Implement the 8XN Mux
         --
         -------------------------------------------------------------
         DO_8XN_NUX : process (sig_mux_sel_int_local,
                               mmap_read_data_in)
           begin
             
             case sig_mux_sel_int_local is
               when 0 =>
                   sig_mux_dout <=  mmap_read_data_in(CHANNEL_DWIDTH-1 downto 0);
               when 1 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*2)-1 downto CHANNEL_DWIDTH*1);
               when 2 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*3)-1 downto CHANNEL_DWIDTH*2);
               when 3 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*4)-1 downto CHANNEL_DWIDTH*3);
               when 4 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*5)-1 downto CHANNEL_DWIDTH*4);
               when 5 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*6)-1 downto CHANNEL_DWIDTH*5);
               when 6 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*7)-1 downto CHANNEL_DWIDTH*6);
               
               when others =>  -- 7 case
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*8)-1 downto CHANNEL_DWIDTH*7);
             end case;
                 
           end process DO_8XN_NUX; 
 
         
       end generate GEN_8XN;
  
 
 
 
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_16XN
    --
    -- If Generate Description:
    --  16 channel input mux case
    --
    --
    ------------------------------------------------------------
    GEN_16XN : if (NUM_MUX_CHANNELS = 16) generate
    
       -- local signals
       signal sig_mux_sel_slice     : std_logic_vector(MUX_SEL_WIDTH-1 downto 0)  := (others => '0');
       signal sig_mux_sel_unsgnd    : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_mux_sel_int       : integer := 0;
       signal sig_mux_sel_int_local : integer := 0;
       signal sig_mux_dout          : std_logic_vector(CHANNEL_DWIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_mux_sel_slice     <= mstr2data_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_mux_sel_unsgnd    <= UNSIGNED(sig_mux_sel_slice);    -- convert to unsigned
        
         sig_mux_sel_int       <= TO_INTEGER(sig_mux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                  -- with locally static subtype error in each of the
                                                                  -- Mux IfGens
        
         sig_mux_sel_int_local <= sig_mux_sel_int;
         
         sig_rdmux_dout        <= sig_mux_dout;
       
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_16XN_NUX
         --
         -- Process Description:
         --  Implement the 16XN Mux
         --
         -------------------------------------------------------------
         DO_16XN_NUX : process (sig_mux_sel_int_local,
                                mmap_read_data_in)
           begin
             
             case sig_mux_sel_int_local is
               when 0 =>
                   sig_mux_dout <=  mmap_read_data_in(CHANNEL_DWIDTH-1 downto 0);
               when 1 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*2)-1 downto CHANNEL_DWIDTH*1);
               when 2 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*3)-1 downto CHANNEL_DWIDTH*2);
               when 3 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*4)-1 downto CHANNEL_DWIDTH*3);
               when 4 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*5)-1 downto CHANNEL_DWIDTH*4);
               when 5 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*6)-1 downto CHANNEL_DWIDTH*5);
               when 6 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*7)-1 downto CHANNEL_DWIDTH*6);
               when 7 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*8)-1 downto CHANNEL_DWIDTH*7);
               when 8 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*9)-1 downto CHANNEL_DWIDTH*8);
               when 9 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*10)-1 downto CHANNEL_DWIDTH*9);
               when 10 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*11)-1 downto CHANNEL_DWIDTH*10);
               when 11 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*12)-1 downto CHANNEL_DWIDTH*11);
               when 12 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*13)-1 downto CHANNEL_DWIDTH*12);
               when 13 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*14)-1 downto CHANNEL_DWIDTH*13);
               when 14 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*15)-1 downto CHANNEL_DWIDTH*14);
               
               when others => -- 15 case
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*16)-1 downto CHANNEL_DWIDTH*15);
             end case;
          
           end process DO_16XN_NUX; 
 
         
       end generate GEN_16XN;
  
 
 
 
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_32XN
    --
    -- If Generate Description:
    --  32 channel input mux case
    --
    --
    ------------------------------------------------------------
    GEN_32XN : if (NUM_MUX_CHANNELS = 32) generate
    
       -- local signals
       signal sig_mux_sel_slice     : std_logic_vector(MUX_SEL_WIDTH-1 downto 0)  := (others => '0');
       signal sig_mux_sel_unsgnd    : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_mux_sel_int       : integer := 0;
       signal sig_mux_sel_int_local : integer := 0;
       signal sig_mux_dout          : std_logic_vector(CHANNEL_DWIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_mux_sel_slice     <= mstr2data_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_mux_sel_unsgnd    <= UNSIGNED(sig_mux_sel_slice);    -- convert to unsigned
        
         sig_mux_sel_int       <= TO_INTEGER(sig_mux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                  -- with locally static subtype error in each of the
                                                                  -- Mux IfGens
        
         sig_mux_sel_int_local <= sig_mux_sel_int;
         
         sig_rdmux_dout        <= sig_mux_dout;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_32XN_NUX
         --
         -- Process Description:
         --  Implement the 32XN Mux
         --
         -------------------------------------------------------------
         DO_32XN_NUX : process (sig_mux_sel_int_local,
                                mmap_read_data_in)
           begin
             
             case sig_mux_sel_int_local is
               
               when 0 =>
                   sig_mux_dout <=  mmap_read_data_in(CHANNEL_DWIDTH-1 downto 0);
               when 1 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*2)-1 downto CHANNEL_DWIDTH*1);
               when 2 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*3)-1 downto CHANNEL_DWIDTH*2);
               when 3 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*4)-1 downto CHANNEL_DWIDTH*3);
               when 4 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*5)-1 downto CHANNEL_DWIDTH*4);
               when 5 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*6)-1 downto CHANNEL_DWIDTH*5);
               when 6 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*7)-1 downto CHANNEL_DWIDTH*6);
               when 7 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*8)-1 downto CHANNEL_DWIDTH*7);
               when 8 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*9)-1 downto CHANNEL_DWIDTH*8);
               when 9 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*10)-1 downto CHANNEL_DWIDTH*9);
               when 10 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*11)-1 downto CHANNEL_DWIDTH*10);
               when 11 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*12)-1 downto CHANNEL_DWIDTH*11);
               when 12 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*13)-1 downto CHANNEL_DWIDTH*12);
               when 13 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*14)-1 downto CHANNEL_DWIDTH*13);
               when 14 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*15)-1 downto CHANNEL_DWIDTH*14);
               when 15 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*16)-1 downto CHANNEL_DWIDTH*15);
               when 16 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*17)-1 downto CHANNEL_DWIDTH*16);
               when 17 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*18)-1 downto CHANNEL_DWIDTH*17);
               when 18 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*19)-1 downto CHANNEL_DWIDTH*18);
               when 19 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*20)-1 downto CHANNEL_DWIDTH*19);
               when 20 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*21)-1 downto CHANNEL_DWIDTH*20);
               when 21 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*22)-1 downto CHANNEL_DWIDTH*21);
               when 22 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*23)-1 downto CHANNEL_DWIDTH*22);
               when 23 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*24)-1 downto CHANNEL_DWIDTH*23);
               when 24 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*25)-1 downto CHANNEL_DWIDTH*24);
               when 25 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*26)-1 downto CHANNEL_DWIDTH*25);
               when 26 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*27)-1 downto CHANNEL_DWIDTH*26);
               when 27 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*28)-1 downto CHANNEL_DWIDTH*27);
               when 28 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*29)-1 downto CHANNEL_DWIDTH*28);
               when 29 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*30)-1 downto CHANNEL_DWIDTH*29);
               when 30 =>
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*31)-1 downto CHANNEL_DWIDTH*30);
               
               when others => -- 31 case
                   sig_mux_dout <=  mmap_read_data_in((CHANNEL_DWIDTH*32)-1 downto CHANNEL_DWIDTH*31);
             end case;
          
           end process DO_32XN_NUX; 
 
         
       end generate GEN_32XN;
  
 
 
 
 
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_64XN
    --
    -- If Generate Description:
    --  64 channel input mux case
    --
    --
    ------------------------------------------------------------
    GEN_64XN : if (NUM_MUX_CHANNELS = 64) generate
    
       -- local signals
       signal sig_mux_sel_slice     : std_logic_vector(MUX_SEL_WIDTH-1 downto 0)  := (others => '0');
       signal sig_mux_sel_unsgnd    : unsigned(MUX_SEL_WIDTH-1 downto 0)          := (others => '0');
       signal sig_mux_sel_int       : integer := 0;
       signal sig_mux_sel_int_local : integer := 0;
       signal sig_mux_dout          : std_logic_vector(CHANNEL_DWIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_mux_sel_slice     <= mstr2data_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_mux_sel_unsgnd    <= UNSIGNED(sig_mux_sel_slice);    -- convert to unsigned
        
         sig_mux_sel_int       <= TO_INTEGER(sig_mux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                  -- with locally static subtype error in each of the
                                                                  -- Mux IfGens
        
         sig_mux_sel_int_local <= sig_mux_sel_int;
         
         sig_rdmux_dout        <= sig_mux_dout;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_64XN_NUX
         --
         -- Process Description:
         --  Implement the 64XN Mux
         --
         -------------------------------------------------------------
         DO_64XN_NUX : process (sig_mux_sel_int_local,
                                mmap_read_data_in)
           begin
             
             case sig_mux_sel_int_local is
             
               when 0 =>
                   sig_mux_dout  <=  mmap_read_data_in(CHANNEL_DWIDTH-1 downto 0)                     ;
               when 1 =>
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*2)-1 downto CHANNEL_DWIDTH*1)  ;
               when 2 =>                                                              
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*3)-1 downto CHANNEL_DWIDTH*2)  ;
               when 3 =>                                                              
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*4)-1 downto CHANNEL_DWIDTH*3)  ;
               when 4 =>                                                              
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*5)-1 downto CHANNEL_DWIDTH*4)  ;
               when 5 =>                                                              
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*6)-1 downto CHANNEL_DWIDTH*5)  ;
               when 6 =>                                                               
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*7)-1 downto CHANNEL_DWIDTH*6)  ;
               when 7 =>                                                               
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*8)-1 downto CHANNEL_DWIDTH*7)  ;
               when 8 =>                                                               
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*9)-1 downto CHANNEL_DWIDTH*8)  ;
               when 9 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*10)-1 downto CHANNEL_DWIDTH*9) ;
               when 10 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*11)-1 downto CHANNEL_DWIDTH*10);
               when 11 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*12)-1 downto CHANNEL_DWIDTH*11);
               when 12 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*13)-1 downto CHANNEL_DWIDTH*12);
               when 13 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*14)-1 downto CHANNEL_DWIDTH*13);
               when 14 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*15)-1 downto CHANNEL_DWIDTH*14);
               when 15 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*16)-1 downto CHANNEL_DWIDTH*15);
               when 16 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*17)-1 downto CHANNEL_DWIDTH*16);
               when 17 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*18)-1 downto CHANNEL_DWIDTH*17);
               when 18 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*19)-1 downto CHANNEL_DWIDTH*18);
               when 19 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*20)-1 downto CHANNEL_DWIDTH*19);
               when 20 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*21)-1 downto CHANNEL_DWIDTH*20);
               when 21 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*22)-1 downto CHANNEL_DWIDTH*21);
               when 22 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*23)-1 downto CHANNEL_DWIDTH*22);
               when 23 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*24)-1 downto CHANNEL_DWIDTH*23);
               when 24 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*25)-1 downto CHANNEL_DWIDTH*24);
               when 25 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*26)-1 downto CHANNEL_DWIDTH*25);
               when 26 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*27)-1 downto CHANNEL_DWIDTH*26);
               when 27 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*28)-1 downto CHANNEL_DWIDTH*27);
               when 28 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*29)-1 downto CHANNEL_DWIDTH*28);
               when 29 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*30)-1 downto CHANNEL_DWIDTH*29);
               when 30 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*31)-1 downto CHANNEL_DWIDTH*30);
               when 31 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*32)-1 downto CHANNEL_DWIDTH*31);
             
             
               when 32 =>
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*33)-1 downto CHANNEL_DWIDTH*32);
               when 33 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*34)-1 downto CHANNEL_DWIDTH*33);
               when 34 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*35)-1 downto CHANNEL_DWIDTH*34);
               when 35 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*36)-1 downto CHANNEL_DWIDTH*35);
               when 36 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*37)-1 downto CHANNEL_DWIDTH*36);
               when 37 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*38)-1 downto CHANNEL_DWIDTH*37);
               when 38 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*39)-1 downto CHANNEL_DWIDTH*38);
               when 39 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*40)-1 downto CHANNEL_DWIDTH*39);
               when 40 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*41)-1 downto CHANNEL_DWIDTH*40);
               when 41 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*42)-1 downto CHANNEL_DWIDTH*41);
               when 42 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*43)-1 downto CHANNEL_DWIDTH*42);
               when 43 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*44)-1 downto CHANNEL_DWIDTH*43);
               when 44 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*45)-1 downto CHANNEL_DWIDTH*44);
               when 45 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*46)-1 downto CHANNEL_DWIDTH*45);
               when 46 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*47)-1 downto CHANNEL_DWIDTH*46);
               when 47 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*48)-1 downto CHANNEL_DWIDTH*47);
               when 48 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*49)-1 downto CHANNEL_DWIDTH*48);
               when 49 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*50)-1 downto CHANNEL_DWIDTH*49);
               when 50 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*51)-1 downto CHANNEL_DWIDTH*50);
               when 51 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*52)-1 downto CHANNEL_DWIDTH*51);
               when 52 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*53)-1 downto CHANNEL_DWIDTH*52);
               when 53 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*54)-1 downto CHANNEL_DWIDTH*53);
               when 54 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*55)-1 downto CHANNEL_DWIDTH*54);
               when 55 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*56)-1 downto CHANNEL_DWIDTH*55);
               when 56 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*57)-1 downto CHANNEL_DWIDTH*56);
               when 57 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*58)-1 downto CHANNEL_DWIDTH*57);
               when 58 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*59)-1 downto CHANNEL_DWIDTH*58);
               when 59 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*60)-1 downto CHANNEL_DWIDTH*59);
               when 60 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*61)-1 downto CHANNEL_DWIDTH*60);
               when 61 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*62)-1 downto CHANNEL_DWIDTH*61);
               when 62 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*63)-1 downto CHANNEL_DWIDTH*62);
               
               when others => -- 63 case
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*64)-1 downto CHANNEL_DWIDTH*63);
             
             end case;
          
           end process DO_64XN_NUX; 
 
         
       end generate GEN_64XN;
  
 
  
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_128XN
    --
    -- If Generate Description:
    --  128 channel input mux case
    --
    --
    ------------------------------------------------------------
    GEN_128XN : if (NUM_MUX_CHANNELS = 128) generate
    
       -- local signals
       signal sig_mux_sel_slice     : std_logic_vector(MUX_SEL_WIDTH-1 downto 0)  := (others => '0');
       signal sig_mux_sel_unsgnd    : unsigned(MUX_SEL_WIDTH-1 downto 0)          := (others => '0');
       signal sig_mux_sel_int       : integer := 0;
       signal sig_mux_sel_int_local : integer := 0;
       signal sig_mux_dout          : std_logic_vector(CHANNEL_DWIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_mux_sel_slice     <= mstr2data_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_mux_sel_unsgnd    <= UNSIGNED(sig_mux_sel_slice);    -- convert to unsigned
        
         sig_mux_sel_int       <= TO_INTEGER(sig_mux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                  -- with locally static subtype error in each of the
                                                                  -- Mux IfGens
        
         sig_mux_sel_int_local <= sig_mux_sel_int;
         
         sig_rdmux_dout        <= sig_mux_dout;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_128XN_NUX
         --
         -- Process Description:
         --  Implement the 64XN Mux
         --
         -------------------------------------------------------------
         DO_128XN_NUX : process (sig_mux_sel_int_local,
                                 mmap_read_data_in)
           begin
             
             case sig_mux_sel_int_local is
               
               when 0 =>
                   sig_mux_dout  <=  mmap_read_data_in(CHANNEL_DWIDTH-1 downto 0)                     ;
               when 1 =>
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*2)-1 downto CHANNEL_DWIDTH*1)  ;
               when 2 =>                                                              
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*3)-1 downto CHANNEL_DWIDTH*2)  ;
               when 3 =>                                                              
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*4)-1 downto CHANNEL_DWIDTH*3)  ;
               when 4 =>                                                              
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*5)-1 downto CHANNEL_DWIDTH*4)  ;
               when 5 =>                                                              
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*6)-1 downto CHANNEL_DWIDTH*5)  ;
               when 6 =>                                                               
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*7)-1 downto CHANNEL_DWIDTH*6)  ;
               when 7 =>                                                               
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*8)-1 downto CHANNEL_DWIDTH*7)  ;
               when 8 =>                                                               
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*9)-1 downto CHANNEL_DWIDTH*8)  ;
               when 9 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*10)-1 downto CHANNEL_DWIDTH*9) ;
               when 10 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*11)-1 downto CHANNEL_DWIDTH*10);
               when 11 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*12)-1 downto CHANNEL_DWIDTH*11);
               when 12 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*13)-1 downto CHANNEL_DWIDTH*12);
               when 13 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*14)-1 downto CHANNEL_DWIDTH*13);
               when 14 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*15)-1 downto CHANNEL_DWIDTH*14);
               when 15 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*16)-1 downto CHANNEL_DWIDTH*15);
               when 16 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*17)-1 downto CHANNEL_DWIDTH*16);
               when 17 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*18)-1 downto CHANNEL_DWIDTH*17);
               when 18 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*19)-1 downto CHANNEL_DWIDTH*18);
               when 19 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*20)-1 downto CHANNEL_DWIDTH*19);
               when 20 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*21)-1 downto CHANNEL_DWIDTH*20);
               when 21 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*22)-1 downto CHANNEL_DWIDTH*21);
               when 22 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*23)-1 downto CHANNEL_DWIDTH*22);
               when 23 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*24)-1 downto CHANNEL_DWIDTH*23);
               when 24 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*25)-1 downto CHANNEL_DWIDTH*24);
               when 25 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*26)-1 downto CHANNEL_DWIDTH*25);
               when 26 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*27)-1 downto CHANNEL_DWIDTH*26);
               when 27 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*28)-1 downto CHANNEL_DWIDTH*27);
               when 28 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*29)-1 downto CHANNEL_DWIDTH*28);
               when 29 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*30)-1 downto CHANNEL_DWIDTH*29);
               when 30 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*31)-1 downto CHANNEL_DWIDTH*30);
               when 31 =>                                                           
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*32)-1 downto CHANNEL_DWIDTH*31);
             
             
               when 32 =>
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*33)-1 downto CHANNEL_DWIDTH*32);
               when 33 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*34)-1 downto CHANNEL_DWIDTH*33);
               when 34 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*35)-1 downto CHANNEL_DWIDTH*34);
               when 35 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*36)-1 downto CHANNEL_DWIDTH*35);
               when 36 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*37)-1 downto CHANNEL_DWIDTH*36);
               when 37 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*38)-1 downto CHANNEL_DWIDTH*37);
               when 38 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*39)-1 downto CHANNEL_DWIDTH*38);
               when 39 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*40)-1 downto CHANNEL_DWIDTH*39);
               when 40 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*41)-1 downto CHANNEL_DWIDTH*40);
               when 41 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*42)-1 downto CHANNEL_DWIDTH*41);
               when 42 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*43)-1 downto CHANNEL_DWIDTH*42);
               when 43 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*44)-1 downto CHANNEL_DWIDTH*43);
               when 44 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*45)-1 downto CHANNEL_DWIDTH*44);
               when 45 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*46)-1 downto CHANNEL_DWIDTH*45);
               when 46 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*47)-1 downto CHANNEL_DWIDTH*46);
               when 47 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*48)-1 downto CHANNEL_DWIDTH*47);
               when 48 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*49)-1 downto CHANNEL_DWIDTH*48);
               when 49 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*50)-1 downto CHANNEL_DWIDTH*49);
               when 50 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*51)-1 downto CHANNEL_DWIDTH*50);
               when 51 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*52)-1 downto CHANNEL_DWIDTH*51);
               when 52 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*53)-1 downto CHANNEL_DWIDTH*52);
               when 53 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*54)-1 downto CHANNEL_DWIDTH*53);
               when 54 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*55)-1 downto CHANNEL_DWIDTH*54);
               when 55 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*56)-1 downto CHANNEL_DWIDTH*55);
               when 56 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*57)-1 downto CHANNEL_DWIDTH*56);
               when 57 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*58)-1 downto CHANNEL_DWIDTH*57);
               when 58 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*59)-1 downto CHANNEL_DWIDTH*58);
               when 59 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*60)-1 downto CHANNEL_DWIDTH*59);
               when 60 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*61)-1 downto CHANNEL_DWIDTH*60);
               when 61 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*62)-1 downto CHANNEL_DWIDTH*61);
               when 62 =>                                                            
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*63)-1 downto CHANNEL_DWIDTH*62);
               when 63 => 
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*64)-1 downto CHANNEL_DWIDTH*63);
             
             
               when 64 =>
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*65)-1 downto CHANNEL_DWIDTH*64) ;
               when 65 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*66)-1 downto CHANNEL_DWIDTH*65) ;
               when 66 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*67)-1 downto CHANNEL_DWIDTH*66) ;
               when 67 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*68)-1 downto CHANNEL_DWIDTH*67) ;
               when 68 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*69)-1 downto CHANNEL_DWIDTH*68) ;
               when 69 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*70)-1 downto CHANNEL_DWIDTH*69) ;
               when 70 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*71)-1 downto CHANNEL_DWIDTH*70) ;
               when 71 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*72)-1 downto CHANNEL_DWIDTH*71) ;
               when 72 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*73)-1 downto CHANNEL_DWIDTH*72) ;
               when 73 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*74)-1 downto CHANNEL_DWIDTH*73) ;
               when 74 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*75)-1 downto CHANNEL_DWIDTH*74) ;
               when 75 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*76)-1 downto CHANNEL_DWIDTH*75) ;
               when 76 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*77)-1 downto CHANNEL_DWIDTH*76) ;
               when 77 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*78)-1 downto CHANNEL_DWIDTH*77) ;
               when 78 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*79)-1 downto CHANNEL_DWIDTH*78) ;
               when 79 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*80)-1 downto CHANNEL_DWIDTH*79) ;
               when 80 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*81)-1 downto CHANNEL_DWIDTH*80) ;
               when 81 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*82)-1 downto CHANNEL_DWIDTH*81) ;
               when 82 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*83)-1 downto CHANNEL_DWIDTH*82) ;
               when 83 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*84)-1 downto CHANNEL_DWIDTH*83) ;
               when 84 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*85)-1 downto CHANNEL_DWIDTH*84) ;
               when 85 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*86)-1 downto CHANNEL_DWIDTH*85) ;
               when 86 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*87)-1 downto CHANNEL_DWIDTH*86) ;
               when 87 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*88)-1 downto CHANNEL_DWIDTH*87) ;
               when 88 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*89)-1 downto CHANNEL_DWIDTH*88) ;
               when 89 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*90)-1 downto CHANNEL_DWIDTH*89) ;
               when 90 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*91)-1 downto CHANNEL_DWIDTH*90) ;
               when 91 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*92)-1 downto CHANNEL_DWIDTH*91) ;
               when 92 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*93)-1 downto CHANNEL_DWIDTH*92) ;
               when 93 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*94)-1 downto CHANNEL_DWIDTH*93) ;
               when 94 =>                                                                   
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*95)-1 downto CHANNEL_DWIDTH*94) ;
               when 95 =>                                                                 
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*96)-1 downto CHANNEL_DWIDTH*95) ;
             
             
               when 96 =>
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*97 )-1 downto CHANNEL_DWIDTH*96 ) ;
               when 97 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*98 )-1 downto CHANNEL_DWIDTH*97 ) ;
               when 98 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*99 )-1 downto CHANNEL_DWIDTH*98 ) ;
               when 99 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*100)-1 downto CHANNEL_DWIDTH*99 ) ;
               when 100 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*101)-1 downto CHANNEL_DWIDTH*100) ;
               when 101 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*102)-1 downto CHANNEL_DWIDTH*101) ;
               when 102 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*103)-1 downto CHANNEL_DWIDTH*102) ;
               when 103 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*104)-1 downto CHANNEL_DWIDTH*103) ;
               when 104 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*105)-1 downto CHANNEL_DWIDTH*104) ;
               when 105 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*106)-1 downto CHANNEL_DWIDTH*105) ;
               when 106 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*107)-1 downto CHANNEL_DWIDTH*106) ;
               when 107 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*108)-1 downto CHANNEL_DWIDTH*107) ;
               when 108 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*109)-1 downto CHANNEL_DWIDTH*108) ;
               when 109 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*110)-1 downto CHANNEL_DWIDTH*109) ;
               when 110 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*111)-1 downto CHANNEL_DWIDTH*110) ;
               when 111 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*112)-1 downto CHANNEL_DWIDTH*111) ;
               when 112 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*113)-1 downto CHANNEL_DWIDTH*112) ;
               when 113 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*114)-1 downto CHANNEL_DWIDTH*113) ;
               when 114 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*115)-1 downto CHANNEL_DWIDTH*114) ;
               when 115 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*116)-1 downto CHANNEL_DWIDTH*115) ;
               when 116 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*117)-1 downto CHANNEL_DWIDTH*116) ;
               when 117 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*118)-1 downto CHANNEL_DWIDTH*117) ;
               when 118 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*119)-1 downto CHANNEL_DWIDTH*118) ;
               when 119 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*120)-1 downto CHANNEL_DWIDTH*119) ;
               when 120 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*121)-1 downto CHANNEL_DWIDTH*120) ;
               when 121 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*122)-1 downto CHANNEL_DWIDTH*121) ;
               when 122 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*123)-1 downto CHANNEL_DWIDTH*122) ;
               when 123 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*124)-1 downto CHANNEL_DWIDTH*123) ;
               when 124 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*125)-1 downto CHANNEL_DWIDTH*124) ;
               when 125 =>                                                                    
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*126)-1 downto CHANNEL_DWIDTH*125) ;
               when 126 =>                                                                 
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*127)-1 downto CHANNEL_DWIDTH*126) ;
               
               when others => -- 127 case
                   sig_mux_dout  <=  mmap_read_data_in((CHANNEL_DWIDTH*128)-1 downto CHANNEL_DWIDTH*127) ;
             
             
             end case;
          
           end process DO_128XN_NUX; 
 
         
       end generate GEN_128XN;
  
 
  
  
  
  
  
  
  end implementation;


----------------------------------------------------------------------------
-- axi_sg_addr_cntl.vhd
----------------------------------------------------------------------------
--
-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_sg_addr_cntl.vhd
  --
  -- Description:     
  --    This file implements the axi_sg Master Address Controller.                 
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  library axi_sg_v4_1_6;
  Use axi_sg_v4_1_6.axi_sg_fifo;
  
  -------------------------------------------------------------------------------
  
  entity axi_sg_addr_cntl is
    generic (
      C_ADDR_FIFO_DEPTH      : Integer range  1 to  32 :=  4;
        -- sets the depth of the Command Queue FIFO
        
      C_ADDR_WIDTH           : Integer range 32 to  64 := 32;
        -- Sets the address bus width
      
      C_ADDR_ID              : Integer range  0 to 255 :=  0;
        -- Sets the value to be on the AxID output
      
      C_ADDR_ID_WIDTH        : Integer range  1 to   8 :=  4;
        -- Sets the width of the AxID output
      
      C_TAG_WIDTH            : Integer range  1 to   8 :=  4;
        -- Sets the width of the Command Tag field width
      
      C_FAMILY               : String  := "virtex7"
        -- Specifies the target FPGA family
        
      );
    port (
      
      -- Clock input ---------------------------------------------
      primary_aclk         : in  std_logic;                     --
         -- Primary synchronization clock for the Master side   --
         -- interface and internal logic. It is also used       --
         -- for the User interface synchronization when         --
         -- C_STSCMD_IS_ASYNC = 0.                              --
                                                                --
      -- Reset input                                            --
      mmap_reset           : in  std_logic;                     --
         -- Reset used for the internal master logic            --
      ------------------------------------------------------------   
      
     
      -- AXI Address Channel I/O  --------------------------------------------
      addr2axi_aid     : out std_logic_vector(C_ADDR_ID_WIDTH-1 downto 0);  --
         -- AXI Address Channel ID output                                   --
                                                                            --
      addr2axi_aaddr   : out std_logic_vector(C_ADDR_WIDTH-1 downto 0);     --
         -- AXI Address Channel Address output                              --
                                                                            --
      addr2axi_alen    : out std_logic_vector(7 downto 0);                  --
         -- AXI Address Channel LEN output                                  --
         -- Sized to support 256 data beat bursts                           --
                                                                            --
      addr2axi_asize   : out std_logic_vector(2 downto 0);                  --
         -- AXI Address Channel SIZE output                                 --
                                                                            --
      addr2axi_aburst  : out std_logic_vector(1 downto 0);                  --
         -- AXI Address Channel BURST output                                --
                                                                            --
      addr2axi_acache  : out std_logic_vector(3 downto 0);                  --
         -- AXI Address Channel BURST output                                --
                                                                            --
      addr2axi_auser  : out std_logic_vector(3 downto 0);                  --
         -- AXI Address Channel BURST output                                --
                                                                            --
      addr2axi_aprot   : out std_logic_vector(2 downto 0);                  --
         -- AXI Address Channel PROT output                                 --
                                                                            --
      addr2axi_avalid  : out std_logic;                                     --
         -- AXI Address Channel VALID output                                --

                                                                            --
      axi2addr_aready  : in  std_logic;                                     --
         -- AXI Address Channel READY input                                 --
      ------------------------------------------------------------------------   
         
         
        
      -- Currently unsupported AXI Address Channel output signals -------
        -- addr2axi_alock   : out std_logic_vector(2 downto 0);        --
        -- addr2axi_acache  : out std_logic_vector(4 downto 0);        --
        -- addr2axi_aqos    : out std_logic_vector(3 downto 0);        --
        -- addr2axi_aregion : out std_logic_vector(3 downto 0);        --
      -------------------------------------------------------------------
      
      
      
      -- Command Calculation Interface -----------------------------------------
      mstr2addr_tag         : In std_logic_vector(C_TAG_WIDTH-1 downto 0);    --
         -- The next command tag                                              --
                                                                              --
      mstr2addr_addr        : In  std_logic_vector(C_ADDR_WIDTH-1 downto 0);  --
         -- The next command address to put on the AXI MMap ADDR              --
                                                                              --
      mstr2addr_len         : In  std_logic_vector(7 downto 0);               --
         -- The next command length to put on the AXI MMap LEN                --
         -- Sized to support 256 data beat bursts                             --
                                                                              --
      mstr2addr_size        : In  std_logic_vector(2 downto 0);               --
         -- The next command size to put on the AXI MMap SIZE                 --
                                                                              --
      mstr2addr_burst       : In  std_logic_vector(1 downto 0);               --
         -- The next command burst type to put on the AXI MMap BURST          --
                                                                              --
      mstr2addr_cache        : In  std_logic_vector(3 downto 0);               --
         -- The next command burst type to put on the AXI MMap BURST          --
                                                                              --
      mstr2addr_user        : In  std_logic_vector(3 downto 0);               --
         -- The next command burst type to put on the AXI MMap BURST          --
                                                                              --
      mstr2addr_cmd_cmplt   : In std_logic;                                   --
         -- The indication to the Address Channel that the current            --
         -- sub-command output is the last one compiled from the              --
         -- parent command pulled from the Command FIFO                       --
                                                                              --
      mstr2addr_calc_error  : In std_logic;                                   --
         -- Indication if the next command in the calculation pipe            --
         -- has a calculation error                                           --
                                                                              --
      mstr2addr_cmd_valid   : in std_logic;                                   --
         -- The next command valid indication to the Address Channel          --
         -- Controller for the AXI MMap                                       --
  
                                                                              --
      addr2mstr_cmd_ready   : out std_logic;                                  --
         -- Indication to the Command Calculator that the                     --
         -- command is being accepted                                         --
      --------------------------------------------------------------------------
     
     
      
      -- Halted Indication to Reset Module ------------------------------
      addr2rst_stop_cmplt     : out std_logic;                         --
         -- Output flag indicating the address controller has stopped  --
         -- posting commands to the Address Channel due to a stop      --
         -- request vai the data2addr_stop_req input port              --
       ------------------------------------------------------------------
       
      
      
      -- Address Generation Control ---------------------------------------
      allow_addr_req    : in std_logic;                                  --
         -- Input used to enable/stall the posting of address requests.  --
         -- 0 = stall address request generation.                        --
         -- 1 = Enable Address request geneartion                        --
                                                                         --
      addr_req_posted   : out std_logic;                                 --
         -- Indication from the Address Channel Controller to external   --
         -- User logic that an address has been posted to the            --
         -- AXI Address Channel.                                         --
      ---------------------------------------------------------------------
    
      
                                                                        
                                                                        
      -- Data Channel Interface ---------------------------------------------
      addr2data_addr_posted    : Out std_logic;                            --
         -- Indication from the Address Channel Controller to the          --
         -- Data Controller that an address has been posted to the         --
         -- AXI Address Channel.                                           --
                                                                           --
      data2addr_data_rdy       : In std_logic;                             --
         -- Indication that the Data Channel is ready to send the first    --
         -- databeat of the next command on the write data channel.        --
         -- This is used for the "wait for data" feature which keeps the   --
         -- address controller from issuing a transfer requset until the   --
         -- corresponding data is ready. This is expected to be held in    --
         -- the asserted state until the addr2data_addr_posted signal is   --
         -- asserted.                                                      --
                                                                           --
      data2addr_stop_req       : In std_logic;                             --
         -- Indication that the Data Channel has encountered an error      --
         -- or a soft shutdown request and needs the Address Controller    --
         -- to stop posting commands to the AXI Address channel            --
      -----------------------------------------------------------------------

       
      -- Status Module Interface ---------------------------------------
      addr2stat_calc_error     : out std_logic;                       --
         -- Indication to the Status Module that the Addr Cntl FIFO   --
         -- is loaded with a Calc error                               --
                                                                      --
      addr2stat_cmd_fifo_empty : out std_logic                        --
         -- Indication to the Status Module that the Addr Cntl FIFO   --
         -- is empty                                                  --
      ------------------------------------------------------------------   
      
     
     
         
      );
  
  end entity axi_sg_addr_cntl;
  
  
  architecture implementation of axi_sg_addr_cntl is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    -- Constant Declarations  --------------------------------------------
    Constant APROT_VALUE      : std_logic_vector(2 downto 0) := (others => '0');
                                --'0' & -- bit 2, Normal Access
                                --'0' & -- bit 1, Nonsecure Access
                                --'0';  -- bit 0, Data Access
    
    Constant LEN_WIDTH        : integer := 8;
    Constant SIZE_WIDTH       : integer := 3;
    Constant BURST_WIDTH      : integer := 2;
    Constant CMD_CMPLT_WIDTH  : integer := 1;
    Constant CALC_ERROR_WIDTH : integer := 1;
    
    Constant ADDR_QUAL_WIDTH  : integer := C_TAG_WIDTH     +  -- Cmd Tag field width
                                           C_ADDR_WIDTH    +  -- Cmd Address field width
                                           LEN_WIDTH       +  -- Cmd Len field width
                                           SIZE_WIDTH      +  -- Cmd Size field width
                                           BURST_WIDTH     +  -- Cmd Burst field width
                                           CMD_CMPLT_WIDTH +  -- Cmd Cmplt filed width
                                           CALC_ERROR_WIDTH +  -- Cmd Calc Error flag
                                           8; -- Cmd Cache, user fields
    
    Constant USE_SYNC_FIFO    : integer := 0;
    Constant REG_FIFO_PRIM    : integer := 0; 
    Constant BRAM_FIFO_PRIM   : integer := 1; 
    Constant SRL_FIFO_PRIM    : integer := 2; 
    Constant FIFO_PRIM_TYPE   : integer := SRL_FIFO_PRIM; 
    
    
    
    
    
    -- Signal Declarations  --------------------------------------------
    signal sig_axi_addr               : std_logic_vector(C_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_axi_alen               : std_logic_vector(7 downto 0) := (others => '0');             
    signal sig_axi_asize              : std_logic_vector(2 downto 0) := (others => '0');             
    signal sig_axi_aburst             : std_logic_vector(1 downto 0) := (others => '0');             
    signal sig_axi_acache             : std_logic_vector(3 downto 0) := (others => '0');             
    signal sig_axi_auser              : std_logic_vector(3 downto 0) := (others => '0');             
    signal sig_axi_avalid             : std_logic := '0';
    signal sig_axi_aready             : std_logic := '0';
    signal sig_addr_posted            : std_logic := '0';
    signal sig_calc_error             : std_logic := '0';
    signal sig_cmd_fifo_empty         : std_logic := '0';
    Signal sig_aq_fifo_data_in        : std_logic_vector(ADDR_QUAL_WIDTH-1 downto 0) := (others => '0');
    Signal sig_aq_fifo_data_out       : std_logic_vector(ADDR_QUAL_WIDTH-1 downto 0) := (others => '0');
    signal sig_fifo_next_tag          : std_logic_vector(C_TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_fifo_next_addr         : std_logic_vector(C_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_fifo_next_len          : std_logic_vector(7 downto 0) := (others => '0');             
    signal sig_fifo_next_size         : std_logic_vector(2 downto 0) := (others => '0');             
    signal sig_fifo_next_burst        : std_logic_vector(1 downto 0) := (others => '0');
    signal sig_fifo_next_user         : std_logic_vector(3 downto 0) := (others => '0');
    signal sig_fifo_next_cache        : std_logic_vector(3 downto 0) := (others => '0');
    signal sig_fifo_next_cmd_cmplt    : std_logic := '0';
    signal sig_fifo_calc_error        : std_logic := '0';
    signal sig_fifo_wr_cmd_valid      : std_logic := '0';
    signal sig_fifo_wr_cmd_ready      : std_logic := '0';
    signal sig_fifo_rd_cmd_valid      : std_logic := '0';
    signal sig_fifo_rd_cmd_ready      : std_logic := '0';
    signal sig_next_tag_reg           : std_logic_vector(C_TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_next_addr_reg          : std_logic_vector(C_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_next_len_reg           : std_logic_vector(7 downto 0) := (others => '0');             
    signal sig_next_size_reg          : std_logic_vector(2 downto 0) := (others => '0');             
    signal sig_next_burst_reg         : std_logic_vector(1 downto 0) := (others => '0');             
    signal sig_next_cache_reg         : std_logic_vector(3 downto 0) := (others => '0');             
    signal sig_next_user_reg          : std_logic_vector(3 downto 0) := (others => '0');             
    signal sig_next_cmd_cmplt_reg     : std_logic := '0';             
    signal sig_addr_valid_reg         : std_logic := '0';
    signal sig_calc_error_reg         : std_logic := '0';
    signal sig_pop_addr_reg           : std_logic := '0';
    signal sig_push_addr_reg          : std_logic := '0';
    signal sig_addr_reg_empty         : std_logic := '0';
    signal sig_addr_reg_full          : std_logic := '0';
    signal sig_posted_to_axi          : std_logic := '0';
    -- obsoleted   signal sig_set_wfd_flop           : std_logic := '0';
    -- obsoleted   signal sig_clr_wfd_flop           : std_logic := '0';
    -- obsoleted   signal sig_wait_for_data          : std_logic := '0';
    -- obsoleted   signal sig_data2addr_data_rdy_reg : std_logic := '0';
    signal sig_allow_addr_req         : std_logic := '0';
    signal sig_posted_to_axi_2        : std_logic := '0';
 
    signal new_cmd_in : std_logic;
    signal first_addr_valid : std_logic;
    signal first_addr_valid_del : std_logic;
    signal first_addr_int : std_logic_vector (C_ADDR_WIDTH-1 downto 0);
    signal last_addr_int : std_logic_vector (C_ADDR_WIDTH-1 downto 0);
    signal addr2axi_cache_int  : std_logic_vector (7 downto 0); 
    signal addr2axi_cache_int1  : std_logic_vector (7 downto 0); 

    signal last_one : std_logic;
    signal latch : std_logic;
    signal first_one : std_logic;
    signal latch_n : std_logic;
    signal latch_n_del : std_logic;
    signal mstr2addr_cache_info_int : std_logic_vector (7 downto 0);
-- Register duplication attribute assignments to control fanout
-- on handshake output signals  
  
  Attribute KEEP : string; -- declaration
  Attribute EQUIVALENT_REGISTER_REMOVAL : string; -- declaration
  
  Attribute KEEP of sig_posted_to_axi   : signal is "TRUE"; -- definition
  Attribute KEEP of sig_posted_to_axi_2 : signal is "TRUE"; -- definition
  
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_posted_to_axi   : signal is "no"; 
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_posted_to_axi_2 : signal is "no"; 
  
  
  
 
    
  begin --(architecture implementation)
  
    -- AXI I/O Port assignments
    addr2axi_aid        <= STD_LOGIC_VECTOR(TO_UNSIGNED(C_ADDR_ID, C_ADDR_ID_WIDTH));
    addr2axi_aaddr      <= sig_axi_addr  ;
    addr2axi_alen       <= sig_axi_alen  ;
    addr2axi_asize      <= sig_axi_asize ;
    addr2axi_aburst     <= sig_axi_aburst;
    addr2axi_acache     <= sig_axi_acache;
    addr2axi_auser      <= sig_axi_auser;
    addr2axi_aprot      <= APROT_VALUE   ; 
    addr2axi_avalid     <= sig_axi_avalid;
    sig_axi_aready      <= axi2addr_aready;
    
    
    -- Command Calculator Handshake output
    sig_fifo_wr_cmd_valid <= mstr2addr_cmd_valid  ;
    addr2mstr_cmd_ready   <= sig_fifo_wr_cmd_ready;
    
    -- Data Channel Controller synchro pulse output                  
    addr2data_addr_posted <= sig_addr_posted;
     
     
    -- Status Module Interface outputs
    addr2stat_calc_error     <= sig_calc_error    ;
    addr2stat_cmd_fifo_empty <= sig_addr_reg_empty and
                                sig_cmd_fifo_empty; 
     
    

   
    -- Flag Indicating the Address Controller has completed a Stop
    addr2rst_stop_cmplt   <=  (data2addr_stop_req and  -- normal shutdown case
                               sig_addr_reg_empty) or
                              (data2addr_stop_req and  -- shutdown after error trap
                               sig_calc_error);
   
    -- Assign the address posting control and status
    sig_allow_addr_req    <=  allow_addr_req      ;
    addr_req_posted       <=  sig_posted_to_axi_2 ;
   
   
    
    -- Internal logic ------------------------------





    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_ADDR_FIFO
    --
    -- If Generate Description:
    --   Implements the case where the cmd qualifier depth is
    -- greater than 1.
    --
    ------------------------------------------------------------
--    GEN_ADDR_FIFO : if (C_ADDR_FIFO_DEPTH > 1) generate
--
--      begin
--
--        -- Format the input FIFO data word 
--        
--        sig_aq_fifo_data_in <= mstr2addr_cache      &
--                               mstr2addr_user      &
--                               mstr2addr_calc_error &
--                               mstr2addr_cmd_cmplt  &
--                               mstr2addr_burst      &
--                               mstr2addr_size       &
--                               mstr2addr_len        &
--                               mstr2addr_addr       &
--                               mstr2addr_tag ;
--        
--    
--    
--        -- Rip fields from FIFO output data word
--        sig_fifo_next_cache     <=  sig_aq_fifo_data_out((C_ADDR_WIDTH  +
--                                                        C_TAG_WIDTH     +
--                                                        LEN_WIDTH       +
--                                                        SIZE_WIDTH      +
--                                                        BURST_WIDTH     +
--                                                        CMD_CMPLT_WIDTH + 
--                                                        CALC_ERROR_WIDTH + 7)
--                                                    downto
--                                                        (C_ADDR_WIDTH  +
--                                                        C_TAG_WIDTH     +
--                                                        LEN_WIDTH       +
--                                                        SIZE_WIDTH      +
--                                                        BURST_WIDTH     +
--                                                        CMD_CMPLT_WIDTH +
--                                                        CALC_ERROR_WIDTH + 4)
--                                                    ); 
--
--        sig_fifo_next_user     <=  sig_aq_fifo_data_out((C_ADDR_WIDTH  +
--                                                        C_TAG_WIDTH     +
--                                                        LEN_WIDTH       +
--                                                        SIZE_WIDTH      +
--                                                        BURST_WIDTH     +
--                                                        CMD_CMPLT_WIDTH + 
--                                                        CALC_ERROR_WIDTH + 3)
--                                                    downto
--                                                        (C_ADDR_WIDTH  +
--                                                        C_TAG_WIDTH     +
--                                                        LEN_WIDTH       +
--                                                        SIZE_WIDTH      +
--                                                        BURST_WIDTH     +
--                                                        CMD_CMPLT_WIDTH +
--                                                        CALC_ERROR_WIDTH)
--                                                    ); 
--
--
--        sig_fifo_calc_error     <=  sig_aq_fifo_data_out((C_ADDR_WIDTH  +
--                                                        C_TAG_WIDTH     +
--                                                        LEN_WIDTH       +
--                                                        SIZE_WIDTH      +
--                                                        BURST_WIDTH     +
--                                                        CMD_CMPLT_WIDTH + 
--                                                        CALC_ERROR_WIDTH)-1); 
--                                                        
--        
--        sig_fifo_next_cmd_cmplt <=  sig_aq_fifo_data_out((C_ADDR_WIDTH  +
--                                                        C_TAG_WIDTH     +
--                                                        LEN_WIDTH       +
--                                                        SIZE_WIDTH      +
--                                                        BURST_WIDTH     +
--                                                        CMD_CMPLT_WIDTH)-1); 
--                                                        
--        
--        sig_fifo_next_burst     <=  sig_aq_fifo_data_out((C_ADDR_WIDTH +
--                                                         C_TAG_WIDTH   +
--                                                         LEN_WIDTH     +
--                                                         SIZE_WIDTH    +
--                                                         BURST_WIDTH)-1 
--                                                downto 
--                                                         C_ADDR_WIDTH  +
--                                                         C_TAG_WIDTH   +
--                                                         LEN_WIDTH     +
--                                                         SIZE_WIDTH) ;
--                                                          
--        sig_fifo_next_size      <=  sig_aq_fifo_data_out((C_ADDR_WIDTH +
--                                                         C_TAG_WIDTH   +
--                                                         LEN_WIDTH     +
--                                                         SIZE_WIDTH)-1
--                                                downto 
--                                                         C_ADDR_WIDTH  +
--                                                         C_TAG_WIDTH   +
--                                                         LEN_WIDTH) ; 
--                                                         
--        sig_fifo_next_len       <=  sig_aq_fifo_data_out((C_ADDR_WIDTH +
--                                                         C_TAG_WIDTH   +
--                                                         LEN_WIDTH)-1 
--                                                downto 
--                                                        C_ADDR_WIDTH   +
--                                                        C_TAG_WIDTH) ;
--                                                         
--        sig_fifo_next_addr      <=  sig_aq_fifo_data_out((C_ADDR_WIDTH +
--                                                         C_TAG_WIDTH)-1 
--                                                downto 
--                                                         C_TAG_WIDTH) ;
--         
--        sig_fifo_next_tag       <=  sig_aq_fifo_data_out(C_TAG_WIDTH-1 downto 0); 
--      
--      
--      
--        ------------------------------------------------------------
--        -- Instance: I_ADDR_QUAL_FIFO 
--        --
--        -- Description:
--        -- Instance for the Address/Qualifier FIFO
--        --
--        ------------------------------------------------------------
--         I_ADDR_QUAL_FIFO : entity axi_sg_v4_1_6.axi_sg_fifo
--         generic map (
--       
--           C_DWIDTH            =>  ADDR_QUAL_WIDTH         ,  
--           C_DEPTH             =>  C_ADDR_FIFO_DEPTH       ,  
--           C_IS_ASYNC          =>  USE_SYNC_FIFO           ,  
--           C_PRIM_TYPE         =>  FIFO_PRIM_TYPE          ,  
--           C_FAMILY            =>  C_FAMILY                   
--          
--           )
--         port map (
--           
--           -- Write Clock and reset
--           fifo_wr_reset        =>   mmap_reset            ,  
--           fifo_wr_clk          =>   primary_aclk          ,  
--           
--           -- Write Side
--           fifo_wr_tvalid       =>   sig_fifo_wr_cmd_valid ,  
--           fifo_wr_tready       =>   sig_fifo_wr_cmd_ready ,  
--           fifo_wr_tdata        =>   sig_aq_fifo_data_in   ,  
--           fifo_wr_full         =>   open                  ,  
--          
--          
--           -- Read Clock and reset
--           fifo_async_rd_reset  =>   mmap_reset            ,     
--           fifo_async_rd_clk    =>   primary_aclk          ,   
--           
--           -- Read Side
--           fifo_rd_tvalid       =>   sig_fifo_rd_cmd_valid ,  
--           fifo_rd_tready       =>   sig_fifo_rd_cmd_ready ,  
--           fifo_rd_tdata        =>   sig_aq_fifo_data_out  ,  
--           fifo_rd_empty        =>   sig_cmd_fifo_empty       
--          
--           );
--       
--      
--      
--      end generate GEN_ADDR_FIFO;
       
   

    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_NO_ADDR_FIFO
    --
    -- If Generate Description:
    --   Implements the case where no additional FIFOing is needed
    -- on the input command address/qualifiers.
    --
    ------------------------------------------------------------
    GEN_NO_ADDR_FIFO : if (C_ADDR_FIFO_DEPTH = 1) generate

      begin

        -- Bypass FIFO
        sig_fifo_next_tag       <=  mstr2addr_tag         ; 
        sig_fifo_next_addr      <=  mstr2addr_addr        ; 
        sig_fifo_next_len       <=  mstr2addr_len         ; 
        sig_fifo_next_size      <=  mstr2addr_size        ; 
        sig_fifo_next_burst     <=  mstr2addr_burst       ; 
        sig_fifo_next_cache     <=  mstr2addr_cache       ; 
        sig_fifo_next_user      <=  mstr2addr_user       ; 
        sig_fifo_next_cmd_cmplt <=  mstr2addr_cmd_cmplt   ;
        sig_fifo_calc_error     <=  mstr2addr_calc_error  ; 
        sig_cmd_fifo_empty      <=  sig_addr_reg_empty    ;
        sig_fifo_wr_cmd_ready   <=  sig_fifo_rd_cmd_ready ;
        sig_fifo_rd_cmd_valid   <=  sig_fifo_wr_cmd_valid ;
        
      
      end generate GEN_NO_ADDR_FIFO;
       
    
    
    -- Output Register Logic ------------------------------------------- 
    
    sig_axi_addr           <= sig_next_addr_reg  ;
    sig_axi_alen           <= sig_next_len_reg   ;
    sig_axi_asize          <= sig_next_size_reg  ;
    sig_axi_aburst         <= sig_next_burst_reg ;
    sig_axi_acache         <= sig_next_cache_reg ;
    sig_axi_auser          <= sig_next_user_reg ;
    sig_axi_avalid         <= sig_addr_valid_reg ;
    
    sig_calc_error         <= sig_calc_error_reg ;
    
    sig_fifo_rd_cmd_ready  <= sig_addr_reg_empty  and
                              sig_allow_addr_req  and   
                              -- obsoleted   not(sig_wait_for_data) and
                              not(data2addr_stop_req);
    
    sig_addr_posted        <= sig_posted_to_axi  ;
    
    
    -- Internal signals
    sig_push_addr_reg      <= sig_addr_reg_empty     and 
                              sig_fifo_rd_cmd_valid  and
                              sig_allow_addr_req     and   
                              -- obsoleted   not(sig_wait_for_data) and
                              not(data2addr_stop_req);
    
    sig_pop_addr_reg       <= not(sig_calc_error_reg) and 
                              sig_axi_aready          and 
                              sig_addr_reg_full;
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_ADDR_FIFO_REG
    --
    -- Process Description:
    --    This process implements a register for the Address 
    -- Control FIFO that operates like a 1 deep Sync FIFO.
    --
    -------------------------------------------------------------
    IMP_ADDR_FIFO_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset       = '1' or
                sig_pop_addr_reg = '1') then
              
              sig_next_tag_reg        <= (others => '0')         ;
              sig_next_addr_reg       <= (others => '0')         ;
              sig_next_len_reg        <= (others => '0')         ;
              sig_next_size_reg       <= (others => '0')         ;
              sig_next_burst_reg      <= (others => '0')         ;
              sig_next_cache_reg      <= (others => '0')         ;
              sig_next_user_reg      <= (others => '0')         ;
              sig_next_cmd_cmplt_reg  <= '0'                     ;
              sig_addr_valid_reg      <= '0'                     ;
              sig_calc_error_reg      <= '0'                     ;
                                                                 
              sig_addr_reg_empty      <= '1'                     ;
              sig_addr_reg_full       <= '0'                     ;
              
            elsif (sig_push_addr_reg = '1') then
              
              sig_next_tag_reg        <= sig_fifo_next_tag       ;
              sig_next_addr_reg       <= sig_fifo_next_addr      ;
              sig_next_len_reg        <= sig_fifo_next_len       ;
              sig_next_size_reg       <= sig_fifo_next_size      ;
              sig_next_burst_reg      <= sig_fifo_next_burst     ;
              sig_next_cache_reg      <= sig_fifo_next_cache     ;
              sig_next_user_reg      <= sig_fifo_next_user     ;
              sig_next_cmd_cmplt_reg  <= sig_fifo_next_cmd_cmplt ;
              sig_addr_valid_reg      <= not(sig_fifo_calc_error);
              sig_calc_error_reg      <= sig_fifo_calc_error     ;
              
              sig_addr_reg_empty      <= '0'                     ;
              sig_addr_reg_full       <= '1'                     ;
              
            else
              null;  -- don't change state
            end if; 
         end if;       
       end process IMP_ADDR_FIFO_REG; 
     

    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_POSTED_FLAG
    --
    -- Process Description:
    --   This implements a FLOP that creates a 1 clock wide pulse
    -- indicating a new address/qualifier set has been posted to
    -- the AXI Addres Channel outputs. This is used to synchronize 
    -- the Data Channel Controller.
    --
    -------------------------------------------------------------
    IMP_POSTED_FLAG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then

              sig_posted_to_axi   <= '0';
              sig_posted_to_axi_2 <= '0';
              
            elsif (sig_push_addr_reg = '1') then

              sig_posted_to_axi   <= '1';
              sig_posted_to_axi_2 <= '1';
              
            else
              
              sig_posted_to_axi   <= '0';
              sig_posted_to_axi_2 <= '0';
              
            end if; 
         end if;       
       end process IMP_POSTED_FLAG; 
  
--     PROC_CMD_DETECT :  process (primary_aclk)
--                     begin
--                     if (mmap_reset = '1') then
--                        first_addr_valid_del <= '0';
--                     elsif (primary_aclk'event and primary_aclk = '1') then
--                        first_addr_valid_del <= first_addr_valid;
--                     end if;
--     end process PROC_CMD_DETECT;
--
--     PROC_ADDR_DET : process (primary_aclk)
--            begin
--              if (mmap_reset = '1') then
--                 first_addr_valid <= '0';
--                 first_addr_int <= (others => '0');
--                 last_addr_int <= (others => '0');
--              elsif (primary_aclk'event and primary_aclk = '1') then
--                if (mstr2addr_cmd_valid = '1' and first_addr_valid = '0') then
--                   first_addr_valid <= '1';
--                   first_addr_int <= mstr2addr_addr;
--                   last_addr_int <= last_addr_int;
--                elsif (mstr2addr_cmd_cmplt = '1') then
--                   first_addr_valid <= '0';
--                   first_addr_int <= first_addr_int;
--                   last_addr_int <= mstr2addr_addr;
--                end if;
--              end if;
--           end process PROC_ADDR_DET;
--
--     latch <= first_addr_valid and (not first_addr_valid_del);
--     latch_n <= (not first_addr_valid) and first_addr_valid_del;
--
--     PROC_CACHE1 : process (primary_aclk)
--          begin
--              if (mmap_reset = '1') then
--                 mstr2addr_cache_info_int <= (others => '0');
--                 latch_n_del <= '0';
--              elsif (primary_aclk'event and primary_aclk = '1') then
--                 if (latch_n = '1') then 
--                   mstr2addr_cache_info_int <= mstr2addr_cache_info;
--                 end if;
--                 latch_n_del <= latch_n;
--              end if;
--          end process PROC_CACHE1;
--       
--
--     PROC_CACHE : process (primary_aclk)
--          begin
--              if (mmap_reset = '1') then
--                 addr2axi_cache_int1 <= (others => '0');
--                 first_one <= '0'; 
--              elsif (primary_aclk'event and primary_aclk = '1') then
--                 first_one <= '0';
----                 if (latch = '1' and first_one = '0') then  -- first one
--                 if (sig_addr_valid_reg = '0' and first_addr_valid = '0') then
--                    addr2axi_cache_int1 <= mstr2addr_cache_info;
----                    first_one <= '1';
----                 elsif (latch_n_del = '1') then
----                    addr2axi_cache_int <= mstr2addr_cache_info_int;
--                 elsif ((first_addr_int = sig_next_addr_reg) and (sig_addr_valid_reg = '1')) then
--                    addr2axi_cache_int1 <= addr2axi_cache_int1; --mstr2addr_cache_info (7 downto 4);
--                 elsif ((last_addr_int >= sig_next_addr_reg) and (sig_addr_valid_reg = '1')) then 
--                    addr2axi_cache_int1 <= addr2axi_cache_int1; --mstr2addr_cache_info (7 downto 4);
--                 end if; 
--              end if;
--          end process PROC_CACHE;
--
--
--     PROC_CACHE2 : process (primary_aclk)
--          begin
--              if (mmap_reset = '1') then
--                 addr2axi_cache_int <= (others => '0');
--              elsif (primary_aclk'event and primary_aclk = '1') then
--                 addr2axi_cache_int <= addr2axi_cache_int1;
--              end if;
--          end process PROC_CACHE2;
-- 
--addr2axi_cache <= addr2axi_cache_int (3 downto 0);
--addr2axi_user  <= addr2axi_cache_int (7 downto 4);
-- 
  end implementation;


-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_sg_rddata_cntl.vhd
  --
  -- Description:     
  --    This file implements the DataMover Master Read Data Controller.                 
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  
  library axi_sg_v4_1_6; 
  use axi_sg_v4_1_6.axi_sg_rdmux;  
  
  
  -------------------------------------------------------------------------------
  
  entity axi_sg_rddata_cntl is
    generic (
      
      C_INCLUDE_DRE          : Integer range  0 to   1 :=  0;
        -- Indicates if the DRE interface is used
        
      C_ALIGN_WIDTH          : Integer range  1 to   3 :=  3;
        -- Sets the width of the DRE Alignment controls
        
      C_SEL_ADDR_WIDTH       : Integer range  1 to   8 :=  5;
        -- Sets the width of the LS bits of the transfer address that
        -- are being used to Mux read data from a wider AXI4 Read
        -- Data Bus
        
      C_DATA_CNTL_FIFO_DEPTH : Integer range  1 to  32 :=  4;
        -- Sets the depth of the internal command fifo used for the
        -- command queue
        
      C_MMAP_DWIDTH          : Integer range 32 to 1024 := 32;
        -- Indicates the native data width of the Read Data port
        
      C_STREAM_DWIDTH        : Integer range  8 to 1024 := 32;
        -- Sets the width of the Stream output data port
        
      C_TAG_WIDTH            : Integer range  1 to   8 :=  4;
        -- Indicates the width of the Tag field of the input command
        
      C_FAMILY               : String                  := "virtex7"
        -- Indicates the device family of the target FPGA
        
      
      );
    port (
      
      -- Clock and Reset inputs ----------------------------------------
                                                                      --
      primary_aclk          : in  std_logic;                          --
         -- Primary synchronization clock for the Master side         --
         -- interface and internal logic. It is also used             --
         -- for the User interface synchronization when               --
         -- C_STSCMD_IS_ASYNC = 0.                                    --
                                                                      --
      -- Reset input                                                  --
      mmap_reset            : in  std_logic;                          --
         -- Reset used for the internal master logic                  --
      ------------------------------------------------------------------
      
     
      
      -- Soft Shutdown internal interface -----------------------------------
                                                                           --
      rst2data_stop_request : in  std_logic;                               --
         -- Active high soft stop request to modules                       --
                                                                           --
      data2addr_stop_req    : Out std_logic;                               --
        -- Active high signal requesting the Address Controller            --
        -- to stop posting commands to the AXI Read Address Channel        --
                                                                           --
      data2rst_stop_cmplt   : Out std_logic;                               --
        -- Active high indication that the Data Controller has completed   --
        -- any pending transfers committed by the Address Controller       --
        -- after a stop has been requested by the Reset module.            --
      -----------------------------------------------------------------------
   
   
        
      -- External Address Pipelining Contol support -------------------------
                                                                           --
      mm2s_rd_xfer_cmplt    : out std_logic;                               --
        -- Active high indication that the Data Controller has completed   --
        -- a single read data transfer on the AXI4 Read Data Channel.      --
        -- This signal escentially echos the assertion of rlast received   --
        -- from the AXI4.                                                  --
      -----------------------------------------------------------------------
      
      
      
        
     -- AXI Read Data Channel I/O  ---------------------------------------------
                                                                              --
      mm2s_rdata            : In  std_logic_vector(C_MMAP_DWIDTH-1 downto 0); --
        -- AXI Read data input                                                --
                                                                              --
      mm2s_rresp            : In  std_logic_vector(1 downto 0);               --
        -- AXI Read response input                                            --
                                                                              --
      mm2s_rlast            : In  std_logic;                                  --
        -- AXI Read LAST input                                                --
                                                                              --
      mm2s_rvalid           : In  std_logic;                                  --
        -- AXI Read VALID input                                               --
                                                                              --
      mm2s_rready           : Out std_logic;                                  --
        -- AXI Read data READY output                                         --
      --------------------------------------------------------------------------
               
                
                
                
     -- MM2S DRE Control  -------------------------------------------------------------
                                                                                     --
      mm2s_dre_new_align      : Out std_logic;                                       --
        -- Active high signal indicating new DRE aligment required                   --
                                                                                     --
      mm2s_dre_use_autodest   : Out std_logic;                                       --
        -- Active high signal indicating to the DRE to use an auto-                  --
        -- calculated desination alignment based on the last transfer                --
                                                                                     --
      mm2s_dre_src_align      : Out std_logic_vector(C_ALIGN_WIDTH-1 downto 0);      --
        -- Bit field indicating the byte lane of the first valid data byte           --
        -- being sent to the DRE                                                     --
                                                                                     --
      mm2s_dre_dest_align     : Out std_logic_vector(C_ALIGN_WIDTH-1 downto 0);      --
        -- Bit field indicating the desired byte lane of the first valid data byte   --
        -- to be output by the DRE                                                   --
                                                                                     --
      mm2s_dre_flush          : Out std_logic;                                       --
        -- Active high signal indicating to the DRE to flush the current             --
        -- contents to the output register in preparation of a new alignment         --
        -- that will be comming on the next transfer input                           --
      ---------------------------------------------------------------------------------
               
                
                
                
     -- AXI Master Stream Channel------------------------------------------------------
                                                                                     --
      mm2s_strm_wvalid   : Out std_logic;                                            --
        -- AXI Stream VALID Output                                                   --
                                                                                     --
      mm2s_strm_wready   : In  Std_logic;                                            --
        -- AXI Stream READY input                                                    --
                                                                                     --
      mm2s_strm_wdata    : Out std_logic_vector(C_STREAM_DWIDTH-1 downto 0);         --
        -- AXI Stream data output                                                    --
                                                                                     --
      mm2s_strm_wstrb    : Out std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);     --    
        -- AXI Stream STRB output                                                    --
                                                                                     --
      mm2s_strm_wlast    : Out std_logic;                                            --
        -- AXI Stream LAST output                                                    --


      ---------------------------------------------------------------------------------
               
                
      
      -- MM2S Store and Forward Supplimental Control   --------------------------------
      -- This output is time aligned and qualified with the AXI Master Stream Channel--
                                                                                     --
      mm2s_data2sf_cmd_cmplt   : out std_logic;                                      --
                                                                                     --
      ---------------------------------------------------------------------------------
                                                                                     
                                                                                     
                                                                                     
                                                                                     
                                                                                     
                
                
      -- Command Calculator Interface -------------------------------------------------
                                                                                     --
      mstr2data_tag        : In std_logic_vector(C_TAG_WIDTH-1 downto 0);            --
         -- The next command tag                                                     --
                                                                                     --
      mstr2data_saddr_lsb  : In std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0);       --
         -- The next command start address LSbs to use for the read data             --
         -- mux (only used if Stream data width is 8 or 16 bits).                    --
                                                                                     --
      mstr2data_len        : In std_logic_vector(7 downto 0);                        --
         -- The LEN value output to the Address Channel                              --
                                                                                     --
      mstr2data_strt_strb  : In std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);    --
         -- The starting strobe value to use for the first stream data beat          --
                                                                                     --
      mstr2data_last_strb  : In std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);    --
         -- The endiing (LAST) strobe value to use for the last stream               --
         -- data beat                                                                --
                                                                                     --
      mstr2data_drr        : In std_logic;                                           --
         -- The starting tranfer of a sequence of transfers                          --
                                                                                     --
      mstr2data_eof        : In std_logic;                                           --
         -- The endiing tranfer of a sequence of transfers                           --
                                                                                     --
      mstr2data_sequential : In std_logic;                                           --
         -- The next sequential tranfer of a sequence of transfers                   --
         -- spawned from a single parent command                                     --
                                                                                     --
      mstr2data_calc_error : In std_logic;                                           --
         -- Indication if the next command in the calculation pipe                   --
         -- has a calculation error                                                  --
                                                                                     --
      mstr2data_cmd_cmplt  : In std_logic;                                           --
         -- The indication to the Data Channel that the current                      --
         -- sub-command output is the last one compiled from the                     --
         -- parent command pulled from the Command FIFO                              --
                                                                                     --
      mstr2data_cmd_valid  : In std_logic;                                           --
         -- The next command valid indication to the Data Channel                    --
         -- Controller for the AXI MMap                                              --
                                                                                     --
      data2mstr_cmd_ready  : Out std_logic ;                                         --
         -- Indication from the Data Channel Controller that the                     --
         -- command is being accepted on the AXI Address Channel                     --
                                                                                     --
      mstr2data_dre_src_align   : In std_logic_vector(C_ALIGN_WIDTH-1 downto 0);     --
         -- The source (input) alignment for the DRE                                 --
                                                                                     --
      mstr2data_dre_dest_align  : In std_logic_vector(C_ALIGN_WIDTH-1 downto 0);     --
         -- The destinstion (output) alignment for the DRE                           --
      ---------------------------------------------------------------------------------
     
      
      
      
        
      -- Address Controller Interface -------------------------------------------------
                                                                                     --
      addr2data_addr_posted : In std_logic ;                                         --
         -- Indication from the Address Channel Controller to the                    --
         -- Data Controller that an address has been posted to the                   --
         -- AXI Address Channel                                                      --
      ---------------------------------------------------------------------------------


      
      -- Data Controller General Halted Status ----------------------------------------
                                                                                     --
      data2all_dcntlr_halted : Out std_logic;                                        --
         -- When asserted, this indicates the data controller has satisfied          --
         -- all pending transfers queued by the Address Controller and is halted.    --
      ---------------------------------------------------------------------------------
      
       
 
      -- Output Stream Skid Buffer Halt control ---------------------------------------
                                                                                     --
      data2skid_halt : Out std_logic;                                                --
         -- The data controller asserts this output for 1 primary clock period       --
         -- The pulse commands the MM2S Stream skid buffer to tun off outputs        --
         -- at the next tlast transmission.                                          --
      ---------------------------------------------------------------------------------
      
       
 
       
      -- Read Status Controller Interface ------------------------------------------------
                                                                                        --
      data2rsc_tag       : Out std_logic_vector(C_TAG_WIDTH-1 downto 0);                --
         -- The propagated command tag from the Command Calculator                      --
                                                                                        --
      data2rsc_calc_err  : Out std_logic ;                                              --
         -- Indication that the current command out from the Cntl FIFO                  --
         -- has a propagated calculation error from the Command Calculator              --
                                                                                        --
      data2rsc_okay      : Out std_logic ;                                              --
         -- Indication that the AXI Read transfer completed with OK status              --
                                                                                        --
      data2rsc_decerr    : Out std_logic ;                                              --
         -- Indication that the AXI Read transfer completed with decode error status    --
                                                                                        --
      data2rsc_slverr    : Out std_logic ;                                              --
         -- Indication that the AXI Read transfer completed with slave error status     --
                                                                                        --
      data2rsc_cmd_cmplt : Out std_logic ;                                              --
         -- Indication by the Data Channel Controller that the                          --
         -- corresponding status is the last status for a parent command                --
         -- pulled from the command FIFO                                                --
                                                                                        --
      rsc2data_ready     : in  std_logic;                                               --
         -- Handshake bit from the Read Status Controller Module indicating             --
         -- that the it is ready to accept a new Read status transfer                   --
                                                                                        --
      data2rsc_valid     : Out  std_logic ;                                             --
         -- Handshake bit output to the Read Status Controller Module                   --
         -- indicating that the Data Controller has valid tag and status                --
         -- indicators to transfer                                                      --
                                                                                        --
      rsc2mstr_halt_pipe : In std_logic                                                 --
         -- Status Flag indicating the Status Controller needs to stall the command     --
         -- execution pipe due to a Status flow issue or internal error. Generally      --
         -- this will occur if the Status FIFO is not being serviced fast enough to     --
         -- keep ahead of the command execution.                                        --
      ------------------------------------------------------------------------------------
      
      );
  
  end entity axi_sg_rddata_cntl;
  
  
  architecture implementation of axi_sg_rddata_cntl is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    
    
    -- Function declaration   ----------------------------------------
    
    
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_set_cnt_width
    --
    -- Function Description:
    --    Sets a count width based on a fifo depth. A depth of 4 or less
    -- is a special case which requires a minimum count width of 3 bits.
    --
    -------------------------------------------------------------------
    function funct_set_cnt_width (fifo_depth : integer) return integer is
    
      Variable temp_cnt_width : Integer := 4;
    
    begin
    
      
      if (fifo_depth <= 4) then
      
         temp_cnt_width := 3;
      
      elsif (fifo_depth <= 8) then
     -- coverage off 
         temp_cnt_width := 4;
      
      elsif (fifo_depth <= 16) then
      
         temp_cnt_width := 5;
      
      elsif (fifo_depth <= 32) then
      
         temp_cnt_width := 6;
      
      else  -- fifo depth <= 64
      
         temp_cnt_width := 7;
      
      end if;
      -- coverage on
      
      Return (temp_cnt_width);
       
       
    end function funct_set_cnt_width;
    
 
 
 
  
    -- Constant Declarations  --------------------------------------------
    
    Constant OKAY                   : std_logic_vector(1 downto 0) := "00";
    Constant EXOKAY                 : std_logic_vector(1 downto 0) := "01";
    Constant SLVERR                 : std_logic_vector(1 downto 0) := "10";
    Constant DECERR                 : std_logic_vector(1 downto 0) := "11";
                                    
    Constant STRM_STRB_WIDTH        : integer := C_STREAM_DWIDTH/8;
    Constant LEN_OF_ZERO            : std_logic_vector(7 downto 0) := (others => '0');
    Constant USE_SYNC_FIFO          : integer := 0;
    Constant REG_FIFO_PRIM          : integer := 0; 
    Constant BRAM_FIFO_PRIM         : integer := 1; 
    Constant SRL_FIFO_PRIM          : integer := 2; 
    Constant FIFO_PRIM_TYPE         : integer := SRL_FIFO_PRIM; 
    Constant TAG_WIDTH              : integer := C_TAG_WIDTH;
    Constant SADDR_LSB_WIDTH        : integer := C_SEL_ADDR_WIDTH;
    Constant LEN_WIDTH              : integer := 8;
    Constant STRB_WIDTH             : integer := C_STREAM_DWIDTH/8;
    Constant SOF_WIDTH              : integer := 1;
    Constant EOF_WIDTH              : integer := 1;
    Constant CMD_CMPLT_WIDTH        : integer := 1;
    Constant SEQUENTIAL_WIDTH       : integer := 1;
    Constant CALC_ERR_WIDTH         : integer := 1;
    Constant DRE_ALIGN_WIDTH        : integer := C_ALIGN_WIDTH;
                                    
    Constant DCTL_FIFO_WIDTH        : Integer := TAG_WIDTH        +  -- Tag field
                                                 SADDR_LSB_WIDTH  +  -- LS Address field width
                                                 LEN_WIDTH        +  -- LEN field
                                                 STRB_WIDTH       +  -- Starting Strobe field
                                                 STRB_WIDTH       +  -- Ending Strobe field
                                                 SOF_WIDTH        +  -- SOF Flag Field
                                                 EOF_WIDTH        +  -- EOF flag field
                                                 SEQUENTIAL_WIDTH +  -- Calc error flag
                                                 CMD_CMPLT_WIDTH  +  -- Sequential command flag
                                                 CALC_ERR_WIDTH   +  -- Command Complete Flag
                                                 DRE_ALIGN_WIDTH  +  -- DRE Source Align width
                                                 DRE_ALIGN_WIDTH ;   -- DRE Dest Align width
                                                 
                                    
    -- Caution, the INDEX calculations are order dependent so don't rearrange
    Constant TAG_STRT_INDEX         : integer := 0;
    Constant SADDR_LSB_STRT_INDEX   : integer := TAG_STRT_INDEX + TAG_WIDTH;
    Constant LEN_STRT_INDEX         : integer := SADDR_LSB_STRT_INDEX + SADDR_LSB_WIDTH;
    Constant STRT_STRB_STRT_INDEX   : integer := LEN_STRT_INDEX + LEN_WIDTH;
    Constant LAST_STRB_STRT_INDEX   : integer := STRT_STRB_STRT_INDEX + STRB_WIDTH;
    Constant SOF_STRT_INDEX         : integer := LAST_STRB_STRT_INDEX + STRB_WIDTH;
    Constant EOF_STRT_INDEX         : integer := SOF_STRT_INDEX + SOF_WIDTH;
    Constant SEQUENTIAL_STRT_INDEX  : integer := EOF_STRT_INDEX + EOF_WIDTH;
    Constant CMD_CMPLT_STRT_INDEX   : integer := SEQUENTIAL_STRT_INDEX + SEQUENTIAL_WIDTH;
    Constant CALC_ERR_STRT_INDEX    : integer := CMD_CMPLT_STRT_INDEX + CMD_CMPLT_WIDTH;
    Constant DRE_SRC_STRT_INDEX     : integer := CALC_ERR_STRT_INDEX + CALC_ERR_WIDTH;
    Constant DRE_DEST_STRT_INDEX    : integer := DRE_SRC_STRT_INDEX + DRE_ALIGN_WIDTH;
    
    Constant ADDR_INCR_VALUE        : integer := C_STREAM_DWIDTH/8;
    
    --Constant ADDR_POSTED_CNTR_WIDTH : integer := 5; -- allows up to 32 entry address queue
    Constant ADDR_POSTED_CNTR_WIDTH : integer := funct_set_cnt_width(C_DATA_CNTL_FIFO_DEPTH); 
    
    
    Constant ADDR_POSTED_ZERO       : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) 
                                      := (others => '0');
    Constant ADDR_POSTED_ONE        : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) 
                                      := TO_UNSIGNED(1, ADDR_POSTED_CNTR_WIDTH);
    Constant ADDR_POSTED_MAX        : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) 
                                       := (others => '1');
                    
    
    
    
    -- Signal Declarations  --------------------------------------------
    
    signal sig_good_dbeat               : std_logic := '0';
    signal sig_get_next_dqual           : std_logic := '0';
    signal sig_last_mmap_dbeat          : std_logic := '0';
    signal sig_last_mmap_dbeat_reg      : std_logic := '0';
    signal sig_data2mmap_ready          : std_logic := '0';
    signal sig_mmap2data_valid          : std_logic := '0';
    signal sig_mmap2data_last           : std_logic := '0';
    signal sig_aposted_cntr_ready       : std_logic := '0';
    signal sig_ld_new_cmd               : std_logic := '0';
    signal sig_ld_new_cmd_reg           : std_logic := '0';
    signal sig_cmd_cmplt_reg            : std_logic := '0';
    signal sig_tag_reg                  : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_lsb_reg             : std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_strt_strb_reg            : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_last_strb_reg            : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_addr_posted              : std_logic := '0';
    signal sig_addr_chan_rdy            : std_logic := '0';
    signal sig_dqual_rdy                : std_logic := '0';
    signal sig_good_mmap_dbeat          : std_logic := '0';
    signal sig_first_dbeat              : std_logic := '0';
    signal sig_last_dbeat               : std_logic := '0';
    signal sig_new_len_eq_0             : std_logic := '0';
    signal sig_dbeat_cntr               : unsigned(7 downto 0) := (others => '0');
    Signal sig_dbeat_cntr_int           : Integer range 0 to 255 := 0;
    signal sig_dbeat_cntr_eq_0          : std_logic := '0';
    signal sig_dbeat_cntr_eq_1          : std_logic := '0';
    signal sig_calc_error_reg           : std_logic := '0';
    signal sig_decerr                   : std_logic := '0';
    signal sig_slverr                   : std_logic := '0';
    signal sig_coelsc_okay_reg          : std_logic := '0';
    signal sig_coelsc_interr_reg        : std_logic := '0';
    signal sig_coelsc_decerr_reg        : std_logic := '0';
    signal sig_coelsc_slverr_reg        : std_logic := '0';
    signal sig_coelsc_cmd_cmplt_reg     : std_logic := '0';
    signal sig_coelsc_tag_reg           : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_pop_coelsc_reg           : std_logic := '0';
    signal sig_push_coelsc_reg          : std_logic := '0';
    signal sig_coelsc_reg_empty         : std_logic := '0';
    signal sig_coelsc_reg_full          : std_logic := '0';
    signal sig_rsc2data_ready           : std_logic := '0';
    signal sig_cmd_cmplt_last_dbeat     : std_logic := '0';
    signal sig_next_tag_reg             : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_next_strt_strb_reg       : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_next_last_strb_reg       : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_next_eof_reg             : std_logic := '0';
    signal sig_next_sequential_reg      : std_logic := '0';
    signal sig_next_cmd_cmplt_reg       : std_logic := '0';
    signal sig_next_calc_error_reg      : std_logic := '0';
    signal sig_next_dre_src_align_reg   : std_logic_vector(C_ALIGN_WIDTH-1 downto 0) := (others => '0');  
    signal sig_next_dre_dest_align_reg  : std_logic_vector(C_ALIGN_WIDTH-1 downto 0) := (others => '0'); 
    signal sig_pop_dqual_reg            : std_logic := '0';
    signal sig_push_dqual_reg           : std_logic := '0';
    signal sig_dqual_reg_empty          : std_logic := '0';
    signal sig_dqual_reg_full           : std_logic := '0';
    signal sig_addr_posted_cntr         : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_posted_cntr_eq_0    : std_logic := '0';
    signal sig_addr_posted_cntr_max     : std_logic := '0';
    signal sig_decr_addr_posted_cntr    : std_logic := '0';
    signal sig_incr_addr_posted_cntr    : std_logic := '0';
    signal sig_ls_addr_cntr             : unsigned(C_SEL_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_incr_ls_addr_cntr        : std_logic := '0';
    signal sig_addr_incr_unsgnd         : unsigned(C_SEL_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_no_posted_cmds           : std_logic := '0';
    Signal sig_cmd_fifo_data_in         : std_logic_vector(DCTL_FIFO_WIDTH-1 downto 0);
    Signal sig_cmd_fifo_data_out        : std_logic_vector(DCTL_FIFO_WIDTH-1 downto 0);
    signal sig_fifo_next_tag            : std_logic_vector(TAG_WIDTH-1 downto 0);
    signal sig_fifo_next_sadddr_lsb     : std_logic_vector(SADDR_LSB_WIDTH-1 downto 0);             
    signal sig_fifo_next_len            : std_logic_vector(LEN_WIDTH-1 downto 0);             
    signal sig_fifo_next_strt_strb      : std_logic_vector(STRB_WIDTH-1 downto 0);             
    signal sig_fifo_next_last_strb      : std_logic_vector(STRB_WIDTH-1 downto 0);             
    signal sig_fifo_next_drr            : std_logic := '0';
    signal sig_fifo_next_eof            : std_logic := '0';
    signal sig_fifo_next_cmd_cmplt      : std_logic := '0';
    signal sig_fifo_next_calc_error     : std_logic := '0';
    signal sig_fifo_next_sequential     : std_logic := '0';
    signal sig_fifo_next_dre_src_align  : std_logic_vector(C_ALIGN_WIDTH-1 downto 0) := (others => '0');  
    signal sig_fifo_next_dre_dest_align : std_logic_vector(C_ALIGN_WIDTH-1 downto 0) := (others => '0'); 
    signal sig_cmd_fifo_empty           : std_logic := '0';
    signal sig_fifo_wr_cmd_valid        : std_logic := '0';
    signal sig_fifo_wr_cmd_ready        : std_logic := '0';
    signal sig_fifo_rd_cmd_valid        : std_logic := '0';
    signal sig_fifo_rd_cmd_ready        : std_logic := '0';
    signal sig_sequential_push          : std_logic := '0';
    signal sig_clr_dqual_reg            : std_logic := '0';
    signal sig_advance_pipe             : std_logic := '0';
    signal sig_halt_reg                 : std_logic := '0';
    signal sig_halt_reg_dly1            : std_logic := '0';
    signal sig_halt_reg_dly2            : std_logic := '0';
    signal sig_halt_reg_dly3            : std_logic := '0';
    signal sig_data2skid_halt           : std_logic := '0';
    signal sig_rd_xfer_cmplt            : std_logic := '0';
    signal mm2s_rlast_del : std_logic; 
    
    
    
  begin --(architecture implementation)
  
    -- AXI MMap Data Channel Port assignments
--    mm2s_rready          <= '1'; --sig_data2mmap_ready;
    
    -- Read Status Block interface
    data2rsc_valid       <= mm2s_rlast_del; --sig_coelsc_reg_full      ;
    data2rsc_cmd_cmplt   <= mm2s_rlast_del;
  --  data2rsc_valid       <= sig_coelsc_reg_full      ;

    mm2s_strm_wvalid <= mm2s_rvalid;-- and sig_data2mmap_ready; 
    mm2s_strm_wlast <= mm2s_rlast; -- and sig_data2mmap_ready; 
    mm2s_strm_wstrb <= (others => '1');
    mm2s_strm_wdata <= mm2s_rdata;

-- Adding a register for rready as OVC error out during reset

    RREADY_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset          = '1' ) then
              mm2s_rready <= '0';
              
            Else
              mm2s_rready <= '1';
            end if; 
         end if;       
       end process RREADY_REG; 


    STATUS_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset          = '1' ) then
              mm2s_rlast_del <= '0';
              
            Else
              mm2s_rlast_del <= mm2s_rlast and mm2s_rvalid;
            end if; 
         end if;       
       end process STATUS_REG; 

    

    STATUS_COELESC_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset          = '1' or
               rsc2data_ready  = '0') then -- and        -- Added more qualification here for simultaneus
--                sig_push_coelsc_reg = '0')) then     -- push and pop condition per CR590244
                                                                                                
              sig_coelsc_tag_reg       <= (others => '0');
              sig_coelsc_interr_reg    <= '0';
              sig_coelsc_decerr_reg    <= '0';
              sig_coelsc_slverr_reg    <= '0';
              sig_coelsc_okay_reg      <= '1';       -- set back to default of "OKAY"
  
            Elsif (mm2s_rvalid = '1') Then
            
              sig_coelsc_tag_reg       <= sig_tag_reg;                             
              sig_coelsc_interr_reg    <= '0';
              sig_coelsc_decerr_reg    <= sig_decerr or sig_coelsc_decerr_reg;
              sig_coelsc_slverr_reg    <= sig_slverr or sig_coelsc_slverr_reg;
              sig_coelsc_okay_reg      <= not(sig_decerr       or 
                                              sig_slverr );
            else
              
              null;  -- hold current state
              
            end if; 
         end if;       
       end process STATUS_COELESC_REG; 



    sig_rsc2data_ready   <= rsc2data_ready           ;
    
    data2rsc_tag         <= sig_coelsc_tag_reg       ;
    data2rsc_calc_err    <= sig_coelsc_interr_reg    ;
    data2rsc_okay        <= sig_coelsc_okay_reg      ;
    data2rsc_decerr      <= sig_coelsc_decerr_reg    ;
    data2rsc_slverr      <= sig_coelsc_slverr_reg    ;
    
   
--                                                    
--    -- AXI MM2S Stream Channel Port assignments               
----    mm2s_strm_wvalid     <= (mm2s_rvalid             and         
----                             sig_advance_pipe)       or
----                            (sig_halt_reg            and  -- Force tvalid high on a Halt and
--  --                           sig_dqual_reg_full      and  -- a transfer is scheduled and
--    --                         not(sig_no_posted_cmds) and  -- there are cmds posted to AXi and
--      --                       not(sig_calc_error_reg));    -- not a calc error       
--    
--            
--                                                    
----    mm2s_strm_wlast      <= (mm2s_rlast              and
--  --                          sig_next_eof_reg)        or
--    --                        (sig_halt_reg            and  -- Force tvalid high on a Halt and
--      --                       sig_dqual_reg_full      and  -- a transfer is scheduled and
--        --                     not(sig_no_posted_cmds) and  -- there are cmds posted to AXi and
--          --                   not(sig_calc_error_reg));    -- not a calc error;        
--    
--    
--    -- Generate the Write Strobes for the Stream interface
----    mm2s_strm_wstrb <= (others => '1')
----      When (sig_halt_reg = '1')        -- Force tstrb high on a Halt
-- --     else sig_strt_strb_reg
--  --    When (sig_first_dbeat = '1')
--   --   Else sig_last_strb_reg
--   --   When (sig_last_dbeat = '1')
--   --   Else (others => '1');
--    
--    
--    
--    
--    
--    -- MM2S Supplimental Controls
--    mm2s_data2sf_cmd_cmplt <= (mm2s_rlast              and
--                               sig_next_cmd_cmplt_reg) or
--                              (sig_halt_reg            and  
--                               sig_dqual_reg_full      and  
--                               not(sig_no_posted_cmds) and  
--                               not(sig_calc_error_reg));    
--    
--    
--    
--    
--    
--    
--    -- Address Channel Controller synchro pulse input                  
--    sig_addr_posted      <= addr2data_addr_posted;
--                                                        
-- 
-- 
--    -- Request to halt the Address Channel Controller               
    data2skid_halt <= '0'; 
    data2all_dcntlr_halted <= '0'; 
    data2mstr_cmd_ready <= '0'; 
    mm2s_data2sf_cmd_cmplt <= '0';
    data2addr_stop_req   <= sig_halt_reg;
    data2rst_stop_cmplt  <= '0';
    mm2s_rd_xfer_cmplt   <= '0';
-- 
--    
--    -- Halted flag to the reset module                  
--    data2rst_stop_cmplt  <= (sig_halt_reg_dly3 and   -- Normal Mode shutdown
--                            sig_no_posted_cmds and 
--                            not(sig_calc_error_reg)) or
--                            (sig_halt_reg_dly3 and   -- Shutdown after error trap
--                             sig_calc_error_reg);
--    
--     
--    
--    -- Read Transfer Completed Status output
--    mm2s_rd_xfer_cmplt <=  sig_rd_xfer_cmplt;                     
--    
--    
--     
--    -- Internal logic ------------------------------
-- 
-- 
--    -------------------------------------------------------------
--    -- Synchronous Process with Sync Reset
--    --
--    -- Label: IMP_RD_CMPLT_FLAG
--    --
--    -- Process Description:
--    --   Implements the status flag indicating that a read data 
--    -- transfer has completed. This is an echo of a rlast assertion
--    -- and a qualified data beat on the AXI4 Read Data Channel 
--    -- inputs.
--    --
--    -------------------------------------------------------------
--    IMP_RD_CMPLT_FLAG : process (primary_aclk)
--      begin
--        if (primary_aclk'event and primary_aclk = '1') then
--           if (mmap_reset = '1') then
--    
--             sig_rd_xfer_cmplt <= '0';
--    
--           else
--    
--             sig_rd_xfer_cmplt <= sig_mmap2data_last and 
--                                  sig_good_mmap_dbeat;
--                                  
--           end if; 
--        end if;       
--      end process IMP_RD_CMPLT_FLAG; 
--     
--    
-- 
--  
--    
--    -- General flag for advancing the MMap Read and the Stream
--    -- data pipelines
--    sig_advance_pipe     <=  sig_addr_chan_rdy        and                                  
--                             sig_dqual_rdy            and                                  
--                             not(sig_coelsc_reg_full) and  -- new status back-pressure term
--                             not(sig_calc_error_reg);                                      
--    
--                                      
--    -- test for Kevin's status throttle case
--    sig_data2mmap_ready  <= (mm2s_strm_wready or 
--                             sig_halt_reg)    and    -- Ignore the Stream ready on a Halt request                              
--                             sig_advance_pipe;          
--    
--     
--     
--    sig_good_mmap_dbeat  <= sig_data2mmap_ready and 
--                            sig_mmap2data_valid;
--    
--    
--    sig_last_mmap_dbeat  <= sig_good_mmap_dbeat and 
--                            sig_mmap2data_last;
--     
--     
--    sig_get_next_dqual   <= sig_last_mmap_dbeat; 
--    
--    
--    
--    
--    
--    
--         
--    ------------------------------------------------------------
--    -- Instance: I_READ_MUX 
--    --
--    -- Description:
--    --  Instance of the MM2S Read Data Channel Read Mux   
--    --
--    ------------------------------------------------------------
--    I_READ_MUX : entity axi_sg_v4_1_6.axi_sg_rdmux
--    generic map (
--  
--      C_SEL_ADDR_WIDTH     =>  C_SEL_ADDR_WIDTH ,   
--      C_MMAP_DWIDTH        =>  C_MMAP_DWIDTH    ,   
--      C_STREAM_DWIDTH      =>  C_STREAM_DWIDTH      
--  
--      )
--    port map (
--  
--      mmap_read_data_in    =>  mm2s_rdata       ,   
--      mux_data_out         =>  open, --mm2s_strm_wdata  ,            
--      mstr2data_saddr_lsb  =>  sig_addr_lsb_reg     
--    
--      );
--   
--    
--    
--    
--    
--    -------------------------------------------------------------
--    -- Synchronous Process with Sync Reset
--    --
--    -- Label: REG_LAST_DBEAT
--    --
--    -- Process Description:
--    --   This implements a FLOP that creates a pulse
--    -- indicating the LAST signal for an incoming read data channel
--    -- has been received. Note that it is possible to have back to 
--    -- back LAST databeats.
--    --
--    -------------------------------------------------------------
--    REG_LAST_DBEAT : process (primary_aclk)
--       begin
--         if (primary_aclk'event and primary_aclk = '1') then
--           if (mmap_reset = '1') then
--
--             sig_last_mmap_dbeat_reg <= '0';
--             
--           else
--             
--             sig_last_mmap_dbeat_reg <= sig_last_mmap_dbeat;
--             
--           end if; 
--         end if;       
--       end process REG_LAST_DBEAT; 
--  
--    
--    
--    
--    
--    
--
--     ------------------------------------------------------------
--     -- If Generate
--     --
--     -- Label: GEN_NO_DATA_CNTL_FIFO
--     --
--     -- If Generate Description:
--     --   Omits the input data control FIFO if the requested FIFO
--     -- depth is 1. The Data Qualifier Register serves as a 
--     -- 1 deep FIFO by itself.
--     --
--     ------------------------------------------------------------
--     GEN_NO_DATA_CNTL_FIFO : if (C_DATA_CNTL_FIFO_DEPTH = 1) generate
--
--       begin
--
--         -- Command Calculator Handshake output
--         data2mstr_cmd_ready    <= sig_fifo_wr_cmd_ready;           
--        
--         sig_fifo_rd_cmd_valid  <= mstr2data_cmd_valid ;
--         
--         
--         
--         -- pre 13.1 sig_fifo_wr_cmd_ready  <= sig_dqual_reg_empty     and                                         
--         -- pre 13.1                           sig_aposted_cntr_ready  and                                         
--         -- pre 13.1                           not(rsc2mstr_halt_pipe) and  -- The Rd Status Controller is not stalling
--         -- pre 13.1                           not(sig_calc_error_reg);     -- the command execution pipe and there is  
--         -- pre 13.1                                                        -- no calculation error being propagated
--
--         sig_fifo_wr_cmd_ready  <= sig_push_dqual_reg;
--
--
--                                                                    
--         
--         sig_fifo_next_tag             <= mstr2data_tag        ;    
--         sig_fifo_next_sadddr_lsb      <= mstr2data_saddr_lsb  ;    
--         sig_fifo_next_len             <= mstr2data_len        ;    
--         sig_fifo_next_strt_strb       <= mstr2data_strt_strb  ;    
--         sig_fifo_next_last_strb       <= mstr2data_last_strb  ;    
--         sig_fifo_next_drr             <= mstr2data_drr        ;    
--         sig_fifo_next_eof             <= mstr2data_eof        ;    
--         sig_fifo_next_sequential      <= mstr2data_sequential ;    
--         sig_fifo_next_cmd_cmplt       <= mstr2data_cmd_cmplt  ;    
--         sig_fifo_next_calc_error      <= mstr2data_calc_error ; 
--            
--         sig_fifo_next_dre_src_align   <= mstr2data_dre_src_align  ; 
--         sig_fifo_next_dre_dest_align  <= mstr2data_dre_dest_align ; 
--                                                              
--             
--   
--       end generate GEN_NO_DATA_CNTL_FIFO;
--         
--    
--    
--    
--    
--      
--     ------------------------------------------------------------
--     -- If Generate
--     --
--     -- Label: GEN_DATA_CNTL_FIFO
--     --
--     -- If Generate Description:
--     --   Includes the input data control FIFO if the requested 
--     -- FIFO depth is more than 1.
--     --
--     ------------------------------------------------------------
----     GEN_DATA_CNTL_FIFO : if (C_DATA_CNTL_FIFO_DEPTH > 1) generate
----
----       begin
----
----       
----         -- Command Calculator Handshake output
----         data2mstr_cmd_ready    <= sig_fifo_wr_cmd_ready;           
----         
----         sig_fifo_wr_cmd_valid  <= mstr2data_cmd_valid  ;
----         
----
----         sig_fifo_rd_cmd_ready    <= sig_push_dqual_reg;  -- pop the fifo when dqual reg is pushed   
----                                                               
----
----         
----         
----                              
----         -- Format the input fifo data word
----         sig_cmd_fifo_data_in  <=   mstr2data_dre_dest_align &
----                                    mstr2data_dre_src_align  &
----                                    mstr2data_calc_error     &
----                                    mstr2data_cmd_cmplt      &
----                                    mstr2data_sequential     &
----                                    mstr2data_eof            &
----                                    mstr2data_drr            &
----                                    mstr2data_last_strb      &
----                                    mstr2data_strt_strb      &
----                                    mstr2data_len            &
----                                    mstr2data_saddr_lsb      &
----                                    mstr2data_tag ;
----         
----          
----         -- Rip the output fifo data word
----         sig_fifo_next_tag        <= sig_cmd_fifo_data_out((TAG_STRT_INDEX+TAG_WIDTH)-1 downto 
----                                                            TAG_STRT_INDEX);                   
----         sig_fifo_next_sadddr_lsb <= sig_cmd_fifo_data_out((SADDR_LSB_STRT_INDEX+SADDR_LSB_WIDTH)-1 downto 
----                                                            SADDR_LSB_STRT_INDEX);
----         sig_fifo_next_len        <= sig_cmd_fifo_data_out((LEN_STRT_INDEX+LEN_WIDTH)-1 downto 
----                                                            LEN_STRT_INDEX);
----         sig_fifo_next_strt_strb  <= sig_cmd_fifo_data_out((STRT_STRB_STRT_INDEX+STRB_WIDTH)-1 downto 
----                                                            STRT_STRB_STRT_INDEX);
----         sig_fifo_next_last_strb  <= sig_cmd_fifo_data_out((LAST_STRB_STRT_INDEX+STRB_WIDTH)-1 downto 
----                                                            LAST_STRB_STRT_INDEX);
----         sig_fifo_next_drr        <= sig_cmd_fifo_data_out(SOF_STRT_INDEX);
----         sig_fifo_next_eof        <= sig_cmd_fifo_data_out(EOF_STRT_INDEX);
----         sig_fifo_next_sequential <= sig_cmd_fifo_data_out(SEQUENTIAL_STRT_INDEX);
----         sig_fifo_next_cmd_cmplt  <= sig_cmd_fifo_data_out(CMD_CMPLT_STRT_INDEX);
----         sig_fifo_next_calc_error <= sig_cmd_fifo_data_out(CALC_ERR_STRT_INDEX);
----
----         sig_fifo_next_dre_src_align   <= sig_cmd_fifo_data_out((DRE_SRC_STRT_INDEX+DRE_ALIGN_WIDTH)-1 downto 
----                                                                 DRE_SRC_STRT_INDEX);
----         sig_fifo_next_dre_dest_align  <= sig_cmd_fifo_data_out((DRE_DEST_STRT_INDEX+DRE_ALIGN_WIDTH)-1 downto 
----                                                                 DRE_DEST_STRT_INDEX);
----         
----                                           
----                                           
----                                           
----         ------------------------------------------------------------
----         -- Instance: I_DATA_CNTL_FIFO 
----         --
----         -- Description:
----         -- Instance for the Command Qualifier FIFO
----         --
----         ------------------------------------------------------------
----          I_DATA_CNTL_FIFO : entity axi_sg_v4_1_6.axi_sg_fifo
----          generic map (
----        
----            C_DWIDTH             =>  DCTL_FIFO_WIDTH        ,  
----            C_DEPTH              =>  C_DATA_CNTL_FIFO_DEPTH ,  
----            C_IS_ASYNC           =>  USE_SYNC_FIFO          ,  
----            C_PRIM_TYPE          =>  FIFO_PRIM_TYPE         ,  
----            C_FAMILY             =>  C_FAMILY                  
----           
----            )
----          port map (
----            
----            -- Write Clock and reset
----            fifo_wr_reset        =>   mmap_reset            ,  
----            fifo_wr_clk          =>   primary_aclk          ,  
----            
----            -- Write Side
----            fifo_wr_tvalid       =>   sig_fifo_wr_cmd_valid ,  
----            fifo_wr_tready       =>   sig_fifo_wr_cmd_ready ,  
----            fifo_wr_tdata        =>   sig_cmd_fifo_data_in  ,  
----            fifo_wr_full         =>   open                  ,  
----           
----            -- Read Clock and reset
----            fifo_async_rd_reset  =>   mmap_reset            ,   
----            fifo_async_rd_clk    =>   primary_aclk          ,  
----            
----            -- Read Side
----            fifo_rd_tvalid       =>   sig_fifo_rd_cmd_valid ,  
----            fifo_rd_tready       =>   sig_fifo_rd_cmd_ready ,  
----            fifo_rd_tdata        =>   sig_cmd_fifo_data_out ,  
----            fifo_rd_empty        =>   sig_cmd_fifo_empty       
----           
----            );
----        
----
----       end generate GEN_DATA_CNTL_FIFO;
----         
--          
--    
--   
--  
--  
--  
--  
--  
--    -- Data Qualifier Register ------------------------------------
--    
--    sig_ld_new_cmd           <= sig_push_dqual_reg      ;
--    sig_addr_chan_rdy        <= not(sig_addr_posted_cntr_eq_0);
--    sig_dqual_rdy            <= sig_dqual_reg_full      ;
--    sig_strt_strb_reg        <= sig_next_strt_strb_reg  ;
--    sig_last_strb_reg        <= sig_next_last_strb_reg  ;
--    sig_tag_reg              <= sig_next_tag_reg        ;
--    sig_cmd_cmplt_reg        <= sig_next_cmd_cmplt_reg  ;
--    sig_calc_error_reg       <= sig_next_calc_error_reg ;
--    
--    
--    -- Flag indicating that there are no posted commands to AXI
--    sig_no_posted_cmds       <= sig_addr_posted_cntr_eq_0;
--    
--    
--    
--    -- new for no bubbles between child requests
--    sig_sequential_push      <= sig_good_mmap_dbeat and -- MMap handshake qualified
--                                sig_last_dbeat      and -- last data beat of transfer
--                                sig_next_sequential_reg;-- next queued command is sequential 
--                                                        -- to the current command
--    
--    
--    -- pre 13.1 sig_push_dqual_reg       <= (sig_sequential_push   or
--    -- pre 13.1                              sig_dqual_reg_empty)  and 
--    -- pre 13.1                             sig_fifo_rd_cmd_valid  and
--    -- pre 13.1                             sig_aposted_cntr_ready and 
--    -- pre 13.1                             not(rsc2mstr_halt_pipe);  -- The Rd Status Controller is not  
--                                                                      -- stalling the command execution pipe
--    
--    sig_push_dqual_reg       <= (sig_sequential_push    or
--                                 sig_dqual_reg_empty)   and 
--                                sig_fifo_rd_cmd_valid   and
--                                sig_aposted_cntr_ready  and 
--                                not(sig_calc_error_reg) and -- 13.1 addition => An error has not been propagated
--                                not(rsc2mstr_halt_pipe);    -- The Rd Status Controller is not  
--                                                            -- stalling the command execution pipe
--                                                        
--                                                        
--    sig_pop_dqual_reg        <= not(sig_next_calc_error_reg) and 
--                                sig_get_next_dqual and 
--                                sig_dqual_reg_full  ; 
--    
--  
--    -- new for no bubbles between child requests
--    sig_clr_dqual_reg        <=  mmap_reset         or
--                                 (sig_pop_dqual_reg and
--                                 not(sig_push_dqual_reg));
--  
--    
--  -------------------------------------------------------------
--  -- Synchronous Process with Sync Reset
--  --
--  -- Label: IMP_DQUAL_REG
--  --
--  -- Process Description:
--  --    This process implements a register for the Data 
--  -- Control and qualifiers. It operates like a 1 deep Sync FIFO.
--  --
--  -------------------------------------------------------------
--  IMP_DQUAL_REG : process (primary_aclk)
--     begin
--       if (primary_aclk'event and primary_aclk = '1') then
--          if (sig_clr_dqual_reg = '1') then
--            
--            sig_next_tag_reg             <= (others => '0');
--            sig_next_strt_strb_reg       <= (others => '0');
--            sig_next_last_strb_reg       <= (others => '0');
--            sig_next_eof_reg             <= '0';
--            sig_next_cmd_cmplt_reg       <= '0';
--            sig_next_sequential_reg      <= '0';
--            sig_next_calc_error_reg      <= '0';
--            sig_next_dre_src_align_reg   <= (others => '0');
--            sig_next_dre_dest_align_reg  <= (others => '0');
--            
--            sig_dqual_reg_empty          <= '1';
--            sig_dqual_reg_full           <= '0';
--            
--          elsif (sig_push_dqual_reg = '1') then
--            
--            sig_next_tag_reg             <= sig_fifo_next_tag            ;
--            sig_next_strt_strb_reg       <= sig_fifo_next_strt_strb      ;
--            sig_next_last_strb_reg       <= sig_fifo_next_last_strb      ;
--            sig_next_eof_reg             <= sig_fifo_next_eof            ;
--            sig_next_cmd_cmplt_reg       <= sig_fifo_next_cmd_cmplt      ;
--            sig_next_sequential_reg      <= sig_fifo_next_sequential     ;
--            sig_next_calc_error_reg      <= sig_fifo_next_calc_error     ;
--            sig_next_dre_src_align_reg   <= sig_fifo_next_dre_src_align  ;  
--            sig_next_dre_dest_align_reg  <= sig_fifo_next_dre_dest_align ;  
--            
--            sig_dqual_reg_empty          <= '0';
--            sig_dqual_reg_full           <= '1';
--            
--          else
--            null;  -- don't change state
--          end if; 
--       end if;       
--     end process IMP_DQUAL_REG; 
--     
--
--  
--  
--  
--    
--    
--    -- Address LS Cntr logic  --------------------------
--    
--    sig_addr_lsb_reg         <= STD_LOGIC_VECTOR(sig_ls_addr_cntr);
--    sig_addr_incr_unsgnd     <= TO_UNSIGNED(ADDR_INCR_VALUE, C_SEL_ADDR_WIDTH);
--    sig_incr_ls_addr_cntr    <= sig_good_mmap_dbeat;
--    
--    -------------------------------------------------------------
--    -- Synchronous Process with Sync Reset
--    --
--    -- Label: DO_ADDR_LSB_CNTR
--    --
--    -- Process Description:
--    --  Implements the LS Address Counter used for controlling
--    -- the Read Data Mux during Burst transfers
--    --
--    -------------------------------------------------------------
--    DO_ADDR_LSB_CNTR : process (primary_aclk)
--       begin
--         if (primary_aclk'event and primary_aclk = '1') then
--            if (mmap_reset         = '1'  or
--               (sig_pop_dqual_reg  = '1'  and
--                sig_push_dqual_reg = '0')) then                 -- Clear the Counter
--              
--              sig_ls_addr_cntr <= (others => '0');
--            
--            elsif (sig_push_dqual_reg = '1') then               -- Load the Counter
--              
--              sig_ls_addr_cntr <= unsigned(sig_fifo_next_sadddr_lsb);
--            
--            elsif (sig_incr_ls_addr_cntr = '1') then            -- Increment the Counter
--              
--              sig_ls_addr_cntr <= sig_ls_addr_cntr + sig_addr_incr_unsgnd;
--            
--            else
--              null;  -- Hold Current value
--            end if; 
--         end if;       
--       end process DO_ADDR_LSB_CNTR; 
--    
--    
--    
--    
--    
--    
--    
--    
--    
--    
--    
--    
--    ----- Address posted Counter logic --------------------------------
--    
--    sig_incr_addr_posted_cntr <= sig_addr_posted              ;
--    
--    
--    sig_decr_addr_posted_cntr <= sig_last_mmap_dbeat_reg      ;
--    
--    
--    sig_aposted_cntr_ready    <= not(sig_addr_posted_cntr_max);
--    
--    sig_addr_posted_cntr_eq_0 <= '1'
--      when (sig_addr_posted_cntr = ADDR_POSTED_ZERO)
--      Else '0';
--    
--    sig_addr_posted_cntr_max <= '1'
--      when (sig_addr_posted_cntr = ADDR_POSTED_MAX)
--      Else '0';
--    
--    
--    
--    
--    
--    -------------------------------------------------------------
--    -- Synchronous Process with Sync Reset
--    --
--    -- Label: IMP_ADDR_POSTED_FIFO_CNTR
--    --
--    -- Process Description:
--    --    This process implements a register for the Address 
--    -- Posted FIFO that operates like a 1 deep Sync FIFO.
--    --
--    -------------------------------------------------------------
--    IMP_ADDR_POSTED_FIFO_CNTR : process (primary_aclk)
--       begin
--         if (primary_aclk'event and primary_aclk = '1') then
--            if (mmap_reset = '1') then
--              
--              sig_addr_posted_cntr <= ADDR_POSTED_ZERO;
--              
--            elsif (sig_incr_addr_posted_cntr = '1' and
--                   sig_decr_addr_posted_cntr  = '0' and
--                   sig_addr_posted_cntr_max = '0') then
--              
--              sig_addr_posted_cntr <= sig_addr_posted_cntr + ADDR_POSTED_ONE ;
--              
--            elsif (sig_incr_addr_posted_cntr  = '0' and
--                   sig_decr_addr_posted_cntr   = '1' and
--                   sig_addr_posted_cntr_eq_0 = '0') then
--              
--              sig_addr_posted_cntr <= sig_addr_posted_cntr - ADDR_POSTED_ONE ;
--              
--            else
--              null;  -- don't change state
--            end if; 
--         end if;       
--       end process IMP_ADDR_POSTED_FIFO_CNTR; 
--     
--         
--      
--      
--      
--      
--      
--      
--    ------- First/Middle/Last Dbeat detirmination -------------------
--     
--     sig_new_len_eq_0 <= '1'
--       When  (sig_fifo_next_len = LEN_OF_ZERO)
--       else '0';
--     
--     
--      
--      
--     -------------------------------------------------------------
--     -- Synchronous Process with Sync Reset
--     --
--     -- Label: DO_FIRST_MID_LAST
--     --
--     -- Process Description:
--     --  Implements the detection of the First/Mid/Last databeat of
--     -- a transfer.
--     --
--     -------------------------------------------------------------
--     DO_FIRST_MID_LAST : process (primary_aclk)
--        begin
--          if (primary_aclk'event and primary_aclk = '1') then
--             if (mmap_reset = '1') then
--               
--               sig_first_dbeat   <= '0';
--               sig_last_dbeat    <= '0';
--             
--             elsif (sig_ld_new_cmd = '1') then
--               
--               sig_first_dbeat   <= not(sig_new_len_eq_0);
--               sig_last_dbeat    <= sig_new_len_eq_0;
--             
--             Elsif (sig_dbeat_cntr_eq_1 = '1' and
--                    sig_good_mmap_dbeat = '1') Then
--             
--               sig_first_dbeat   <= '0';
--               sig_last_dbeat    <= '1';
--             
--             Elsif (sig_dbeat_cntr_eq_0 = '0' and
--                    sig_dbeat_cntr_eq_1 = '0' and
--                    sig_good_mmap_dbeat = '1') Then
--             
--               sig_first_dbeat   <= '0';
--               sig_last_dbeat    <= '0';
--             
--             else
--               null; -- hols current state
--             end if; 
--          end if;       
--        end process DO_FIRST_MID_LAST; 
--    
--    
--   
--   
--   
--   -------  Data Controller Halted Indication ------------------------------- 
--    
-- 
--    data2all_dcntlr_halted <= sig_no_posted_cmds  and
--                              (sig_calc_error_reg or
--                               rst2data_stop_request);
-- 
-- 
--    
--    
--   -------  Data Beat counter logic ------------------------------- 
--    sig_dbeat_cntr_int  <= TO_INTEGER(sig_dbeat_cntr);
--    
--    sig_dbeat_cntr_eq_0 <= '1'
--      when (sig_dbeat_cntr_int = 0)
--      Else '0';
--    
--    sig_dbeat_cntr_eq_1 <= '1'
--      when (sig_dbeat_cntr_int = 1)
--      Else '0';
--    
--    
--    
--    
--    -------------------------------------------------------------
--    -- Synchronous Process with Sync Reset
--    --
--    -- Label: DO_DBEAT_CNTR
--    --
--    -- Process Description:
--    --
--    --
--    -------------------------------------------------------------
--    DO_DBEAT_CNTR : process (primary_aclk)
--       begin
--         if (primary_aclk'event and primary_aclk = '1') then
--            if (mmap_reset = '1') then
--              sig_dbeat_cntr <= (others => '0');
--            elsif (sig_ld_new_cmd = '1') then
--              sig_dbeat_cntr <= unsigned(sig_fifo_next_len);
--            Elsif (sig_good_mmap_dbeat = '1' and
--                   sig_dbeat_cntr_eq_0 = '0') Then
--              sig_dbeat_cntr <= sig_dbeat_cntr-1;
--            else
--              null; -- Hold current state
--            end if; 
--         end if;       
--       end process DO_DBEAT_CNTR; 
--      
--  
-- 
-- 
--  
--  
--   ------  Read Response Status Logic  ------------------------------
--  
--   -------------------------------------------------------------
--   -- Synchronous Process with Sync Reset
--   --
--   -- Label: LD_NEW_CMD_PULSE
--   --
--   -- Process Description:
--   -- Generate a 1 Clock wide pulse when a new command has been
--   -- loaded into the Command Register
--   --
--   -------------------------------------------------------------
--   LD_NEW_CMD_PULSE : process (primary_aclk)
--      begin
--        if (primary_aclk'event and primary_aclk = '1') then
--           if (mmap_reset         = '1' or
--               sig_ld_new_cmd_reg = '1') then
--             sig_ld_new_cmd_reg <= '0';
--           elsif (sig_ld_new_cmd = '1') then
--             sig_ld_new_cmd_reg <= '1';
--           else
--             null; -- hold State
--           end if; 
--        end if;       
--      end process LD_NEW_CMD_PULSE; 
--  
--                               
--                               
--   sig_pop_coelsc_reg  <= sig_coelsc_reg_full and
--                          sig_rsc2data_ready ; 
--                          
--   sig_push_coelsc_reg <= (sig_good_mmap_dbeat and  
--                           not(sig_coelsc_reg_full)) or
--                          (sig_ld_new_cmd_reg and 
--                           sig_calc_error_reg) ; 
--   
--   sig_cmd_cmplt_last_dbeat <= (sig_cmd_cmplt_reg and sig_mmap2data_last) or
--                                sig_calc_error_reg;
--   
--   
--      
  -------  Read Response Decode   
    
   -- Decode the AXI MMap Read Response       
   sig_decerr  <= '1'
     When mm2s_rresp = DECERR
     Else '0'; 
          
   sig_slverr  <= '1'
     When mm2s_rresp = SLVERR
     Else '0';      
   
--   
--   
--          
--          
--    -------------------------------------------------------------
--    -- Synchronous Process with Sync Reset
--    --
--    -- Label: RD_RESP_COELESC_REG
--    --
--    -- Process Description:
--    --   Implement the Read error/status coelescing register. 
--    -- Once a bit is set it will remain set until the overall 
--    -- status is written to the Status Controller. 
--    -- Tag bits are just registered at each valid dbeat.
--    --
--    -------------------------------------------------------------
----    STATUS_COELESC_REG : process (primary_aclk)
----       begin
----         if (primary_aclk'event and primary_aclk = '1') then
----            if (mmap_reset          = '1' or
----               (sig_pop_coelsc_reg  = '1' and        -- Added more qualification here for simultaneus
----                sig_push_coelsc_reg = '0')) then     -- push and pop condition per CR590244
----                                                                                                
----              sig_coelsc_tag_reg       <= (others => '0');
----              sig_coelsc_cmd_cmplt_reg <= '0';
----              sig_coelsc_interr_reg    <= '0';
----              sig_coelsc_decerr_reg    <= '0';
----              sig_coelsc_slverr_reg    <= '0';
----              sig_coelsc_okay_reg      <= '1';       -- set back to default of "OKAY"
----  
----              sig_coelsc_reg_full      <= '0';
----              sig_coelsc_reg_empty     <= '1';
----  
----  
----              
----            Elsif (sig_push_coelsc_reg = '1') Then
----            
----              sig_coelsc_tag_reg       <= sig_tag_reg;                             
----              sig_coelsc_cmd_cmplt_reg <= sig_cmd_cmplt_last_dbeat;                  
----              sig_coelsc_interr_reg    <= sig_calc_error_reg or 
----                                          sig_coelsc_interr_reg;
----              sig_coelsc_decerr_reg    <= sig_decerr or sig_coelsc_decerr_reg;
----              sig_coelsc_slverr_reg    <= sig_slverr or sig_coelsc_slverr_reg;
----              sig_coelsc_okay_reg      <= not(sig_decerr       or 
----                                              sig_slverr       or 
----                                              sig_calc_error_reg );
----              
----              sig_coelsc_reg_full      <= sig_cmd_cmplt_last_dbeat;
----              sig_coelsc_reg_empty     <= not(sig_cmd_cmplt_last_dbeat);
----                                            
----              
----            else
----              
----              null;  -- hold current state
----              
----            end if; 
----         end if;       
----       end process STATUS_COELESC_REG; 
--   
--   
--   
--   
--   
--   
--   
--   
--   
--    
--    ------------------------------------------------------------
--    -- If Generate
--    --
--    -- Label: GEN_NO_DRE
--    --
--    -- If Generate Description:
--    --  Ties off DRE Control signals to logic low when DRE is
--    -- omitted from the MM2S functionality.
--    --
--    --
--    ------------------------------------------------------------
--    GEN_NO_DRE : if (C_INCLUDE_DRE = 0) generate
--    
--       begin
--    
         mm2s_dre_new_align     <= '0';
         mm2s_dre_use_autodest  <= '0';
         mm2s_dre_src_align     <= (others => '0');
         mm2s_dre_dest_align    <= (others => '0');
         mm2s_dre_flush         <= '0';
--        
--       end generate GEN_NO_DRE;
--   
--    
--    
--    
--    
--    
--    
--    
--    
--    
--    
--    
--    
--    ------------------------------------------------------------
--    -- If Generate
--    --
--    -- Label: GEN_INCLUDE_DRE_CNTLS
--    --
--    -- If Generate Description:
--    -- Implements the DRE Control logic when MM2S DRE is enabled.
--    --
--    --  - The DRE needs to have forced alignment at a SOF assertion
--    --
--    --
--    ------------------------------------------------------------
--    GEN_INCLUDE_DRE_CNTLS : if (C_INCLUDE_DRE = 1) generate
--    
--       -- local signals
--       signal lsig_s_h_dre_autodest  : std_logic := '0';
--       signal lsig_s_h_dre_new_align : std_logic := '0';
--    
--       begin
--    
--        
--         mm2s_dre_new_align     <= lsig_s_h_dre_new_align;
--         
--         
--         
--         
--         -- Autodest is asserted on a new parent command and the 
--         -- previous parent command was not delimited with a EOF
--         mm2s_dre_use_autodest  <= lsig_s_h_dre_autodest;
--         
--         
--         
--         
--         -- Assign the DRE Source and Destination Alignments
--         -- Only used when mm2s_dre_new_align is asserted 
--         mm2s_dre_src_align     <= sig_next_dre_src_align_reg ;
--         mm2s_dre_dest_align    <= sig_next_dre_dest_align_reg;
--         
--         
--         -- Assert the Flush flag when the MMap Tlast input of the current transfer is
--         -- asserted and the next transfer is not sequential and not the last 
--         -- transfer of a packet.
--         mm2s_dre_flush         <= mm2s_rlast and
--                                   not(sig_next_sequential_reg) and   
--                                   not(sig_next_eof_reg);
--        
--        
--        
--         
--         
--         -------------------------------------------------------------
--         -- Synchronous Process with Sync Reset
--         --
--         -- Label: IMP_S_H_NEW_ALIGN
--         --
--         -- Process Description:
--         --   Generates the new alignment command flag to the DRE.
--         --
--         -------------------------------------------------------------
--         IMP_S_H_NEW_ALIGN : process (primary_aclk)
--            begin
--              if (primary_aclk'event and primary_aclk = '1') then
--                 if (mmap_reset   = '1') then
--                   
--                   lsig_s_h_dre_new_align <= '0';
--                 
--                 
--                 Elsif (sig_push_dqual_reg = '1' and
--                        sig_fifo_next_drr  = '1') Then
--                 
--                   lsig_s_h_dre_new_align <= '1';
--                 
--                 elsif (sig_pop_dqual_reg = '1') then
--                   
--                   lsig_s_h_dre_new_align <=  sig_next_cmd_cmplt_reg and
--                                             not(sig_next_sequential_reg) and
--                                             not(sig_next_eof_reg);
--                 
--                 Elsif (sig_good_mmap_dbeat = '1') Then
--                 
--                   lsig_s_h_dre_new_align <= '0';
--                 
--                 
--                 else
--                   
--                   null; -- hold current state
--                 
--                 end if; 
--              end if;       
--            end process IMP_S_H_NEW_ALIGN; 
--        
--        
--        
--         
--         
--        
--         -------------------------------------------------------------
--         -- Synchronous Process with Sync Reset
--         --
--         -- Label: IMP_S_H_AUTODEST
--         --
--         -- Process Description:
--         --   Generates the control for the DRE indicating whether the
--         -- DRE destination alignment should be derived from the write
--         -- strobe stat of the last completed data-beat to the AXI 
--         -- stream output.
--         --
--         -------------------------------------------------------------
--         IMP_S_H_AUTODEST : process (primary_aclk)
--            begin
--              if (primary_aclk'event and primary_aclk = '1') then
--                 if (mmap_reset   = '1') then
--                   
--                   lsig_s_h_dre_autodest <= '0';
--                 
--                 
--                 Elsif (sig_push_dqual_reg = '1' and
--                        sig_fifo_next_drr  = '1') Then
--                   
--                   lsig_s_h_dre_autodest <= '0';
--                 
--                 elsif (sig_pop_dqual_reg = '1') then
--                   
--                   lsig_s_h_dre_autodest <=  sig_next_cmd_cmplt_reg and
--                                             not(sig_next_sequential_reg) and
--                                             not(sig_next_eof_reg);
--                 
--                 Elsif (lsig_s_h_dre_new_align = '1' and
--                        sig_good_mmap_dbeat    = '1') Then
--                 
--                   lsig_s_h_dre_autodest <= '0';
--                 
--                 
--                 else
--                   
--                   null; -- hold current state
--                 
--                 end if; 
--              end if;       
--            end process IMP_S_H_AUTODEST; 
--        
--        
--        
--        
--       end generate GEN_INCLUDE_DRE_CNTLS;
--   
--   
--   
--   
--   
--   
--    
--    
--    
--    
--    
--    
--    
--    
--    
--    
--    
--    
--    
--    
--   -------  Soft Shutdown Logic ------------------------------- 
--    
--    
--    -- Assign the output port skid buf control
--    data2skid_halt      <= sig_data2skid_halt;
--    
--    -- Create a 1 clock wide pulse to tell the output
--    -- stream skid buffer to shut down its outputs
--    sig_data2skid_halt  <=  sig_halt_reg_dly2 and 
--                            not(sig_halt_reg_dly3);
--    
--    
--    
--    -------------------------------------------------------------
--    -- Synchronous Process with Sync Reset
--    --
--    -- Label: IMP_HALT_REQ_REG
--    --
--    -- Process Description:
--    --   Implements the flop for capturing the Halt request from 
--    -- the Reset module.
--    --
--    -------------------------------------------------------------
    IMP_HALT_REQ_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              
              sig_halt_reg      <= '0';
            
            elsif (rst2data_stop_request = '1') then
              
              sig_halt_reg <= '1';
            
            else
              null;  -- Hold current State
            end if; 
         end if;       
       end process IMP_HALT_REQ_REG; 
--  
--  
--   
--   
--    -------------------------------------------------------------
--    -- Synchronous Process with Sync Reset
--    --
--    -- Label: IMP_HALT_REQ_REG_DLY
--    --
--    -- Process Description:
--    --   Implements the flops for delaying the halt request by 3
--    -- clocks to allow the Address Controller to halt before the
--    -- Data Contoller can safely indicate it has exhausted all
--    -- transfers committed to the AXI Address Channel by the Address
--    -- Controller.
--    --
--    -------------------------------------------------------------
--    IMP_HALT_REQ_REG_DLY : process (primary_aclk)
--       begin
--         if (primary_aclk'event and primary_aclk = '1') then
--            if (mmap_reset = '1') then
--              
--              sig_halt_reg_dly1 <= '0';
--              sig_halt_reg_dly2 <= '0';
--              sig_halt_reg_dly3 <= '0';
--            
--            else
--              
--              sig_halt_reg_dly1 <= sig_halt_reg;
--              sig_halt_reg_dly2 <= sig_halt_reg_dly1;
--              sig_halt_reg_dly3 <= sig_halt_reg_dly2;
--            
--            end if; 
--         end if;       
--       end process IMP_HALT_REQ_REG_DLY; 
--  
--  
--   
--   
--   
--   
--   
--   
--   
   
   
   
      
      
  end implementation;


-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_sg_rd_status_cntl.vhd
  --
  -- Description:     
  --    This file implements the DataMover Master Read Status Controller.                 
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  
  -------------------------------------------------------------------------------
  
  entity axi_sg_rd_status_cntl is
    generic (
      
      C_STS_WIDTH          : Integer               := 8;
        -- sets the width of the Status ports
      
      C_TAG_WIDTH          : Integer range  1 to 8 := 4
        -- Sets the width of the Tag field in the Status reply
      
      );
    port (
      
      -- Clock and Reset input --------------------------------------
                                                                   --
      primary_aclk           : in  std_logic;                      --
         -- Primary synchronization clock for the Master side      --
         -- interface and internal logic. It is also used          --
         -- for the User interface synchronization when            --
         -- C_STSCMD_IS_ASYNC = 0.                                 --
                                                                   --
      -- Reset input                                               --
      mmap_reset             : in  std_logic;                      --
         -- Reset used for the internal master logic               --
      ---------------------------------------------------------------
      
      
      
                
                
      -- Command Calculator Status Interface  ---------------------------
                                                                       --
      calc2rsc_calc_error    : in std_logic ;                          --
         -- Indication from the Command Calculator that a calculation  --
         -- error has occured.                                         --
      -------------------------------------------------------------------
     
     
     
      
        
      -- Address Controller Status Interface ----------------------------
                                                                       --
      addr2rsc_calc_error    : In std_logic ;                          --
         -- Indication from the Data Channel Controller FIFO that it   --
         -- is empty (no commands pending)                             --
                                                                       --
      addr2rsc_fifo_empty    : In std_logic ;                          --
         -- Indication from the Address Controller FIFO that it        --
         -- is empty (no commands pending)                             --
      -------------------------------------------------------------------

                    
                    
                    
      --  Data Controller Status Interface ---------------------------------------------
                                                                                      --
      data2rsc_tag           : In std_logic_vector(C_TAG_WIDTH-1 downto 0);           --
         -- The command tag                                                           --
                                                                                      --
      data2rsc_calc_error    : In std_logic ;                                         --
         -- Indication from the Data Channel Controller FIFO that it                  --
         -- is empty (no commands pending)                                            --
                                                                                      --
      data2rsc_okay          : In std_logic ;                                         --
         -- Indication that the AXI Read transfer completed with OK status            --
                                                                                      --
      data2rsc_decerr        : In std_logic ;                                         --
         -- Indication that the AXI Read transfer completed with decode error status  --
                                                                                      --
      data2rsc_slverr        : In std_logic ;                                         --
         -- Indication that the AXI Read transfer completed with slave error status   --
                                                                                      --
      data2rsc_cmd_cmplt     : In std_logic ;                                         --
         -- Indication by the Data Channel Controller that the                        --
         -- corresponding status is the last status for a parent command              --
         -- pulled from the command FIFO                                              --
                                                                                      --
      rsc2data_ready         : Out  std_logic;                                        --
         -- Handshake bit from the Read Status Controller Module indicating           --
         -- that the it is ready to accept a new Read status transfer                 --
                                                                                      --
      data2rsc_valid         : in  std_logic ;                                        --
         -- Handshake bit output to the Read Status Controller Module                 --
         -- indicating that the Data Controller has valid tag and status              --
         -- indicators to transfer                                                    --
      ----------------------------------------------------------------------------------


      
      -- Command/Status Module Interface ----------------------------------------------
                                                                                     --
      rsc2stat_status        : Out std_logic_vector(C_STS_WIDTH-1 downto 0);         --
         -- Read Status value collected during a Read Data transfer                  --
         -- Output to the Command/Status Module                                      --
                                                                                     --
      stat2rsc_status_ready  : In  std_logic;                                        --
         -- Input from the Command/Status Module indicating that the                 --
         -- Status Reg/FIFO is ready to accept a transfer                            --
                                                                                     --
      rsc2stat_status_valid  : Out std_logic ;                                       --
         -- Control Signal to the Status Reg/FIFO indicating a new status            --
         -- output value is valid and ready for transfer                             --
      ---------------------------------------------------------------------------------

  
    
    
      -- Address and Data Controller Pipe halt ----------------------------------
                                                                               --
      rsc2mstr_halt_pipe     : Out std_logic                                   --
         -- Indication to Halt the Data and Address Command pipeline due       --
         -- to the Status FIFO going full or an internal error being logged    --
      ---------------------------------------------------------------------------

  
      );
  
  end entity axi_sg_rd_status_cntl;
  
  
  architecture implementation of axi_sg_rd_status_cntl is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    -- Constant Declarations  --------------------------------------------
    
    Constant OKAY               : std_logic_vector(1 downto 0) := "00";
    Constant EXOKAY             : std_logic_vector(1 downto 0) := "01";
    Constant SLVERR             : std_logic_vector(1 downto 0) := "10";
    Constant DECERR             : std_logic_vector(1 downto 0) := "11";
    Constant STAT_RSVD          : std_logic_vector(3 downto 0) := "0000";
    Constant TAG_WIDTH          : integer := C_TAG_WIDTH;
    Constant STAT_REG_TAG_WIDTH : integer := 4;
    
    
    -- Signal Declarations  --------------------------------------------
    
    signal sig_tag2status            : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_rsc2status_valid      : std_logic := '0';
    signal sig_rsc2data_ready        : std_logic := '0';
    signal sig_rd_sts_okay_reg       : std_logic := '0';
    signal sig_rd_sts_interr_reg     : std_logic := '0';
    signal sig_rd_sts_decerr_reg     : std_logic := '0';
    signal sig_rd_sts_slverr_reg     : std_logic := '0';
    signal sig_rd_sts_tag_reg        : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_pop_rd_sts_reg        : std_logic := '0';
    signal sig_push_rd_sts_reg       : std_logic := '0';
    Signal sig_rd_sts_push_ok        : std_logic := '0';
    signal sig_rd_sts_reg_empty      : std_logic := '0';
    signal sig_rd_sts_reg_full       : std_logic := '0';
    
    
    
    
    
            
  begin --(architecture implementation)
  
    -- Assign the status write output control
    rsc2stat_status_valid  <= sig_rsc2status_valid ;
    
    sig_rsc2status_valid   <= sig_rd_sts_reg_full;
    
                                           
    -- Formulate the status outout value (assumes an 8-bit status width)
    rsc2stat_status        <=  sig_rd_sts_okay_reg    &   
                               sig_rd_sts_slverr_reg  &
                               sig_rd_sts_decerr_reg  & 
                               sig_rd_sts_interr_reg  &
                               sig_tag2status;
    
    -- Detect that a push of a new status word is completing
    sig_rd_sts_push_ok   <= sig_rsc2status_valid and 
                            stat2rsc_status_ready;
    
    -- Signal a halt to the execution pipe if new status
    -- is valid but the Status FIFO is not accepting it
    rsc2mstr_halt_pipe   <=  sig_rsc2status_valid and
                             (not(stat2rsc_status_ready) ); 
   
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_TAG_LE_STAT
    --
    -- If Generate Description:
    -- Populates the TAG bits into the availble Status bits when
    -- the TAG width is less than or equal to the available number
    -- of bits in the Status word. 
    --
    ------------------------------------------------------------
    GEN_TAG_LE_STAT : if (TAG_WIDTH <= STAT_REG_TAG_WIDTH) generate
    
       -- local signals
         signal lsig_temp_tag_small : std_logic_vector(STAT_REG_TAG_WIDTH-1 downto 0) := (others => '0');
         
         
       begin
    
         sig_tag2status <= lsig_temp_tag_small;
         
         
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: POPULATE_SMALL_TAG
         --
         -- Process Description:
         --
         --
         -------------------------------------------------------------
         POPULATE_SMALL_TAG : process (sig_rd_sts_tag_reg)
            begin
         
              -- Set default value
              lsig_temp_tag_small <= (others => '0');
          
              -- Now overload actual TAG bits
              lsig_temp_tag_small(TAG_WIDTH-1 downto 0) <= sig_rd_sts_tag_reg;
          
         
            end process POPULATE_SMALL_TAG; 
         
         
       end generate GEN_TAG_LE_STAT;
     
     
     
     
     
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_TAG_GT_STAT
    --
    -- If Generate Description:
    -- Populates the TAG bits into the availble Status bits when
    -- the TAG width is greater than the available number of 
    -- bits in the Status word. The upper bits of the TAG are 
    -- clipped off (discarded). 
    --
    ------------------------------------------------------------
    GEN_TAG_GT_STAT : if (TAG_WIDTH > STAT_REG_TAG_WIDTH) generate
    
       -- local signals
         signal lsig_temp_tag_big : std_logic_vector(STAT_REG_TAG_WIDTH-1 downto 0);
         
         
       begin
    
         
         sig_tag2status <= lsig_temp_tag_big;
         
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: POPULATE_BIG_TAG
         --
         -- Process Description:
         --
         --
         -------------------------------------------------------------
         POPULATE_SMALL_TAG : process (sig_rd_sts_tag_reg)
            begin
         
              -- Set default value
              lsig_temp_tag_big <= (others => '0');
          
              -- Now overload actual TAG bits
              lsig_temp_tag_big <= sig_rd_sts_tag_reg(STAT_REG_TAG_WIDTH-1 downto 0);
          
         
            end process POPULATE_SMALL_TAG; 
         
         
       end generate GEN_TAG_GT_STAT;
     
     
     
     
     
          
   ------- Read Status Collection Logic --------------------------------       
    
    rsc2data_ready      <=  sig_rsc2data_ready ;
    
    sig_rsc2data_ready  <= sig_rd_sts_reg_empty;
    
    
    sig_push_rd_sts_reg <= data2rsc_valid and
                           sig_rsc2data_ready;
          
    sig_pop_rd_sts_reg  <= sig_rd_sts_push_ok;
    
    
          
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: RD_STATUS_FIFO_REG
    --
    -- Process Description:
    --   Implement Read status FIFO register. 
    -- This register holds the Read status from the Data Controller
    -- until it is transfered to the Status FIFO.
    --
    -------------------------------------------------------------
    RD_STATUS_FIFO_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset         = '1' or
                sig_pop_rd_sts_reg = '1') then
              
              sig_rd_sts_tag_reg       <= (others => '0');
              sig_rd_sts_interr_reg    <= '0';
              sig_rd_sts_decerr_reg    <= '0';
              sig_rd_sts_slverr_reg    <= '0';
              sig_rd_sts_okay_reg      <= '1'; -- set back to default of "OKAY"
  
              sig_rd_sts_reg_full      <= '0';
              sig_rd_sts_reg_empty     <= '1';
  
  
              
            Elsif (sig_push_rd_sts_reg = '1') Then
            
              sig_rd_sts_tag_reg       <= data2rsc_tag;                             
              sig_rd_sts_interr_reg    <= data2rsc_calc_error or 
                                          sig_rd_sts_interr_reg;
              sig_rd_sts_decerr_reg    <= data2rsc_decerr or sig_rd_sts_decerr_reg;
              sig_rd_sts_slverr_reg    <= data2rsc_slverr or sig_rd_sts_slverr_reg;
              sig_rd_sts_okay_reg      <= data2rsc_okay and 
                                          not(data2rsc_decerr          or 
                                              sig_rd_sts_decerr_reg    or
                                              data2rsc_slverr          or 
                                              sig_rd_sts_slverr_reg    or
                                              data2rsc_calc_error      or
                                              sig_rd_sts_interr_reg      
                                              );
              
              sig_rd_sts_reg_full      <= data2rsc_cmd_cmplt or
                                          data2rsc_calc_error;
              sig_rd_sts_reg_empty     <= not(data2rsc_cmd_cmplt or
                                              data2rsc_calc_error);
                                            
            else
              
              null;  -- hold current state
              
            end if; 
         end if;       
       end process RD_STATUS_FIFO_REG; 
      
    
    
    
    
          
          
          
  
  end implementation;


-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_sg_scc.vhd
  --
  -- Description:     
  --    This file implements the DataMover Lite Master Simple Command Calculator (SCC).                 
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  
  
  -------------------------------------------------------------------------------
  
  entity axi_sg_scc is
    generic (
      
      C_SEL_ADDR_WIDTH     : Integer range  1 to  8 :=  5;
        -- Sets the width of the LS address bus used for 
        -- Muxing/Demuxing data to/from a wider AXI4 data bus
      
      C_ADDR_WIDTH         : Integer range 32 to 64 := 32;
        -- Sets the width of the AXi Address Channel 
      
      C_STREAM_DWIDTH      : Integer range  8 to 64 := 32;
        -- Sets the width of the Native Data width that
        -- is being supported by the PCC
      
      C_MAX_BURST_LEN      : Integer range 16 to 64 := 16;
        -- Indicates the max allowed burst length to use for
        -- AXI4 transfer calculations
      
      C_CMD_WIDTH          : Integer                := 68;
        -- Sets the width of the input command port
      
      C_TAG_WIDTH          : Integer range  1 to  8 := 4;
        -- Sets the width of the Tag field in the input command

      C_ENABLE_EXTRA_FIELD   : Integer range 0 to 1 := 1
      
      );
    port (
      
      -- Clock and Reset inputs -------------------------------------
      primary_aclk         : in  std_logic;                        --
         -- Primary synchronization clock for the Master side      --
         -- interface and internal logic. It is also used          --
         -- for the User interface synchronization when            --
         -- C_STSCMD_IS_ASYNC = 0.                                 --
                                                                   --
      -- Reset input                                               --
      mmap_reset           : in  std_logic;                        --
        -- Reset used for the internal master logic                --
      ---------------------------------------------------------------
      
      
     
      -- Command Input Interface ---------------------------------------------------------
                                                                                        --
      cmd2mstr_command      : in std_logic_vector(C_CMD_WIDTH-1 downto 0);              --
         -- The next command value available from the Command FIFO/Register             --
                                                                                        --
      cache2mstr_command      : in std_logic_vector(7 downto 0);              --
         -- The next command value available from the Command FIFO/Register             --
                                                                                        --
      cmd2mstr_cmd_valid    : in std_logic;                                             --
         -- Handshake bit indicating if the Command FIFO/Register has at leasdt 1 entry --
                                                                                        --
      mst2cmd_cmd_ready     : out  std_logic;                                           --
         -- Handshake bit indicating the Command Calculator is ready to accept          --
         -- another command                                                             --
      ------------------------------------------------------------------------------------
      
      
      
      -- Address Channel Controller Interface --------------------------------------------
                                                                                        --
      mstr2addr_tag       : out std_logic_vector(C_TAG_WIDTH-1 downto 0);               --
         -- The next command tag                                                        --
                                                                                        --
      mstr2addr_addr      : out std_logic_vector(C_ADDR_WIDTH-1 downto 0);              --
         -- The next command address to put on the AXI MMap ADDR                        --
                                                                                        --
      mstr2addr_len       : out std_logic_vector(7 downto 0);                           --
         -- The next command length to put on the AXI MMap LEN                          --
                                                                                        --
      mstr2addr_size      : out std_logic_vector(2 downto 0);                           --
         -- The next command size to put on the AXI MMap SIZE                           --
                                                                                        --
      mstr2addr_burst     : out std_logic_vector(1 downto 0);                           --
         -- The next command burst type to put on the AXI MMap BURST                    --
                                                                                        --
      mstr2addr_cache     : out std_logic_vector(3 downto 0);                           --
         -- The next command burst type to put on the AXI MMap BURST                    --
                                                                                        --
      mstr2addr_user      : out std_logic_vector(3 downto 0);                           --
         -- The next command burst type to put on the AXI MMap BURST                    --
                                                                                        --
      mstr2addr_cmd_cmplt : out std_logic;                                              --
         -- The indication to the Address Channel that the current                      --
         -- sub-command output is the last one compiled from the                        --
         -- parent command pulled from the Command FIFO                                 --
                                                                                        --
      mstr2addr_calc_error : out std_logic;                                             --
         -- Indication if the next command in the calculation pipe                      --
         -- has a calcualtion error                                                     --
                                                                                        --
      mstr2addr_cmd_valid : out std_logic;                                              --
         -- The next command valid indication to the Address Channel                    --
         -- Controller for the AXI MMap                                                 --
                                                                                        --
      addr2mstr_cmd_ready : In std_logic;                                               --
         -- Indication from the Address Channel Controller that the                     --
         -- command is being accepted                                                   --
      ------------------------------------------------------------------------------------
      
      
      
      -- Data Channel Controller Interface  ----------------------------------------------
                                                                                        --
      mstr2data_tag        : out std_logic_vector(C_TAG_WIDTH-1 downto 0);              --
         -- The next command tag                                                        --
                                                                                        --
      mstr2data_saddr_lsb  : out std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0);         --
         -- The next command start address LSbs to use for the read data                --
         -- mux (only used if Stream data width is 8 or 16 bits).                       --
                                                                                        --
      mstr2data_len        : out std_logic_vector(7 downto 0);                          --
         -- The LEN value output to the Address Channel                                 --
                                                                                        --
      mstr2data_strt_strb  : out std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);      --
         -- The starting strobe value to use for the data transfer                      --
                                                                                        --
      mstr2data_last_strb  : out std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);      --
         -- The endiing (LAST) strobe value to use for the data transfer                --
                                                                                        --
      mstr2data_sof        : out std_logic;                                             --
         -- The starting tranfer of a sequence of transfers                             --
                                                                                        --
      mstr2data_eof        : out std_logic;                                             --
         -- The endiing tranfer of a sequence of parent transfer commands               --
                                                                                        --
      mstr2data_calc_error : out std_logic;                                             --
         -- Indication if the next command in the calculation pipe                      --
         -- has a calculation error                                                     --
                                                                                        --
      mstr2data_cmd_cmplt  : out std_logic;                                             --
         -- The indication to the Data Channel that the current                         --
         -- sub-command output is the last one compiled from the                        --
         -- parent command pulled from the Command FIFO                                 --
                                                                                        --
      mstr2data_cmd_valid  : out std_logic;                                             --
         -- The next command valid indication to the Data Channel                       --
         -- Controller for the AXI MMap                                                 --
                                                                                        --
      data2mstr_cmd_ready  : In std_logic ;                                             --
         -- Indication from the Data Channel Controller that the                        --
         -- command is being accepted on the AXI Address                                --
         -- Channel                                                                     --
                                                                                        --
      calc_error           : Out std_logic                                              --
         -- Indication from the Command Calculator that a calculation                   --
         -- error has occured.                                                          --
      ------------------------------------------------------------------------------------
     
      );
  
  end entity axi_sg_scc;
  
  
  architecture implementation of axi_sg_scc is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    
  
  
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_get_slice_width
    --
    -- Function Description:
    -- Calculates the bits to rip from the Command BTT field to calculate
    -- the LEN value output to the AXI Address Channel.
    --
    -------------------------------------------------------------------
    function funct_get_slice_width (max_burst_len : integer) return integer is
                                    
    
      Variable temp_slice_width : Integer := 0;
    
    begin
  
      case max_burst_len is
       -- coverage off 
        when 64 =>
          temp_slice_width := 7;
        when 32 =>
          temp_slice_width := 6;
        when others =>   -- assume 16 dbeats is max LEN
          temp_slice_width := 5;
       -- coverage on
      end case;
      
      Return (temp_slice_width);
     
    end function funct_get_slice_width;
    
    
    
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_get_residue_width
    --
    -- Function Description:
    --  Calculates the number of Least significant bits of the BTT field
    -- that are unused for the LEN calculation
    --
    -------------------------------------------------------------------
    function funct_get_btt_ls_unused (transfer_width : integer) return integer is
    
      Variable temp_btt_ls_unused : Integer := 0; -- 8-bit stream
      
    begin
    
      case transfer_width is
       -- coverage off
        when 64 =>
            temp_btt_ls_unused := 3;
       -- coverage on
        when 32 =>
            temp_btt_ls_unused := 2;
       -- coverage off
        when 16 =>
            temp_btt_ls_unused := 1;
        when others =>  -- assume 8-bit transfers
            temp_btt_ls_unused := 0;
       -- coverage on
      end case;
      
      Return (temp_btt_ls_unused);
     
    end function funct_get_btt_ls_unused;
    
    
    
    
    
    
    
    
    
    -- Constant Declarations  ----------------------------------------
    
    Constant BASE_CMD_WIDTH      : integer := 32; -- Bit Width of Command LS (no address)
    Constant CMD_TYPE_INDEX      : integer := 23;
    Constant CMD_ADDR_LS_INDEX   : integer := BASE_CMD_WIDTH;
    Constant CMD_ADDR_MS_INDEX   : integer := (C_ADDR_WIDTH+BASE_CMD_WIDTH)-1;
    Constant CMD_TAG_WIDTH       : integer := C_TAG_WIDTH;
    Constant CMD_TAG_LS_INDEX    : integer := C_ADDR_WIDTH+BASE_CMD_WIDTH;
    Constant CMD_TAG_MS_INDEX    : integer := (CMD_TAG_LS_INDEX+CMD_TAG_WIDTH)-1;
    Constant AXI_BURST_FIXED     : std_logic_vector(1 downto 0) := "00";
    Constant AXI_BURST_INCR      : std_logic_vector(1 downto 0) := "01";
    Constant AXI_BURST_WRAP      : std_logic_vector(1 downto 0) := "10";
    Constant AXI_BURST_RESVD     : std_logic_vector(1 downto 0) := "11";
    Constant AXI_SIZE_1BYTE      : std_logic_vector(2 downto 0) := "000"; 
    Constant AXI_SIZE_2BYTE      : std_logic_vector(2 downto 0) := "001"; 
    Constant AXI_SIZE_4BYTE      : std_logic_vector(2 downto 0) := "010"; 
    Constant AXI_SIZE_8BYTE      : std_logic_vector(2 downto 0) := "011"; 
    Constant AXI_SIZE_16BYTE     : std_logic_vector(2 downto 0) := "100"; 
    Constant AXI_SIZE_32BYTE     : std_logic_vector(2 downto 0) := "101"; 
    Constant AXI_SIZE_64BYTE     : std_logic_vector(2 downto 0) := "110"; 
    Constant AXI_SIZE_128BYTE    : std_logic_vector(2 downto 0) := "111"; 
    Constant BTT_SLICE_SIZE      : integer := funct_get_slice_width(C_MAX_BURST_LEN);
    Constant MAX_BURST_LEN_US    : unsigned(BTT_SLICE_SIZE-1 downto 0) := 
                                   TO_UNSIGNED(C_MAX_BURST_LEN-1, BTT_SLICE_SIZE);
    Constant BTT_LS_UNUSED_WIDTH : integer := funct_get_btt_ls_unused(C_STREAM_DWIDTH);
    Constant CMD_BTT_WIDTH       : integer :=  BTT_SLICE_SIZE+BTT_LS_UNUSED_WIDTH;
    Constant CMD_BTT_LS_INDEX    : integer :=  0;
    Constant CMD_BTT_MS_INDEX    : integer :=  CMD_BTT_WIDTH-1;
    Constant BTT_ZEROS           : std_logic_vector(CMD_BTT_WIDTH-1 downto 0) := (others => '0'); 
    Constant BTT_RESIDUE_ZEROS   : unsigned(BTT_LS_UNUSED_WIDTH-1 downto 0) := (others => '0'); 
    Constant BTT_SLICE_ONE       : unsigned(BTT_SLICE_SIZE-1 downto 0) := TO_UNSIGNED(1, BTT_SLICE_SIZE); 
    Constant STRB_WIDTH          : integer := C_STREAM_DWIDTH/8; -- Number of bytes in the Stream
    Constant LEN_WIDTH           : integer := 8; 
    
               
               
    -- Type Declarations  --------------------------------------------
    
    type SCC_SM_STATE_TYPE is (
                INIT,
                POP_RECOVER,
                GET_NXT_CMD,
                CHK_AND_CALC,
                PUSH_TO_AXI,
                ERROR_TRAP
                );
  
  
    
    
    
    -- Signal Declarations  --------------------------------------------
  
    signal sm_scc_state              : SCC_SM_STATE_TYPE := INIT;
    signal sm_scc_state_ns           : SCC_SM_STATE_TYPE := INIT;
    signal sm_pop_input_cmd          : std_logic := '0';
    signal sm_pop_input_cmd_ns       : std_logic := '0';
    signal sm_set_push2axi           : std_logic := '0';
    signal sm_set_push2axi_ns        : std_logic := '0';
    signal sm_set_error              : std_logic := '0';
    signal sm_set_error_ns           : std_logic := '0';
    Signal sm_scc_sm_ready           : std_logic := '0';
    Signal sm_scc_sm_ready_ns        : std_logic := '0';
    signal sig_cmd2data_valid        : std_logic := '0';
    signal sig_clr_cmd2data_valid    : std_logic := '0';
    signal sig_cmd2addr_valid        : std_logic := '0';
    signal sig_clr_cmd2addr_valid    : std_logic := '0';
    signal sig_addr_data_rdy_pending : std_logic := '0';
    signal sig_cmd_btt_slice         : std_logic_vector(CMD_BTT_WIDTH-1 downto 0) := (others => '0');
    signal sig_load_input_cmd        : std_logic := '0';
    signal sig_cmd_reg_empty         : std_logic := '0';
    signal sig_cmd_reg_full          : std_logic := '0';
    signal sig_cmd_addr_reg          : std_logic_vector(C_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_cmd_btt_reg           : std_logic_vector(CMD_BTT_WIDTH-1 downto 0) := (others => '0');
    signal sig_cmd_type_reg          : std_logic := '0';
    signal sig_cmd_burst_reg         : std_logic_vector (1 downto 0) := "00";
    signal sig_cmd_tag_reg           : std_logic_vector(CMD_TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_data_rdy4cmd     : std_logic := '0';
    signal sig_btt_raw               : std_logic := '0';
    signal sig_btt_is_zero           : std_logic := '0';
    signal sig_btt_is_zero_reg       : std_logic := '0';
    signal sig_next_tag              : std_logic_vector(CMD_TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_next_addr             : std_logic_vector(C_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_next_len              : std_logic_vector(LEN_WIDTH-1 downto 0) := (others => '0');
    signal sig_next_size             : std_logic_vector(2 downto 0) := (others => '0');
    signal sig_next_burst            : std_logic_vector(1 downto 0) := (others => '0');
    signal sig_next_cache            : std_logic_vector(3 downto 0) := (others => '0');
    signal sig_next_user             : std_logic_vector(3 downto 0) := (others => '0');
    signal sig_next_strt_strb        : std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0) := (others => '0');
    signal sig_next_end_strb         : std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0) := (others => '0');
    signal sig_cmd2addr_valid1 : std_logic; 
  
    
    
  begin --(architecture implementation)
     
    -- Assign calculation error output 
    calc_error            <= sm_set_error;
   
    -- Assign the ready output to the Command FIFO 
    mst2cmd_cmd_ready     <= sig_cmd_reg_empty and addr2mstr_cmd_ready; --sm_scc_sm_ready;
    
    -- Assign the Address Channel Controller Qualifiers
    mstr2addr_tag         <= "0000"; --sig_next_tag  ; 
    mstr2addr_addr        <= sig_next_addr ; 
    mstr2addr_len         <= sig_next_len  ; 
    mstr2addr_size        <= sig_next_size ;
    mstr2addr_burst       <= sig_cmd_burst_reg; 
    mstr2addr_cache       <= sig_next_cache; 
    mstr2addr_user        <= sig_next_user; 
    mstr2addr_cmd_valid   <= sig_cmd2addr_valid1;
    mstr2addr_calc_error  <= sm_set_error  ; 
    mstr2addr_cmd_cmplt   <= '1'           ;   -- Lite mode is always 1 
    
    -- Assign the Data Channel Controller Qualifiers
    mstr2data_tag         <= "0000"; --sig_next_tag ; 
    mstr2data_saddr_lsb   <= sig_cmd_addr_reg(C_SEL_ADDR_WIDTH-1 downto 0);
    
    mstr2data_len         <= sig_next_len ;
    
    mstr2data_strt_strb   <= sig_next_strt_strb;
    mstr2data_last_strb   <= sig_next_end_strb;
    mstr2data_sof         <= '1';  -- Lite mode is always 1 cmd
    mstr2data_eof         <= '1';  -- Lite mode is always 1 cmd
    mstr2data_cmd_cmplt   <= '1';  -- Lite mode is always 1 cmd
    mstr2data_cmd_valid   <= sig_cmd2data_valid;
    mstr2data_calc_error  <= sm_set_error;   
    
    
    -- Internal logic ------------------------------
    sig_addr_data_rdy_pending  <=  sig_cmd2addr_valid or 
                                   sig_cmd2data_valid;
   
    sig_clr_cmd2data_valid     <=  sig_cmd2data_valid and data2mstr_cmd_ready;
    
    sig_clr_cmd2addr_valid     <=  sig_cmd2addr_valid and addr2mstr_cmd_ready;
    
    
    sig_load_input_cmd         <=  cmd2mstr_cmd_valid and 
                                   sig_cmd_reg_empty;--  and
                                 --  sm_scc_sm_ready;
    
    sig_next_tag               <=  sig_cmd_tag_reg;
    
    sig_next_addr              <=  sig_cmd_addr_reg;
    
    sig_addr_data_rdy4cmd      <=  addr2mstr_cmd_ready and data2mstr_cmd_ready;
    
    sig_cmd_btt_slice          <=  cmd2mstr_command(CMD_BTT_MS_INDEX downto CMD_BTT_LS_INDEX);
    
    sig_btt_is_zero  <= '1'
      when  (sig_cmd_btt_slice = BTT_ZEROS)
      Else '0';
    
  
  
    
 
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_NO_RESIDUE_BITS
    --
    -- If Generate Description:
    --
    --
    --
    ------------------------------------------------------------
    GEN_NO_RESIDUE_BITS : if (BTT_LS_UNUSED_WIDTH = 0) generate
    
    
       -- signals 
       signal sig_len_btt_slice         : unsigned(BTT_SLICE_SIZE-1 downto 0) := (others => '0');
       signal sig_len_btt_slice_minus_1 : unsigned(BTT_SLICE_SIZE-1 downto 0) := (others => '0');
       signal sig_len2use               : unsigned(BTT_SLICE_SIZE-1 downto 0) := (others => '0');
       
     
       begin
    
       -- LEN Calculation logic ------------------------------------------ 
        
         sig_next_len         <= STD_LOGIC_VECTOR(RESIZE(sig_len2use, LEN_WIDTH));
        
         sig_len_btt_slice    <= UNSIGNED(sig_cmd_btt_reg(CMD_BTT_MS_INDEX downto 0));
         
         sig_len_btt_slice_minus_1 <= sig_len_btt_slice-BTT_SLICE_ONE
          when sig_btt_is_zero_reg = '0'
          else (others => '0');    -- clip at zero
         
         
         -- If most significant bit of BTT set then limit to 
         -- Max Burst Len, else rip it from the BTT value,
         -- otheriwse subtract 1 from the BTT ripped value
         -- 1 from the BTT ripped value
         sig_len2use <= MAX_BURST_LEN_US 
           When (sig_cmd_btt_reg(CMD_BTT_MS_INDEX) = '1')
           Else sig_len_btt_slice_minus_1;
           
        
       end generate GEN_NO_RESIDUE_BITS;
        
    
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_HAS_RESIDUE_BITS
    --
    -- If Generate Description:
    --
    --
    --
    ------------------------------------------------------------
    GEN_HAS_RESIDUE_BITS : if (BTT_LS_UNUSED_WIDTH > 0) generate
    
       -- signals 
       signal sig_btt_len_residue       : unsigned(BTT_LS_UNUSED_WIDTH-1 downto 0) := (others => '0');
       signal sig_len_btt_slice         : unsigned(BTT_SLICE_SIZE-1 downto 0) := (others => '0');
       signal sig_len_btt_slice_minus_1 : unsigned(BTT_SLICE_SIZE-1 downto 0) := (others => '0');
       signal sig_len2use               : unsigned(BTT_SLICE_SIZE-1 downto 0) := (others => '0');
       
     
       begin
    
       -- LEN Calculation logic ------------------------------------------ 
       RD_EXTRA_FIELDS : if (C_ENABLE_EXTRA_FIELD = 1) generate 
         sig_next_len         <= "00001100" when sig_cmd_tag_reg (0) = '1'
                                 else "00000111"; --STD_LOGIC_VECTOR(RESIZE(sig_len2use, LEN_WIDTH));
       end generate RD_EXTRA_FIELDS;

       NORD_EXTRA_FIELDS : if (C_ENABLE_EXTRA_FIELD = 0) generate 
         sig_next_len         <= "00000111";
       end generate NORD_EXTRA_FIELDS;
        
         sig_len_btt_slice    <= UNSIGNED(sig_cmd_btt_reg(CMD_BTT_MS_INDEX downto BTT_LS_UNUSED_WIDTH));
         
         sig_len_btt_slice_minus_1 <= sig_len_btt_slice-BTT_SLICE_ONE
          when sig_btt_is_zero_reg = '0'
          else (others => '0');    -- clip at zero
         
         sig_btt_len_residue  <= UNSIGNED(sig_cmd_btt_reg(BTT_LS_UNUSED_WIDTH-1 downto 0));
            
         
         -- If most significant bit of BTT set then limit to 
         -- Max Burst Len, else rip it from the BTT value
         -- However if residue bits are zeroes then subtract
         -- 1 from the BTT ripped value
         sig_len2use <= MAX_BURST_LEN_US 
           When (sig_cmd_btt_reg(CMD_BTT_MS_INDEX) = '1')
           Else sig_len_btt_slice_minus_1
           when (sig_btt_len_residue = BTT_RESIDUE_ZEROS)
           Else sig_len_btt_slice;

        
       end generate GEN_HAS_RESIDUE_BITS;
        
        
        
        
        
        
        
     
     
     
    
    
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: REG_INPUT_CMD
    --
    -- Process Description:
    --  Implements the input command holding registers
    --
    -------------------------------------------------------------
    REG_INPUT_CMD : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset       = '1' or
                addr2mstr_cmd_ready = '0') then
--                sm_pop_input_cmd = '1') then
            
              sig_cmd_btt_reg      <=  (others => '0');
              sig_cmd_type_reg     <=  '0';
              sig_cmd_addr_reg     <=  (others => '0');
              sig_cmd_tag_reg      <=  (others => '0');
              sig_btt_is_zero_reg  <=  '0';
              
              sig_cmd_reg_empty    <=  '1';
              sig_cmd_reg_full     <=  '0';
              sig_cmd_burst_reg    <=  "00";
              sig_cmd2addr_valid1 <= '0';
            
            elsif (sig_load_input_cmd = '1') then
              
              sig_cmd_btt_reg      <= sig_cmd_btt_slice;
              sig_cmd_type_reg     <= cmd2mstr_command(CMD_TYPE_INDEX);
              sig_cmd_addr_reg     <= cmd2mstr_command(CMD_ADDR_MS_INDEX downto CMD_ADDR_LS_INDEX);  
              sig_cmd_tag_reg      <=  cmd2mstr_command(CMD_TAG_MS_INDEX downto CMD_TAG_LS_INDEX);
              sig_btt_is_zero_reg  <= sig_btt_is_zero;
              
              sig_cmd_reg_empty    <=  '0';
              sig_cmd_reg_full     <=  '1';

              sig_cmd_burst_reg    <= sig_next_burst;
              sig_cmd2addr_valid1 <= '1';
            
            else
              null; -- Hold current State
            end if; 
         end if;       
       end process REG_INPUT_CMD; 
   
   
    
    
    -- Only Incrementing Burst type supported (per Interface_X guidelines)
    sig_next_burst <= AXI_BURST_INCR when (cmd2mstr_command(CMD_TYPE_INDEX) = '1') else
                      AXI_BURST_FIXED;
    sig_next_user <= cache2mstr_command (7 downto 4);   
    sig_next_cache  <= cache2mstr_command (3 downto 0);   
 
 
  
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_LEN_SDWIDTH_64
    --
    -- If Generate Description:
    --  This IfGen implements the AXI LEN qualifier calculation
    -- and the Stream data channel start/end STRB value.  
    --
    -- This IfGen is for the 64-bit Stream data Width case.
    --
    ------------------------------------------------------------
    GEN_LEN_SDWIDTH_64 : if (C_STREAM_DWIDTH = 64) generate
    
       -- Local Constants
       Constant AXI_SIZE2USE      : std_logic_vector(2 downto 0) := AXI_SIZE_8BYTE;
       Constant RESIDUE_BIT_WIDTH : integer := 3;
       
       
       
       -- local signals
       signal sig_last_strb2use              : std_logic_vector(STRB_WIDTH-1 downto 0) := (others => '0');
       signal sig_last_strb                  : std_logic_vector(STRB_WIDTH-1 downto 0) := (others => '0');
       Signal sig_btt_ms_bit_value           : std_logic := '0';
       signal lsig_btt_len_residue           : std_logic_vector(BTT_LS_UNUSED_WIDTH-1 downto 0) := (others => '0');
       signal sig_btt_len_residue_composite  : std_logic_vector(RESIDUE_BIT_WIDTH downto 0) := (others => '0');
                                                                -- note 1 extra bit implied
                                                                
       
       begin
           
         -- Assign the Address Channel Controller Size Qualifier Value
         sig_next_size        <= AXI_SIZE2USE;
         
         -- Assign the Strobe Values
         sig_next_strt_strb   <= (others => '1'); -- always aligned on first databeat for LITE DataMover
         sig_next_end_strb    <= sig_last_strb;
        
        
         -- Local calculations ------------------------------
         
         lsig_btt_len_residue  <= sig_cmd_btt_reg(BTT_LS_UNUSED_WIDTH-1 downto 0);
         
         sig_btt_ms_bit_value  <= sig_cmd_btt_reg(CMD_BTT_MS_INDEX);
         
         sig_btt_len_residue_composite <= sig_btt_ms_bit_value &
                                          lsig_btt_len_residue;
         
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: IMP_LAST_STRB_8bit
         --
         -- Process Description:
         -- Generates the Strobe values for the LAST databeat of the
         -- Burst to MMap when the Stream is 64 bits wide and 8 strobe
         -- bits are required.
         --
         -------------------------------------------------------------
         IMP_LAST_STRB_8bit : process (sig_btt_len_residue_composite)
            begin
         
             case sig_btt_len_residue_composite is
               when "0001" =>
                 sig_last_strb <= "00000001";
               when "0010" =>
                 sig_last_strb <= "00000011";
               when "0011" =>
                 sig_last_strb <= "00000111";
               when "0100" =>
                 sig_last_strb <= "00001111";
               when "0101" =>
                 sig_last_strb <= "00011111";
               when "0110" =>
                 sig_last_strb <= "00111111";
               when "0111" =>
                 sig_last_strb <= "01111111";
               when others =>
                 sig_last_strb <= "11111111";
             end case;
             
             
            end process IMP_LAST_STRB_8bit; 
         
         
       end generate GEN_LEN_SDWIDTH_64;
  
  
  
  
  
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_LEN_SDWIDTH_32
    --
    -- If Generate Description:
    --  This IfGen implements the AXI LEN qualifier calculation
    -- and the Stream data channel start/end STRB value.  
    --
    -- This IfGen is for the 32-bit Stream data Width case.
    --
    ------------------------------------------------------------
    GEN_LEN_SDWIDTH_32 : if (C_STREAM_DWIDTH = 32) generate
    
       -- Local Constants
       Constant AXI_SIZE2USE                 : std_logic_vector(2 downto 0) := AXI_SIZE_4BYTE;
       Constant RESIDUE_BIT_WIDTH            : integer := 2;
       
       -- local signals
       signal sig_last_strb2use              : std_logic_vector(STRB_WIDTH-1 downto 0) := (others => '0');
       signal sig_last_strb                  : std_logic_vector(STRB_WIDTH-1 downto 0) := (others => '0');
       Signal sig_btt_ms_bit_value           : std_logic := '0';
       signal sig_btt_len_residue_composite  : std_logic_vector(RESIDUE_BIT_WIDTH downto 0) := (others => '0'); -- 1 extra bit
       signal lsig_btt_len_residue           : std_logic_vector(BTT_LS_UNUSED_WIDTH-1 downto 0) := (others => '0');
       
       
       begin
           
         -- Assign the Address Channel Controller Size Qualifier Value
         sig_next_size        <= AXI_SIZE2USE;
         
         -- Assign the Strobe Values
         sig_next_strt_strb   <= (others => '1'); -- always aligned on first databeat for LITE DataMover
         sig_next_end_strb    <= sig_last_strb;
        
        
         -- Local calculations ------------------------------
         
         lsig_btt_len_residue  <= sig_cmd_btt_reg(BTT_LS_UNUSED_WIDTH-1 downto 0);
         
         sig_btt_ms_bit_value <= sig_cmd_btt_reg(CMD_BTT_MS_INDEX);
         
         sig_btt_len_residue_composite <= sig_btt_ms_bit_value &
                                          lsig_btt_len_residue;
         
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: IMP_LAST_STRB_4bit
         --
         -- Process Description:
         -- Generates the Strobe values for the LAST databeat of the
         -- Burst to MMap when the Stream is 32 bits wide and 4 strobe
         -- bits are required.
         --
         -------------------------------------------------------------
         IMP_LAST_STRB_4bit : process (sig_btt_len_residue_composite)
            begin
         
             case sig_btt_len_residue_composite is
       -- coverage off 
               when "001" =>
                 sig_last_strb <= "0001";
               when "010" =>
                 sig_last_strb <= "0011";
               when "011" =>
                 sig_last_strb <= "0111";
       -- coverage on
               when others =>
                 sig_last_strb <= "1111";
             end case;
             
             
            end process IMP_LAST_STRB_4bit; 
          
       end generate GEN_LEN_SDWIDTH_32;
  
  
  
  
  
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_LEN_SDWIDTH_16
    --
    -- If Generate Description:
    --  This IfGen implements the AXI LEN qualifier calculation
    -- and the Stream data channel start/end STRB value.  
    --
    -- This IfGen is for the 16-bit Stream data Width case.
    --
    ------------------------------------------------------------
    GEN_LEN_SDWIDTH_16 : if (C_STREAM_DWIDTH = 16) generate
    
       -- Local Constants
       Constant AXI_SIZE2USE      : std_logic_vector(2 downto 0) := AXI_SIZE_2BYTE;
       Constant RESIDUE_BIT_WIDTH : integer := 1;
       
       
       -- local signals
       signal sig_last_strb2use              : std_logic_vector(STRB_WIDTH-1 downto 0) := (others => '0');
       signal sig_last_strb                  : std_logic_vector(STRB_WIDTH-1 downto 0) := (others => '0');
       Signal sig_btt_ms_bit_value           : std_logic := '0';
       signal sig_btt_len_residue_composite  : std_logic_vector(RESIDUE_BIT_WIDTH downto 0) := (others => '0'); -- 1 extra bit
       signal lsig_btt_len_residue           : std_logic_vector(BTT_LS_UNUSED_WIDTH-1 downto 0) := (others => '0');
       
       
       begin
           
         -- Assign the Address Channel Controller Size Qualifier Value
         sig_next_size        <= AXI_SIZE2USE;
         
         -- Assign the Strobe Values
         sig_next_strt_strb   <= (others => '1'); -- always aligned on first databeat for LITE DataMover
         sig_next_end_strb    <= sig_last_strb;
        
        
         -- Local calculations ------------------------------
         
         lsig_btt_len_residue  <= sig_cmd_btt_reg(BTT_LS_UNUSED_WIDTH-1 downto 0);
         
         sig_btt_ms_bit_value  <= sig_cmd_btt_reg(CMD_BTT_MS_INDEX);
         
         sig_btt_len_residue_composite <= sig_btt_ms_bit_value &
                                          lsig_btt_len_residue;
         
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: IMP_LAST_STRB_2bit
         --
         -- Process Description:
         -- Generates the Strobe values for the LAST databeat of the
         -- Burst to MMap when the Stream is 16 bits wide and 2 strobe
         -- bits are required.
         --
         -------------------------------------------------------------
         IMP_LAST_STRB_2bit : process (sig_btt_len_residue_composite)
            begin
         
             case sig_btt_len_residue_composite is
               when "01" =>
                 sig_last_strb <= "01";
               when others =>
                 sig_last_strb <= "11";
             end case;
             
             
            end process IMP_LAST_STRB_2bit; 
         
         
       end generate GEN_LEN_SDWIDTH_16;
  
  
  
  
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_LEN_SDWIDTH_8
    --
    -- If Generate Description:
    --  This IfGen implements the AXI LEN qualifier calculation
    -- and the Stream data channel start/end STRB value.  
    --
    -- This IfGen is for the 8-bit Stream data Width case.
    --
    ------------------------------------------------------------
    GEN_LEN_SDWIDTH_8 : if (C_STREAM_DWIDTH = 8) generate
       
       -- Local Constants
       Constant AXI_SIZE2USE : std_logic_vector(2 downto 0) := AXI_SIZE_1BYTE;
       
       begin
    
           -- Assign the Address Channel Controller Qualifiers
         sig_next_size        <= AXI_SIZE2USE;
           
         -- Assign the Data Channel Controller Qualifiers
         sig_next_strt_strb   <= (others => '1');
         sig_next_end_strb    <= (others => '1');
          
          
       end generate GEN_LEN_SDWIDTH_8;
  
  
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: SCC_SM_REG
    --
    -- Process Description:
    -- Implements registered portion of state machine
    --
    -------------------------------------------------------------
    SCC_SM_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
--              
--              sm_scc_state      <= INIT;
--              sm_pop_input_cmd  <= '0' ;           
--              sm_set_push2axi   <= '0' ;           
              sm_set_error      <= '0' ;
--              sm_scc_sm_ready   <= '0' ;
--              
            elsif (sig_btt_is_zero_reg = '1') then
--              
--              sm_scc_state      <= sm_scc_state_ns     ;
--              sm_pop_input_cmd  <= sm_pop_input_cmd_ns ;           
--              sm_set_push2axi   <= sm_set_push2axi_ns  ;           
              sm_set_error      <= '1'     ;
--              sm_scc_sm_ready   <= sm_scc_sm_ready_ns  ;
--              
            end if; 
         end if;       
       end process SCC_SM_REG; 
    
    
  
  
  
  end implementation;


-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_sg_wr_demux.vhd
  --
  -- Description:     
  --    This file implements the DataMover Master Write Strobe De-Multiplexer.                 
  --  This is needed when the native data width of the DataMover is narrower 
  --  than the AXI4 Write Data Channel.                
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  
  
  -------------------------------------------------------------------------------
  
  entity axi_sg_wr_demux is
    generic (
      
      C_SEL_ADDR_WIDTH     : Integer range  1  to  8 :=  5;
        -- Sets the width of the select control bus
      
      C_MMAP_DWIDTH        : Integer range 32 to 1024 := 32;
        -- Indicates the width of the AXI4 Write Data Channel
      
      C_STREAM_DWIDTH      : Integer range  8 to 1024 := 32
        -- Indicates the native data width of the DataMover S2MM. If 
        -- S2MM Store and Forward with upsizer is enabled, the width is 
        -- the AXi4 Write Data Channel, else it is the S2MM Stream data width.
      
      );
    port (
      
     
      -- AXI MMap Data Channel Input  --------------------------------------------
                                                                                --
      wstrb_in         : In  std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);  --
        -- data input                                                           --
      ----------------------------------------------------------------------------
     
      
      
      -- AXI Master Stream  ------------------------------------------------------
                                                                                --
      demux_wstrb_out    : Out std_logic_vector((C_MMAP_DWIDTH/8)-1 downto 0);  --       
        --De-Mux strb output                                                    --
      ----------------------------------------------------------------------------
               
                
                
      -- Command Calculator Interface --------------------------------------------
                                                                                --
      debeat_saddr_lsb : In std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0)       --
         -- The next command start address LSbs to use for the read data        --
         -- mux (only used if Stream data width is less than the MMap Data      --
         -- Width).                                                             --
      ----------------------------------------------------------------------------
      
         
      );
  
  end entity axi_sg_wr_demux;
  
  
  architecture implementation of axi_sg_wr_demux is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    
    -- Function Decalarations -------------------------------------------------
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: func_mux_sel_width
    --
    -- Function Description:
    --   Calculates the number of needed bits for the Mux Select control
    -- based on the number of input channels to the mux.
    --
    -- Note that the number of input mux channels are always a 
    -- power of 2.
    --
    -------------------------------------------------------------------
    function func_mux_sel_width (num_channels : integer) return integer is
    
     Variable var_sel_width : integer := 0;
    
    begin
    
       case num_channels is
         --when 2 =>
         --    var_sel_width := 1;
         when 4 =>
             var_sel_width := 2;
         when 8 =>
             var_sel_width := 3;
         when 16 =>
             var_sel_width := 4;
         when 32 =>
             var_sel_width := 5;
         when 64 =>
             var_sel_width := 6;
         when 128 =>
             var_sel_width := 7;
         when others =>
             var_sel_width := 1; 
       end case;
       
       Return (var_sel_width);
        
        
    end function func_mux_sel_width;
    
    
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: func_sel_ls_index
    --
    -- Function Description:
    --   Calculates the LS index of the select field to rip from the
    -- input select bus.
    --
    -- Note that the number of input mux channels are always a 
    -- power of 2.
    --
    -------------------------------------------------------------------
    function func_sel_ls_index (stream_width : integer) return integer is
    
     Variable var_sel_ls_index : integer := 0;
    
    begin
    
       case stream_width is
         when 8 =>
             var_sel_ls_index := 0;
         when 16 =>
             var_sel_ls_index := 1;
         when 32 =>
             var_sel_ls_index := 2;
         when 64 =>
             var_sel_ls_index := 3;
         when 128 =>
             var_sel_ls_index := 4;
         when 256 =>
             var_sel_ls_index := 5;
         when 512 =>
             var_sel_ls_index := 6;
         when others =>  -- assume 1024 bit width
             var_sel_ls_index := 7;
       end case;
       
       Return (var_sel_ls_index);
        
        
    end function func_sel_ls_index;
    
    
    
    
    
    -- Constant Decalarations -------------------------------------------------
    
    Constant OMIT_DEMUX    : boolean := (C_STREAM_DWIDTH = C_MMAP_DWIDTH);
    Constant INCLUDE_DEMUX : boolean := not(OMIT_DEMUX);
    
    
    
    
    Constant STREAM_WSTB_WIDTH   : integer := C_STREAM_DWIDTH/8;
    Constant MMAP_WSTB_WIDTH     : integer := C_MMAP_DWIDTH/8;
    Constant NUM_MUX_CHANNELS    : integer := MMAP_WSTB_WIDTH/STREAM_WSTB_WIDTH;
    Constant MUX_SEL_WIDTH       : integer := func_mux_sel_width(NUM_MUX_CHANNELS);
    Constant MUX_SEL_LS_INDEX    : integer := func_sel_ls_index(C_STREAM_DWIDTH);
    
    
    -- Signal Declarations  --------------------------------------------
 
    signal sig_demux_wstrb_out   : std_logic_vector(MMAP_WSTB_WIDTH-1 downto 0) := (others => '0');



    
  begin --(architecture implementation)
  
  
  
  
   -- Assign the Output data port 
    demux_wstrb_out        <= sig_demux_wstrb_out;
  


    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_STRM_EQ_MMAP
    --
    -- If Generate Description:
    --   This IfGen implements the case where the Stream Data Width is 
    -- the same as the Memeory Map read Data width.
    --
    --
    ------------------------------------------------------------
    GEN_STRM_EQ_MMAP : if (OMIT_DEMUX) generate
        
       begin
        
          sig_demux_wstrb_out <= wstrb_in;
        
        
       end generate GEN_STRM_EQ_MMAP;
   
   
    
    
    
     
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_2XN
    --
    -- If Generate Description:
    --  2 channel demux case
    --
    --
    ------------------------------------------------------------
    GEN_2XN : if (INCLUDE_DEMUX and 
                  NUM_MUX_CHANNELS = 2) generate
    
       -- local signals
       signal sig_demux_sel_slice      : std_logic_vector(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_unsgnd     : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_int        : integer := 0;
       signal lsig_demux_sel_int_local : integer := 0;
       signal lsig_demux_wstrb_out     : std_logic_vector(MMAP_WSTB_WIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_demux_sel_slice   <= debeat_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_demux_sel_unsgnd  <=  UNSIGNED(sig_demux_sel_slice);  -- convert to unsigned
        
         sig_demux_sel_int     <=  TO_INTEGER(sig_demux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                     -- with locally static subtype error in each of the
                                                                     -- Mux IfGens
        
         lsig_demux_sel_int_local <= sig_demux_sel_int;
         
         sig_demux_wstrb_out      <= lsig_demux_wstrb_out;
       
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_2XN_DEMUX
         --
         -- Process Description:
         --  Implement the 2XN DeMux
         --
         -------------------------------------------------------------
         DO_2XN_DEMUX : process (lsig_demux_sel_int_local,
                                  wstrb_in)
            begin
              
              -- Set default value
              lsig_demux_wstrb_out <=  (others => '0');
              
              case lsig_demux_sel_int_local is
                when 0 =>
                    lsig_demux_wstrb_out(STREAM_WSTB_WIDTH-1 downto 0) <=  wstrb_in;
                
                when others => -- 1 case
                    lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*2)-1 downto STREAM_WSTB_WIDTH*1) <=  wstrb_in;
              end case;
              
            end process DO_2XN_DEMUX; 
 
         
       end generate GEN_2XN;
  
 
 
 
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_4XN
    --
    -- If Generate Description:
    --  4 channel demux case
    --
    --
    ------------------------------------------------------------
    GEN_4XN : if (INCLUDE_DEMUX and 
                  NUM_MUX_CHANNELS = 4) generate
    
       -- local signals
       signal sig_demux_sel_slice      : std_logic_vector(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_unsgnd     : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_int        : integer := 0;
       signal lsig_demux_sel_int_local : integer := 0;
       signal lsig_demux_wstrb_out     : std_logic_vector(MMAP_WSTB_WIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_demux_sel_slice   <= debeat_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_demux_sel_unsgnd  <=  UNSIGNED(sig_demux_sel_slice);  -- convert to unsigned
        
         sig_demux_sel_int     <=  TO_INTEGER(sig_demux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                 -- with locally static subtype error in each of the
                                                                 -- Mux IfGens
        
         lsig_demux_sel_int_local <= sig_demux_sel_int;
         
         sig_demux_wstrb_out      <= lsig_demux_wstrb_out;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_4XN_DEMUX
         --
         -- Process Description:
         --  Implement the 4XN DeMux
         --
         -------------------------------------------------------------
         DO_4XN_DEMUX : process (lsig_demux_sel_int_local,
                                 wstrb_in)
           begin
              
             -- Set default value
             lsig_demux_wstrb_out <=  (others => '0');
              
             case lsig_demux_sel_int_local is
               when 0 =>
                   lsig_demux_wstrb_out(STREAM_WSTB_WIDTH-1 downto 0) <=  wstrb_in;
               when 1 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*2)-1 downto STREAM_WSTB_WIDTH*1) <=  wstrb_in;
               when 2 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*3)-1 downto STREAM_WSTB_WIDTH*2) <=  wstrb_in;
               
               when others =>  -- 3 case
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*4)-1 downto STREAM_WSTB_WIDTH*3) <=  wstrb_in;
             end case;
             
           end process DO_4XN_DEMUX; 
  
         
       end generate GEN_4XN;
  
 
 
 
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_8XN
    --
    -- If Generate Description:
    --  8 channel demux case
    --
    --
    ------------------------------------------------------------
    GEN_8XN : if (INCLUDE_DEMUX and 
                  NUM_MUX_CHANNELS = 8) generate
    
       -- local signals
       signal sig_demux_sel_slice      : std_logic_vector(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_unsgnd     : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_int        : integer := 0;
       signal lsig_demux_sel_int_local : integer := 0;
       signal lsig_demux_wstrb_out     : std_logic_vector(MMAP_WSTB_WIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_demux_sel_slice   <= debeat_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_demux_sel_unsgnd  <=  UNSIGNED(sig_demux_sel_slice);    -- convert to unsigned
        
         sig_demux_sel_int     <=  TO_INTEGER(sig_demux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                     -- with locally static subtype error in each of the
                                                                     -- Mux IfGens
        
         lsig_demux_sel_int_local <= sig_demux_sel_int;
         
         sig_demux_wstrb_out      <= lsig_demux_wstrb_out;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_8XN_DEMUX
         --
         -- Process Description:
         --  Implement the 8XN DeMux
         --
         -------------------------------------------------------------
         DO_8XN_DEMUX : process (lsig_demux_sel_int_local,
                                 wstrb_in)
           begin
             
             -- Set default value
             lsig_demux_wstrb_out <=  (others => '0');
              
             case lsig_demux_sel_int_local is
               when 0 =>
                   lsig_demux_wstrb_out(STREAM_WSTB_WIDTH-1 downto 0) <=  wstrb_in;
               when 1 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*2)-1 downto STREAM_WSTB_WIDTH*1) <=  wstrb_in;
               when 2 =>                                                             
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*3)-1 downto STREAM_WSTB_WIDTH*2) <=  wstrb_in;
               when 3 =>                                                             
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*4)-1 downto STREAM_WSTB_WIDTH*3) <=  wstrb_in;
               when 4 =>                                                             
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*5)-1 downto STREAM_WSTB_WIDTH*4) <=  wstrb_in;
               when 5 =>                                                             
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*6)-1 downto STREAM_WSTB_WIDTH*5) <=  wstrb_in;
               when 6 =>                                                             
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*7)-1 downto STREAM_WSTB_WIDTH*6) <=  wstrb_in;
               
               when others => -- 7 case
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*8)-1 downto STREAM_WSTB_WIDTH*7) <=  wstrb_in;
             end case;
                 
           end process DO_8XN_DEMUX; 
 
         
       end generate GEN_8XN;
  
 
 
 
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_16XN
    --
    -- If Generate Description:
    --  16 channel demux case
    --
    --
    ------------------------------------------------------------
    GEN_16XN : if (INCLUDE_DEMUX and 
                   NUM_MUX_CHANNELS = 16) generate
    
       -- local signals
       signal sig_demux_sel_slice      : std_logic_vector(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_unsgnd     : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_int        : integer := 0;
       signal lsig_demux_sel_int_local : integer := 0;
       signal lsig_demux_wstrb_out     : std_logic_vector(MMAP_WSTB_WIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_demux_sel_slice   <= debeat_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_demux_sel_unsgnd  <=  UNSIGNED(sig_demux_sel_slice);  -- convert to unsigned
        
         sig_demux_sel_int     <=  TO_INTEGER(sig_demux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                 -- with locally static subtype error in each of the
                                                                 -- Mux IfGens
        
         lsig_demux_sel_int_local <= sig_demux_sel_int;
         
         sig_demux_wstrb_out      <= lsig_demux_wstrb_out;
       
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_16XN_DEMUX
         --
         -- Process Description:
         --  Implement the 16XN DeMux
         --
         -------------------------------------------------------------
         DO_16XN_DEMUX : process (lsig_demux_sel_int_local,
                                  wstrb_in)
           begin
             
             -- Set default value
             lsig_demux_wstrb_out <=  (others => '0');
              
             case lsig_demux_sel_int_local is
               when 0 =>
                   lsig_demux_wstrb_out(STREAM_WSTB_WIDTH-1 downto 0) <=  wstrb_in;
               when 1 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*2)-1 downto STREAM_WSTB_WIDTH*1)   <=  wstrb_in;
               when 2 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*3)-1 downto STREAM_WSTB_WIDTH*2)   <=  wstrb_in;
               when 3 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*4)-1 downto STREAM_WSTB_WIDTH*3)   <=  wstrb_in;
               when 4 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*5)-1 downto STREAM_WSTB_WIDTH*4)   <=  wstrb_in;
               when 5 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*6)-1 downto STREAM_WSTB_WIDTH*5)   <=  wstrb_in;
               when 6 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*7)-1 downto STREAM_WSTB_WIDTH*6)   <=  wstrb_in;
               when 7 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*8)-1 downto STREAM_WSTB_WIDTH*7)   <=  wstrb_in;
               when 8 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*9)-1 downto STREAM_WSTB_WIDTH*8)   <=  wstrb_in;
               when 9 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*10)-1 downto STREAM_WSTB_WIDTH*9)  <=  wstrb_in;
               when 10 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*11)-1 downto STREAM_WSTB_WIDTH*10) <=  wstrb_in;
               when 11 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*12)-1 downto STREAM_WSTB_WIDTH*11) <=  wstrb_in;
               when 12 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*13)-1 downto STREAM_WSTB_WIDTH*12) <=  wstrb_in;
               when 13 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*14)-1 downto STREAM_WSTB_WIDTH*13) <=  wstrb_in;
               when 14 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*15)-1 downto STREAM_WSTB_WIDTH*14) <=  wstrb_in;
               
               when others => -- 15 case
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*16)-1 downto STREAM_WSTB_WIDTH*15) <=  wstrb_in;
             end case;
          
           end process DO_16XN_DEMUX; 
 
         
       end generate GEN_16XN;
  
 
 
 
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_32XN
    --
    -- If Generate Description:
    --  32 channel demux case
    --
    --
    ------------------------------------------------------------
    GEN_32XN : if (INCLUDE_DEMUX and 
                   NUM_MUX_CHANNELS = 32) generate
    
       -- local signals
       signal sig_demux_sel_slice      : std_logic_vector(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_unsgnd     : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_int        : integer := 0;
       signal lsig_demux_sel_int_local : integer := 0;
       signal lsig_demux_wstrb_out     : std_logic_vector(MMAP_WSTB_WIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_demux_sel_slice   <= debeat_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_demux_sel_unsgnd  <=  UNSIGNED(sig_demux_sel_slice);  -- convert to unsigned
        
         sig_demux_sel_int     <=  TO_INTEGER(sig_demux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                 -- with locally static subtype error in each of the
                                                                 -- Mux IfGens
        
         lsig_demux_sel_int_local <= sig_demux_sel_int;
         
         sig_demux_wstrb_out      <= lsig_demux_wstrb_out;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_32XN_DEMUX
         --
         -- Process Description:
         --  Implement the 32XN DeMux
         --
         -------------------------------------------------------------
         DO_32XN_DEMUX : process (lsig_demux_sel_int_local,
                                  wstrb_in)
           begin
             
             -- Set default value
             lsig_demux_wstrb_out <=  (others => '0');
              
             case lsig_demux_sel_int_local is
               when 0 =>
                   lsig_demux_wstrb_out(STREAM_WSTB_WIDTH-1 downto 0) <=  wstrb_in;
               when 1 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*2)-1 downto STREAM_WSTB_WIDTH*1)   <=  wstrb_in;
               when 2 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*3)-1 downto STREAM_WSTB_WIDTH*2)   <=  wstrb_in;
               when 3 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*4)-1 downto STREAM_WSTB_WIDTH*3)   <=  wstrb_in;
               when 4 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*5)-1 downto STREAM_WSTB_WIDTH*4)   <=  wstrb_in;
               when 5 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*6)-1 downto STREAM_WSTB_WIDTH*5)   <=  wstrb_in;
               when 6 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*7)-1 downto STREAM_WSTB_WIDTH*6)   <=  wstrb_in;
               when 7 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*8)-1 downto STREAM_WSTB_WIDTH*7)   <=  wstrb_in;
               when 8 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*9)-1 downto STREAM_WSTB_WIDTH*8)   <=  wstrb_in;
               when 9 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*10)-1 downto STREAM_WSTB_WIDTH*9)  <=  wstrb_in;
               when 10 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*11)-1 downto STREAM_WSTB_WIDTH*10) <=  wstrb_in;
               when 11 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*12)-1 downto STREAM_WSTB_WIDTH*11) <=  wstrb_in;
               when 12 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*13)-1 downto STREAM_WSTB_WIDTH*12) <=  wstrb_in;
               when 13 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*14)-1 downto STREAM_WSTB_WIDTH*13) <=  wstrb_in;
               when 14 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*15)-1 downto STREAM_WSTB_WIDTH*14) <=  wstrb_in;
               when 15 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*16)-1 downto STREAM_WSTB_WIDTH*15) <=  wstrb_in;
               when 16 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*17)-1 downto STREAM_WSTB_WIDTH*16) <=  wstrb_in;
               when 17 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*18)-1 downto STREAM_WSTB_WIDTH*17) <=  wstrb_in;
               when 18 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*19)-1 downto STREAM_WSTB_WIDTH*18) <=  wstrb_in;
               when 19 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*20)-1 downto STREAM_WSTB_WIDTH*19) <=  wstrb_in;
               when 20 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*21)-1 downto STREAM_WSTB_WIDTH*20) <=  wstrb_in;
               when 21 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*22)-1 downto STREAM_WSTB_WIDTH*21) <=  wstrb_in;
               when 22 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*23)-1 downto STREAM_WSTB_WIDTH*22) <=  wstrb_in;
               when 23 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*24)-1 downto STREAM_WSTB_WIDTH*23) <=  wstrb_in;
               when 24 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*25)-1 downto STREAM_WSTB_WIDTH*24) <=  wstrb_in;
               when 25 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*26)-1 downto STREAM_WSTB_WIDTH*25) <=  wstrb_in;
               when 26 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*27)-1 downto STREAM_WSTB_WIDTH*26) <=  wstrb_in;
               when 27 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*28)-1 downto STREAM_WSTB_WIDTH*27) <=  wstrb_in;
               when 28 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*29)-1 downto STREAM_WSTB_WIDTH*28) <=  wstrb_in;
               when 29 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*30)-1 downto STREAM_WSTB_WIDTH*29) <=  wstrb_in;
               when 30 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*31)-1 downto STREAM_WSTB_WIDTH*30) <=  wstrb_in;
               
               when others => -- 31 case
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*32)-1 downto STREAM_WSTB_WIDTH*31) <=  wstrb_in;
             end case;
          
           end process DO_32XN_DEMUX; 
 
         
       end generate GEN_32XN;
  
 
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_64XN
    --
    -- If Generate Description:
    --  64 channel demux case
    --
    --
    ------------------------------------------------------------
    GEN_64XN : if (INCLUDE_DEMUX and 
                   NUM_MUX_CHANNELS = 64) generate
    
       -- local signals
       signal sig_demux_sel_slice      : std_logic_vector(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_unsgnd     : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_int        : integer := 0;
       signal lsig_demux_sel_int_local : integer := 0;
       signal lsig_demux_wstrb_out     : std_logic_vector(MMAP_WSTB_WIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_demux_sel_slice   <= debeat_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_demux_sel_unsgnd  <=  UNSIGNED(sig_demux_sel_slice);  -- convert to unsigned
        
         sig_demux_sel_int     <=  TO_INTEGER(sig_demux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                 -- with locally static subtype error in each of the
                                                                 -- Mux IfGens
        
         lsig_demux_sel_int_local <= sig_demux_sel_int;
         
         sig_demux_wstrb_out      <= lsig_demux_wstrb_out;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_64XN_DEMUX
         --
         -- Process Description:
         --  Implement the 32XN DeMux
         --
         -------------------------------------------------------------
         DO_64XN_DEMUX : process (lsig_demux_sel_int_local,
                                  wstrb_in)
           begin
             
             -- Set default value
             lsig_demux_wstrb_out <=  (others => '0');
              
             case lsig_demux_sel_int_local is
               
               when 0 =>
                   lsig_demux_wstrb_out(STREAM_WSTB_WIDTH-1 downto 0) <=  wstrb_in;
               when 1 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*2)-1 downto STREAM_WSTB_WIDTH*1)   <=  wstrb_in;
               when 2 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*3)-1 downto STREAM_WSTB_WIDTH*2)   <=  wstrb_in;
               when 3 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*4)-1 downto STREAM_WSTB_WIDTH*3)   <=  wstrb_in;
               when 4 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*5)-1 downto STREAM_WSTB_WIDTH*4)   <=  wstrb_in;
               when 5 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*6)-1 downto STREAM_WSTB_WIDTH*5)   <=  wstrb_in;
               when 6 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*7)-1 downto STREAM_WSTB_WIDTH*6)   <=  wstrb_in;
               when 7 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*8)-1 downto STREAM_WSTB_WIDTH*7)   <=  wstrb_in;
               when 8 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*9)-1 downto STREAM_WSTB_WIDTH*8)   <=  wstrb_in;
               when 9 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*10)-1 downto STREAM_WSTB_WIDTH*9)  <=  wstrb_in;
               when 10 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*11)-1 downto STREAM_WSTB_WIDTH*10) <=  wstrb_in;
               when 11 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*12)-1 downto STREAM_WSTB_WIDTH*11) <=  wstrb_in;
               when 12 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*13)-1 downto STREAM_WSTB_WIDTH*12) <=  wstrb_in;
               when 13 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*14)-1 downto STREAM_WSTB_WIDTH*13) <=  wstrb_in;
               when 14 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*15)-1 downto STREAM_WSTB_WIDTH*14) <=  wstrb_in;
               when 15 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*16)-1 downto STREAM_WSTB_WIDTH*15) <=  wstrb_in;
               when 16 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*17)-1 downto STREAM_WSTB_WIDTH*16) <=  wstrb_in;
               when 17 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*18)-1 downto STREAM_WSTB_WIDTH*17) <=  wstrb_in;
               when 18 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*19)-1 downto STREAM_WSTB_WIDTH*18) <=  wstrb_in;
               when 19 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*20)-1 downto STREAM_WSTB_WIDTH*19) <=  wstrb_in;
               when 20 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*21)-1 downto STREAM_WSTB_WIDTH*20) <=  wstrb_in;
               when 21 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*22)-1 downto STREAM_WSTB_WIDTH*21) <=  wstrb_in;
               when 22 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*23)-1 downto STREAM_WSTB_WIDTH*22) <=  wstrb_in;
               when 23 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*24)-1 downto STREAM_WSTB_WIDTH*23) <=  wstrb_in;
               when 24 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*25)-1 downto STREAM_WSTB_WIDTH*24) <=  wstrb_in;
               when 25 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*26)-1 downto STREAM_WSTB_WIDTH*25) <=  wstrb_in;
               when 26 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*27)-1 downto STREAM_WSTB_WIDTH*26) <=  wstrb_in;
               when 27 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*28)-1 downto STREAM_WSTB_WIDTH*27) <=  wstrb_in;
               when 28 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*29)-1 downto STREAM_WSTB_WIDTH*28) <=  wstrb_in;
               when 29 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*30)-1 downto STREAM_WSTB_WIDTH*29) <=  wstrb_in;
               when 30 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*31)-1 downto STREAM_WSTB_WIDTH*30) <=  wstrb_in;
               when 31 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*32)-1 downto STREAM_WSTB_WIDTH*31) <=  wstrb_in;
             
             
               when 32 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*33)-1 downto STREAM_WSTB_WIDTH*32) <=  wstrb_in;
               when 33 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*34)-1 downto STREAM_WSTB_WIDTH*33) <=  wstrb_in;
               when 34 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*35)-1 downto STREAM_WSTB_WIDTH*34) <=  wstrb_in;
               when 35 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*36)-1 downto STREAM_WSTB_WIDTH*35) <=  wstrb_in;
               when 36 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*37)-1 downto STREAM_WSTB_WIDTH*36) <=  wstrb_in;
               when 37 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*38)-1 downto STREAM_WSTB_WIDTH*37) <=  wstrb_in;
               when 38 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*39)-1 downto STREAM_WSTB_WIDTH*38) <=  wstrb_in;
               when 39 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*40)-1 downto STREAM_WSTB_WIDTH*39) <=  wstrb_in;
               when 40 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*41)-1 downto STREAM_WSTB_WIDTH*40) <=  wstrb_in;
               when 41 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*42)-1 downto STREAM_WSTB_WIDTH*41) <=  wstrb_in;
               when 42 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*43)-1 downto STREAM_WSTB_WIDTH*42) <=  wstrb_in;
               when 43 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*44)-1 downto STREAM_WSTB_WIDTH*43) <=  wstrb_in;
               when 44 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*45)-1 downto STREAM_WSTB_WIDTH*44) <=  wstrb_in;
               when 45 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*46)-1 downto STREAM_WSTB_WIDTH*45) <=  wstrb_in;
               when 46 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*47)-1 downto STREAM_WSTB_WIDTH*46) <=  wstrb_in;
               when 47 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*48)-1 downto STREAM_WSTB_WIDTH*47) <=  wstrb_in;
               when 48 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*49)-1 downto STREAM_WSTB_WIDTH*48) <=  wstrb_in;
               when 49 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*50)-1 downto STREAM_WSTB_WIDTH*49) <=  wstrb_in;
               when 50 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*51)-1 downto STREAM_WSTB_WIDTH*50) <=  wstrb_in;
               when 51 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*52)-1 downto STREAM_WSTB_WIDTH*51) <=  wstrb_in;
               when 52 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*53)-1 downto STREAM_WSTB_WIDTH*52) <=  wstrb_in;
               when 53 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*54)-1 downto STREAM_WSTB_WIDTH*53) <=  wstrb_in;
               when 54 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*55)-1 downto STREAM_WSTB_WIDTH*54) <=  wstrb_in;
               when 55 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*56)-1 downto STREAM_WSTB_WIDTH*55) <=  wstrb_in;
               when 56 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*57)-1 downto STREAM_WSTB_WIDTH*56) <=  wstrb_in;
               when 57 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*58)-1 downto STREAM_WSTB_WIDTH*57) <=  wstrb_in;
               when 58 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*59)-1 downto STREAM_WSTB_WIDTH*58) <=  wstrb_in;
               when 59 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*60)-1 downto STREAM_WSTB_WIDTH*59) <=  wstrb_in;
               when 60 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*61)-1 downto STREAM_WSTB_WIDTH*60) <=  wstrb_in;
               when 61 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*62)-1 downto STREAM_WSTB_WIDTH*61) <=  wstrb_in;
               when 62 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*63)-1 downto STREAM_WSTB_WIDTH*62) <=  wstrb_in;
               
               when others => -- 63 case
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*64)-1 downto STREAM_WSTB_WIDTH*63) <=  wstrb_in;
             
             
             end case;
          
           end process DO_64XN_DEMUX; 
 
         
       end generate GEN_64XN;
  
 
  
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_128XN
    --
    -- If Generate Description:
    --  128 channel demux case
    --
    --
    ------------------------------------------------------------
    GEN_128XN : if (INCLUDE_DEMUX and 
                    NUM_MUX_CHANNELS = 128) generate
    
       -- local signals
       signal sig_demux_sel_slice      : std_logic_vector(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_unsgnd     : unsigned(MUX_SEL_WIDTH-1 downto 0) := (others => '0');
       signal sig_demux_sel_int        : integer := 0;
       signal lsig_demux_sel_int_local : integer := 0;
       signal lsig_demux_wstrb_out     : std_logic_vector(MMAP_WSTB_WIDTH-1 downto 0) := (others => '0');
       
       begin
    
         
        -- Rip the Mux Select bits needed for the Mux case from the input select bus
         sig_demux_sel_slice   <= debeat_saddr_lsb((MUX_SEL_LS_INDEX + MUX_SEL_WIDTH)-1 downto MUX_SEL_LS_INDEX);
        
         sig_demux_sel_unsgnd  <=  UNSIGNED(sig_demux_sel_slice);    -- convert to unsigned
        
         sig_demux_sel_int     <=  TO_INTEGER(sig_demux_sel_unsgnd); -- convert to integer for MTI compile issue
                                                                     -- with locally static subtype error in each of the
                                                                     -- Mux IfGens
        
         lsig_demux_sel_int_local <= sig_demux_sel_int;
         
         sig_demux_wstrb_out      <= lsig_demux_wstrb_out;
       
          
          
          
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: DO_128XN_DEMUX
         --
         -- Process Description:
         --  Implement the 32XN DeMux
         --
         -------------------------------------------------------------
         DO_128XN_DEMUX : process (lsig_demux_sel_int_local,
                                  wstrb_in)
           begin
             
             -- Set default value
             lsig_demux_wstrb_out <=  (others => '0');
              
             case lsig_demux_sel_int_local is
               
               when 0 =>
                   lsig_demux_wstrb_out(STREAM_WSTB_WIDTH-1 downto 0) <=  wstrb_in;
               when 1 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*2)-1 downto STREAM_WSTB_WIDTH*1)   <=  wstrb_in;
               when 2 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*3)-1 downto STREAM_WSTB_WIDTH*2)   <=  wstrb_in;
               when 3 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*4)-1 downto STREAM_WSTB_WIDTH*3)   <=  wstrb_in;
               when 4 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*5)-1 downto STREAM_WSTB_WIDTH*4)   <=  wstrb_in;
               when 5 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*6)-1 downto STREAM_WSTB_WIDTH*5)   <=  wstrb_in;
               when 6 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*7)-1 downto STREAM_WSTB_WIDTH*6)   <=  wstrb_in;
               when 7 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*8)-1 downto STREAM_WSTB_WIDTH*7)   <=  wstrb_in;
               when 8 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*9)-1 downto STREAM_WSTB_WIDTH*8)   <=  wstrb_in;
               when 9 =>                                                               
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*10)-1 downto STREAM_WSTB_WIDTH*9)  <=  wstrb_in;
               when 10 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*11)-1 downto STREAM_WSTB_WIDTH*10) <=  wstrb_in;
               when 11 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*12)-1 downto STREAM_WSTB_WIDTH*11) <=  wstrb_in;
               when 12 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*13)-1 downto STREAM_WSTB_WIDTH*12) <=  wstrb_in;
               when 13 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*14)-1 downto STREAM_WSTB_WIDTH*13) <=  wstrb_in;
               when 14 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*15)-1 downto STREAM_WSTB_WIDTH*14) <=  wstrb_in;
               when 15 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*16)-1 downto STREAM_WSTB_WIDTH*15) <=  wstrb_in;
               when 16 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*17)-1 downto STREAM_WSTB_WIDTH*16) <=  wstrb_in;
               when 17 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*18)-1 downto STREAM_WSTB_WIDTH*17) <=  wstrb_in;
               when 18 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*19)-1 downto STREAM_WSTB_WIDTH*18) <=  wstrb_in;
               when 19 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*20)-1 downto STREAM_WSTB_WIDTH*19) <=  wstrb_in;
               when 20 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*21)-1 downto STREAM_WSTB_WIDTH*20) <=  wstrb_in;
               when 21 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*22)-1 downto STREAM_WSTB_WIDTH*21) <=  wstrb_in;
               when 22 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*23)-1 downto STREAM_WSTB_WIDTH*22) <=  wstrb_in;
               when 23 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*24)-1 downto STREAM_WSTB_WIDTH*23) <=  wstrb_in;
               when 24 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*25)-1 downto STREAM_WSTB_WIDTH*24) <=  wstrb_in;
               when 25 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*26)-1 downto STREAM_WSTB_WIDTH*25) <=  wstrb_in;
               when 26 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*27)-1 downto STREAM_WSTB_WIDTH*26) <=  wstrb_in;
               when 27 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*28)-1 downto STREAM_WSTB_WIDTH*27) <=  wstrb_in;
               when 28 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*29)-1 downto STREAM_WSTB_WIDTH*28) <=  wstrb_in;
               when 29 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*30)-1 downto STREAM_WSTB_WIDTH*29) <=  wstrb_in;
               when 30 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*31)-1 downto STREAM_WSTB_WIDTH*30) <=  wstrb_in;
               when 31 =>                                                              
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*32)-1 downto STREAM_WSTB_WIDTH*31) <=  wstrb_in;
             
             
               when 32 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*33)-1 downto STREAM_WSTB_WIDTH*32) <=  wstrb_in;
               when 33 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*34)-1 downto STREAM_WSTB_WIDTH*33) <=  wstrb_in;
               when 34 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*35)-1 downto STREAM_WSTB_WIDTH*34) <=  wstrb_in;
               when 35 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*36)-1 downto STREAM_WSTB_WIDTH*35) <=  wstrb_in;
               when 36 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*37)-1 downto STREAM_WSTB_WIDTH*36) <=  wstrb_in;
               when 37 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*38)-1 downto STREAM_WSTB_WIDTH*37) <=  wstrb_in;
               when 38 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*39)-1 downto STREAM_WSTB_WIDTH*38) <=  wstrb_in;
               when 39 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*40)-1 downto STREAM_WSTB_WIDTH*39) <=  wstrb_in;
               when 40 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*41)-1 downto STREAM_WSTB_WIDTH*40) <=  wstrb_in;
               when 41 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*42)-1 downto STREAM_WSTB_WIDTH*41) <=  wstrb_in;
               when 42 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*43)-1 downto STREAM_WSTB_WIDTH*42) <=  wstrb_in;
               when 43 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*44)-1 downto STREAM_WSTB_WIDTH*43) <=  wstrb_in;
               when 44 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*45)-1 downto STREAM_WSTB_WIDTH*44) <=  wstrb_in;
               when 45 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*46)-1 downto STREAM_WSTB_WIDTH*45) <=  wstrb_in;
               when 46 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*47)-1 downto STREAM_WSTB_WIDTH*46) <=  wstrb_in;
               when 47 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*48)-1 downto STREAM_WSTB_WIDTH*47) <=  wstrb_in;
               when 48 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*49)-1 downto STREAM_WSTB_WIDTH*48) <=  wstrb_in;
               when 49 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*50)-1 downto STREAM_WSTB_WIDTH*49) <=  wstrb_in;
               when 50 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*51)-1 downto STREAM_WSTB_WIDTH*50) <=  wstrb_in;
               when 51 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*52)-1 downto STREAM_WSTB_WIDTH*51) <=  wstrb_in;
               when 52 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*53)-1 downto STREAM_WSTB_WIDTH*52) <=  wstrb_in;
               when 53 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*54)-1 downto STREAM_WSTB_WIDTH*53) <=  wstrb_in;
               when 54 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*55)-1 downto STREAM_WSTB_WIDTH*54) <=  wstrb_in;
               when 55 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*56)-1 downto STREAM_WSTB_WIDTH*55) <=  wstrb_in;
               when 56 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*57)-1 downto STREAM_WSTB_WIDTH*56) <=  wstrb_in;
               when 57 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*58)-1 downto STREAM_WSTB_WIDTH*57) <=  wstrb_in;
               when 58 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*59)-1 downto STREAM_WSTB_WIDTH*58) <=  wstrb_in;
               when 59 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*60)-1 downto STREAM_WSTB_WIDTH*59) <=  wstrb_in;
               when 60 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*61)-1 downto STREAM_WSTB_WIDTH*60) <=  wstrb_in;
               when 61 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*62)-1 downto STREAM_WSTB_WIDTH*61) <=  wstrb_in;
               when 62 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*63)-1 downto STREAM_WSTB_WIDTH*62) <=  wstrb_in;
               when 63 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*64)-1 downto STREAM_WSTB_WIDTH*63) <=  wstrb_in;
              
               
               when 64 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*65)-1 downto STREAM_WSTB_WIDTH*64) <=  wstrb_in;
               when 65 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*66)-1 downto STREAM_WSTB_WIDTH*65) <=  wstrb_in;
               when 66 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*67)-1 downto STREAM_WSTB_WIDTH*66) <=  wstrb_in;
               when 67 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*68)-1 downto STREAM_WSTB_WIDTH*67) <=  wstrb_in;
               when 68 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*69)-1 downto STREAM_WSTB_WIDTH*68) <=  wstrb_in;
               when 69 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*70)-1 downto STREAM_WSTB_WIDTH*69) <=  wstrb_in;
               when 70 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*71)-1 downto STREAM_WSTB_WIDTH*70) <=  wstrb_in;
               when 71 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*72)-1 downto STREAM_WSTB_WIDTH*71) <=  wstrb_in;
               when 72 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*73)-1 downto STREAM_WSTB_WIDTH*72) <=  wstrb_in;
               when 73 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*74)-1 downto STREAM_WSTB_WIDTH*73) <=  wstrb_in;
               when 74 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*75)-1 downto STREAM_WSTB_WIDTH*74) <=  wstrb_in;
               when 75 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*76)-1 downto STREAM_WSTB_WIDTH*75) <=  wstrb_in;
               when 76 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*77)-1 downto STREAM_WSTB_WIDTH*76) <=  wstrb_in;
               when 77 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*78)-1 downto STREAM_WSTB_WIDTH*77) <=  wstrb_in;
               when 78 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*79)-1 downto STREAM_WSTB_WIDTH*78) <=  wstrb_in;
               when 79 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*80)-1 downto STREAM_WSTB_WIDTH*79) <=  wstrb_in;
               when 80 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*81)-1 downto STREAM_WSTB_WIDTH*80) <=  wstrb_in;
               when 81 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*82)-1 downto STREAM_WSTB_WIDTH*81) <=  wstrb_in;
               when 82 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*83)-1 downto STREAM_WSTB_WIDTH*82) <=  wstrb_in;
               when 83 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*84)-1 downto STREAM_WSTB_WIDTH*83) <=  wstrb_in;
               when 84 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*85)-1 downto STREAM_WSTB_WIDTH*84) <=  wstrb_in;
               when 85 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*86)-1 downto STREAM_WSTB_WIDTH*85) <=  wstrb_in;
               when 86 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*87)-1 downto STREAM_WSTB_WIDTH*86) <=  wstrb_in;
               when 87 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*88)-1 downto STREAM_WSTB_WIDTH*87) <=  wstrb_in;
               when 88 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*89)-1 downto STREAM_WSTB_WIDTH*88) <=  wstrb_in;
               when 89 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*90)-1 downto STREAM_WSTB_WIDTH*89) <=  wstrb_in;
               when 90 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*91)-1 downto STREAM_WSTB_WIDTH*90) <=  wstrb_in;
               when 91 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*92)-1 downto STREAM_WSTB_WIDTH*91) <=  wstrb_in;
               when 92 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*93)-1 downto STREAM_WSTB_WIDTH*92) <=  wstrb_in;
               when 93 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*94)-1 downto STREAM_WSTB_WIDTH*93) <=  wstrb_in;
               when 94 =>                                                                   
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*95)-1 downto STREAM_WSTB_WIDTH*94) <=  wstrb_in;
               when 95 =>                                                                 
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*96)-1 downto STREAM_WSTB_WIDTH*95) <=  wstrb_in;
             
             
               when 96 =>
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*97 )-1 downto STREAM_WSTB_WIDTH*96 ) <=  wstrb_in;
               when 97 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*98 )-1 downto STREAM_WSTB_WIDTH*97 ) <=  wstrb_in;
               when 98 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*99 )-1 downto STREAM_WSTB_WIDTH*98 ) <=  wstrb_in;
               when 99 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*100)-1 downto STREAM_WSTB_WIDTH*99 ) <=  wstrb_in;
               when 100 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*101)-1 downto STREAM_WSTB_WIDTH*100) <=  wstrb_in;
               when 101 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*102)-1 downto STREAM_WSTB_WIDTH*101) <=  wstrb_in;
               when 102 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*103)-1 downto STREAM_WSTB_WIDTH*102) <=  wstrb_in;
               when 103 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*104)-1 downto STREAM_WSTB_WIDTH*103) <=  wstrb_in;
               when 104 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*105)-1 downto STREAM_WSTB_WIDTH*104) <=  wstrb_in;
               when 105 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*106)-1 downto STREAM_WSTB_WIDTH*105) <=  wstrb_in;
               when 106 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*107)-1 downto STREAM_WSTB_WIDTH*106) <=  wstrb_in;
               when 107 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*108)-1 downto STREAM_WSTB_WIDTH*107) <=  wstrb_in;
               when 108 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*109)-1 downto STREAM_WSTB_WIDTH*108) <=  wstrb_in;
               when 109 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*110)-1 downto STREAM_WSTB_WIDTH*109) <=  wstrb_in;
               when 110 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*111)-1 downto STREAM_WSTB_WIDTH*110) <=  wstrb_in;
               when 111 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*112)-1 downto STREAM_WSTB_WIDTH*111) <=  wstrb_in;
               when 112 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*113)-1 downto STREAM_WSTB_WIDTH*112) <=  wstrb_in;
               when 113 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*114)-1 downto STREAM_WSTB_WIDTH*113) <=  wstrb_in;
               when 114 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*115)-1 downto STREAM_WSTB_WIDTH*114) <=  wstrb_in;
               when 115 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*116)-1 downto STREAM_WSTB_WIDTH*115) <=  wstrb_in;
               when 116 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*117)-1 downto STREAM_WSTB_WIDTH*116) <=  wstrb_in;
               when 117 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*118)-1 downto STREAM_WSTB_WIDTH*117) <=  wstrb_in;
               when 118 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*119)-1 downto STREAM_WSTB_WIDTH*118) <=  wstrb_in;
               when 119 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*120)-1 downto STREAM_WSTB_WIDTH*119) <=  wstrb_in;
               when 120 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*121)-1 downto STREAM_WSTB_WIDTH*120) <=  wstrb_in;
               when 121 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*122)-1 downto STREAM_WSTB_WIDTH*121) <=  wstrb_in;
               when 122 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*123)-1 downto STREAM_WSTB_WIDTH*122) <=  wstrb_in;
               when 123 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*124)-1 downto STREAM_WSTB_WIDTH*123) <=  wstrb_in;
               when 124 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*125)-1 downto STREAM_WSTB_WIDTH*124) <=  wstrb_in;
               when 125 =>                                                                    
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*126)-1 downto STREAM_WSTB_WIDTH*125) <=  wstrb_in;
               when 126 =>                                                                 
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*127)-1 downto STREAM_WSTB_WIDTH*126) <=  wstrb_in;
               
               when others => -- 127 case
                   lsig_demux_wstrb_out((STREAM_WSTB_WIDTH*128)-1 downto STREAM_WSTB_WIDTH*127) <=  wstrb_in;
             
             
             
             end case;
          
           end process DO_128XN_DEMUX; 
 
         
       end generate GEN_128XN;
  
 
  
  
  
  
  
  
  
  
  end implementation;


-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_sg_scc_wr.vhd
  --
  -- Description:     
  --    This file implements the DataMover Lite Master Simple Command Calculator (SCC).                 
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  
  
  -------------------------------------------------------------------------------
  
  entity axi_sg_scc_wr is
    generic (
      
      C_SEL_ADDR_WIDTH     : Integer range  1 to  8 :=  5;
        -- Sets the width of the LS address bus used for 
        -- Muxing/Demuxing data to/from a wider AXI4 data bus
      
      C_ADDR_WIDTH         : Integer range 32 to 64 := 32;
        -- Sets the width of the AXi Address Channel 
      
      C_STREAM_DWIDTH      : Integer range  8 to 64 := 32;
        -- Sets the width of the Native Data width that
        -- is being supported by the PCC
      
      C_MAX_BURST_LEN      : Integer range 16 to 64 := 16;
        -- Indicates the max allowed burst length to use for
        -- AXI4 transfer calculations
      
      C_CMD_WIDTH          : Integer                := 68;
        -- Sets the width of the input command port
      
      C_TAG_WIDTH          : Integer range  1 to  8 := 4;
        -- Sets the width of the Tag field in the input command

      C_ENABLE_EXTRA_FIELD   : Integer range 0 to 1 := 1

      );
    port (
      
      -- Clock and Reset inputs -------------------------------------
      primary_aclk         : in  std_logic;                        --
         -- Primary synchronization clock for the Master side      --
         -- interface and internal logic. It is also used          --
         -- for the User interface synchronization when            --
         -- C_STSCMD_IS_ASYNC = 0.                                 --
                                                                   --
      -- Reset input                                               --
      mmap_reset           : in  std_logic;                        --
        -- Reset used for the internal master logic                --
      ---------------------------------------------------------------
      
      
     
      -- Command Input Interface ---------------------------------------------------------
                                                                                        --
      cmd2mstr_command      : in std_logic_vector(C_CMD_WIDTH-1 downto 0);              --
         -- The next command value available from the Command FIFO/Register             --
                                                                                        --
      cache2mstr_command      : in std_logic_vector(7 downto 0);              --
         -- The next command value available from the Command FIFO/Register             --
                                                                                        --
      cmd2mstr_cmd_valid    : in std_logic;                                             --
         -- Handshake bit indicating if the Command FIFO/Register has at leasdt 1 entry --
                                                                                        --
      mst2cmd_cmd_ready     : out  std_logic;                                           --
         -- Handshake bit indicating the Command Calculator is ready to accept          --
         -- another command                                                             --
      ------------------------------------------------------------------------------------
      
      
      
      -- Address Channel Controller Interface --------------------------------------------
                                                                                        --
      mstr2addr_tag       : out std_logic_vector(C_TAG_WIDTH-1 downto 0);               --
         -- The next command tag                                                        --
                                                                                        --
      mstr2addr_addr      : out std_logic_vector(C_ADDR_WIDTH-1 downto 0);              --
         -- The next command address to put on the AXI MMap ADDR                        --
                                                                                        --
      mstr2addr_len       : out std_logic_vector(7 downto 0);                           --
         -- The next command length to put on the AXI MMap LEN                          --
                                                                                        --
      mstr2addr_size      : out std_logic_vector(2 downto 0);                           --
         -- The next command size to put on the AXI MMap SIZE                           --
                                                                                        --
      mstr2addr_burst     : out std_logic_vector(1 downto 0);                           --
         -- The next command burst type to put on the AXI MMap BURST                    --
                                                                                        --
      mstr2addr_cache     : out std_logic_vector(3 downto 0);                           --
         -- The next command burst type to put on the AXI MMap BURST                    --
                                                                                        --
      mstr2addr_user      : out std_logic_vector(3 downto 0);                           --
         -- The next command burst type to put on the AXI MMap BURST                    --
                                                                                        --
      mstr2addr_cmd_cmplt : out std_logic;                                              --
         -- The indication to the Address Channel that the current                      --
         -- sub-command output is the last one compiled from the                        --
         -- parent command pulled from the Command FIFO                                 --
                                                                                        --
      mstr2addr_calc_error : out std_logic;                                             --
         -- Indication if the next command in the calculation pipe                      --
         -- has a calcualtion error                                                     --
                                                                                        --
      mstr2addr_cmd_valid : out std_logic;                                              --
         -- The next command valid indication to the Address Channel                    --
         -- Controller for the AXI MMap                                                 --
                                                                                        --
      addr2mstr_cmd_ready : In std_logic;                                               --
         -- Indication from the Address Channel Controller that the                     --
         -- command is being accepted                                                   --
      ------------------------------------------------------------------------------------
      
      
      
      -- Data Channel Controller Interface  ----------------------------------------------
                                                                                        --
      mstr2data_tag        : out std_logic_vector(C_TAG_WIDTH-1 downto 0);              --
         -- The next command tag                                                        --
                                                                                        --
      mstr2data_saddr_lsb  : out std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0);         --
         -- The next command start address LSbs to use for the read data                --
         -- mux (only used if Stream data width is 8 or 16 bits).                       --
                                                                                        --
      mstr2data_len        : out std_logic_vector(7 downto 0);                          --
         -- The LEN value output to the Address Channel                                 --
                                                                                        --
      mstr2data_strt_strb  : out std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);      --
         -- The starting strobe value to use for the data transfer                      --
                                                                                        --
      mstr2data_last_strb  : out std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);      --
         -- The endiing (LAST) strobe value to use for the data transfer                --
                                                                                        --
      mstr2data_sof        : out std_logic;                                             --
         -- The starting tranfer of a sequence of transfers                             --
                                                                                        --
      mstr2data_eof        : out std_logic;                                             --
         -- The endiing tranfer of a sequence of parent transfer commands               --
                                                                                        --
      mstr2data_calc_error : out std_logic;                                             --
         -- Indication if the next command in the calculation pipe                      --
         -- has a calculation error                                                     --
                                                                                        --
      mstr2data_cmd_cmplt  : out std_logic;                                             --
         -- The indication to the Data Channel that the current                         --
         -- sub-command output is the last one compiled from the                        --
         -- parent command pulled from the Command FIFO                                 --
                                                                                        --
      mstr2data_cmd_valid  : out std_logic;                                             --
         -- The next command valid indication to the Data Channel                       --
         -- Controller for the AXI MMap                                                 --
                                                                                        --
      data2mstr_cmd_ready  : In std_logic ;                                             --
         -- Indication from the Data Channel Controller that the                        --
         -- command is being accepted on the AXI Address                                --
         -- Channel                                                                     --
                                                                                        --
      calc_error           : Out std_logic                                              --
         -- Indication from the Command Calculator that a calculation                   --
         -- error has occured.                                                          --
      ------------------------------------------------------------------------------------
     
      );
  
  end entity axi_sg_scc_wr;
  
  
  architecture implementation of axi_sg_scc_wr is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    
  
  
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_get_slice_width
    --
    -- Function Description:
    -- Calculates the bits to rip from the Command BTT field to calculate
    -- the LEN value output to the AXI Address Channel.
    --
    -------------------------------------------------------------------
    function funct_get_slice_width (max_burst_len : integer) return integer is
                                    
    
      Variable temp_slice_width : Integer := 0;
    
    begin
  
      case max_burst_len is
       -- coverage off 
        when 64 =>
          temp_slice_width := 7;
        when 32 =>
          temp_slice_width := 6;
        when others =>   -- assume 16 dbeats is max LEN
          temp_slice_width := 5;
       -- coverage on 
      end case;
      
      Return (temp_slice_width);
     
    end function funct_get_slice_width;
    
    
    
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_get_residue_width
    --
    -- Function Description:
    --  Calculates the number of Least significant bits of the BTT field
    -- that are unused for the LEN calculation
    --
    -------------------------------------------------------------------
    function funct_get_btt_ls_unused (transfer_width : integer) return integer is
    
      Variable temp_btt_ls_unused : Integer := 0; -- 8-bit stream
      
    begin
    
      case transfer_width is
       -- coverage off 
        when 64 =>
            temp_btt_ls_unused := 3;
       -- coverage on 
        when 32 =>
            temp_btt_ls_unused := 2;
       -- coverage off 
        when 16 =>
            temp_btt_ls_unused := 1;
        when others =>  -- assume 8-bit transfers
            temp_btt_ls_unused := 0;
       -- coverage on 
      end case;
      
      Return (temp_btt_ls_unused);
     
    end function funct_get_btt_ls_unused;
    
    
    
    
    
    
    
    
    
    -- Constant Declarations  ----------------------------------------
    
    Constant BASE_CMD_WIDTH      : integer := 32; -- Bit Width of Command LS (no address)
    Constant CMD_TYPE_INDEX      : integer := 23;
    Constant CMD_ADDR_LS_INDEX   : integer := BASE_CMD_WIDTH;
    Constant CMD_ADDR_MS_INDEX   : integer := (C_ADDR_WIDTH+BASE_CMD_WIDTH)-1;
    Constant CMD_TAG_WIDTH       : integer := C_TAG_WIDTH;
    Constant CMD_TAG_LS_INDEX    : integer := C_ADDR_WIDTH+BASE_CMD_WIDTH;
    Constant CMD_TAG_MS_INDEX    : integer := (CMD_TAG_LS_INDEX+CMD_TAG_WIDTH)-1;
    Constant AXI_BURST_FIXED     : std_logic_vector(1 downto 0) := "00";
    Constant AXI_BURST_INCR      : std_logic_vector(1 downto 0) := "01";
    Constant AXI_BURST_WRAP      : std_logic_vector(1 downto 0) := "10";
    Constant AXI_BURST_RESVD     : std_logic_vector(1 downto 0) := "11";
    Constant AXI_SIZE_1BYTE      : std_logic_vector(2 downto 0) := "000"; 
    Constant AXI_SIZE_2BYTE      : std_logic_vector(2 downto 0) := "001"; 
    Constant AXI_SIZE_4BYTE      : std_logic_vector(2 downto 0) := "010"; 
    Constant AXI_SIZE_8BYTE      : std_logic_vector(2 downto 0) := "011"; 
    Constant AXI_SIZE_16BYTE     : std_logic_vector(2 downto 0) := "100"; 
    Constant AXI_SIZE_32BYTE     : std_logic_vector(2 downto 0) := "101"; 
    Constant AXI_SIZE_64BYTE     : std_logic_vector(2 downto 0) := "110"; 
    Constant AXI_SIZE_128BYTE    : std_logic_vector(2 downto 0) := "111"; 
    Constant BTT_SLICE_SIZE      : integer := funct_get_slice_width(C_MAX_BURST_LEN);
    Constant MAX_BURST_LEN_US    : unsigned(BTT_SLICE_SIZE-1 downto 0) := 
                                   TO_UNSIGNED(C_MAX_BURST_LEN-1, BTT_SLICE_SIZE);
    Constant BTT_LS_UNUSED_WIDTH : integer := funct_get_btt_ls_unused(C_STREAM_DWIDTH);
    Constant CMD_BTT_WIDTH       : integer :=  BTT_SLICE_SIZE+BTT_LS_UNUSED_WIDTH;
    Constant CMD_BTT_LS_INDEX    : integer :=  0;
    Constant CMD_BTT_MS_INDEX    : integer :=  CMD_BTT_WIDTH-1;
    Constant BTT_ZEROS           : std_logic_vector(CMD_BTT_WIDTH-1 downto 0) := (others => '0'); 
    Constant BTT_RESIDUE_ZEROS   : unsigned(BTT_LS_UNUSED_WIDTH-1 downto 0) := (others => '0'); 
    Constant BTT_SLICE_ONE       : unsigned(BTT_SLICE_SIZE-1 downto 0) := TO_UNSIGNED(1, BTT_SLICE_SIZE); 
    Constant STRB_WIDTH          : integer := C_STREAM_DWIDTH/8; -- Number of bytes in the Stream
    Constant LEN_WIDTH           : integer := 8; 
    
               
               
    -- Type Declarations  --------------------------------------------
    
    type SCC_SM_STATE_TYPE is (
                INIT,
                POP_RECOVER,
                GET_NXT_CMD,
                CHK_AND_CALC,
                PUSH_TO_AXI,
                ERROR_TRAP
                );
  
  
    
    
    
    -- Signal Declarations  --------------------------------------------
  
    signal sm_scc_state              : SCC_SM_STATE_TYPE := INIT;
    signal sm_scc_state_ns           : SCC_SM_STATE_TYPE := INIT;
    signal sm_pop_input_cmd          : std_logic := '0';
    signal sm_pop_input_cmd_ns       : std_logic := '0';
    signal sm_set_push2axi           : std_logic := '0';
    signal sm_set_push2axi_ns        : std_logic := '0';
    signal sm_set_error              : std_logic := '0';
    signal sm_set_error_ns           : std_logic := '0';
    Signal sm_scc_sm_ready           : std_logic := '0';
    Signal sm_scc_sm_ready_ns        : std_logic := '0';
    signal sig_cmd2data_valid        : std_logic := '0';
    signal sig_clr_cmd2data_valid    : std_logic := '0';
    signal sig_cmd2addr_valid        : std_logic := '0';
    signal sig_cmd2addr_valid1        : std_logic := '0';
    signal sig_clr_cmd2addr_valid    : std_logic := '0';
    signal sig_addr_data_rdy_pending : std_logic := '0';
    signal sig_cmd_btt_slice         : std_logic_vector(CMD_BTT_WIDTH-1 downto 0) := (others => '0');
    signal sig_load_input_cmd        : std_logic := '0';
    signal sig_cmd_reg_empty         : std_logic := '0';
    signal sig_cmd_reg_full          : std_logic := '0';
    signal sig_cmd_addr_reg          : std_logic_vector(C_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_cmd_btt_reg           : std_logic_vector(CMD_BTT_WIDTH-1 downto 0) := (others => '0');
    signal sig_cmd_type_reg          : std_logic := '0';
    signal sig_cmd_burst_reg         : std_logic_vector (1 downto 0) := "00";
    signal sig_cmd_tag_reg           : std_logic_vector(CMD_TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_data_rdy4cmd     : std_logic := '0';
    signal sig_btt_raw               : std_logic := '0';
    signal sig_btt_is_zero           : std_logic := '0';
    signal sig_btt_is_zero_reg       : std_logic := '0';
    signal sig_next_tag              : std_logic_vector(CMD_TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_next_addr             : std_logic_vector(C_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_next_len              : std_logic_vector(LEN_WIDTH-1 downto 0) := (others => '0');
    signal sig_next_size             : std_logic_vector(2 downto 0) := (others => '0');
    signal sig_next_burst            : std_logic_vector(1 downto 0) := (others => '0');
    signal sig_next_cache            : std_logic_vector(3 downto 0) := (others => '0');
    signal sig_next_user             : std_logic_vector(3 downto 0) := (others => '0');
    signal sig_next_strt_strb        : std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0) := (others => '0');
    signal sig_next_end_strb         : std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0) := (others => '0');
    
  
    
    
  begin --(architecture implementation)
     
    -- Assign calculation error output 
    calc_error            <= sm_set_error;
   
    -- Assign the ready output to the Command FIFO 
    mst2cmd_cmd_ready     <= sig_cmd_reg_empty and addr2mstr_cmd_ready; --sm_scc_sm_ready;
    
    -- Assign the Address Channel Controller Qualifiers
    mstr2addr_tag         <= sig_next_tag  ; 
    mstr2addr_addr        <= sig_next_addr ; 
    mstr2addr_len         <= sig_next_len  ; 
    mstr2addr_size        <= sig_next_size ;
    mstr2addr_burst       <= sig_cmd_burst_reg; 
    mstr2addr_cache       <= sig_next_cache; 
    mstr2addr_user        <= sig_next_user; 
    mstr2addr_cmd_valid   <= sig_cmd2addr_valid1;
    mstr2addr_calc_error  <= sm_set_error  ; 
    mstr2addr_cmd_cmplt   <= '1'           ;   -- Lite mode is always 1 
    
    -- Assign the Data Channel Controller Qualifiers
    mstr2data_tag         <= sig_next_tag ; 
    mstr2data_saddr_lsb   <= sig_cmd_addr_reg(C_SEL_ADDR_WIDTH-1 downto 0);
    
    mstr2data_len         <= sig_next_len ;
    
    mstr2data_strt_strb   <= (others => '1'); --sig_next_strt_strb; -- always F
    mstr2data_last_strb   <= (others => '1'); --sig_next_end_strb;  -- always F
    mstr2data_sof         <= '1';  -- Lite mode is always 1 cmd
    mstr2data_eof         <= '1';  -- Lite mode is always 1 cmd
    mstr2data_cmd_cmplt   <= '1';  -- Lite mode is always 1 cmd
 --   mstr2data_cmd_valid   <= sig_cmd2data_valid;
    mstr2data_cmd_valid   <= sig_cmd2addr_valid1; --sig_cmd2data_valid;
    mstr2data_calc_error  <= sm_set_error;   
    
    
    -- Internal logic ------------------------------
    sig_addr_data_rdy_pending  <=  sig_cmd2addr_valid or 
                                   sig_cmd2data_valid;
   
    sig_clr_cmd2data_valid     <=  sig_cmd2data_valid and data2mstr_cmd_ready;
    
    sig_clr_cmd2addr_valid     <=  sig_cmd2addr_valid and addr2mstr_cmd_ready;
    
       sig_load_input_cmd         <=  cmd2mstr_cmd_valid and
                                   sig_cmd_reg_empty;--  and
                                 --  sm_scc_sm_ready;
 
    
    sig_next_tag               <=  sig_cmd_tag_reg;
    
    sig_next_addr              <=  sig_cmd_addr_reg;
    
    sig_addr_data_rdy4cmd      <=  addr2mstr_cmd_ready and data2mstr_cmd_ready;
    
    sig_cmd_btt_slice          <=  cmd2mstr_command(CMD_BTT_MS_INDEX downto CMD_BTT_LS_INDEX);
    
    sig_btt_is_zero  <= '1'
      when  (sig_cmd_btt_slice = BTT_ZEROS)
      Else '0';
    
  
  
    
 
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_NO_RESIDUE_BITS
    --
    -- If Generate Description:
    --
    --
    --
    ------------------------------------------------------------
    GEN_NO_RESIDUE_BITS : if (BTT_LS_UNUSED_WIDTH = 0) generate
    
    
       -- signals 
       signal sig_len_btt_slice         : unsigned(BTT_SLICE_SIZE-1 downto 0) := (others => '0');
       signal sig_len_btt_slice_minus_1 : unsigned(BTT_SLICE_SIZE-1 downto 0) := (others => '0');
       signal sig_len2use               : unsigned(BTT_SLICE_SIZE-1 downto 0) := (others => '0');
       
     
       begin
    
       -- LEN Calculation logic ------------------------------------------ 
        
         sig_next_len         <= STD_LOGIC_VECTOR(RESIZE(sig_len2use, LEN_WIDTH));
        
         sig_len_btt_slice    <= UNSIGNED(sig_cmd_btt_reg(CMD_BTT_MS_INDEX downto 0));
         
         sig_len_btt_slice_minus_1 <= sig_len_btt_slice-BTT_SLICE_ONE
          when sig_btt_is_zero_reg = '0'
          else (others => '0');    -- clip at zero
         
         
         -- If most significant bit of BTT set then limit to 
         -- Max Burst Len, else rip it from the BTT value,
         -- otheriwse subtract 1 from the BTT ripped value
         -- 1 from the BTT ripped value
         sig_len2use <= MAX_BURST_LEN_US 
           When (sig_cmd_btt_reg(CMD_BTT_MS_INDEX) = '1')
           Else sig_len_btt_slice_minus_1;
           
        
       end generate GEN_NO_RESIDUE_BITS;
        
    
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_HAS_RESIDUE_BITS
    --
    -- If Generate Description:
    --
    --
    --
    ------------------------------------------------------------
    GEN_HAS_RESIDUE_BITS : if (BTT_LS_UNUSED_WIDTH > 0) generate
    
       -- signals 
       signal sig_btt_len_residue       : unsigned(BTT_LS_UNUSED_WIDTH-1 downto 0) := (others => '0');
       signal sig_len_btt_slice         : unsigned(BTT_SLICE_SIZE-1 downto 0) := (others => '0');
       signal sig_len_btt_slice_minus_1 : unsigned(BTT_SLICE_SIZE-1 downto 0) := (others => '0');
       signal sig_len2use               : unsigned(BTT_SLICE_SIZE-1 downto 0) := (others => '0');
       
     
       begin
    
       -- LEN Calculation logic ------------------------------------------ 
       WR_EXTRA_FIELDS : if (C_ENABLE_EXTRA_FIELD = 1) generate
         sig_next_len         <= "00000000" when sig_cmd_tag_reg (0) = '1'
                                 else "00000101"; --STD_LOGIC_VECTOR(RESIZE(sig_len2use, LEN_WIDTH));
       end generate WR_EXTRA_FIELDS;

       NOWR_EXTRA_FIELDS : if (C_ENABLE_EXTRA_FIELD = 0) generate
         sig_next_len         <= "00000000";
       end generate NOWR_EXTRA_FIELDS;
 
--         sig_next_len         <= STD_LOGIC_VECTOR(RESIZE(sig_len2use, LEN_WIDTH));
        
         sig_len_btt_slice    <= UNSIGNED(sig_cmd_btt_reg(CMD_BTT_MS_INDEX downto BTT_LS_UNUSED_WIDTH));
         
         sig_len_btt_slice_minus_1 <= sig_len_btt_slice-BTT_SLICE_ONE
          when sig_btt_is_zero_reg = '0'
          else (others => '0');    -- clip at zero
         
         sig_btt_len_residue  <= UNSIGNED(sig_cmd_btt_reg(BTT_LS_UNUSED_WIDTH-1 downto 0));
            
         
         -- If most significant bit of BTT set then limit to 
         -- Max Burst Len, else rip it from the BTT value
         -- However if residue bits are zeroes then subtract
         -- 1 from the BTT ripped value
         sig_len2use <= MAX_BURST_LEN_US 
           When (sig_cmd_btt_reg(CMD_BTT_MS_INDEX) = '1')
           Else sig_len_btt_slice_minus_1
           when (sig_btt_len_residue = BTT_RESIDUE_ZEROS)
           Else sig_len_btt_slice;

        
       end generate GEN_HAS_RESIDUE_BITS;
        
        
        
        
        
        
        
     
     
     
    
    
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: REG_INPUT_CMD
    --
    -- Process Description:
    --  Implements the input command holding registers
    --
    -------------------------------------------------------------
    REG_INPUT_CMD : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset       = '1' or
                addr2mstr_cmd_ready = '0') then
--                sm_pop_input_cmd = '1') then
            
              sig_cmd_btt_reg      <=  (others => '0');
              sig_cmd_type_reg     <=  '0';
              sig_cmd_addr_reg     <=  (others => '0');
              sig_cmd_tag_reg      <=  (others => '0');
              sig_btt_is_zero_reg  <=  '0';
              
              sig_cmd_reg_empty    <=  '1';
              sig_cmd_reg_full     <=  '0';
              sig_cmd_burst_reg    <=  "00";
              sig_cmd2addr_valid1 <= '0';
            
            elsif (sig_load_input_cmd = '1') then
              
              sig_cmd_btt_reg      <= sig_cmd_btt_slice;
              sig_cmd_type_reg     <= cmd2mstr_command(CMD_TYPE_INDEX);
              sig_cmd_addr_reg     <= cmd2mstr_command(CMD_ADDR_MS_INDEX downto CMD_ADDR_LS_INDEX);  
              sig_cmd_tag_reg      <=  cmd2mstr_command(CMD_TAG_MS_INDEX downto CMD_TAG_LS_INDEX);
              sig_btt_is_zero_reg  <= sig_btt_is_zero;
              
              sig_cmd_reg_empty    <=  '0';
              sig_cmd_reg_full     <=  '1';
              sig_cmd2addr_valid1 <= '1';
              sig_cmd_burst_reg    <= sig_next_burst;
            
            else
              null; -- Hold current State
            end if; 
         end if;       
       end process REG_INPUT_CMD; 
   
   
    
    
    -- Only Incrementing Burst type supported (per Interface_X guidelines)
    sig_next_burst <= AXI_BURST_INCR when (cmd2mstr_command(CMD_TYPE_INDEX) = '1') else
                      AXI_BURST_FIXED;
    sig_next_user <= cache2mstr_command (7 downto 4);   
    sig_next_cache  <= cache2mstr_command (3 downto 0);   
 
 
  
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_LEN_SDWIDTH_64
    --
    -- If Generate Description:
    --  This IfGen implements the AXI LEN qualifier calculation
    -- and the Stream data channel start/end STRB value.  
    --
    -- This IfGen is for the 64-bit Stream data Width case.
    --
    ------------------------------------------------------------
    GEN_LEN_SDWIDTH_64 : if (C_STREAM_DWIDTH = 64) generate
    
       -- Local Constants
       Constant AXI_SIZE2USE      : std_logic_vector(2 downto 0) := AXI_SIZE_8BYTE;
       Constant RESIDUE_BIT_WIDTH : integer := 3;
       
       
       
       -- local signals
       signal sig_last_strb2use              : std_logic_vector(STRB_WIDTH-1 downto 0) := (others => '0');
       signal sig_last_strb                  : std_logic_vector(STRB_WIDTH-1 downto 0) := (others => '0');
       Signal sig_btt_ms_bit_value           : std_logic := '0';
       signal lsig_btt_len_residue           : std_logic_vector(BTT_LS_UNUSED_WIDTH-1 downto 0) := (others => '0');
       signal sig_btt_len_residue_composite  : std_logic_vector(RESIDUE_BIT_WIDTH downto 0) := (others => '0');
                                                                -- note 1 extra bit implied
                                                                
       
       begin
           
         -- Assign the Address Channel Controller Size Qualifier Value
         sig_next_size        <= AXI_SIZE2USE;
         
         -- Assign the Strobe Values
         sig_next_strt_strb   <= (others => '1'); -- always aligned on first databeat for LITE DataMover
         sig_next_end_strb    <= sig_last_strb;
        
        
         -- Local calculations ------------------------------
         
         lsig_btt_len_residue  <= sig_cmd_btt_reg(BTT_LS_UNUSED_WIDTH-1 downto 0);
         
         sig_btt_ms_bit_value  <= sig_cmd_btt_reg(CMD_BTT_MS_INDEX);
         
         sig_btt_len_residue_composite <= sig_btt_ms_bit_value &
                                          lsig_btt_len_residue;
         
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: IMP_LAST_STRB_8bit
         --
         -- Process Description:
         -- Generates the Strobe values for the LAST databeat of the
         -- Burst to MMap when the Stream is 64 bits wide and 8 strobe
         -- bits are required.
         --
         -------------------------------------------------------------
         IMP_LAST_STRB_8bit : process (sig_btt_len_residue_composite)
            begin
         
             case sig_btt_len_residue_composite is
               when "0001" =>
                 sig_last_strb <= "00000001";
               when "0010" =>
                 sig_last_strb <= "00000011";
               when "0011" =>
                 sig_last_strb <= "00000111";
               when "0100" =>
                 sig_last_strb <= "00001111";
               when "0101" =>
                 sig_last_strb <= "00011111";
               when "0110" =>
                 sig_last_strb <= "00111111";
               when "0111" =>
                 sig_last_strb <= "01111111";
               when others =>
                 sig_last_strb <= "11111111";
             end case;
             
             
            end process IMP_LAST_STRB_8bit; 
         
         
       end generate GEN_LEN_SDWIDTH_64;
  
  
  
  
  
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_LEN_SDWIDTH_32
    --
    -- If Generate Description:
    --  This IfGen implements the AXI LEN qualifier calculation
    -- and the Stream data channel start/end STRB value.  
    --
    -- This IfGen is for the 32-bit Stream data Width case.
    --
    ------------------------------------------------------------
    GEN_LEN_SDWIDTH_32 : if (C_STREAM_DWIDTH = 32) generate
    
       -- Local Constants
       Constant AXI_SIZE2USE                 : std_logic_vector(2 downto 0) := AXI_SIZE_4BYTE;
       Constant RESIDUE_BIT_WIDTH            : integer := 2;
       
       -- local signals
       signal sig_last_strb2use              : std_logic_vector(STRB_WIDTH-1 downto 0) := (others => '0');
       signal sig_last_strb                  : std_logic_vector(STRB_WIDTH-1 downto 0) := (others => '0');
       Signal sig_btt_ms_bit_value           : std_logic := '0';
       signal sig_btt_len_residue_composite  : std_logic_vector(RESIDUE_BIT_WIDTH downto 0) := (others => '0'); -- 1 extra bit
       signal lsig_btt_len_residue           : std_logic_vector(BTT_LS_UNUSED_WIDTH-1 downto 0) := (others => '0');
       
       
       begin
           
         -- Assign the Address Channel Controller Size Qualifier Value
         sig_next_size        <= AXI_SIZE2USE;
         
         -- Assign the Strobe Values
         sig_next_strt_strb   <= (others => '1'); -- always aligned on first databeat for LITE DataMover
         sig_next_end_strb    <= sig_last_strb;
        
        
         -- Local calculations ------------------------------
         
         lsig_btt_len_residue  <= sig_cmd_btt_reg(BTT_LS_UNUSED_WIDTH-1 downto 0);
         
         sig_btt_ms_bit_value <= sig_cmd_btt_reg(CMD_BTT_MS_INDEX);
         
         sig_btt_len_residue_composite <= sig_btt_ms_bit_value &
                                          lsig_btt_len_residue;
         
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: IMP_LAST_STRB_4bit
         --
         -- Process Description:
         -- Generates the Strobe values for the LAST databeat of the
         -- Burst to MMap when the Stream is 32 bits wide and 4 strobe
         -- bits are required.
         --
         -------------------------------------------------------------
         IMP_LAST_STRB_4bit : process (sig_btt_len_residue_composite)
            begin
         
             case sig_btt_len_residue_composite is
       -- coverage off 
               when "001" =>
                 sig_last_strb <= "0001";
               when "010" =>
                 sig_last_strb <= "0011";
               when "011" =>
                 sig_last_strb <= "0111";
       -- coverage on 
               when others =>
                 sig_last_strb <= "1111";
             end case;
             
             
            end process IMP_LAST_STRB_4bit; 
          
       end generate GEN_LEN_SDWIDTH_32;
  
  
  
  
  
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_LEN_SDWIDTH_16
    --
    -- If Generate Description:
    --  This IfGen implements the AXI LEN qualifier calculation
    -- and the Stream data channel start/end STRB value.  
    --
    -- This IfGen is for the 16-bit Stream data Width case.
    --
    ------------------------------------------------------------
    GEN_LEN_SDWIDTH_16 : if (C_STREAM_DWIDTH = 16) generate
    
       -- Local Constants
       Constant AXI_SIZE2USE      : std_logic_vector(2 downto 0) := AXI_SIZE_2BYTE;
       Constant RESIDUE_BIT_WIDTH : integer := 1;
       
       
       -- local signals
       signal sig_last_strb2use              : std_logic_vector(STRB_WIDTH-1 downto 0) := (others => '0');
       signal sig_last_strb                  : std_logic_vector(STRB_WIDTH-1 downto 0) := (others => '0');
       Signal sig_btt_ms_bit_value           : std_logic := '0';
       signal sig_btt_len_residue_composite  : std_logic_vector(RESIDUE_BIT_WIDTH downto 0) := (others => '0'); -- 1 extra bit
       signal lsig_btt_len_residue           : std_logic_vector(BTT_LS_UNUSED_WIDTH-1 downto 0) := (others => '0');
       
       
       begin
           
         -- Assign the Address Channel Controller Size Qualifier Value
         sig_next_size        <= AXI_SIZE2USE;
         
         -- Assign the Strobe Values
         sig_next_strt_strb   <= (others => '1'); -- always aligned on first databeat for LITE DataMover
         sig_next_end_strb    <= sig_last_strb;
        
        
         -- Local calculations ------------------------------
         
         lsig_btt_len_residue  <= sig_cmd_btt_reg(BTT_LS_UNUSED_WIDTH-1 downto 0);
         
         sig_btt_ms_bit_value  <= sig_cmd_btt_reg(CMD_BTT_MS_INDEX);
         
         sig_btt_len_residue_composite <= sig_btt_ms_bit_value &
                                          lsig_btt_len_residue;
         
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: IMP_LAST_STRB_2bit
         --
         -- Process Description:
         -- Generates the Strobe values for the LAST databeat of the
         -- Burst to MMap when the Stream is 16 bits wide and 2 strobe
         -- bits are required.
         --
         -------------------------------------------------------------
         IMP_LAST_STRB_2bit : process (sig_btt_len_residue_composite)
            begin
         
             case sig_btt_len_residue_composite is
               when "01" =>
                 sig_last_strb <= "01";
               when others =>
                 sig_last_strb <= "11";
             end case;
             
             
            end process IMP_LAST_STRB_2bit; 
         
         
       end generate GEN_LEN_SDWIDTH_16;
  
  
  
  
  
  
  
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_LEN_SDWIDTH_8
    --
    -- If Generate Description:
    --  This IfGen implements the AXI LEN qualifier calculation
    -- and the Stream data channel start/end STRB value.  
    --
    -- This IfGen is for the 8-bit Stream data Width case.
    --
    ------------------------------------------------------------
    GEN_LEN_SDWIDTH_8 : if (C_STREAM_DWIDTH = 8) generate
       
       -- Local Constants
       Constant AXI_SIZE2USE : std_logic_vector(2 downto 0) := AXI_SIZE_1BYTE;
       
       begin
    
           -- Assign the Address Channel Controller Qualifiers
         sig_next_size        <= AXI_SIZE2USE;
           
         -- Assign the Data Channel Controller Qualifiers
         sig_next_strt_strb   <= (others => '1');
         sig_next_end_strb    <= (others => '1');
          
          
       end generate GEN_LEN_SDWIDTH_8;
  
  
  
  
  
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: CMD2DATA_VALID_FLOP
    --
    -- Process Description:
    --  Implements the set/reset flop for the Command Ready control
    -- to the Data Controller Module.
    --
    -------------------------------------------------------------
    CMD2DATA_VALID_FLOP : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset             = '1' or
                sig_clr_cmd2data_valid = '1') then
    
              sig_cmd2data_valid <= '0';
              
            elsif (sm_set_push2axi_ns = '1') then
    
              sig_cmd2data_valid <= '1';
              
            else
              null; -- hold current state
            end if; 
         end if;       
       end process CMD2DATA_VALID_FLOP; 
      
      
      
      
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: CMD2ADDR_VALID_FLOP
    --
    -- Process Description:
    --  Implements the set/reset flop for the Command Ready control
    -- to the Address Controller Module.
    --
    -------------------------------------------------------------
    CMD2ADDR_VALID_FLOP : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset             = '1' or
                sig_clr_cmd2addr_valid = '1') then
    
              sig_cmd2addr_valid <= '0';
              
            elsif (sm_set_push2axi_ns = '1') then
    
              sig_cmd2addr_valid <= '1';
              
            else
              null; -- hold current state
            end if; 
         end if;       
       end process CMD2ADDR_VALID_FLOP; 
      
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: SCC_SM_REG
    --
    -- Process Description:
    -- Implements registered portion of state machine
    --
    -------------------------------------------------------------
    SCC_SM_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              
--              sm_scc_state      <= INIT;
--              sm_pop_input_cmd  <= '0' ;           
--              sm_set_push2axi   <= '0' ;           
              sm_set_error      <= '0' ;
--              sm_scc_sm_ready   <= '0' ;
              
            elsif (sig_btt_is_zero_reg = '1') then
              sm_set_error <= '1';
              
 --             sm_scc_state      <= sm_scc_state_ns     ;
 --             sm_pop_input_cmd  <= sm_pop_input_cmd_ns ;           
 --             sm_set_push2axi   <= sm_set_push2axi_ns  ;          
 --                  sm_set_error      <= sm_set_error_ns     ;
               
 --             sm_scc_sm_ready   <= sm_scc_sm_ready_ns  ;
              
            end if; 
         end if;       
       end process SCC_SM_REG; 
    
    
  
  
  
  end implementation;


-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:        axi_sg_skid2mm_buf.vhd
--
-- Description:     
--  Implements the AXi Skid Buffer in the Option 2 (Registerd outputs) mode.                
--
--  This Module also provides Write Data Bus Mirroring and WSTRB
--  Demuxing to match a narrow Stream to a wider MMap Write 
--  Channel. By doing this in the skid buffer, the resource 
--  utilization of the skid buffer can be minimized by only
--  having to buffer/mux the Stream data width, not the MMap
--  Data width.   
--                  
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;


library axi_sg_v4_1_6; 
use axi_sg_v4_1_6.axi_sg_wr_demux;

-------------------------------------------------------------------------------

entity axi_sg_skid2mm_buf is
  generic (
    
    C_MDATA_WIDTH         : INTEGER range 32 to 1024 := 32 ;
       --  Width of the MMap Write Data bus (in bits)
    C_SDATA_WIDTH         : INTEGER range 8 to 1024 := 32 ;
       --  Width of the Stream Data bus (in bits)
    C_ADDR_LSB_WIDTH      : INTEGER range 1 to 8 := 5
       --  Width of the LS address bus needed to Demux the WSTRB
       
    );
  port (
     
     -- Clock and Reset Inputs -------------------------------------------
                                                                        --
     ACLK         : In  std_logic ;                                     --
     ARST         : In  std_logic ;                                     --
     ---------------------------------------------------------------------
     
      
     -- Slave Side (Wr Data Controller Input Side) -----------------------
                                                                        --
     S_ADDR_LSB   : in  std_logic_vector(C_ADDR_LSB_WIDTH-1 downto 0);  --
     S_VALID      : In  std_logic ;                                     --
     S_READY      : Out std_logic ;                                     --
     S_DATA       : In  std_logic_vector(C_SDATA_WIDTH-1 downto 0);     --
     S_STRB       : In  std_logic_vector((C_SDATA_WIDTH/8)-1 downto 0); --
     S_LAST       : In  std_logic ;                                     --
     ---------------------------------------------------------------------
     

     -- Master Side (MMap Write Data Output Side) ------------------------
     M_VALID      : Out std_logic ;                                     --
     M_READY      : In  std_logic ;                                     --
     M_DATA       : Out std_logic_vector(C_MDATA_WIDTH-1 downto 0);     --
     M_STRB       : Out std_logic_vector((C_MDATA_WIDTH/8)-1 downto 0); --
     M_LAST       : Out std_logic                                       --
     ---------------------------------------------------------------------
     
    );

end entity axi_sg_skid2mm_buf;


architecture implementation of axi_sg_skid2mm_buf is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


  Constant IN_DATA_WIDTH       : integer := C_SDATA_WIDTH;
  Constant MM2STRM_WIDTH_RATIO : integer := C_MDATA_WIDTH/C_SDATA_WIDTH;
  
  
-- Signals decalrations -------------------------

  Signal sig_reset_reg         : std_logic := '0';
  signal sig_spcl_s_ready_set  : std_logic := '0';
  signal sig_data_skid_reg     : std_logic_vector(IN_DATA_WIDTH-1 downto 0) := (others => '0');
  signal sig_strb_skid_reg     : std_logic_vector((C_MDATA_WIDTH/8)-1 downto 0) := (others => '0');
  signal sig_last_skid_reg     : std_logic := '0';
  signal sig_skid_reg_en       : std_logic := '0';
  signal sig_data_skid_mux_out : std_logic_vector(IN_DATA_WIDTH-1 downto 0) := (others => '0');
  signal sig_strb_skid_mux_out : std_logic_vector((C_MDATA_WIDTH/8)-1 downto 0) := (others => '0');
  signal sig_last_skid_mux_out : std_logic := '0';
  signal sig_skid_mux_sel      : std_logic := '0';
  signal sig_data_reg_out      : std_logic_vector(IN_DATA_WIDTH-1 downto 0) := (others => '0');
  signal sig_strb_reg_out      : std_logic_vector((C_MDATA_WIDTH/8)-1 downto 0) := (others => '0');
  signal sig_last_reg_out      : std_logic := '0';
  signal sig_data_reg_out_en   : std_logic := '0';
  signal sig_m_valid_out       : std_logic := '0';
  signal sig_m_valid_dup       : std_logic := '0';
  signal sig_m_valid_comb      : std_logic := '0';
  signal sig_s_ready_out       : std_logic := '0';
  signal sig_s_ready_dup       : std_logic := '0';
  signal sig_s_ready_comb      : std_logic := '0';
  signal sig_mirror_data_out   : std_logic_vector(C_MDATA_WIDTH-1 downto 0) := (others => '0');
  signal sig_wstrb_demux_out   : std_logic_vector((C_MDATA_WIDTH/8)-1 downto 0) := (others => '0');
                                         
                                         
                                         
 
-- Register duplication attribute assignments to control fanout
-- on handshake output signals  
  
  Attribute KEEP : string; -- declaration
  Attribute EQUIVALENT_REGISTER_REMOVAL : string; -- declaration
  
  Attribute KEEP of sig_m_valid_out : signal is "TRUE"; -- definition
  Attribute KEEP of sig_m_valid_dup : signal is "TRUE"; -- definition
  Attribute KEEP of sig_s_ready_out : signal is "TRUE"; -- definition
  Attribute KEEP of sig_s_ready_dup : signal is "TRUE"; -- definition
  
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_m_valid_out : signal is "no"; 
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_m_valid_dup : signal is "no"; 
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_s_ready_out : signal is "no"; 
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_s_ready_dup : signal is "no"; 
  
  
  

begin --(architecture implementation)

   M_VALID <= sig_m_valid_out;         
   S_READY <= sig_s_ready_out; 
    
   M_STRB  <= sig_strb_reg_out;
   M_LAST  <= sig_last_reg_out;                
   M_DATA  <= sig_mirror_data_out;
            
   -- Assign the special S_READY FLOP set signal
   sig_spcl_s_ready_set <= sig_reset_reg;
  
            
   -- Generate the ouput register load enable control
   sig_data_reg_out_en <= M_READY or not(sig_m_valid_dup);

   -- Generate the skid inpit register load enable control
   sig_skid_reg_en     <= sig_s_ready_dup;
  
   -- Generate the skid mux select control
   sig_skid_mux_sel    <= not(sig_s_ready_dup);
   
   
   -- Skid Mux  
   sig_data_skid_mux_out <=  sig_data_skid_reg
     When (sig_skid_mux_sel = '1')
     Else  S_DATA;
  
   sig_strb_skid_mux_out <=  sig_strb_skid_reg
     When (sig_skid_mux_sel = '1')
     --Else  S_STRB;
     Else  sig_wstrb_demux_out;
  
   sig_last_skid_mux_out <=  sig_last_skid_reg
     When (sig_skid_mux_sel = '1')
     Else  S_LAST;
  
   
   -- m_valid combinational logic        
   sig_m_valid_comb <= S_VALID or
                      (sig_m_valid_dup and
                      (not(sig_s_ready_dup) or
                       not(M_READY)));
   
   
   
   -- s_ready combinational logic        
   sig_s_ready_comb <= M_READY or
                      (sig_s_ready_dup and
                      (not(sig_m_valid_dup) or
                       not(S_VALID)));
   
   
   
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: REG_THE_RST
   --
   -- Process Description:
   -- Register input reset
   --
   -------------------------------------------------------------
   REG_THE_RST : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           
            sig_reset_reg <= ARST;
           
        end if;       
      end process REG_THE_RST; 
   
   
   
   
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: S_READY_FLOP
   --
   -- Process Description:
   -- Registers S_READY handshake signals per Skid Buffer 
   -- Option 2 scheme
   --
   -------------------------------------------------------------
   S_READY_FLOP : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST = '1') then

             sig_s_ready_out  <= '0';
             sig_s_ready_dup  <= '0';
            
           Elsif (sig_spcl_s_ready_set = '1') Then
           
             sig_s_ready_out  <= '1';
             sig_s_ready_dup  <= '1';
           
           else

             sig_s_ready_out  <= sig_s_ready_comb;
             sig_s_ready_dup  <= sig_s_ready_comb;
            
           end if; 
        end if;       
      end process S_READY_FLOP; 
   
   
   
   
            
            
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: M_VALID_FLOP
   --
   -- Process Description:
   -- Registers M_VALID handshake signals per Skid Buffer 
   -- Option 2 scheme
   --
   -------------------------------------------------------------
   M_VALID_FLOP : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST                 = '1' or
               sig_spcl_s_ready_set = '1') then -- Fix from AXI DMA

             sig_m_valid_out  <= '0';
             sig_m_valid_dup  <= '0';
            
           else

             sig_m_valid_out  <= sig_m_valid_comb;
             sig_m_valid_dup  <= sig_m_valid_comb;
            
           end if; 
        end if;       
      end process M_VALID_FLOP; 
   
   
   
   
            
            
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: SKID_DATA_REG
   --
   -- Process Description:
   -- This process implements the Skid register for the 
   -- Skid Buffer Data signals.
   --
   -------------------------------------------------------------
   SKID_DATA_REG : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           
           if  (sig_skid_reg_en = '1') then
             
             sig_data_skid_reg <= S_DATA;
             
           else
             null;  -- hold current state
           end if;
            
        end if;       
      end process SKID_DATA_REG; 
            
            
   
            
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: SKID_CNTL_REG
   --
   -- Process Description:
   -- This process implements the Output registers for the 
   -- Skid Buffer Control signals
   --
   -------------------------------------------------------------
   SKID_CNTL_REG : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST = '1') then
             
             sig_strb_skid_reg <= (others => '0');
             sig_last_skid_reg <= '0';
             
           elsif (sig_skid_reg_en = '1') then
             
             sig_strb_skid_reg <= sig_wstrb_demux_out;
             sig_last_skid_reg <= S_LAST;
             
           else
             null;  -- hold current state
           end if; 
        end if;       
      end process SKID_CNTL_REG; 
            
            
   
            
            
            
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: OUTPUT_DATA_REG
   --
   -- Process Description:
   -- This process implements the Output register for the 
   -- Data signals.
   --
   -------------------------------------------------------------
   OUTPUT_DATA_REG : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           
           if (sig_data_reg_out_en = '1') then
             
             sig_data_reg_out <= sig_data_skid_mux_out;
             
           else
             null;  -- hold current state
           end if;
            
        end if;       
      end process OUTPUT_DATA_REG; 
            
            
 
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: OUTPUT_CNTL_REG
   --
   -- Process Description:
   -- This process implements the Output registers for the 
   -- control signals.
   --
   -------------------------------------------------------------
   OUTPUT_CNTL_REG : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST = '1') then
             
             sig_strb_reg_out <= (others => '0');
             sig_last_reg_out <= '0';
             
           elsif (sig_data_reg_out_en = '1') then
             
             sig_strb_reg_out <= sig_strb_skid_mux_out;
             sig_last_reg_out <= sig_last_skid_mux_out;
             
           else
             null;  -- hold current state
           end if; 
        end if;       
      end process OUTPUT_CNTL_REG; 
            
            
 
 
 
 
 
 
   -------------------------------------------------------------
   -- Combinational Process
   --
   -- Label: DO_WR_DATA_MIRROR
   --
   -- Process Description:
   -- Implement the Write Data Mirror structure
   -- 
   -- Note that it is required that the Stream Width be less than
   -- or equal to the MMap WData width.
   --
   -------------------------------------------------------------
   DO_WR_DATA_MIRROR : process (sig_data_reg_out)
      begin
   
        for slice_index in 0 to MM2STRM_WIDTH_RATIO-1 loop
        
          sig_mirror_data_out(((C_SDATA_WIDTH*slice_index)+C_SDATA_WIDTH)-1 
                              downto C_SDATA_WIDTH*slice_index)
                              
                              <= sig_data_reg_out;
        
        end loop;
 
   
      end process DO_WR_DATA_MIRROR; 
   
 
 
 
 
        
    ------------------------------------------------------------
    -- Instance: I_WSTRB_DEMUX 
    --
    -- Description:
    -- Instance for the Write Strobe DeMux.    
    --
    ------------------------------------------------------------
     I_WSTRB_DEMUX : entity axi_sg_v4_1_6.axi_sg_wr_demux
     generic map (
      
       C_SEL_ADDR_WIDTH     =>  C_ADDR_LSB_WIDTH   ,  
       C_MMAP_DWIDTH        =>  C_MDATA_WIDTH      ,  
       C_STREAM_DWIDTH      =>  C_SDATA_WIDTH         
      
       )
     port map (
   
       wstrb_in             =>  S_STRB              , 
       demux_wstrb_out      =>  sig_wstrb_demux_out ,        
       debeat_saddr_lsb     =>  S_ADDR_LSB            
   
       );
   
 
 

end implementation;


-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_sg_wrdata_cntl.vhd
  --
  -- Description:     
  --    This file implements the DataMover Master Write Data Controller.                 
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  library axi_sg_v4_1_6;
  use axi_sg_v4_1_6.axi_sg_fifo;
  
  -------------------------------------------------------------------------------
  
  entity axi_sg_wrdata_cntl is
    generic (
      
      C_REALIGNER_INCLUDED   : Integer range  0 to   1 :=  0;
        -- Indicates the Data Realignment function is included (external
        -- to this module)
      
      C_ENABLE_INDET_BTT     : Integer range  0 to   1 :=  0;
        -- Indicates the INDET BTT function is included (external
        -- to this module)
            
      C_SF_BYTES_RCVD_WIDTH  : Integer range  1 to  23 :=  1;
        -- Sets the width of the data2wsc_bytes_rcvd port used for 
        -- relaying the actual number of bytes received when Idet BTT is 
        -- enabled (C_ENABLE_INDET_BTT = 1)
      
      C_SEL_ADDR_WIDTH       : Integer range  1 to   8 :=  5;
        -- Sets the width of the LS bits of the transfer address that
        -- are being used to Demux write data to a wider AXI4 Write
        -- Data Bus
        
      C_DATA_CNTL_FIFO_DEPTH : Integer range  1 to  32 :=  4;
        -- Sets the depth of the internal command fifo used for the
        -- command queue
        
      C_MMAP_DWIDTH          : Integer range 32 to 1024 := 32;
        -- Indicates the native data width of the Read Data port
        
      C_STREAM_DWIDTH        : Integer range  8 to 1024 := 32;
        -- Sets the width of the Stream output data port
        
      C_TAG_WIDTH            : Integer range  1 to   8 :=  4;
        -- Indicates the width of the Tag field of the input command
        
      C_FAMILY               : String                  := "virtex7"
        -- Indicates the device family of the target FPGA
        
      
      );
    port (
      
      -- Clock and Reset inputs ----------------------------------------------
                                                                            --
      primary_aclk         : in  std_logic;                                 --
         -- Primary synchronization clock for the Master side               --
         -- interface and internal logic. It is also used                   --
         -- for the User interface synchronization when                     --
         -- C_STSCMD_IS_ASYNC = 0.                                          --
                                                                            --
      -- Reset input                                                        --
      mmap_reset           : in  std_logic;                                 --
         -- Reset used for the internal master logic                        --
      ------------------------------------------------------------------------
      

     
      
      -- Soft Shutdown internal interface ------------------------------------
                                                                            --
      rst2data_stop_request : in  std_logic;                                --
         -- Active high soft stop request to modules                        --
                                                                            --
      data2addr_stop_req    : Out std_logic;                                --
        -- Active high signal requesting the Address Controller             --
        -- to stop posting commands to the AXI Read Address Channel         --
                                                                            --
      data2rst_stop_cmplt   : Out std_logic;                                --
        -- Active high indication that the Data Controller has completed    --
        -- any pending transfers committed by the Address Controller        --
        -- after a stop has been requested by the Reset module.             --
      ------------------------------------------------------------------------
      
      
        
  
      -- Store and Forward support signals for external User logic ------------
                                                                             --
      wr_xfer_cmplt         : Out std_logic;                                 --
        -- Active high indication that the Data Controller has completed     --
        -- a single write data transfer on the AXI4 Write Data Channel.      --
        -- This signal is escentially echos the assertion of wlast sent      --
        -- to the AXI4.                                                      --
                                                                             --
      s2mm_ld_nxt_len        : out std_logic;                                --
        -- Active high pulse indicating a new xfer length has been queued    --
        -- to the WDC Cmd FIFO                                               --
                                                                             --
      s2mm_wr_len            : out std_logic_vector(7 downto 0);             --
        -- Bus indicating the AXI LEN value associated with the xfer command --
        -- loaded into the WDC Command FIFO.                                 --
      -------------------------------------------------------------------------
     
     
     
      -- AXI Write Data Channel Skid buffer I/O  ---------------------------------------
                                                                                      --
      data2skid_saddr_lsb   : out std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0);      --
        -- Write DATA output to skid buffer                                           --
                                                                                      --
      data2skid_wdata       : Out  std_logic_vector(C_STREAM_DWIDTH-1 downto 0);      --
        -- Write DATA output to skid buffer                                           --
                                                                                      --
      data2skid_wstrb       : Out  std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);  --
        -- Write DATA output to skid buffer                                           --
                                                                                      --
      data2skid_wlast       : Out  std_logic;                                         --
        -- Write LAST output to skid buffer                                           --
                                                                                      --
      data2skid_wvalid      : Out  std_logic;                                         --
        -- Write VALID output to skid buffer                                          --
                                                                                      --
      skid2data_wready      : In  std_logic;                                          --
        -- Write READY input from skid buffer                                         --
      ----------------------------------------------------------------------------------
     
 
 
      -- AXI Slave Stream In -----------------------------------------------------------
                                                                                      --
      s2mm_strm_wvalid   : In  std_logic;                                             --
        -- AXI Stream VALID input                                                     --
                                                                                      --
      s2mm_strm_wready   : Out  Std_logic;                                            --
        -- AXI Stream READY Output                                                    --
                                                                                      --
      s2mm_strm_wdata    : In  std_logic_vector(C_STREAM_DWIDTH-1 downto 0);          --
        -- AXI Stream data input                                                      --
                                                                                      --
      s2mm_strm_wstrb    : In std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);       --  
        -- AXI Stream STRB input                                                      --
                                                                                      --
      s2mm_strm_wlast    : In std_logic;                                              --
        -- AXI Stream LAST input                                                      --
      ----------------------------------------------------------------------------------
               
      
      
      -- Stream input sideband signal from Indeterminate BTT and/or DRE ----------------
                                                                                      --
      s2mm_strm_eop      : In std_logic;                                              --
        -- Stream End of Packet marker input. This is only used when Indeterminate    --
        -- BTT mode is enable. Otherwise it is ignored                                --
                                                                                      --
                                                                                      --
      s2mm_stbs_asserted : in  std_logic_vector(7 downto 0);                          --
        -- Indicates the number of asserted WSTRB bits for the                        --
        -- associated input stream data beat                                          --
                                                                                      --
                                                                                      --
      -- Realigner Underrun/overrun error flag used in non Indeterminate BTT          --
      -- Mode                                                                         --
      realign2wdc_eop_error  : In  std_logic ;                                        --
        -- Asserted active high and will only clear with reset. It is only used       --
        -- when Indeterminate BTT is not enabled and the Realigner Module is          --
        -- instantiated upstream from the WDC. The Realigner will detect overrun      --
        -- underrun conditions and will will relay these conditions via this signal.  --
      ----------------------------------------------------------------------------------        
      
      
                
                
                
      -- Command Calculator Interface --------------------------------------------------
                                                                                      --
      mstr2data_tag        : In std_logic_vector(C_TAG_WIDTH-1 downto 0);             --
         -- The next command tag                                                      --
                                                                                      --
      mstr2data_saddr_lsb  : In std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0);        --
         -- The next command start address LSbs to use for the write strb             --
         -- demux (only used if Stream data width is less than the MMap Dwidth).      --
                                                                                      --
      mstr2data_len        : In std_logic_vector(7 downto 0);                         --
         -- The LEN value output to the Address Channel                               --
                                                                                      --
      mstr2data_strt_strb  : In std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);     --
         -- The starting strobe value to use for the first stream data beat           --
                                                                                      --
      mstr2data_last_strb  : In std_logic_vector((C_STREAM_DWIDTH/8)-1 downto 0);     --
         -- The endiing (LAST) strobe value to use for the last stream                --
         -- data beat                                                                 --
                                                                                      --
      mstr2data_drr        : In std_logic;                                            --
         -- The starting tranfer of a sequence of transfers                           --
                                                                                      --
      mstr2data_eof        : In std_logic;                                            --
         -- The endiing tranfer of a sequence of transfers                            --
                                                                                      --
      mstr2data_sequential : In std_logic;                                            --
         -- The next sequential tranfer of a sequence of transfers                    --
         -- spawned from a single parent command                                      --
                                                                                      --
      mstr2data_calc_error : In std_logic;                                            --
         -- Indication if the next command in the calculation pipe                    --
         -- has a calculation error                                                   --
                                                                                      --
      mstr2data_cmd_cmplt  : In std_logic;                                            --
         -- The final child tranfer of a parent command fetched from                  --
         -- the Command FIFO (not necessarily an EOF command)                         --
                                                                                      --
      mstr2data_cmd_valid  : In std_logic;                                            --
         -- The next command valid indication to the Data Channel                     --
         -- Controller for the AXI MMap                                               --
                                                                                      --
      data2mstr_cmd_ready  : Out std_logic ;                                          --
         -- Indication from the Data Channel Controller that the                      --
         -- command is being accepted on the AXI Address                              --
         -- Channel                                                                   --
      ----------------------------------------------------------------------------------
      
      
      
        
      -- Address Controller Interface --------------------------------------------------
                                                                                      --
      addr2data_addr_posted    : In std_logic ;                                       --
         -- Indication from the Address Channel Controller to the                     --
         -- Data Controller that an address has been posted to the                    --
         -- AXI Address Channel                                                       --
                                                                                      --
                                                                                      --
      data2addr_data_rdy       : out std_logic;                                       --
         -- Indication that the Data Channel is ready to send the first               --
         -- databeat of the next command on the write data channel.                   --
         -- This is used for the "wait for data" feature which keeps the              --
         -- address controller from issuing a transfer request until the              --
         -- corresponding data valid is asserted on the stream input. The             --
         -- WDC will continue to assert the output until an assertion on              --
         -- the addr2data_addr_posted is received.                                    --
       ---------------------------------------------------------------------------------
  
  
      
      -- Premature TLAST assertion error flag ------------------------------------------
                                                                                      --
      data2all_tlast_error : Out std_logic;                                           --
         -- When asserted, this indicates the data controller detected                --
         -- a premature TLAST assertion on the incoming data stream.                  --
       ---------------------------------------------------------------------------------     
      
      
      -- Data Controller Halted Status -------------------------------------------------
                                                                                      --
      data2all_dcntlr_halted : Out std_logic;                                         --
         -- When asserted, this indicates the data controller has satisfied           --
         -- all pending transfers queued by the Address Controller and is halted.     --
      ----------------------------------------------------------------------------------
      
       
 
      -- Input Stream Skid Buffer Halt control -----------------------------------------
                                                                                      --
      data2skid_halt       : Out std_logic;                                           --
         -- The data controller asserts this output for 1 primary clock period        --
         -- The pulse commands the MM2S Stream skid buffer to tun off outputs         --
         -- at the next tlast transmission.                                           --
      ----------------------------------------------------------------------------------
      
       
 
       
      -- Write Status Controller Interface ---------------------------------------------
                                                                                      --
      data2wsc_tag         : Out std_logic_vector(C_TAG_WIDTH-1 downto 0);            --
         -- The command tag                                                           --
                                                                                      --
      data2wsc_calc_err    : Out std_logic ;                                          --
         -- Indication that the current command out from the Cntl FIFO                --
         -- has a calculation error                                                   --
                                                                                      --
      data2wsc_last_err    : Out std_logic ;                                          --
        -- Indication that the current write transfer encountered a premature         --
        -- TLAST assertion on the incoming Stream Channel                             --
                                                                                      --
      data2wsc_cmd_cmplt   : Out std_logic ;                                          --
         -- Indication by the Data Channel Controller that the                        --
         -- corresponding status is the last status for a command                     --
         -- pulled from the command FIFO                                              --
                                                                                      --
      wsc2data_ready       : in  std_logic;                                           --
         -- Input from the Write Status Module indicating that the                    --
         -- Status Reg/FIFO is ready to accept data                                   --
                                                                                      --
      data2wsc_valid       : Out  std_logic;                                          --
         -- Output to the Command/Status Module indicating that the                   --
         -- Data Controller has valid tag and err indicators to write                 --
         -- to the Status module                                                      --
                                                                                      --
      data2wsc_eop         : Out  std_logic;                                          --
         -- Output to the Write Status Controller indicating that the                 --
         -- associated command status also corresponds to a End of Packet             --
         -- marker for the input Stream. This is only used when Inderminate           --
         -- BTT is enabled in the S2MM.                                               --
                                                                                      --
      data2wsc_bytes_rcvd  : Out std_logic_vector(C_SF_BYTES_RCVD_WIDTH-1 downto 0);  --
         -- Output to the Write Status Controller indicating the actual               --
         -- number of bytes received from the Stream input for the                    --
         -- corresponding command status. This is only used when Inderminate          --
         -- BTT is enabled in the S2MM.                                               --
                                                                                      --
      wsc2mstr_halt_pipe   : In  std_logic                                            --
         -- Indication to Halt the Data and Address Command pipeline due              --
         -- to the Status FIFO going full or an internal error being logged           --
      ----------------------------------------------------------------------------------

    
      
      );
  
  end entity axi_sg_wrdata_cntl;
  
  
  architecture implementation of axi_sg_wrdata_cntl  is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    
    
    -- Function declaration   ----------------------------------------
    
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_get_dbeat_residue_width
    --
    -- Function Description:
    --  Calculates the number of Least significant bits of the BTT field
    -- that are unused for the LEN calculation
    --
    -------------------------------------------------------------------
                    -- coverage off  
    function funct_get_dbeat_residue_width (bytes_per_beat : integer) return integer is

      Variable temp_dbeat_residue_width : Integer := 0; -- 8-bit stream

    begin

      case bytes_per_beat is
        
        when 128 =>  -- 1024 bits          -- Added per Per CR616409
            temp_dbeat_residue_width := 7; -- Added per Per CR616409
        when 64 =>   -- 512 bits           -- Added per Per CR616409
            temp_dbeat_residue_width := 6; -- Added per Per CR616409
        when 32 =>   -- 256 bits           
            temp_dbeat_residue_width := 5;
        when 16 =>   -- 128 bits
            temp_dbeat_residue_width := 4;
        when 8 =>    -- 64 bits
            temp_dbeat_residue_width := 3;
        when 4 =>    -- 32 bits
            temp_dbeat_residue_width := 2;
        when 2 =>    -- 16 bits
            temp_dbeat_residue_width := 1;
        when others =>  -- assume 1-byte transfers
            temp_dbeat_residue_width := 0;
      end case;

      Return (temp_dbeat_residue_width);

    end function funct_get_dbeat_residue_width;
                    -- coverage on  




    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_set_cnt_width
    --
    -- Function Description:
    --    Sets a count width based on a fifo depth. A depth of 4 or less
    -- is a special case which requires a minimum count width of 3 bits.
    --
    -------------------------------------------------------------------
    function funct_set_cnt_width (fifo_depth : integer) return integer is
    
      Variable temp_cnt_width : Integer := 4;
    
    begin
    
      
      if (fifo_depth <= 4) then
      
         temp_cnt_width := 3;
      
                    -- coverage off  
      elsif (fifo_depth <= 8) then
      
         temp_cnt_width := 4;
      
      elsif (fifo_depth <= 16) then
      
         temp_cnt_width := 5;
      
      elsif (fifo_depth <= 32) then
      
         temp_cnt_width := 6;
      
      else  -- fifo depth <= 64
      
         temp_cnt_width := 7;
      
      end if;
                    -- coverage on  
      
      Return (temp_cnt_width);
       
       
    end function funct_set_cnt_width;
    
 
 
 
  
    -- Constant Declarations  --------------------------------------------
    
    Constant STRM_STRB_WIDTH        : integer := C_STREAM_DWIDTH/8;
    Constant LEN_OF_ZERO            : std_logic_vector(7 downto 0) := (others => '0');
    Constant USE_SYNC_FIFO          : integer := 0;
    Constant REG_FIFO_PRIM          : integer := 0; 
    Constant BRAM_FIFO_PRIM         : integer := 1; 
    Constant SRL_FIFO_PRIM          : integer := 2; 
    Constant FIFO_PRIM_TYPE         : integer := SRL_FIFO_PRIM; 
    Constant TAG_WIDTH              : integer := C_TAG_WIDTH;
    Constant SADDR_LSB_WIDTH        : integer := C_SEL_ADDR_WIDTH;
    Constant LEN_WIDTH              : integer := 8;
    Constant STRB_WIDTH             : integer := C_STREAM_DWIDTH/8;
    Constant DRR_WIDTH              : integer := 1;
    Constant EOF_WIDTH              : integer := 1;
    Constant CALC_ERR_WIDTH         : integer := 1;
    Constant CMD_CMPLT_WIDTH        : integer := 1;
    Constant SEQUENTIAL_WIDTH       : integer := 1;
    Constant DCTL_FIFO_WIDTH        : Integer := TAG_WIDTH        +  -- Tag field
                                                 SADDR_LSB_WIDTH  +  -- LS Address field width
                                                 LEN_WIDTH        +  -- LEN field
                                                 STRB_WIDTH       +  -- Starting Strobe field
                                                 STRB_WIDTH       +  -- Ending Strobe field
                                                 DRR_WIDTH        +  -- DRE Re-alignment Request Flag Field
                                                 EOF_WIDTH        +  -- EOF flag field
                                                 SEQUENTIAL_WIDTH +  -- Sequential command flag
                                                 CMD_CMPLT_WIDTH  +  -- Command Complete Flag
                                                 CALC_ERR_WIDTH;     -- Calc error flag
    
    Constant TAG_STRT_INDEX         : integer := 0;
    Constant SADDR_LSB_STRT_INDEX   : integer := TAG_STRT_INDEX + TAG_WIDTH;
    Constant LEN_STRT_INDEX         : integer := SADDR_LSB_STRT_INDEX + SADDR_LSB_WIDTH;
    Constant STRT_STRB_STRT_INDEX   : integer := LEN_STRT_INDEX + LEN_WIDTH;
    Constant LAST_STRB_STRT_INDEX   : integer := STRT_STRB_STRT_INDEX + STRB_WIDTH;
    Constant DRR_STRT_INDEX         : integer := LAST_STRB_STRT_INDEX + STRB_WIDTH;
    Constant EOF_STRT_INDEX         : integer := DRR_STRT_INDEX + DRR_WIDTH;
    Constant SEQUENTIAL_STRT_INDEX  : integer := EOF_STRT_INDEX + EOF_WIDTH;
    Constant CMD_CMPLT_STRT_INDEX   : integer := SEQUENTIAL_STRT_INDEX+SEQUENTIAL_WIDTH;
    Constant CALC_ERR_STRT_INDEX    : integer := CMD_CMPLT_STRT_INDEX+CMD_CMPLT_WIDTH;        
    Constant ADDR_INCR_VALUE        : integer := C_STREAM_DWIDTH/8;
    
    Constant ADDR_POSTED_CNTR_WIDTH : integer := funct_set_cnt_width(C_DATA_CNTL_FIFO_DEPTH); 
    
    
    
    Constant ADDR_POSTED_ZERO       : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) 
                                      := (others => '0');
    Constant ADDR_POSTED_ONE        : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) 
                                      := TO_UNSIGNED(1, ADDR_POSTED_CNTR_WIDTH);
    Constant ADDR_POSTED_MAX        : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) 
                                      := (others => '1');
                    
    
    
    
    
    -- Signal Declarations  --------------------------------------------
    
    signal sig_get_next_dqual        : std_logic := '0';
    signal sig_last_mmap_dbeat       : std_logic := '0';
    signal sig_last_mmap_dbeat_reg   : std_logic := '0';
    signal sig_mmap2data_ready       : std_logic := '0';
    signal sig_data2mmap_valid       : std_logic := '0';
    signal sig_data2mmap_last        : std_logic := '0';
    signal sig_data2mmap_data        : std_logic_vector(C_STREAM_DWIDTH-1 downto 0) := (others => '0');
    signal sig_ld_new_cmd            : std_logic := '0';
    signal sig_ld_new_cmd_reg        : std_logic := '0';
    signal sig_cmd_cmplt_reg         : std_logic := '0';
    signal sig_calc_error_reg        : std_logic := '0';
    signal sig_tag_reg               : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_lsb_reg          : std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_strt_strb_reg         : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_last_strb_reg         : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_addr_posted           : std_logic := '0';
    signal sig_dqual_rdy             : std_logic := '0';
    signal sig_good_mmap_dbeat       : std_logic := '0';
    signal sig_first_dbeat           : std_logic := '0';
    signal sig_last_dbeat            : std_logic := '0';
    signal sig_single_dbeat          : std_logic := '0';
    signal sig_new_len_eq_0          : std_logic := '0';
    signal sig_dbeat_cntr            : unsigned(7 downto 0) := (others => '0');
    Signal sig_dbeat_cntr_int        : Integer range 0 to 255 := 0;
    signal sig_dbeat_cntr_eq_0       : std_logic := '0';
    signal sig_dbeat_cntr_eq_1       : std_logic := '0';
    signal sig_wsc_ready             : std_logic := '0';
    signal sig_push_to_wsc           : std_logic := '0';
    signal sig_push_to_wsc_cmplt     : std_logic := '0';
    signal sig_set_push2wsc          : std_logic := '0';
    signal sig_data2wsc_tag          : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_data2wsc_calc_err     : std_logic := '0';
    signal sig_data2wsc_last_err     : std_logic := '0';
    signal sig_data2wsc_cmd_cmplt    : std_logic := '0';
    signal sig_tlast_error           : std_logic := '0';
    signal sig_tlast_error_strbs     : std_logic := '0';
    signal sig_end_stbs_match_err    : std_logic := '0';
    signal sig_tlast_error_reg       : std_logic := '0';
    signal sig_cmd_is_eof            : std_logic := '0';
    signal sig_push_err2wsc          : std_logic := '0';
    signal sig_tlast_error_ovrrun    : std_logic := '0';
    signal sig_tlast_error_undrrun   : std_logic := '0';
    signal sig_next_tag_reg          : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_next_strt_strb_reg    : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_next_last_strb_reg    : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_next_eof_reg          : std_logic := '0';
    signal sig_next_sequential_reg   : std_logic := '0';
    signal sig_next_cmd_cmplt_reg    : std_logic := '0';
    signal sig_next_calc_error_reg   : std_logic := '0';
    signal sig_pop_dqual_reg         : std_logic := '0';
    signal sig_push_dqual_reg        : std_logic := '0';
    signal sig_dqual_reg_empty       : std_logic := '0';
    signal sig_dqual_reg_full        : std_logic := '0';
    signal sig_addr_posted_cntr      : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_posted_cntr_eq_0 : std_logic := '0';
    signal sig_addr_posted_cntr_max  : std_logic := '0';
    signal sig_decr_addr_posted_cntr : std_logic := '0';
    signal sig_incr_addr_posted_cntr : std_logic := '0';
    signal sig_addr_posted_cntr_eq_1 : std_logic := '0';
    signal sig_apc_going2zero        : std_logic := '0';
    signal sig_aposted_cntr_ready    : std_logic := '0';
    signal sig_addr_chan_rdy         : std_logic := '0';
    Signal sig_no_posted_cmds        : std_logic := '0';
    signal sig_ls_addr_cntr          : unsigned(C_SEL_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_incr_ls_addr_cntr     : std_logic := '0';
    signal sig_addr_incr_unsgnd      : unsigned(C_SEL_ADDR_WIDTH-1 downto 0) := (others => '0');
    Signal sig_cmd_fifo_data_in      : std_logic_vector(DCTL_FIFO_WIDTH-1 downto 0) := (others => '0');
    Signal sig_cmd_fifo_data_out     : std_logic_vector(DCTL_FIFO_WIDTH-1 downto 0) := (others => '0');
    signal sig_fifo_next_tag         : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_fifo_next_sadddr_lsb  : std_logic_vector(C_SEL_ADDR_WIDTH-1 downto 0) := (others => '0');
    signal sig_fifo_next_len         : std_logic_vector(7 downto 0) := (others => '0');
    signal sig_fifo_next_strt_strb   : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_fifo_next_last_strb   : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_fifo_next_drr         : std_logic := '0';
    signal sig_fifo_next_eof         : std_logic := '0';
    signal sig_fifo_next_cmd_cmplt   : std_logic := '0';
    signal sig_fifo_next_sequential  : std_logic := '0';
    signal sig_fifo_next_calc_error  : std_logic := '0';
    signal sig_cmd_fifo_empty        : std_logic := '0';
    signal sig_fifo_wr_cmd_valid     : std_logic := '0';
    signal sig_fifo_wr_cmd_ready     : std_logic := '0';
    signal sig_fifo_rd_cmd_valid     : std_logic := '0';
    signal sig_fifo_rd_cmd_ready     : std_logic := '0';
    signal sig_sequential_push       : std_logic := '0';
    signal sig_clr_dqual_reg         : std_logic := '0';
    signal sig_tlast_err_stop        : std_logic := '0';
    signal sig_halt_reg              : std_logic := '0';
    signal sig_halt_reg_dly1         : std_logic := '0';
    signal sig_halt_reg_dly2         : std_logic := '0';
    signal sig_halt_reg_dly3         : std_logic := '0';
    signal sig_data2skid_halt        : std_logic := '0';
    signal sig_stop_wvalid           : std_logic := '0';
    signal sig_data2rst_stop_cmplt   : std_logic := '0';
    signal sig_s2mm_strm_wready      : std_logic := '0';
    signal sig_s2mm_strm_wready_del      : std_logic := '0';
    signal sig_good_strm_dbeat       : std_logic := '0';
    signal sig_halt_strb             : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_sfhalt_next_strt_strb : std_logic_vector(STRM_STRB_WIDTH-1 downto 0) := (others => '0');             
    signal sig_wfd_simult_clr_set    : std_logic := '0';
    signal sig_wr_xfer_cmplt         : std_logic := '0';
    signal sig_s2mm_ld_nxt_len       : std_logic := '0';
    signal sig_s2mm_wr_len           : std_logic_vector(7 downto 0) := (others => '0');
    signal sig_data2mstr_cmd_ready   : std_logic := '0';
    signal sig_spcl_push_err2wsc     : std_logic := '0';
    
    
                               
  begin --(architecture implementation)
  
    -- Command calculator handshake
    data2mstr_cmd_ready <= sig_data2mstr_cmd_ready;
    
    
    -- Write Data Channel Skid Buffer Port assignments
    sig_mmap2data_ready  <= skid2data_wready     ;
    data2skid_wvalid     <= sig_data2mmap_valid  ;
    data2skid_wlast      <= sig_data2mmap_last   ;
    data2skid_wdata      <= sig_data2mmap_data   ;
    data2skid_saddr_lsb  <= sig_addr_lsb_reg     ;
    
    -- AXI MM2S Stream Channel Port assignments           
    sig_data2mmap_data   <= s2mm_strm_wdata      ;

    
    -- Premature TLAST assertion indication
    data2all_tlast_error <= sig_tlast_error_reg  ;
    
    
    
    
   
    -- Stream Input Ready Handshake
    s2mm_strm_wready     <= sig_s2mm_strm_wready ;                                           
    
    
    
    sig_good_strm_dbeat  <= s2mm_strm_wvalid and
                            sig_s2mm_strm_wready;
                        --    sig_s2mm_strm_wready_del;
    
    
    sig_data2mmap_last   <= sig_dbeat_cntr_eq_0 and
                            sig_dqual_rdy;

                      

    -- Write Status Block interface signals
    data2wsc_valid       <= sig_push_to_wsc and
                            not(sig_tlast_err_stop) ; -- only allow 1 status write on TLAST errror
    sig_wsc_ready        <= wsc2data_ready          ;
    data2wsc_tag         <= sig_data2wsc_tag        ;   
    data2wsc_calc_err    <= sig_data2wsc_calc_err   ; 
    data2wsc_last_err    <= sig_data2wsc_last_err   ; 
    data2wsc_cmd_cmplt   <= sig_data2wsc_cmd_cmplt  ;   
    
                                      
    -- Address Channel Controller synchro pulse input                  
    sig_addr_posted      <= addr2data_addr_posted;
                                                        
    
    
    -- Request to halt the Address Channel Controller                  
    data2addr_stop_req   <= sig_halt_reg or
                            sig_tlast_error_reg;
 
    
    -- Halted flag to the reset module                  
    data2rst_stop_cmplt  <= sig_data2rst_stop_cmplt;
    
    
    -- Indicate the Write Data Controller is always ready
    data2addr_data_rdy   <= '1'; 
    
    
    
    -- Write Transfer Completed Status output 
    wr_xfer_cmplt        <= sig_wr_xfer_cmplt ;
    
    -- New LEN value is being loaded 
    s2mm_ld_nxt_len      <= sig_s2mm_ld_nxt_len;
    
    -- The new LEN value
    s2mm_wr_len          <= sig_s2mm_wr_len;
    
    
     
     
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_WR_CMPLT_FLAG
    --
    -- Process Description:
    --   Implements the status flag indicating that a write data 
    -- transfer has completed. This is an echo of a wlast assertion
    -- and a qualified data beat on the AXI4 Write Data Channel.
    --
    -------------------------------------------------------------
    IMP_WR_CMPLT_FLAG : process (primary_aclk)
      begin
        if (primary_aclk'event and primary_aclk = '1') then
           if (mmap_reset = '1') then
    
             sig_wr_xfer_cmplt <= '0';
             sig_s2mm_strm_wready_del <= '0';
    
           else
    
             sig_wr_xfer_cmplt <= sig_data2mmap_last and 
                                  sig_good_strm_dbeat;

             sig_s2mm_strm_wready_del <= sig_s2mm_strm_wready;
                                  
           end if; 
        end if;       
      end process IMP_WR_CMPLT_FLAG; 
     
    
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_OMIT_INDET_BTT
    --
    -- If Generate Description:
    --   Omits any Indeterminate BTT Support logic and includes
    -- any error detection needed in Non Indeterminate BTT mode.
    --
    ------------------------------------------------------------
    GEN_OMIT_INDET_BTT : if (C_ENABLE_INDET_BTT = 0) generate
    
      begin
   
       
        
        
        sig_sfhalt_next_strt_strb <= sig_fifo_next_strt_strb;
        
        
        
        
        -- Just housekeep the output port signals
        
        data2wsc_eop         <= '0';
   
        data2wsc_bytes_rcvd  <= (others => '0');
   
       
        
        
        
        -- WRSTRB logic ------------------------------
                  

        -- Generate the Write Strobes for the MMap Write Data Channel
        -- for the non Indeterminate BTT Case
        data2skid_wstrb  <= (others => '1') when mmap_reset = '0' else (others => '0'); --sig_strt_strb_reg 
--        data2skid_wstrb  <= sig_strt_strb_reg 
--          When (sig_first_dbeat  = '1')
--          Else  sig_last_strb_reg
--          When  (sig_last_dbeat  = '1')
--          Else (others => '1');

        
                 
        -- Generate the Stream Ready for the Stream input side
        sig_s2mm_strm_wready <= sig_halt_reg            or  -- force tready if a halt requested
                                (sig_mmap2data_ready    and
                                sig_addr_chan_rdy       and -- This puts combinational logic in the stream WREADY path
                                sig_dqual_rdy           and
                                not(sig_calc_error_reg) and
                                not(sig_tlast_error_reg));   -- Stop the stream channel at a overrun/underrun detection
        
         
        
        -- MMap Write Data Channel Valid Handshaking
        sig_data2mmap_valid <= (s2mm_strm_wvalid       or
                               sig_tlast_error_reg     or  -- force valid if TLAST error 
                               sig_halt_reg       )    and -- force valid if halt requested       
                               sig_addr_chan_rdy       and -- xfers are commited on the address channel and       
                               sig_dqual_rdy           and -- there are commands in the command fifo        
                               not(sig_calc_error_reg) and
                               not(sig_stop_wvalid);       -- gate off wvalid immediately after a wlast for 1 clk
                                                           -- or when the soft shutdown has completed
              
      
        
        
        
        
        
        ------------------------------------------------------------
        -- If Generate
        --
        -- Label: GEN_LOCAL_ERR_DETECT
        --
        -- If Generate Description:
        --  Implements the local overrun and underrun detection when
        -- the S2MM Realigner is not included.
        --
        --
        ------------------------------------------------------------
        GEN_LOCAL_ERR_DETECT : if (C_REALIGNER_INCLUDED = 0) generate
        
        
           begin
        
             -------  Input Stream TLAST assertion error ------------------------------- 
             
             
             sig_tlast_error_ovrrun <= sig_cmd_is_eof       and
                                       sig_dbeat_cntr_eq_0  and
                                       sig_good_mmap_dbeat  and
                                       not(s2mm_strm_wlast);
             
             
             
             sig_tlast_error_undrrun <= s2mm_strm_wlast     and
                                        sig_good_mmap_dbeat and
                                        (not(sig_dbeat_cntr_eq_0) or
                                         not(sig_cmd_is_eof));
             
                    
                    
             sig_end_stbs_match_err  <=  '1'                            -- Set flag if the calculated end strobe value
               When ((s2mm_strm_wstrb    /= sig_next_last_strb_reg) and -- does not match the received strobe value 
                    (s2mm_strm_wlast     = '1') and                     -- at TLAST assertion
                    (sig_good_mmap_dbeat = '1'))                        -- Qualified databeat
               Else '0';
                    
                                 
             sig_tlast_error <=  (sig_tlast_error_ovrrun  or
                                  sig_tlast_error_undrrun or
                                  sig_end_stbs_match_err) and
                                  not(sig_halt_reg);          -- Suppress TLAST error when in soft shutdown 
                                 
                                 
             
             -- Just housekeep this when local TLAST error detection is used
             sig_spcl_push_err2wsc <= '0';
            
            
            
           
           end generate GEN_LOCAL_ERR_DETECT;
        
        
 
 
 
        ------------------------------------------------------------
        -- If Generate
        --
        -- Label: GEN_EXTERN_ERR_DETECT
        --
        -- If Generate Description:
        --  Omits the local overrun and underrun detection and relies
        -- on the S2MM Realigner for the detection.
        --
        ------------------------------------------------------------
        GEN_EXTERN_ERR_DETECT : if (C_REALIGNER_INCLUDED = 1) generate
        
        
           begin
        
 
              sig_tlast_error_undrrun <= '0';  -- not used here
 
              sig_tlast_error_ovrrun  <= '0';  -- not used here
 
              sig_end_stbs_match_err  <= '0';  -- not used here
 
 
           
              sig_tlast_error <= realign2wdc_eop_error and  -- External error detection asserted
                                 not(sig_halt_reg);         -- Suppress TLAST error when in soft shutdown
               
             
             
              -- Special case for pushing error status when timing is such that no 
              -- addresses have been posted to AXI and a TLAST error has been detected  
              -- by the Realigner module and propagated in from the Stream input side.
              sig_spcl_push_err2wsc <= sig_tlast_error_reg     and
                                       not(sig_tlast_err_stop) and
                                       not(sig_addr_chan_rdy );
             
             
             
             
             
             
               
               
           
           end generate GEN_EXTERN_ERR_DETECT;
 
 
 
 
 
 
 
        
        
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: IMP_TLAST_ERR_REG
        --
        -- Process Description:
        --  Implements a sample and hold flop for the flag indicating
        -- that the input Stream TLAST assertion was not at the expected
        -- data beat relative to the commanded number of databeats
        -- from the associated command from the SCC or PCC.
        -------------------------------------------------------------
        IMP_TLAST_ERR_REG : process (primary_aclk)
           begin
             if (primary_aclk'event and primary_aclk = '1') then
                if (mmap_reset = '1') then
                  sig_tlast_error_reg <= '0';
-- coverage off
                elsif (sig_tlast_error = '1') then
                  sig_tlast_error_reg <= '1';
-- coverage on
                else
                  null;  -- hold current state
                end if; 
             end if;       
           end process IMP_TLAST_ERR_REG; 
           
           
           
           
           
        -------------------------------------------------------------
        -- Synchronous Process with Sync Reset
        --
        -- Label: IMP_TLAST_ERROR_STOP
        --
        -- Process Description:
        --  Implements the flop to generate a stop flag once the TLAST
        -- error condition has been relayed to the Write Status 
        -- Controller. This stop flag is used to prevent any more 
        -- pushes to the Write Status Controller.
        --
        -------------------------------------------------------------
        IMP_TLAST_ERROR_STOP : process (primary_aclk)
           begin
             if (primary_aclk'event and primary_aclk = '1') then
                if (mmap_reset = '1') then
                  sig_tlast_err_stop <= '0';
-- coverage off
                elsif (sig_tlast_error_reg   = '1' and
                       sig_push_to_wsc_cmplt = '1') then
                  sig_tlast_err_stop <= '1';
-- coverage on
                else
                  null; -- Hold State
                end if; 
             end if;       
           end process IMP_TLAST_ERROR_STOP; 
           
           
           
      
      
      
      end generate GEN_OMIT_INDET_BTT;
   
   













    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_INDET_BTT
    --
    -- If Generate Description:
    --   Includes any Indeterminate BTT Support logic. Primarily
    -- this is a counter for the input stream bytes received. The
    -- received byte count is relayed to the Write Status Controller
    -- for each parent command completed.
    -- When a packet completion is indicated via the EOP marker
    -- assertion, the status to the Write Status Controller also
    -- indicates the EOP condition.
    -- Note that underrun and overrun detection/error flagging
    -- is disabled in Indeterminate BTT Mode.
    --
    ------------------------------------------------------------
--    GEN_INDET_BTT : if (C_ENABLE_INDET_BTT = 1) generate
--    
--      -- local constants
--      Constant BYTE_CNTR_WIDTH          : integer := C_SF_BYTES_RCVD_WIDTH;
--      Constant NUM_ZEROS_WIDTH          : integer := 8;
--      Constant BYTES_PER_DBEAT          : integer := C_STREAM_DWIDTH/8;
--      Constant STRBGEN_ADDR_SLICE_WIDTH : integer := 
--                                          funct_get_dbeat_residue_width(BYTES_PER_DBEAT);
--      
--      Constant STRBGEN_ADDR_0            : std_logic_vector(STRBGEN_ADDR_SLICE_WIDTH-1 downto 0) := (others => '0');
--      
--      
--      
--      -- local signals
--      signal lsig_byte_cntr             : unsigned(BYTE_CNTR_WIDTH-1 downto 0) := (others => '0');
--      signal lsig_byte_cntr_incr_value  : unsigned(BYTE_CNTR_WIDTH-1 downto 0) := (others => '0');
--      signal lsig_ld_byte_cntr          : std_logic := '0';
--      signal lsig_incr_byte_cntr        : std_logic := '0';
--      signal lsig_clr_byte_cntr         : std_logic := '0';
--      signal lsig_end_of_cmd_reg        : std_logic := '0';
--      signal lsig_eop_s_h_reg           : std_logic := '0';
--      signal lsig_eop_reg               : std_logic := '0';
--      signal sig_strbgen_addr           : std_logic_vector(STRBGEN_ADDR_SLICE_WIDTH-1 downto 0) := (others => '0');
--      signal sig_strbgen_bytes          : std_logic_vector(STRBGEN_ADDR_SLICE_WIDTH   downto 0) := (others => '0');
--      
--      
--      
--      
--      begin
--   
--       
--        -- Assign the outputs to the Write Status Controller
--        data2wsc_eop         <= lsig_eop_reg and 
--                                not(sig_next_calc_error_reg);
--         
--        data2wsc_bytes_rcvd  <= STD_LOGIC_VECTOR(lsig_byte_cntr);
--        
-- 
-- 
--        -- WRSTRB logic ------------------------------
--      
--      
--      
--        --sig_strbgen_bytes <= (others => '1'); -- set to the max value
--      
--        
--        -- set the length to the max number of bytes per databeat
--        sig_strbgen_bytes <=  STD_LOGIC_VECTOR(TO_UNSIGNED(BYTES_PER_DBEAT, STRBGEN_ADDR_SLICE_WIDTH+1));
--        
--        
--        
--        
--        
--        
--        sig_strbgen_addr  <= STD_LOGIC_VECTOR(RESIZE(UNSIGNED(sig_fifo_next_sadddr_lsb), 
--                                                     STRBGEN_ADDR_SLICE_WIDTH)) ;
-- 
-- 
-- 
--      
--      ------------------------------------------------------------
--      -- Instance: I_STRT_STRB_GEN 
--      --
--      -- Description:
--      --  Strobe generator used to generate the starting databeat
--      -- strobe value for soft shutdown case where the S2MM has to 
--      -- flush out all of the transfers that have been committed
--      -- to the AXI Write address channel. Starting Strobes must
--      -- match the committed address offest for each transfer. 
--      -- 
--      ------------------------------------------------------------
--      I_STRT_STRB_GEN : entity axi_sg_v4_1_6.axi_sg_strb_gen2
--      generic map (
--                            
--        C_OP_MODE            =>  0                         , -- 0 = Offset/Length mode
--        C_STRB_WIDTH         =>  BYTES_PER_DBEAT           ,   
--        C_OFFSET_WIDTH       =>  STRBGEN_ADDR_SLICE_WIDTH  ,   
--        C_NUM_BYTES_WIDTH    =>  STRBGEN_ADDR_SLICE_WIDTH+1           
--    
--        )
--      port map (
--        
--        start_addr_offset    =>  sig_strbgen_addr         , 
--        end_addr_offset      =>  STRBGEN_ADDR_0           , -- not used in op mode 0
--        num_valid_bytes      =>  sig_strbgen_bytes        , 
--        strb_out             =>  sig_sfhalt_next_strt_strb   
--    
--        );
--                                  
--     
--
--
--   
--   
--
--        -- Generate the WSTRB to use during soft shutdown 
--        sig_halt_strb  <= sig_strt_strb_reg 
--          When (sig_first_dbeat   = '1' or
--                sig_single_dbeat  = '1')
--          Else  (others => '1');
--
--           
--                  
--        -- Generate the Write Strobes for the MMap Write Data Channel
--        -- for the Indeterminate BTT case. Strobes come from the Stream
--        -- input from the Indeterminate BTT module during normal operation.
--        -- However, during soft shutdown, those strobes become unpredictable
--        -- so generated strobes have to be used.
--        data2skid_wstrb <=  sig_halt_strb
--          When (sig_halt_reg = '1')
--        
--          Else s2mm_strm_wstrb;
--          
--          
--          
--        -- Generate the Stream Ready for the Stream input side
--        sig_s2mm_strm_wready <=  sig_halt_reg             or -- force tready if a halt requested
--                                 (sig_mmap2data_ready    and -- MMap is accepting the xfers
--                                 sig_addr_chan_rdy       and -- xfers are commited on the address channel and 
--                                 sig_dqual_rdy           and -- there are commands in the command fifo        
--                                 not(sig_calc_error_reg) and -- No internal error                             
--                                 not(sig_stop_wvalid));      -- Gate off stream ready immediately after a wlast for 1 clk
--                                                             -- or when the soft shutdown has completed
--         
--        
--        -- MMap Write Data Channel Valid Handshaking
--        sig_data2mmap_valid <= (s2mm_strm_wvalid        or -- Normal Stream input valid       
--                               sig_halt_reg       )    and -- force valid if halt requested       
--                               sig_addr_chan_rdy       and -- xfers are commited on the address channel and       
--                               sig_dqual_rdy           and -- there are commands in the command fifo        
--                               not(sig_calc_error_reg) and -- No internal error
--                               not(sig_stop_wvalid);       -- Gate off wvalid immediately after a wlast for 1 clk
--                                                           -- or when the soft shutdown has completed
--              
--      
--         
--        -- TLAST Error housekeeping for Indeterminate BTT Mode
--        -- There is no Underrun/overrun in Stroe and Forward mode 
--         
--        sig_tlast_error_ovrrun  <= '0'; -- Not used with Indeterminate BTT
--        sig_tlast_error_undrrun <= '0'; -- Not used with Indeterminate BTT
--        sig_end_stbs_match_err  <= '0'; -- Not used with Indeterminate BTT
--        sig_tlast_error         <= '0'; -- Not used with Indeterminate BTT
--        sig_tlast_error_reg     <= '0'; -- Not used with Indeterminate BTT
--        sig_tlast_err_stop      <= '0'; -- Not used with Indeterminate BTT
--        
--        
--        
--        
--        
--        -------------------------------------------------------------
--        -- Synchronous Process with Sync Reset
--        --
--        -- Label: IMP_EOP_REG_FLOP
--        --
--        -- Process Description:
--        --  Register the End of Packet marker.
--        --
--        -------------------------------------------------------------
--        IMP_EOP_REG_FLOP : process (primary_aclk)
--          begin
--            if (primary_aclk'event and primary_aclk = '1') then
--               if (mmap_reset = '1') then
--        
--                 lsig_end_of_cmd_reg <= '0';
--                 lsig_eop_reg        <= '0';
--               
--               
--               Elsif (sig_good_strm_dbeat = '1') Then
--               
--        
--                 lsig_end_of_cmd_reg <= sig_next_cmd_cmplt_reg and
--                                        s2mm_strm_wlast;
--                 
--                 lsig_eop_reg        <= s2mm_strm_eop;
--               
--               else
--
--                 null; -- hold current state  
--                   
--               end if; 
--            end if;       
--          end process IMP_EOP_REG_FLOP; 
--        
--        
--        
--        
-- 
--        -----  Byte Counter Logic -----------------------------------------------
--        -- The Byte counter reflects the actual byte count received on the 
--        -- Stream input for each parent command loaded into the S2MM command
--        -- FIFO. Thus it counts input bytes until the command complete qualifier
--        -- is set and the TLAST input from the Stream input.
--      
--      
--        lsig_clr_byte_cntr        <= lsig_end_of_cmd_reg and   -- Clear if a new stream packet does not start 
--                                     not(sig_good_strm_dbeat); -- immediately after the previous one finished.    
--        
--     
--        lsig_ld_byte_cntr         <= lsig_end_of_cmd_reg and -- Only load if a new stream packet starts       
--                                     sig_good_strm_dbeat;    -- immediately after the previous one finished.       
--        
--        lsig_incr_byte_cntr       <= sig_good_strm_dbeat; 
--        
--        
--        lsig_byte_cntr_incr_value <=  RESIZE(UNSIGNED(s2mm_stbs_asserted), 
--                                                       BYTE_CNTR_WIDTH);
--     
--        -------------------------------------------------------------
--        -- Synchronous Process with Sync Reset
--        --
--        -- Label: IMP_BYTE_CMTR
--        --
--        -- Process Description:
--        -- Keeps a running byte count per burst packet loaded into the 
--        -- xfer FIFO. It is based on the strobes set on the incoming
--        -- Stream dbeat.
--        --
--        -------------------------------------------------------------
--        IMP_BYTE_CMTR : process (primary_aclk)
--           begin
--             if (primary_aclk'event and primary_aclk = '1') then
--               if (mmap_reset         = '1' or
--                   lsig_clr_byte_cntr = '1') then 
--
--                 lsig_byte_cntr <= (others => '0');
--                 
--               elsif (lsig_ld_byte_cntr = '1') then
--
--                 lsig_byte_cntr <= lsig_byte_cntr_incr_value;
--                 
--               elsif (lsig_incr_byte_cntr = '1') then
--
--                 lsig_byte_cntr <= lsig_byte_cntr + lsig_byte_cntr_incr_value;
--                 
--               else
--                 null;  -- hold current value
--               end if; 
--             end if;       
--           end process IMP_BYTE_CMTR; 
--     
--     
--        
-- 
--   
--      end generate GEN_INDET_BTT;
--   
   
    
    
    
    
    
    
              
    
    -- Internal logic ------------------------------
    
    sig_good_mmap_dbeat  <= sig_mmap2data_ready and 
                            sig_data2mmap_valid;
    
    
    sig_last_mmap_dbeat  <= sig_good_mmap_dbeat and 
                            sig_data2mmap_last;
     
     
    sig_get_next_dqual   <= sig_last_mmap_dbeat; 
    
    
    
    
    
    
         
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: REG_LAST_DBEAT
    --
    -- Process Description:
    --   This implements a FLOP that creates a pulse
    -- indicating the LAST signal for an outgoing write data channel
    -- has been sent. Note that it is possible to have back to 
    -- back LAST databeats.
    --
    -------------------------------------------------------------
    REG_LAST_DBEAT : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
           if (mmap_reset = '1') then

             sig_last_mmap_dbeat_reg <= '0';
             
           else
             
             sig_last_mmap_dbeat_reg <= sig_last_mmap_dbeat;
             
           end if; 
         end if;       
       end process REG_LAST_DBEAT; 
  
 
 
 
 
    
    
    -----  Write Status Interface Stuff --------------------------
    
    sig_push_to_wsc_cmplt <= sig_push_to_wsc and sig_wsc_ready;
    
    
    sig_set_push2wsc      <= (sig_good_mmap_dbeat and
                             sig_dbeat_cntr_eq_0) or
                             sig_push_err2wsc     or
                             sig_spcl_push_err2wsc;   -- Special case from CR616212
                             
    
    
    
    
    
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_INTERR_PUSH_FLOP
    --
    -- Process Description:
    -- Generate a 1 clock wide pulse when a calc error has propagated
    -- from the Command Calculator. This pulse is used to force a 
    -- push of the error status to the Write Status Controller
    -- without a AXI transfer completion.
    --
    -------------------------------------------------------------
    IMP_INTERR_PUSH_FLOP : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset       = '1' or
                sig_push_err2wsc = '1') then
              sig_push_err2wsc <= '0';
            elsif (sig_ld_new_cmd_reg = '1' and
                   sig_calc_error_reg = '1') then
              sig_push_err2wsc <= '1';
            else
              null; -- hold state
            end if; 
         end if;       
       end process IMP_INTERR_PUSH_FLOP; 
    
    
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_PUSH2WSC_FLOP
    --
    -- Process Description:
    -- Implements a Sample and hold register for the outbound status
    -- signals to the Write Status Controller (WSC). This register
    -- has to support back to back transfer completions.
    --
    -------------------------------------------------------------
    IMP_PUSH2WSC_FLOP : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset             = '1'  or
               (sig_push_to_wsc_cmplt = '1'  and
                sig_set_push2wsc      = '0')) then
              
              sig_push_to_wsc        <= '0';
              sig_data2wsc_tag       <=  (others => '0');
              sig_data2wsc_calc_err  <=  '0';
              sig_data2wsc_last_err  <=  '0';
              sig_data2wsc_cmd_cmplt <=  '0';
              
            elsif (sig_set_push2wsc   = '1' and 
                   sig_tlast_err_stop = '0') then
              
              sig_push_to_wsc        <= '1';
              sig_data2wsc_tag       <= sig_tag_reg          ;
              sig_data2wsc_calc_err  <= sig_calc_error_reg   ;
              sig_data2wsc_last_err  <= sig_tlast_error_reg or 
                                        sig_tlast_error      ;
              sig_data2wsc_cmd_cmplt <= sig_cmd_cmplt_reg   or 
                                        sig_tlast_error_reg or
                                        sig_tlast_error      ;
              
            else
              null;  -- hold current state
            end if; 
         end if;       
       end process IMP_PUSH2WSC_FLOP; 
     
  
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_LD_NEW_CMD_REG
    --
    -- Process Description:
    -- Registers the flag indicating a new command has been 
    -- loaded. Needs to be a 1 clk wide pulse.
    --
    -------------------------------------------------------------
    IMP_LD_NEW_CMD_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset         = '1' or
                sig_ld_new_cmd_reg = '1') then
              sig_ld_new_cmd_reg <= '0';
            else
              sig_ld_new_cmd_reg <= sig_ld_new_cmd;
            end if; 
         end if;       
       end process IMP_LD_NEW_CMD_REG; 
    
    
    
    
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_NXT_LEN_REG
    --
    -- Process Description:
    -- Registers the load control and length value for a command 
    -- passed to the WDC input command interface. The registered
    -- signals are used for the external Indeterminate BTT support
    -- ports.
    --
    -------------------------------------------------------------
    IMP_NXT_LEN_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset         = '1') then

              sig_s2mm_ld_nxt_len <= '0';
              sig_s2mm_wr_len     <= (others => '0');
              
            else
              sig_s2mm_ld_nxt_len <= mstr2data_cmd_valid and
                                     sig_data2mstr_cmd_ready;
              sig_s2mm_wr_len     <= mstr2data_len;
                                     
                                     
            end if; 
         end if;       
       end process IMP_NXT_LEN_REG; 
    
    
    
    
    
    
    
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_NO_DATA_CNTL_FIFO
    --
    -- If Generate Description:
     --   Omits the input data control FIFO if the requested FIFO
     -- depth is 1. The Data Qualifier Register serves as a 
     -- 1 deep FIFO by itself.
    --
    ------------------------------------------------------------
    GEN_NO_DATA_CNTL_FIFO : if (C_DATA_CNTL_FIFO_DEPTH = 1) generate
    
       
      begin

        -- Command Calculator Handshake output
        sig_data2mstr_cmd_ready <= sig_fifo_wr_cmd_ready;           
       
        sig_fifo_rd_cmd_valid   <= mstr2data_cmd_valid ;
        
        
        
        -- pre 13.1 sig_fifo_wr_cmd_ready  <= sig_dqual_reg_empty     and                                         
        -- pre 13.1                           sig_aposted_cntr_ready  and                                         
        -- pre 13.1                           not(wsc2mstr_halt_pipe) and -- The Wr Status Controller is not stalling
        -- pre 13.1                           not(sig_calc_error_reg);    -- the command execution pipe and there is  
        -- pre 13.1                                                       -- no calculation error being propagated    
        
        sig_fifo_wr_cmd_ready  <= sig_push_dqual_reg;
        
                                                              
        
        sig_fifo_next_tag         <= mstr2data_tag        ;    
        sig_fifo_next_sadddr_lsb  <= mstr2data_saddr_lsb  ;    
        sig_fifo_next_len         <= mstr2data_len        ;    
        sig_fifo_next_strt_strb   <= mstr2data_strt_strb  ;    
        sig_fifo_next_last_strb   <= mstr2data_last_strb  ;    
        sig_fifo_next_drr         <= mstr2data_drr        ;    
        sig_fifo_next_eof         <= mstr2data_eof        ;    
        sig_fifo_next_sequential  <= mstr2data_sequential ;    
        sig_fifo_next_cmd_cmplt   <= mstr2data_cmd_cmplt  ;    
        sig_fifo_next_calc_error  <= mstr2data_calc_error ;    
                                                             
             
   
      end generate GEN_NO_DATA_CNTL_FIFO;
  
  
 
    
    
    
     ------------------------------------------------------------
     -- If Generate
     --
     -- Label: GEN_DATA_CNTL_FIFO
     --
     -- If Generate Description:
     --   Includes the input data control FIFO if the requested 
     -- FIFO depth is more than 1.
     --
     ------------------------------------------------------------
     GEN_DATA_CNTL_FIFO : if (C_DATA_CNTL_FIFO_DEPTH > 1) generate

       begin

       
         -- Command Calculator Handshake output
         sig_data2mstr_cmd_ready  <= sig_fifo_wr_cmd_ready;           
         
         sig_fifo_wr_cmd_valid    <= mstr2data_cmd_valid  ;
         
         
         -- pop the fifo when dqual reg is pushed
         sig_fifo_rd_cmd_ready    <= sig_push_dqual_reg;     
                                                               

         
         
                              
         -- Format the input fifo data word
         sig_cmd_fifo_data_in  <=   mstr2data_calc_error &
                                    mstr2data_cmd_cmplt  &
                                    mstr2data_sequential &
                                    mstr2data_eof        &
                                    mstr2data_drr        &
                                    mstr2data_last_strb  &
                                    mstr2data_strt_strb  &
                                    mstr2data_len        &
                                    mstr2data_saddr_lsb  &
                                    mstr2data_tag ;
         
          
         -- Rip the output fifo data word
         sig_fifo_next_tag        <= sig_cmd_fifo_data_out((TAG_STRT_INDEX+TAG_WIDTH)-1 downto 
                                                            TAG_STRT_INDEX);                   
         sig_fifo_next_sadddr_lsb <= sig_cmd_fifo_data_out((SADDR_LSB_STRT_INDEX+SADDR_LSB_WIDTH)-1 downto 
                                                            SADDR_LSB_STRT_INDEX);
         sig_fifo_next_len        <= sig_cmd_fifo_data_out((LEN_STRT_INDEX+LEN_WIDTH)-1 downto 
                                                            LEN_STRT_INDEX);
         sig_fifo_next_strt_strb  <= sig_cmd_fifo_data_out((STRT_STRB_STRT_INDEX+STRB_WIDTH)-1 downto 
                                                            STRT_STRB_STRT_INDEX);
         sig_fifo_next_last_strb  <= sig_cmd_fifo_data_out((LAST_STRB_STRT_INDEX+STRB_WIDTH)-1 downto 
                                                            LAST_STRB_STRT_INDEX);
         sig_fifo_next_drr        <= sig_cmd_fifo_data_out(DRR_STRT_INDEX);
         sig_fifo_next_eof        <= sig_cmd_fifo_data_out(EOF_STRT_INDEX);
         sig_fifo_next_sequential <= sig_cmd_fifo_data_out(SEQUENTIAL_STRT_INDEX);
         sig_fifo_next_cmd_cmplt  <= sig_cmd_fifo_data_out(CMD_CMPLT_STRT_INDEX);
         sig_fifo_next_calc_error <= sig_cmd_fifo_data_out(CALC_ERR_STRT_INDEX);

         
         
         
         ------------------------------------------------------------
         -- Instance: I_DATA_CNTL_FIFO 
         --
         -- Description:
         -- Instance for the Command Qualifier FIFO
         --
         ------------------------------------------------------------
          I_DATA_CNTL_FIFO : entity axi_sg_v4_1_6.axi_sg_fifo
          generic map (
        
            C_DWIDTH             =>  DCTL_FIFO_WIDTH        , 
            C_DEPTH              =>  C_DATA_CNTL_FIFO_DEPTH , 
            C_IS_ASYNC           =>  USE_SYNC_FIFO          , 
            C_PRIM_TYPE          =>  FIFO_PRIM_TYPE         , 
            C_FAMILY             =>  C_FAMILY                 
           
            )
          port map (
            
            -- Write Clock and reset
            fifo_wr_reset        =>   mmap_reset            , 
            fifo_wr_clk          =>   primary_aclk          , 
            
            -- Write Side
            fifo_wr_tvalid       =>   sig_fifo_wr_cmd_valid , 
            fifo_wr_tready       =>   sig_fifo_wr_cmd_ready , 
            fifo_wr_tdata        =>   sig_cmd_fifo_data_in  , 
            fifo_wr_full         =>   open                  , 
           
           
            -- Read Clock and reset
            fifo_async_rd_reset  =>   mmap_reset            ,   
            fifo_async_rd_clk    =>   primary_aclk          , 
            
            -- Read Side
            fifo_rd_tvalid       =>   sig_fifo_rd_cmd_valid , 
            fifo_rd_tready       =>   sig_fifo_rd_cmd_ready , 
            fifo_rd_tdata        =>   sig_cmd_fifo_data_out , 
            fifo_rd_empty        =>   sig_cmd_fifo_empty      
           
            );
        

       end generate GEN_DATA_CNTL_FIFO;
         
          
    
   
  
  
  
  
  
    -- Data Qualifier Register ------------------------------------
    
    
    sig_ld_new_cmd           <= sig_push_dqual_reg              ;
    sig_dqual_rdy            <= sig_dqual_reg_full              ;
    sig_strt_strb_reg        <= sig_next_strt_strb_reg          ;
    sig_last_strb_reg        <= sig_next_last_strb_reg          ;
    sig_tag_reg              <= sig_next_tag_reg                ;
    sig_cmd_cmplt_reg        <= sig_next_cmd_cmplt_reg          ;
    sig_calc_error_reg       <= sig_next_calc_error_reg         ;
    
    sig_cmd_is_eof           <= sig_next_eof_reg                ;
    
    
    
    -- new for no bubbles between child requests
    sig_sequential_push      <= sig_good_mmap_dbeat and -- MMap handshake qualified
                                sig_last_dbeat      and -- last data beat of transfer
                                sig_next_sequential_reg;-- next queued command is sequential 
                                                        -- to the current command
    
    
    -- pre 13.1 sig_push_dqual_reg        <= (sig_sequential_push   or
    -- pre 13.1                               sig_dqual_reg_empty)  and 
    -- pre 13.1                              sig_fifo_rd_cmd_valid  and
    -- pre 13.1                              sig_aposted_cntr_ready and 
    -- pre 13.1                              not(wsc2mstr_halt_pipe);  -- The Wr Status Controller is not     
    -- pre 13.1                                                        -- stalling the command execution pipe 

    
    sig_push_dqual_reg       <= (sig_sequential_push    or
                                 sig_dqual_reg_empty)   and 
                                sig_fifo_rd_cmd_valid   and
                                sig_aposted_cntr_ready  and 
                                not(sig_calc_error_reg) and -- 13.1 addition => An error has not been propagated
                                not(wsc2mstr_halt_pipe);    -- The Wr Status Controller is not  
                                                            -- stalling the command execution pipe
                                                        







                                                         
    sig_pop_dqual_reg         <= not(sig_next_calc_error_reg) and 
                                 sig_get_next_dqual and 
                                 sig_dqual_reg_full  ; 
    
  
    -- new for no bubbles between child requests
    sig_clr_dqual_reg        <=  mmap_reset         or
                                 (sig_pop_dqual_reg and
                                 not(sig_push_dqual_reg));
  
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_DQUAL_REG
    --
    -- Process Description:
    --    This process implements a register for the Data 
    -- Control and qualifiers. It operates like a 1 deep Sync FIFO.
    --
    -------------------------------------------------------------
    IMP_DQUAL_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (sig_clr_dqual_reg = '1') then
              
              sig_next_tag_reg         <= (others => '0');
              sig_next_strt_strb_reg   <= (others => '0');
              sig_next_last_strb_reg   <= (others => '0');
              sig_next_eof_reg         <= '0'            ;
              sig_next_sequential_reg  <= '0'            ;
              sig_next_cmd_cmplt_reg   <= '0'            ;
              sig_next_calc_error_reg  <= '0'            ;
                                                        
              sig_dqual_reg_empty      <= '1'            ;
              sig_dqual_reg_full       <= '0'            ;
                                                        
            elsif (sig_push_dqual_reg = '1') then
              
              sig_next_tag_reg        <= sig_fifo_next_tag         ;
              sig_next_strt_strb_reg  <= sig_sfhalt_next_strt_strb ;
              sig_next_last_strb_reg  <= sig_fifo_next_last_strb   ;
              sig_next_eof_reg        <= sig_fifo_next_eof         ;
              sig_next_sequential_reg <= sig_fifo_next_sequential  ;
              sig_next_cmd_cmplt_reg  <= sig_fifo_next_cmd_cmplt   ;
              sig_next_calc_error_reg <= sig_fifo_next_calc_error  ;
              
              sig_dqual_reg_empty     <= '0';
              sig_dqual_reg_full      <= '1';
              
            else
              null;  -- don't change state
            end if; 
         end if;       
       end process IMP_DQUAL_REG; 
     

  
  
  
    
    
    -- Address LS Cntr logic  --------------------------
   
    sig_addr_lsb_reg         <= STD_LOGIC_VECTOR(sig_ls_addr_cntr);
    sig_addr_incr_unsgnd     <= TO_UNSIGNED(ADDR_INCR_VALUE, C_SEL_ADDR_WIDTH);
    sig_incr_ls_addr_cntr    <= sig_good_mmap_dbeat;
    
   
   
   
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: DO_ADDR_LSB_CNTR
    --
    -- Process Description:
    --  Implements the LS Address Counter used for controlling
    -- the Write STRB  DeMux during Burst transfers
    --
    -------------------------------------------------------------
    DO_ADDR_LSB_CNTR : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset         = '1' or
               (sig_pop_dqual_reg  = '1'and
                sig_push_dqual_reg = '0')) then                 -- Clear the Counter
              
              sig_ls_addr_cntr <= (others => '0');
            
            elsif (sig_push_dqual_reg = '1') then               -- Load the Counter
              
              sig_ls_addr_cntr <= unsigned(sig_fifo_next_sadddr_lsb);
            
            elsif (sig_incr_ls_addr_cntr = '1') then            -- Increment the Counter
              
              sig_ls_addr_cntr <= sig_ls_addr_cntr + sig_addr_incr_unsgnd;
            
            else
              null;  -- Hold Current value
            end if; 
         end if;       
       end process DO_ADDR_LSB_CNTR; 
    
    
    
    
    
    
    
    
    
    
    
    
   -- Address Posted Counter Logic --------------------------------------
   
    sig_addr_chan_rdy         <= not(sig_addr_posted_cntr_eq_0 or 
                                     sig_apc_going2zero)         ; -- Gates data channel xfer handshake
    
    sig_aposted_cntr_ready    <= not(sig_addr_posted_cntr_max)   ; -- Gates new command fetching
    
    sig_no_posted_cmds        <= sig_addr_posted_cntr_eq_0       ; -- Used for flushing cmds that are posted
    
 
 
 
    
    sig_incr_addr_posted_cntr <= sig_addr_posted         ;
    
    sig_decr_addr_posted_cntr <= sig_last_mmap_dbeat_reg ;
    
    sig_addr_posted_cntr_eq_0 <= '1'
      when (sig_addr_posted_cntr = ADDR_POSTED_ZERO)
      Else '0';
    
    sig_addr_posted_cntr_max <= '1'
      when (sig_addr_posted_cntr = ADDR_POSTED_MAX)
      Else '0';
    
    
    sig_addr_posted_cntr_eq_1 <= '1'
      when (sig_addr_posted_cntr = ADDR_POSTED_ONE)
      Else '0';
    
    sig_apc_going2zero  <= sig_addr_posted_cntr_eq_1 and
                           sig_decr_addr_posted_cntr and
                           not(sig_incr_addr_posted_cntr);
    
    
    
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_ADDR_POSTED_FIFO_CNTR
    --
    -- Process Description:
    --    This process implements a counter for the tracking  
    -- if an Address has been posted on the AXI address channel.
    -- The Data Controller must wait for an address to be posted
    -- before proceeding with the corresponding data transfer on
    -- the Data Channel. The counter is also used to track flushing
    -- operations where all transfers commited on the  AXI Address
    -- Channel have to be completed before a halt can occur.
    -------------------------------------------------------------
    IMP_ADDR_POSTED_FIFO_CNTR : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              
              sig_addr_posted_cntr <= ADDR_POSTED_ZERO;
              
            elsif (sig_incr_addr_posted_cntr = '1' and
                   sig_decr_addr_posted_cntr = '0' and
                   sig_addr_posted_cntr_max  = '0') then
              
              sig_addr_posted_cntr <= sig_addr_posted_cntr + ADDR_POSTED_ONE ;
              
            elsif (sig_incr_addr_posted_cntr = '0' and
                   sig_decr_addr_posted_cntr = '1' and
                   sig_addr_posted_cntr_eq_0 = '0') then
              
              sig_addr_posted_cntr <= sig_addr_posted_cntr - ADDR_POSTED_ONE ;
              
            else
              null;  -- don't change state
            end if; 
         end if;       
       end process IMP_ADDR_POSTED_FIFO_CNTR; 
 
 
      
      
      
    ------- First/Middle/Last Dbeat detimination -------------------
     
     sig_new_len_eq_0 <= '1'
       When  (sig_fifo_next_len = LEN_OF_ZERO)
       else '0';
     
     
      
      
     -------------------------------------------------------------
     -- Synchronous Process with Sync Reset
     --
     -- Label: DO_FIRST_MID_LAST
     --
     -- Process Description:
     --  Implements the detection of the First/Mid/Last databeat of
     -- a transfer.
     --
     -------------------------------------------------------------
     DO_FIRST_MID_LAST : process (primary_aclk)
        begin
          if (primary_aclk'event and primary_aclk = '1') then
             if (mmap_reset = '1') then
               
               sig_first_dbeat   <= '0';
               sig_last_dbeat    <= '0';
               sig_single_dbeat  <= '0';
             
             elsif (sig_ld_new_cmd = '1') then
               
               sig_first_dbeat   <= not(sig_new_len_eq_0);
               sig_last_dbeat    <= sig_new_len_eq_0;
               sig_single_dbeat  <= sig_new_len_eq_0;
             
             Elsif (sig_dbeat_cntr_eq_1 = '1' and
                    sig_good_mmap_dbeat = '1') Then
             
               sig_first_dbeat   <= '0';
               sig_last_dbeat    <= '1';
               sig_single_dbeat  <= '0';
             
             Elsif (sig_dbeat_cntr_eq_0 = '0' and
                    sig_dbeat_cntr_eq_1 = '0' and
                    sig_good_mmap_dbeat = '1') Then
             
               sig_first_dbeat   <= '0';
               sig_last_dbeat    <= '0';
               sig_single_dbeat  <= '0';
             
             else
               null; -- hold current state
             end if; 
          end if;       
        end process DO_FIRST_MID_LAST; 
    
    
   
   
   
   -------  Data Controller Halted Indication ------------------------------- 
    
 
    data2all_dcntlr_halted <= sig_no_posted_cmds or
                              sig_calc_error_reg;
 
 
    
    
       
       
       
       
       
       
    
    
   -------  Data Beat counter logic ------------------------------- 
    
    
    
    
    
    sig_dbeat_cntr_int  <= TO_INTEGER(sig_dbeat_cntr);
    
    sig_dbeat_cntr_eq_0 <= '1'
      when (sig_dbeat_cntr_int = 0)
      Else '0';
    
    sig_dbeat_cntr_eq_1 <= '1'
      when (sig_dbeat_cntr_int = 1)
      Else '0';
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: DO_DBEAT_CNTR
    --
    -- Process Description:
    -- Implements the transfer data beat counter used to track 
    -- progress of the transfer.
    --
    -------------------------------------------------------------
    DO_DBEAT_CNTR : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              sig_dbeat_cntr <= (others => '0');
            elsif (sig_ld_new_cmd = '1') then
              sig_dbeat_cntr <= unsigned(sig_fifo_next_len);
            Elsif (sig_good_mmap_dbeat = '1' and
                   sig_dbeat_cntr_eq_0 = '0') Then
              sig_dbeat_cntr <= sig_dbeat_cntr-1;
            else
              null; -- Hold current state
            end if; 
         end if;       
       end process DO_DBEAT_CNTR; 
  
  
  
  
  
  
  
  
  
  
  
  
   -------  Soft Shutdown Logic ------------------------------- 
    
    
    
    
    
    -- Formulate the soft shutdown complete flag
    sig_data2rst_stop_cmplt  <= (sig_halt_reg_dly3        and   -- Normal Mode shutdown
                                 sig_no_posted_cmds       and 
                                 not(sig_calc_error_reg)) or
                                (sig_halt_reg_dly3  and         -- Shutdown after error trap
                                 sig_calc_error_reg);
    
    
              
    
    -- Generate a gate signal to deassert the WVALID output
    -- for 1 clock cycle after a WLAST is issued. This only 
    -- occurs when in soft shutdown mode. 
    sig_stop_wvalid  <= (sig_last_mmap_dbeat_reg and
                        sig_halt_reg) or
                        sig_data2rst_stop_cmplt;
  
  
    
    
    
    -- Assign the output port skid buf control for the
    -- input Stream skid buffer
    data2skid_halt      <= sig_data2skid_halt;
    
    -- Create a 1 clock wide pulse to tell the input
    -- stream skid buffer to shut down.
    sig_data2skid_halt  <=  sig_halt_reg_dly2 and 
                            not(sig_halt_reg_dly3);
    
    
  
  
     
     
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_HALT_REQ_REG
    --
    -- Process Description:
    --   Implements the flop for capturing the Halt request from 
    -- the Reset module.
    --
    -------------------------------------------------------------
    IMP_HALT_REQ_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              
              sig_halt_reg      <= '0';
            
            elsif (rst2data_stop_request = '1') then
              
              sig_halt_reg <= '1';
            
            else
              null;  -- Hold current State
            end if; 
         end if;       
       end process IMP_HALT_REQ_REG; 
  
  
   
   
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_HALT_REQ_REG_DLY
    --
    -- Process Description:
    --   Implements the flops for delaying the halt request by 3
    -- clocks to allow the Address Controller to halt before the
    -- Data Contoller can safely indicate it has exhausted all
    -- transfers committed to the AXI Address Channel by the Address
    -- Controller.
    --
    -------------------------------------------------------------
    IMP_HALT_REQ_REG_DLY : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              
              sig_halt_reg_dly1 <= '0';
              sig_halt_reg_dly2 <= '0';
              sig_halt_reg_dly3 <= '0';
            
            else
              
              sig_halt_reg_dly1 <= sig_halt_reg;
              sig_halt_reg_dly2 <= sig_halt_reg_dly1;
              sig_halt_reg_dly3 <= sig_halt_reg_dly2;
            
            end if; 
         end if;       
       end process IMP_HALT_REQ_REG_DLY; 
  
  
   
    
    
    
    
    
    
 
  end implementation;


-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_sg_wr_status_cntl.vhd
  --
  -- Description:     
  --    This file implements the DataMover Master Write Status Controller.                 
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  library axi_sg_v4_1_6;
  use axi_sg_v4_1_6.axi_sg_fifo;
  
  -------------------------------------------------------------------------------
  
  entity axi_sg_wr_status_cntl is
    generic (
      
      C_ENABLE_INDET_BTT     : Integer range  0 to   1 :=  0;
        -- Specifies if the Indeterminate BTT Module is enabled
        -- for use (outside of this module)
      
      C_SF_BYTES_RCVD_WIDTH  : Integer range  1 to  23 :=  1;
        -- Sets the width of the data2wsc_bytes_rcvd port used for 
        -- relaying the actual number of bytes received when Idet BTT is 
        -- enabled (C_ENABLE_INDET_BTT = 1)
      
      C_STS_FIFO_DEPTH       : Integer range  1 to  32 :=  8;
        -- Specifies the depth of the internal status queue fifo
      
      C_STS_WIDTH            : Integer range  8 to  32 :=  8;
        -- sets the width of the Status ports
      
      C_TAG_WIDTH            : Integer range  1 to 8   :=  4;
        -- Sets the width of the Tag field in the Status reply
      
      C_FAMILY               : String                  := "virtex7"
        -- Specifies the target FPGA device family
      
      
      );
    port (
      
      -- Clock and Reset inputs ------------------------------------------
                                                                        --
      primary_aclk         : in  std_logic;                             --
         -- Primary synchronization clock for the Master side           --
         -- interface and internal logic. It is also used               --
         -- for the User interface synchronization when                 --
         -- C_STSCMD_IS_ASYNC = 0.                                      --
                                                                        --
      -- Reset input                                                    --
      mmap_reset           : in  std_logic;                             --
         -- Reset used for the internal master logic                    --
      --------------------------------------------------------------------
      

     
      
      -- Soft Shutdown Control interface  --------------------------------
                                                                        --
      rst2wsc_stop_request : in  std_logic;                             --
         -- Active high soft stop request to modules                    --
                                                                        --
      wsc2rst_stop_cmplt   : Out std_logic;                             --
        -- Active high indication that the Write status Controller      --
        -- has completed any pending transfers committed by the         --
        -- Address Controller after a stop has been requested by        --
        -- the Reset module.                                            --
                                                                        --
      addr2wsc_addr_posted : In std_logic ;                             --
         -- Indication from the Address Channel Controller to the       --
         -- write Status Controller that an address has been posted     --
         -- to the AXI Address Channel                                  --
      --------------------------------------------------------------------


     
     
      --  Write Response Channel Interface -------------------------------
                                                                        --
      s2mm_bresp          : In std_logic_vector(1 downto 0);            --
         -- The Write response value                                    --
                                                                        --
      s2mm_bvalid         : In std_logic ;                              --
         -- Indication from the Write Response Channel that a new       --
         -- write status input is valid                                 --
                                                                        --
      s2mm_bready         : out std_logic ;                             --
         -- Indication to the Write Response Channel that the           --
         -- Status module is ready for a new status input               --
      --------------------------------------------------------------------


       
       
      -- Command Calculator Interface -------------------------------------
                                                                         --
      calc2wsc_calc_error    : in std_logic ;                            --
         -- Indication from the Command Calculator that a calculation    --
         -- error has occured.                                           --
      ---------------------------------------------------------------------
     
     
      
        
      -- Address Controller Status ----------------------------------------
                                                                         --
      addr2wsc_calc_error    : In std_logic ;                            --
         -- Indication from the Address Channel Controller that it       --
         -- has encountered a calculation error from the command         --
         -- Calculator                                                   --
                                                                         --
      addr2wsc_fifo_empty    : In std_logic ;                            --
         -- Indication from the Address Controller FIFO that it          --
         -- is empty (no commands pending)                               --
      ---------------------------------------------------------------------

                    
                    
                    
      --  Data Controller Status ---------------------------------------------------------
                                                                                        --
      data2wsc_tag           : In std_logic_vector(C_TAG_WIDTH-1 downto 0);             --
         -- The command tag                                                             --
                                                                                        --
      data2wsc_calc_error    : In std_logic ;                                           --
         -- Indication from the Data Channel Controller FIFO that it                    --
         -- has encountered a Calculation error in the command pipe                     --
                                                                                        --
      data2wsc_last_error    : In std_logic ;                                           --
         -- Indication from the Write Data Channel Controller that a                    --
         -- premature TLAST assertion was encountered on the incoming                   --
         -- Stream Channel                                                              --
                                                                                        --
      data2wsc_cmd_cmplt    : In std_logic ;                                            --
         -- Indication from the Data Channel Controller that the                        --
         -- corresponding status is the final status for a parent                       --
         -- command fetched from the command FIFO                                       --
                                                                                        --
      data2wsc_valid         : In std_logic ;                                           --
         -- Indication from the Data Channel Controller FIFO that it                    --
         -- has a new tag/error status to transfer                                      --
                                                                                        --
      wsc2data_ready         : out std_logic ;                                          --
         -- Indication to the Data Channel Controller FIFO that the                     --
         -- Status module is ready for a new tag/error status input                     --
                                                                                        --
                                                                                        --
      data2wsc_eop           : In  std_logic;                                           --
         -- Input from the Write Data Controller indicating that the                    --
         -- associated command status also corresponds to a End of Packet               --
         -- marker for the input Stream. This is only used when Store and               --
         -- Forward is enabled in the S2MM.                                             --
                                                                                        --
      data2wsc_bytes_rcvd    : In  std_logic_vector(C_SF_BYTES_RCVD_WIDTH-1 downto 0);  --
         -- Input from the Write Data Controller indicating the actual                  --
         -- number of bytes received from the Stream input for the                      --
         -- corresponding command status. This is only used when Store and              --
         -- Forward is enabled in the S2MM.                                             --
      ------------------------------------------------------------------------------------
      
      
      
      -- Command/Status Interface --------------------------------------------------------
                                                                                        --
      wsc2stat_status       : Out std_logic_vector(C_STS_WIDTH-1 downto 0);             --
         -- Read Status value collected during a Read Data transfer                     --
         -- Output to the Command/Status Module                                         --
                                                                                        --
      stat2wsc_status_ready : In  std_logic;                                            --
         -- Input from the Command/Status Module indicating that the                    --
         -- Status Reg/FIFO is Full and cannot accept more staus writes                 --
                                                                                        --
      wsc2stat_status_valid : Out std_logic ;                                           --
         -- Control Signal to Write the Status value to the Status                      --
         -- Reg/FIFO                                                                    --
      ------------------------------------------------------------------------------------

  
                                                                             
                                                                             
      -- Address and Data Controller Pipe halt --------------------------------
                                                                             --
      wsc2mstr_halt_pipe    : Out std_logic                                  --
         -- Indication to Halt the Data and Address Command pipeline due     --
         -- to the Status pipe getting full at some point                    --
      -------------------------------------------------------------------------

  
      );
  
  end entity axi_sg_wr_status_cntl;
  
  
  architecture implementation of axi_sg_wr_status_cntl is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

 
 
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: funct_set_cnt_width
    --
    -- Function Description:
    --    Sets a count width based on a fifo depth. A depth of 4 or less
    -- is a special case which requires a minimum count width of 3 bits.
    --
    -------------------------------------------------------------------
    function funct_set_cnt_width (fifo_depth : integer) return integer is
    
      Variable temp_cnt_width : Integer := 4;
    
    begin
    
      
      if (fifo_depth <= 4) then
      
         temp_cnt_width := 3;
      
                    -- coverage off  
      elsif (fifo_depth <= 8) then
      
         temp_cnt_width := 4;
      
      elsif (fifo_depth <= 16) then
      
         temp_cnt_width := 5;
      
      elsif (fifo_depth <= 32) then
      
         temp_cnt_width := 6;
      
      else  -- fifo depth <= 64
      
         temp_cnt_width := 7;
      
                    -- coverage on  
      end if;
      
      Return (temp_cnt_width);
       
       
    end function funct_set_cnt_width;
    
 
 
 
  
    -- Constant Declarations  --------------------------------------------
    
    Constant OKAY                   : std_logic_vector(1 downto 0) := "00";
    Constant EXOKAY                 : std_logic_vector(1 downto 0) := "01";
    Constant SLVERR                 : std_logic_vector(1 downto 0) := "10";
    Constant DECERR                 : std_logic_vector(1 downto 0) := "11";
    Constant STAT_RSVD              : std_logic_vector(3 downto 0) := "0000";
    Constant TAG_WIDTH              : integer := C_TAG_WIDTH;
    Constant STAT_REG_TAG_WIDTH     : integer := 4;
    Constant SYNC_FIFO_SELECT       : integer := 0;
    Constant SRL_FIFO_TYPE          : integer := 2;
    Constant DCNTL_SFIFO_DEPTH      : integer := C_STS_FIFO_DEPTH;
    Constant DCNTL_STATCNT_WIDTH    : integer := funct_set_cnt_width(C_STS_FIFO_DEPTH);-- bits
    Constant DCNTL_HALT_THRES       : unsigned(DCNTL_STATCNT_WIDTH-1 downto 0) := 
                                      TO_UNSIGNED(DCNTL_SFIFO_DEPTH-2,DCNTL_STATCNT_WIDTH);
    Constant DCNTL_STATCNT_ZERO     : unsigned(DCNTL_STATCNT_WIDTH-1 downto 0) := (others => '0');
    Constant DCNTL_STATCNT_MAX      : unsigned(DCNTL_STATCNT_WIDTH-1 downto 0) := 
                                      TO_UNSIGNED(DCNTL_SFIFO_DEPTH,DCNTL_STATCNT_WIDTH);
    Constant DCNTL_STATCNT_ONE      : unsigned(DCNTL_STATCNT_WIDTH-1 downto 0) := 
                                      TO_UNSIGNED(1, DCNTL_STATCNT_WIDTH);
    Constant WRESP_WIDTH            : integer := 2;
    Constant WRESP_SFIFO_WIDTH      : integer := WRESP_WIDTH;
    Constant WRESP_SFIFO_DEPTH      : integer := DCNTL_SFIFO_DEPTH;
    
    Constant ADDR_POSTED_CNTR_WIDTH : integer := funct_set_cnt_width(C_STS_FIFO_DEPTH);-- bits 
    
    
    Constant ADDR_POSTED_ZERO       : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) 
                                      := (others => '0');
    Constant ADDR_POSTED_ONE        : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) 
                                      := TO_UNSIGNED(1, ADDR_POSTED_CNTR_WIDTH);
    Constant ADDR_POSTED_MAX        : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) 
                                      := (others => '1');
    
    
    -- Signal Declarations  --------------------------------------------
    
    signal sig_valid_status_rdy      : std_logic := '0';
    signal sig_decerr                : std_logic := '0';
    signal sig_slverr                : std_logic := '0';
    signal sig_coelsc_okay_reg       : std_logic := '0';
    signal sig_coelsc_interr_reg     : std_logic := '0';
    signal sig_coelsc_decerr_reg     : std_logic := '0';
    signal sig_coelsc_slverr_reg     : std_logic := '0';
    signal sig_coelsc_tag_reg        : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_pop_coelsc_reg        : std_logic := '0';
    signal sig_push_coelsc_reg       : std_logic := '0';
    signal sig_coelsc_reg_empty      : std_logic := '0';
    signal sig_coelsc_reg_full       : std_logic := '0';
    signal sig_tag2status            : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_data_tag_reg          : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
    signal sig_data_err_reg          : std_logic := '0';
    signal sig_data_last_err_reg     : std_logic := '0';
    signal sig_data_cmd_cmplt_reg    : std_logic := '0';
    signal sig_bresp_reg             : std_logic_vector(1 downto 0) := (others => '0');
    signal sig_push_status           : std_logic := '0';
    Signal sig_status_push_ok        : std_logic := '0';
    signal sig_status_valid          : std_logic := '0';
    signal sig_wsc2data_ready        : std_logic := '0';
    signal sig_s2mm_bready           : std_logic := '0';
    signal sig_wresp_sfifo_in        : std_logic_vector(WRESP_SFIFO_WIDTH-1 downto 0) := (others => '0');
    signal sig_wresp_sfifo_out       : std_logic_vector(WRESP_SFIFO_WIDTH-1 downto 0) := (others => '0');
    signal sig_wresp_sfifo_wr_valid  : std_logic := '0';
    signal sig_wresp_sfifo_wr_ready  : std_logic := '0';
    signal sig_wresp_sfifo_wr_full   : std_logic := '0';
    signal sig_wresp_sfifo_rd_valid  : std_logic := '0';
    signal sig_wresp_sfifo_rd_ready  : std_logic := '0';
    signal sig_wresp_sfifo_rd_empty  : std_logic := '0';
    signal sig_halt_reg              : std_logic := '0';
    signal sig_halt_reg_dly1         : std_logic := '0';
    signal sig_halt_reg_dly2         : std_logic := '0';
    signal sig_halt_reg_dly3         : std_logic := '0';
    signal sig_addr_posted_cntr      : unsigned(ADDR_POSTED_CNTR_WIDTH-1 downto 0) := (others => '0');
    signal sig_addr_posted_cntr_eq_0 : std_logic := '0';
    signal sig_addr_posted_cntr_eq_1 : std_logic := '0';
    signal sig_addr_posted_cntr_max  : std_logic := '0';
    signal sig_decr_addr_posted_cntr : std_logic := '0';
    signal sig_incr_addr_posted_cntr : std_logic := '0';
    signal sig_no_posted_cmds        : std_logic := '0';
    signal sig_addr_posted           : std_logic := '0';
    signal sig_all_cmds_done         : std_logic := '0';
    signal sig_wsc2stat_status       : std_logic_vector(C_STS_WIDTH-1 downto 0) := (others => '0');
    signal sig_dcntl_sfifo_wr_valid  : std_logic := '0';
    signal sig_dcntl_sfifo_wr_ready  : std_logic := '0';
    signal sig_dcntl_sfifo_wr_full   : std_logic := '0';
    signal sig_dcntl_sfifo_rd_valid  : std_logic := '0';
    signal sig_dcntl_sfifo_rd_ready  : std_logic := '0';
    signal sig_dcntl_sfifo_rd_empty  : std_logic := '0';
    signal sig_wdc_statcnt           : unsigned(DCNTL_STATCNT_WIDTH-1 downto 0) := (others => '0');
    signal sig_incr_statcnt          : std_logic := '0';
    signal sig_decr_statcnt          : std_logic := '0';
    signal sig_statcnt_eq_max        : std_logic := '0';
    signal sig_statcnt_eq_0          : std_logic := '0';
    signal sig_statcnt_gt_eq_thres   : std_logic := '0';
    signal sig_wdc_status_going_full : std_logic := '0';
                                           
                                           
                                           
            
  begin --(architecture implementation)
  
    
    -- Assign the ready output to the AXI Write Response Channel
    s2mm_bready           <= sig_s2mm_bready or
                             sig_halt_reg;     -- force bready if a Halt is requested
    
    -- Assign the ready output to the Data Controller status interface
    wsc2data_ready        <= sig_wsc2data_ready;
    
    -- Assign the status valid output control to the Status FIFO
    wsc2stat_status_valid <= sig_status_valid ;
    
    -- Formulate the status output value to the Status FIFO
    wsc2stat_status       <=  sig_wsc2stat_status;
    
     
    -- Formulate the status write request signal
    sig_status_valid      <= sig_push_status;
    
                                           
     
    -- Indicate the desire to push a coelesced status word 
    -- to the Status FIFO
    sig_push_status       <= sig_coelsc_reg_full;
    
    
    
    -- Detect that a push of a new status word is completing
    sig_status_push_ok    <= sig_status_valid and
                             stat2wsc_status_ready; 
    
    sig_pop_coelsc_reg    <= sig_status_push_ok;
    
    
    -- Signal a halt to the execution pipe if new status
    -- is valid but the Status FIFO is not accepting it or
    -- the WDC Status FIFO is going full
    wsc2mstr_halt_pipe    <= (sig_status_valid and
                             not(stat2wsc_status_ready)) or
                             sig_wdc_status_going_full;
 
    
    -- Monitor the Status capture registers to detect a 
    -- qualified Status set and push to the coelescing register
    -- when available to do so
    sig_push_coelsc_reg   <= sig_valid_status_rdy and 
                             sig_coelsc_reg_empty;
    
    --    pre CR616212  sig_valid_status_rdy  <= (sig_wresp_sfifo_rd_valid  and 
    --    pre CR616212                            sig_dcntl_sfifo_rd_valid) or
    --    pre CR616212                           (sig_data_err_reg and
    --    pre CR616212                            sig_dcntl_sfifo_rd_valid);     
          
    sig_valid_status_rdy  <= (sig_wresp_sfifo_rd_valid  and 
                              sig_dcntl_sfifo_rd_valid) or
                             (sig_data_err_reg and
                              sig_dcntl_sfifo_rd_valid) or  -- or Added for CR616212
                             (sig_data_last_err_reg and     -- Added for CR616212
                              sig_dcntl_sfifo_rd_valid);    -- Added for CR616212 
          
 
 
 
    -- Decode the AXI MMap Read Respose       
    sig_decerr  <= '1'
      When sig_bresp_reg = DECERR
      Else '0'; 
           
    sig_slverr  <= '1'
      When sig_bresp_reg = SLVERR
      Else '0';      
          
    
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_TAG_LE_STAT
    --
    -- If Generate Description:
    -- Populates the TAG bits into the availble Status bits when
    -- the TAG width is less than or equal to the available number
    -- of bits in the Status word. 
    --
    ------------------------------------------------------------
    GEN_TAG_LE_STAT : if (TAG_WIDTH <= STAT_REG_TAG_WIDTH) generate
    
       -- local signals
         signal lsig_temp_tag_small : std_logic_vector(STAT_REG_TAG_WIDTH-1 downto 0) := (others => '0');
         
         
       begin
    
         sig_tag2status <= lsig_temp_tag_small;
         
         
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: POPULATE_SMALL_TAG
         --
         -- Process Description:
         --
         --
         -------------------------------------------------------------
         POPULATE_SMALL_TAG : process (sig_coelsc_tag_reg)
            begin
         
              -- Set default value
              lsig_temp_tag_small <= (others => '0');
          
              -- Now overload actual TAG bits
              lsig_temp_tag_small(TAG_WIDTH-1 downto 0) <= sig_coelsc_tag_reg;
          
         
            end process POPULATE_SMALL_TAG; 
         
         
       end generate GEN_TAG_LE_STAT;
     
     
     
     
     
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_TAG_GT_STAT
    --
    -- If Generate Description:
    -- Populates the TAG bits into the availble Status bits when
    -- the TAG width is greater than the available number of 
    -- bits in the Status word. The upper bits of the TAG are 
    -- clipped off (discarded). 
    --
    ------------------------------------------------------------
    GEN_TAG_GT_STAT : if (TAG_WIDTH > STAT_REG_TAG_WIDTH) generate
    
       -- local signals
         signal lsig_temp_tag_big : std_logic_vector(STAT_REG_TAG_WIDTH-1 downto 0) := (others => '0');
         
         
       begin
    
         
         sig_tag2status <= lsig_temp_tag_big;
         
         
         -------------------------------------------------------------
         -- Combinational Process
         --
         -- Label: POPULATE_BIG_TAG
         --
         -- Process Description:
         --
         --
         -------------------------------------------------------------
         POPULATE_SMALL_TAG : process (sig_coelsc_tag_reg)
            begin
         
              -- Set default value
              lsig_temp_tag_big <= (others => '0');
          
              -- Now overload actual TAG bits
              lsig_temp_tag_big <= sig_coelsc_tag_reg(STAT_REG_TAG_WIDTH-1 downto 0);
          
         
            end process POPULATE_SMALL_TAG; 
         
         
       end generate GEN_TAG_GT_STAT;
     
     
     
    
    
    
    
    
    
    
    
    
     
     
    
    
    
    -------------------------------------------------------------------------
    -- Write Response Channel input FIFO and logic
    
                        
    -- BRESP is the only fifo data
    sig_wresp_sfifo_in       <=  s2mm_bresp;     
    
    
    -- The fifo output is already in the right format
    sig_bresp_reg            <=  sig_wresp_sfifo_out;
    
    
    -- Write Side assignments
    sig_wresp_sfifo_wr_valid <=  s2mm_bvalid;
    
    sig_s2mm_bready          <=  sig_wresp_sfifo_wr_ready;
    
    
    -- read Side ready assignment
    sig_wresp_sfifo_rd_ready <=  sig_push_coelsc_reg;
    
    
    
    ------------------------------------------------------------
    -- Instance: I_WRESP_STATUS_FIFO 
    --
    -- Description:
    -- Instance for the AXI Write Response FIFO
    --
    ------------------------------------------------------------
     I_WRESP_STATUS_FIFO : entity axi_sg_v4_1_6.axi_sg_fifo
     generic map (
   
       C_DWIDTH             =>  WRESP_SFIFO_WIDTH         ,  
       C_DEPTH              =>  WRESP_SFIFO_DEPTH         ,  
       C_IS_ASYNC           =>  SYNC_FIFO_SELECT          ,  
       C_PRIM_TYPE          =>  SRL_FIFO_TYPE             ,  
       C_FAMILY             =>  C_FAMILY                     
      
       )
     port map (
       
       -- Write Clock and reset
       fifo_wr_reset        =>   mmap_reset               ,  
       fifo_wr_clk          =>   primary_aclk             ,  
       
       -- Write Side
       fifo_wr_tvalid       =>   sig_wresp_sfifo_wr_valid ,  
       fifo_wr_tready       =>   sig_wresp_sfifo_wr_ready ,  
       fifo_wr_tdata        =>   sig_wresp_sfifo_in       ,  
       fifo_wr_full         =>   sig_wresp_sfifo_wr_full  ,  
      
      
       -- Read Clock and reset (not used in Sync mode)
       fifo_async_rd_reset  =>   mmap_reset               ,    
       fifo_async_rd_clk    =>   primary_aclk             ,  
       
       -- Read Side
       fifo_rd_tvalid       =>   sig_wresp_sfifo_rd_valid ,  
       fifo_rd_tready       =>   sig_wresp_sfifo_rd_ready ,  
       fifo_rd_tdata        =>   sig_wresp_sfifo_out      ,  
       fifo_rd_empty        =>   sig_wresp_sfifo_rd_empty    
      
       );
   

    
    
    
    
    
    --------  Write Data Controller Status FIFO Going Full Logic -------------
    
    
    sig_incr_statcnt   <= sig_dcntl_sfifo_wr_valid and
                          sig_dcntl_sfifo_wr_ready;
    
    sig_decr_statcnt   <= sig_dcntl_sfifo_rd_valid and
                          sig_dcntl_sfifo_rd_ready;
    
          
    sig_statcnt_eq_max <= '1'
      when (sig_wdc_statcnt = DCNTL_STATCNT_MAX)
      Else '0';
          
    sig_statcnt_eq_0   <= '1'
      when (sig_wdc_statcnt = DCNTL_STATCNT_ZERO)
      Else '0';
          
    sig_statcnt_gt_eq_thres <= '1'
      when (sig_wdc_statcnt >= DCNTL_HALT_THRES)
      Else '0';
          
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_WDC_GOING_FULL_FLOP
    --
    -- Process Description:
    --  Implements a flop for the WDC Status FIFO going full flag.
    --
    -------------------------------------------------------------
    IMP_WDC_GOING_FULL_FLOP : process (primary_aclk)
      begin
        if (primary_aclk'event and primary_aclk = '1') then
           if (mmap_reset = '1') then
    
             sig_wdc_status_going_full <= '0';
           
           else
    
             sig_wdc_status_going_full <= sig_statcnt_gt_eq_thres;
    
           end if; 
        end if;       
      end process IMP_WDC_GOING_FULL_FLOP; 
    
          
          
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_DCNTL_FIFO_CNTR
    --
    -- Process Description:
    --   Implements a simple counter keeping track of the number
    -- of entries in the WDC Status FIFO. If the Status FIFO gets
    -- too full, the S2MM Data Pipe has to be halted.
    --
    -------------------------------------------------------------
    IMP_DCNTL_FIFO_CNTR : process (primary_aclk)
      begin
        if (primary_aclk'event and primary_aclk = '1') then
           if (mmap_reset = '1') then
    
             sig_wdc_statcnt <= (others => '0');
    
           elsif (sig_incr_statcnt   = '1' and 
                  sig_decr_statcnt   = '0' and
                  sig_statcnt_eq_max = '0') then
    
             sig_wdc_statcnt <= sig_wdc_statcnt + DCNTL_STATCNT_ONE;
    
           elsif (sig_incr_statcnt = '0' and 
                  sig_decr_statcnt = '1' and
                  sig_statcnt_eq_0 = '0') then
    
             sig_wdc_statcnt <= sig_wdc_statcnt - DCNTL_STATCNT_ONE;
    
           else
    
             null; -- Hold current count value
    
           end if; 
        end if;       
      end process IMP_DCNTL_FIFO_CNTR; 
          
          
          
          
          
          
          
          
          
          
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_OMIT_INDET_BTT
    --
    -- If Generate Description:
    --  Implements the logic needed when Indeterminate BTT is
    -- not enabled in the S2MM function.
    --
    ------------------------------------------------------------
    GEN_OMIT_INDET_BTT : if (C_ENABLE_INDET_BTT = 0) generate
    
       -- Local Constants
       Constant DCNTL_SFIFO_WIDTH           : integer := STAT_REG_TAG_WIDTH+3;
       Constant DCNTL_SFIFO_CMD_CMPLT_INDEX : integer := 0;
       Constant DCNTL_SFIFO_TLAST_ERR_INDEX : integer := 1;
       Constant DCNTL_SFIFO_CALC_ERR_INDEX  : integer := 2;
       Constant DCNTL_SFIFO_TAG_INDEX       : integer := DCNTL_SFIFO_CALC_ERR_INDEX+1;
    
       
       -- local signals
       signal sig_dcntl_sfifo_in        : std_logic_vector(DCNTL_SFIFO_WIDTH-1 downto 0) := (others => '0');
       signal sig_dcntl_sfifo_out       : std_logic_vector(DCNTL_SFIFO_WIDTH-1 downto 0) := (others => '0');
    
    
       begin
    
       
         sig_wsc2stat_status       <=  sig_coelsc_okay_reg    &   
                                       sig_coelsc_slverr_reg  &
                                       sig_coelsc_decerr_reg  & 
                                       sig_coelsc_interr_reg  &
                                       sig_tag2status;
         
         
         
         -----------------------------------------------------------------------------
         -- Data Controller Status FIFO and Logic
         
         
         -- Concatonate Input bits to build Dcntl fifo data word
         sig_dcntl_sfifo_in      <=  data2wsc_tag        &   -- bit 3 to tag Width+2
                                     data2wsc_calc_error &   -- bit 2
                                     data2wsc_last_error &   -- bit 1
                                     data2wsc_cmd_cmplt  ;   -- bit 0
         
                       
         -- Rip the DCntl fifo outputs back to constituant pieces              
         sig_data_tag_reg        <= sig_dcntl_sfifo_out((DCNTL_SFIFO_TAG_INDEX+STAT_REG_TAG_WIDTH)-1 downto
                                                        DCNTL_SFIFO_TAG_INDEX);
                                                        
         sig_data_err_reg        <= sig_dcntl_sfifo_out(DCNTL_SFIFO_CALC_ERR_INDEX) ;
         
         sig_data_last_err_reg   <= sig_dcntl_sfifo_out(DCNTL_SFIFO_TLAST_ERR_INDEX);
         
         sig_data_cmd_cmplt_reg  <= sig_dcntl_sfifo_out(DCNTL_SFIFO_CMD_CMPLT_INDEX);
        
         
         
         -- Data Control Valid/Ready assignments
         sig_dcntl_sfifo_wr_valid <= data2wsc_valid     ;
         
         sig_wsc2data_ready       <= sig_dcntl_sfifo_wr_ready; 
         
         
         
         -- read side ready assignment
         sig_dcntl_sfifo_rd_ready <= sig_push_coelsc_reg;
               
                                                  
         
         ------------------------------------------------------------
         -- Instance: I_DATA_CNTL_STATUS_FIFO 
         --
         -- Description:
         -- Instance for the Command Qualifier FIFO
         --
         ------------------------------------------------------------
          I_DATA_CNTL_STATUS_FIFO : entity axi_sg_v4_1_6.axi_sg_fifo
          generic map (
        
            C_DWIDTH             =>  DCNTL_SFIFO_WIDTH         , 
            C_DEPTH              =>  DCNTL_SFIFO_DEPTH         , 
            C_IS_ASYNC           =>  SYNC_FIFO_SELECT          , 
            C_PRIM_TYPE          =>  SRL_FIFO_TYPE             , 
            C_FAMILY             =>  C_FAMILY                    
           
            )
          port map (
            
            -- Write Clock and reset
            fifo_wr_reset        =>   mmap_reset               , 
            fifo_wr_clk          =>   primary_aclk             , 
            
            -- Write Side
            fifo_wr_tvalid       =>   sig_dcntl_sfifo_wr_valid , 
            fifo_wr_tready       =>   sig_dcntl_sfifo_wr_ready , 
            fifo_wr_tdata        =>   sig_dcntl_sfifo_in       , 
            fifo_wr_full         =>   sig_dcntl_sfifo_wr_full  , 
           
           
            -- Read Clock and reset (not used in Sync mode)
            fifo_async_rd_reset  =>   mmap_reset               ,   
            fifo_async_rd_clk    =>   primary_aclk             , 
            
            -- Read Side
            fifo_rd_tvalid       =>   sig_dcntl_sfifo_rd_valid , 
            fifo_rd_tready       =>   sig_dcntl_sfifo_rd_ready , 
            fifo_rd_tdata        =>   sig_dcntl_sfifo_out      , 
            fifo_rd_empty        =>   sig_dcntl_sfifo_rd_empty   
           
            );
        

         
         
         
            
            
       
       
         -------------------------------------------------------------
         -- Synchronous Process with Sync Reset
         --
         -- Label: STATUS_COELESC_REG
         --
         -- Process Description:
         --   Implement error status coelescing register. 
         -- Once a bit is set it will remain set until the overall 
         -- status is written to the Status FIFO. 
         -- Tag bits are just registered at each valid dbeat.
         --
         -------------------------------------------------------------
         STATUS_COELESC_REG : process (primary_aclk)
            begin
              if (primary_aclk'event and primary_aclk = '1') then
                 if (mmap_reset         = '1' or
                     sig_pop_coelsc_reg = '1') then
                   
                   sig_coelsc_tag_reg       <= (others => '0');
                   sig_coelsc_interr_reg    <= '0';
                   sig_coelsc_decerr_reg    <= '0';
                   sig_coelsc_slverr_reg    <= '0';
                   sig_coelsc_okay_reg      <= '1'; -- set back to default of "OKAY"
       
                   sig_coelsc_reg_full      <= '0';
                   sig_coelsc_reg_empty     <= '1';
       
       
                   
                 Elsif (sig_push_coelsc_reg = '1') Then
                 
                   sig_coelsc_tag_reg       <= sig_data_tag_reg;                             
                   sig_coelsc_interr_reg    <= sig_data_err_reg      or 
                                               sig_data_last_err_reg or 
                                               sig_coelsc_interr_reg;
                   sig_coelsc_decerr_reg    <= not(sig_data_err_reg) and 
                                               (sig_decerr           or 
                                                sig_coelsc_decerr_reg);
                   sig_coelsc_slverr_reg    <= not(sig_data_err_reg) and 
                                               (sig_slverr           or 
                                                sig_coelsc_slverr_reg);
                   sig_coelsc_okay_reg      <= not(sig_decerr            or 
                                                   sig_coelsc_decerr_reg or
                                                   sig_slverr            or 
                                                   sig_coelsc_slverr_reg or
                                                   sig_data_err_reg      or
                                                   sig_data_last_err_reg or    
                                                   sig_coelsc_interr_reg      
                                                   );
       
                   sig_coelsc_reg_full      <= sig_data_cmd_cmplt_reg;
                   sig_coelsc_reg_empty     <= not(sig_data_cmd_cmplt_reg);
                                                 
                   
                 else
                   
                   null;  -- hold current state
                   
                 end if; 
              end if;       
            end process STATUS_COELESC_REG; 
       
       
       
       
       
       
       end generate GEN_OMIT_INDET_BTT;
    
    
    
    
    
    
    
    
    
    
    
   

    
    
    
    
    
    
    
    
    ------------------------------------------------------------
    -- If Generate
    --
    -- Label: GEN_ENABLE_INDET_BTT
    --
    -- If Generate Description:
    --  Implements the logic needed when Indeterminate BTT is
    -- enabled in the S2MM function. Primary difference is the
    -- addition to the reported status of the End of Packet 
    -- marker (EOP) and the received byte count for the parent 
    -- command.
    --
    ------------------------------------------------------------
    GEN_ENABLE_INDET_BTT : if (C_ENABLE_INDET_BTT = 1) generate
    
       -- Local Constants
       Constant SF_DCNTL_SFIFO_WIDTH         : integer := TAG_WIDTH +
                                                          C_SF_BYTES_RCVD_WIDTH + 3;
       Constant SF_SFIFO_LS_TAG_INDEX        : integer := 0;
       Constant SF_SFIFO_MS_TAG_INDEX        : integer := SF_SFIFO_LS_TAG_INDEX + (TAG_WIDTH-1);
       Constant SF_SFIFO_CALC_ERR_INDEX      : integer := SF_SFIFO_MS_TAG_INDEX+1;
       Constant SF_SFIFO_CMD_CMPLT_INDEX     : integer := SF_SFIFO_CALC_ERR_INDEX+1;
       Constant SF_SFIFO_LS_BYTES_RCVD_INDEX : integer := SF_SFIFO_CMD_CMPLT_INDEX+1;
       Constant SF_SFIFO_MS_BYTES_RCVD_INDEX : integer := SF_SFIFO_LS_BYTES_RCVD_INDEX+
                                                          (C_SF_BYTES_RCVD_WIDTH-1);
       Constant SF_SFIFO_EOP_INDEX           : integer := SF_SFIFO_MS_BYTES_RCVD_INDEX+1;
       Constant BYTES_RCVD_FIELD_WIDTH       : integer := 23;
       
       
       -- local signals
       signal sig_dcntl_sfifo_in        : std_logic_vector(SF_DCNTL_SFIFO_WIDTH-1 downto 0) := (others => '0');
       signal sig_dcntl_sfifo_out       : std_logic_vector(SF_DCNTL_SFIFO_WIDTH-1 downto 0) := (others => '0');
       signal sig_data_bytes_rcvd       : std_logic_vector(C_SF_BYTES_RCVD_WIDTH-1 downto 0) := (others => '0');
       signal sig_data_eop              : std_logic := '0';
       signal sig_coelsc_bytes_rcvd     : std_logic_vector(C_SF_BYTES_RCVD_WIDTH-1 downto 0) := (others => '0');
       signal sig_coelsc_eop            : std_logic := '0';
       signal sig_coelsc_bytes_rcvd_pad : std_logic_vector(BYTES_RCVD_FIELD_WIDTH-1 downto 0) := (others => '0');
 
    
       begin
    
       
                                                                
         sig_wsc2stat_status     <=  sig_coelsc_eop            &
                                     sig_coelsc_bytes_rcvd_pad &
                                     sig_coelsc_okay_reg       &   
                                     sig_coelsc_slverr_reg     &
                                     sig_coelsc_decerr_reg     & 
                                     sig_coelsc_interr_reg     &
                                     sig_tag2status;
         
            
         
         
         -----------------------------------------------------------------------------
         -- Data Controller Status FIFO and Logic


     
         
         -- Concatonate Input bits to build Dcntl fifo input data word
         sig_dcntl_sfifo_in      <=  data2wsc_eop        & -- ms bit
                                     data2wsc_bytes_rcvd & -- bit 7 to C_SF_BYTES_RCVD_WIDTH+7  
                                     data2wsc_cmd_cmplt  & -- bit 6 
                                     data2wsc_calc_error & -- bit 4  
                                     data2wsc_tag;         -- bits 0 to 3
                       
                       
         -- Rip the DCntl fifo outputs back to constituant pieces              
         
         sig_data_eop            <= sig_dcntl_sfifo_out(SF_SFIFO_EOP_INDEX);
         
         sig_data_bytes_rcvd     <= sig_dcntl_sfifo_out(SF_SFIFO_MS_BYTES_RCVD_INDEX downto 
                                                        SF_SFIFO_LS_BYTES_RCVD_INDEX);
         
         sig_data_cmd_cmplt_reg  <= sig_dcntl_sfifo_out(SF_SFIFO_CMD_CMPLT_INDEX);
          
         sig_data_err_reg        <= sig_dcntl_sfifo_out(SF_SFIFO_CALC_ERR_INDEX);
         
         sig_data_tag_reg        <= sig_dcntl_sfifo_out(SF_SFIFO_MS_TAG_INDEX downto 
                                                        SF_SFIFO_LS_TAG_INDEX) ;
         
         
         
         
         -- Data Control Valid/Ready assignments
         sig_dcntl_sfifo_wr_valid <= data2wsc_valid ;
         
         sig_wsc2data_ready       <= sig_dcntl_sfifo_wr_ready; 
         
         
         
         -- read side ready assignment
         sig_dcntl_sfifo_rd_ready <= sig_push_coelsc_reg;
               
                                                  
         
         ------------------------------------------------------------
         -- Instance: I_SF_DATA_CNTL_STATUS_FIFO 
         --
         -- Description:
         -- Instance for the Command Qualifier FIFO when Store and
         -- Forward is included.
         --
         ------------------------------------------------------------
          I_SF_DATA_CNTL_STATUS_FIFO : entity axi_sg_v4_1_6.axi_sg_fifo
          generic map (
        
            C_DWIDTH             =>  SF_DCNTL_SFIFO_WIDTH      ,  
            C_DEPTH              =>  DCNTL_SFIFO_DEPTH         ,  
            C_IS_ASYNC           =>  SYNC_FIFO_SELECT          ,  
            C_PRIM_TYPE          =>  SRL_FIFO_TYPE             ,  
            C_FAMILY             =>  C_FAMILY                     
           
            )
          port map (
            
            -- Write Clock and reset
            fifo_wr_reset        =>   mmap_reset               ,  
            fifo_wr_clk          =>   primary_aclk             ,  
            
            -- Write Side
            fifo_wr_tvalid       =>   sig_dcntl_sfifo_wr_valid ,  
            fifo_wr_tready       =>   sig_dcntl_sfifo_wr_ready ,  
            fifo_wr_tdata        =>   sig_dcntl_sfifo_in       ,  
            fifo_wr_full         =>   sig_dcntl_sfifo_wr_full  ,  
           
            -- Read Clock and reset (not used in Sync mode)
            fifo_async_rd_reset  =>   mmap_reset               ,     
            fifo_async_rd_clk    =>   primary_aclk             ,   
            
            -- Read Side
            fifo_rd_tvalid       =>   sig_dcntl_sfifo_rd_valid ,  
            fifo_rd_tready       =>   sig_dcntl_sfifo_rd_ready ,  
            fifo_rd_tdata        =>   sig_dcntl_sfifo_out      ,  
            fifo_rd_empty        =>   sig_dcntl_sfifo_rd_empty    
           
            );
        

         
         
         
            
            
       
       
         -------------------------------------------------------------
         -- Synchronous Process with Sync Reset
         --
         -- Label: SF_STATUS_COELESC_REG
         --
         -- Process Description:
         --   Implement error status coelescing register. 
         -- Once a bit is set it will remain set until the overall 
         -- status is written to the Status FIFO. 
         -- Tag bits are just registered at each valid dbeat.
         --
         -------------------------------------------------------------
         SF_STATUS_COELESC_REG : process (primary_aclk)
            begin
              if (primary_aclk'event and primary_aclk = '1') then
                 if (mmap_reset         = '1' or
                     sig_pop_coelsc_reg = '1') then
                   
                   sig_coelsc_tag_reg       <= (others => '0');
                   sig_coelsc_interr_reg    <= '0';
                   sig_coelsc_decerr_reg    <= '0';
                   sig_coelsc_slverr_reg    <= '0';
                   sig_coelsc_okay_reg      <= '1'; -- set back to default of "OKAY"
                   sig_coelsc_bytes_rcvd    <= (others => '0');
                   sig_coelsc_eop           <= '0';
      
       
                   sig_coelsc_reg_full      <= '0';
                   sig_coelsc_reg_empty     <= '1';
       
       
                   
                 Elsif (sig_push_coelsc_reg = '1') Then
                 
                   sig_coelsc_tag_reg       <= sig_data_tag_reg;                             
                   sig_coelsc_interr_reg    <= sig_data_err_reg      or 
                                               sig_coelsc_interr_reg;
                   sig_coelsc_decerr_reg    <= not(sig_data_err_reg) and 
                                               (sig_decerr           or 
                                                sig_coelsc_decerr_reg);
                   sig_coelsc_slverr_reg    <= not(sig_data_err_reg) and 
                                               (sig_slverr           or 
                                                sig_coelsc_slverr_reg);
                   sig_coelsc_okay_reg      <= not(sig_decerr            or 
                                                   sig_coelsc_decerr_reg or
                                                   sig_slverr            or 
                                                   sig_coelsc_slverr_reg or
                                                   sig_data_err_reg      or
                                                   sig_coelsc_interr_reg      
                                                   );
                   sig_coelsc_bytes_rcvd    <= sig_data_bytes_rcvd;
                   sig_coelsc_eop           <= sig_data_eop;
      
       
                   sig_coelsc_reg_full      <= sig_data_cmd_cmplt_reg;
                   sig_coelsc_reg_empty     <= not(sig_data_cmd_cmplt_reg);
                                                 
                   
                 else
                   
                   null;  -- hold current state
                   
                 end if; 
              end if;       
            end process SF_STATUS_COELESC_REG; 
       
       
       
       
       
       
         
         ------------------------------------------------------------
         -- If Generate
         --
         -- Label: SF_GEN_PAD_BYTES_RCVD
         --
         -- If Generate Description:
         --  Pad the bytes received value with zeros to fill in the 
         -- status field width.
         --
         --
         ------------------------------------------------------------
         SF_GEN_PAD_BYTES_RCVD : if (C_SF_BYTES_RCVD_WIDTH < BYTES_RCVD_FIELD_WIDTH) generate
         
           begin
            
             sig_coelsc_bytes_rcvd_pad(BYTES_RCVD_FIELD_WIDTH-1 downto 
                                       C_SF_BYTES_RCVD_WIDTH) <= (others => '0');
            
            
             sig_coelsc_bytes_rcvd_pad(C_SF_BYTES_RCVD_WIDTH-1 downto 0) <=  sig_coelsc_bytes_rcvd;
            
           
           end generate SF_GEN_PAD_BYTES_RCVD;
       
       
       
         
         
         ------------------------------------------------------------
         -- If Generate
         --
         -- Label: SF_GEN_NO_PAD_BYTES_RCVD
         --
         -- If Generate Description:
         --  No padding required for the bytes received value.
         --
         --
         ------------------------------------------------------------
         SF_GEN_NO_PAD_BYTES_RCVD : if (C_SF_BYTES_RCVD_WIDTH = BYTES_RCVD_FIELD_WIDTH) generate
         
           begin
            
               sig_coelsc_bytes_rcvd_pad <= sig_coelsc_bytes_rcvd; -- no pad required
            
           
           end generate SF_GEN_NO_PAD_BYTES_RCVD;
       
       
          
          
       
       end generate GEN_ENABLE_INDET_BTT;
    
    
    
    
    
    
    
    
    
    
    
   -------  Soft Shutdown Logic ------------------------------- 
    
    
  
  
   -- Address Posted Counter Logic ---------------------t-----------------
   -- Supports soft shutdown by tracking when all commited Write 
   -- transfers to the AXI Bus have had corresponding Write Status 
   -- Reponses Received.   

    
    sig_addr_posted           <= addr2wsc_addr_posted ;
    
    sig_incr_addr_posted_cntr <= sig_addr_posted      ;
    
    sig_decr_addr_posted_cntr <= sig_s2mm_bready  and
                                 s2mm_bvalid          ;
        
    sig_addr_posted_cntr_eq_0 <= '1'
      when (sig_addr_posted_cntr = ADDR_POSTED_ZERO)
      Else '0';

    sig_addr_posted_cntr_eq_1 <= '1'
      when (sig_addr_posted_cntr = ADDR_POSTED_ONE)
      Else '0';

    
    sig_addr_posted_cntr_max <= '1'
      when (sig_addr_posted_cntr = ADDR_POSTED_MAX)
      Else '0';
    
    
    
    
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_ADDR_POSTED_FIFO_CNTR
    --
    -- Process Description:
    --    This process implements a counter for the tracking  
    -- if an Address has been posted on the AXI address channel.
    -- The counter is used to track flushing operations where all
    -- transfers committed on the AXI Address Channel have to
    --  be completed before a halt can occur.
    -------------------------------------------------------------
    IMP_ADDR_POSTED_FIFO_CNTR : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              
              sig_addr_posted_cntr <= ADDR_POSTED_ZERO;
              
            elsif (sig_incr_addr_posted_cntr = '1' and
                   sig_decr_addr_posted_cntr = '0' and
                   sig_addr_posted_cntr_max  = '0') then
              
              sig_addr_posted_cntr <= sig_addr_posted_cntr + ADDR_POSTED_ONE ;
              
            elsif (sig_incr_addr_posted_cntr = '0' and
                   sig_decr_addr_posted_cntr = '1' and
                   sig_addr_posted_cntr_eq_0 = '0') then
              
              sig_addr_posted_cntr <= sig_addr_posted_cntr - ADDR_POSTED_ONE ;
              
            else
              null;  -- don't change state
            end if; 
         end if;       
       end process IMP_ADDR_POSTED_FIFO_CNTR; 
 
 
    
    
    wsc2rst_stop_cmplt <= sig_all_cmds_done;
    
    
    
    
    sig_no_posted_cmds <= (sig_addr_posted_cntr_eq_0 and
                            not(addr2wsc_calc_error)) or
                           (sig_addr_posted_cntr_eq_1 and
                            addr2wsc_calc_error);
    
  
                         
    sig_all_cmds_done <=  sig_no_posted_cmds and
                          sig_halt_reg_dly3;   
                               
                               
                               
    
    
  
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_HALT_REQ_REG
    --
    -- Process Description:
    --   Implements the flop for capturing the Halt request from 
    -- the Reset module.
    --
    -------------------------------------------------------------
    IMP_HALT_REQ_REG : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              
              sig_halt_reg      <= '0';
            
            elsif (rst2wsc_stop_request = '1') then
              
              sig_halt_reg <= '1';
            
            else
              null;  -- Hold current State
            end if; 
         end if;       
       end process IMP_HALT_REQ_REG; 
  
  
   
   
    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_HALT_REQ_REG_DLY
    --
    -- Process Description:
    --   Implements the flops for delaying the halt request by 3
    -- clocks to allow the Address Controller to halt before the
    -- Data Contoller can safely indicate it has exhausted all
    -- transfers committed to the AXI Address Channel by the Address
    -- Controller.
    --
    -------------------------------------------------------------
    IMP_HALT_REQ_REG_DLY : process (primary_aclk)
       begin
         if (primary_aclk'event and primary_aclk = '1') then
            if (mmap_reset = '1') then
              
              sig_halt_reg_dly1 <= '0';
              sig_halt_reg_dly2 <= '0';
              sig_halt_reg_dly3 <= '0';
            
            else
              
              sig_halt_reg_dly1 <= sig_halt_reg;
              sig_halt_reg_dly2 <= sig_halt_reg_dly1;
              sig_halt_reg_dly3 <= sig_halt_reg_dly2;
            
            end if; 
         end if;       
       end process IMP_HALT_REQ_REG_DLY; 
  
  
   
      
          
          
  
  end implementation;


-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:        axi_sg_skid_buf.vhd
--
-- Description:     
--  Implements the AXi Skid Buffer in the Option 2 (Registerd outputs) mode.                
--                  
--                  
--                  
--                  
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;



-------------------------------------------------------------------------------

entity axi_sg_skid_buf is
  generic (
    C_WDATA_WIDTH : INTEGER := 32  
       --  Width of the Stream Data bus (in bits) 
               
    );
  port (
    
    -- Clock and Reset Inputs ---------------------------------------------
    aclk         : In  std_logic ;                                       --
    arst         : In  std_logic ;                                       --
    -----------------------------------------------------------------------
    
    
    -- Shutdown control (assert for 1 clk pulse) --------------------------
                                                                         --
    skid_stop    : In std_logic  ;                                       --
    -----------------------------------------------------------------------
    
    
    -- Slave Side (Stream Data Input) -------------------------------------
    s_valid      : In  std_logic ;                                       --
    s_ready      : Out std_logic ;                                       --
    s_data       : In  std_logic_vector(C_WDATA_WIDTH-1 downto 0);       --
    s_strb       : In  std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0);   --
    s_last       : In  std_logic ;                                       --
    -----------------------------------------------------------------------
    

    -- Master Side (Stream Data Output ------------------------------------
    m_valid      : Out std_logic ;                                       --
    m_ready      : In  std_logic ;                                       --
    m_data       : Out std_logic_vector(C_WDATA_WIDTH-1 downto 0);       --
    m_strb       : Out std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0);   --
    m_last       : Out std_logic                                         --
    -----------------------------------------------------------------------
    );

end entity axi_sg_skid_buf;


architecture implementation of axi_sg_skid_buf is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";



-- Signals decalrations -------------------------

  Signal sig_reset_reg         : std_logic := '0';
  signal sig_spcl_s_ready_set  : std_logic := '0';
  signal sig_data_skid_reg     : std_logic_vector(C_WDATA_WIDTH-1 downto 0) := (others => '0');
  signal sig_strb_skid_reg     : std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0) := (others => '0');
  signal sig_last_skid_reg     : std_logic := '0';
  signal sig_skid_reg_en       : std_logic := '0';
  signal sig_data_skid_mux_out : std_logic_vector(C_WDATA_WIDTH-1 downto 0) := (others => '0');
  signal sig_strb_skid_mux_out : std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0) := (others => '0');
  signal sig_last_skid_mux_out : std_logic := '0';
  signal sig_skid_mux_sel      : std_logic := '0';
  signal sig_data_reg_out      : std_logic_vector(C_WDATA_WIDTH-1 downto 0) := (others => '0');
  signal sig_strb_reg_out      : std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0) := (others => '0');
  signal sig_last_reg_out      : std_logic := '0';
  signal sig_data_reg_out_en   : std_logic := '0';
  signal sig_m_valid_out       : std_logic := '0';
  signal sig_m_valid_dup       : std_logic := '0';
  signal sig_m_valid_comb      : std_logic := '0';
  signal sig_s_ready_out       : std_logic := '0';
  signal sig_s_ready_dup       : std_logic := '0';
  signal sig_s_ready_comb      : std_logic := '0';
  signal sig_stop_request      : std_logic := '0';
  signal sig_stopped           : std_logic := '0';
  signal sig_sready_stop       : std_logic := '0';
  signal sig_sready_early_stop : std_logic := '0';
  signal sig_sready_stop_set   : std_logic := '0';
  signal sig_sready_stop_reg   : std_logic := '0';
  signal sig_mvalid_stop_reg   : std_logic := '0';
  signal sig_mvalid_stop       : std_logic := '0';
  signal sig_mvalid_early_stop : std_logic := '0';
  signal sig_mvalid_stop_set   : std_logic := '0';
  signal sig_slast_with_stop   : std_logic := '0';
  signal sig_sstrb_stop_mask   : std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0) := (others => '0');
  signal sig_sstrb_with_stop   : std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0) := (others => '0');
  
  

 
-- Register duplication attribute assignments to control fanout
-- on handshake output signals  
  
  Attribute KEEP : string; -- declaration
  Attribute EQUIVALENT_REGISTER_REMOVAL : string; -- declaration
  
  Attribute KEEP of sig_m_valid_out : signal is "TRUE"; -- definition
  Attribute KEEP of sig_m_valid_dup : signal is "TRUE"; -- definition
  Attribute KEEP of sig_s_ready_out : signal is "TRUE"; -- definition
  Attribute KEEP of sig_s_ready_dup : signal is "TRUE"; -- definition
  
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_m_valid_out : signal is "no"; 
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_m_valid_dup : signal is "no"; 
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_s_ready_out : signal is "no"; 
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_s_ready_dup : signal is "no"; 
  
  
  

begin --(architecture implementation)

   m_valid <= sig_m_valid_out;         
   s_ready <= sig_s_ready_out; 
    
   m_strb  <= sig_strb_reg_out;
   m_last  <= sig_last_reg_out;                
   m_data  <= sig_data_reg_out;    
            
  
  
  
   -- Special shutdown logic version od Slast.
   -- A halt request forces a tlast through the skig buffer
   sig_slast_with_stop <= s_last or sig_stop_request;
   sig_sstrb_with_stop <= s_strb or sig_sstrb_stop_mask;
  
  
   -- Assign the special s_ready FLOP set signal
   sig_spcl_s_ready_set <= sig_reset_reg;
  
            
   -- Generate the ouput register load enable control
   sig_data_reg_out_en <= m_ready or not(sig_m_valid_dup);

   -- Generate the skid input register load enable control
   sig_skid_reg_en     <= sig_s_ready_dup;
  
   -- Generate the skid mux select control
   sig_skid_mux_sel    <= not(sig_s_ready_dup);
   
   
   -- Skid Mux  
   sig_data_skid_mux_out <=  sig_data_skid_reg
     When (sig_skid_mux_sel = '1')
     Else  s_data;
  
   sig_strb_skid_mux_out <=  sig_strb_skid_reg
     When (sig_skid_mux_sel = '1')
     Else  sig_sstrb_with_stop;
  
   sig_last_skid_mux_out <=  sig_last_skid_reg
     When (sig_skid_mux_sel = '1')
     Else  sig_slast_with_stop;
  
   
   -- m_valid combinational logic        
   sig_m_valid_comb <= s_valid or
                      (sig_m_valid_dup and
                      (not(sig_s_ready_dup) or
                       not(m_ready)));
   
   
   
   -- s_ready combinational logic        
   sig_s_ready_comb <= m_ready or
                      (sig_s_ready_dup and
                      (not(sig_m_valid_dup) or
                       not(s_valid)));
   
   
   
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: REG_THE_RST
   --
   -- Process Description:
   -- Register input reset
   --
   -------------------------------------------------------------
   REG_THE_RST : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           
            sig_reset_reg <= ARST;
           
        end if;       
      end process REG_THE_RST; 
   
   
   
   
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: S_READY_FLOP
   --
   -- Process Description:
   -- Registers s_ready handshake signals per Skid Buffer 
   -- Option 2 scheme
   --
   -------------------------------------------------------------
   S_READY_FLOP : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST                  = '1' or
               sig_sready_stop       = '1' or
               sig_sready_early_stop = '1') then  -- Special stop condition

             sig_s_ready_out  <= '0';
             sig_s_ready_dup  <= '0';
            
           Elsif (sig_spcl_s_ready_set = '1') Then
           
             sig_s_ready_out  <= '1';
             sig_s_ready_dup  <= '1';
           
           else

             sig_s_ready_out  <= sig_s_ready_comb;
             sig_s_ready_dup  <= sig_s_ready_comb;
            
           end if; 
        end if;       
      end process S_READY_FLOP; 
   
   
   
   
            
            
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: M_VALID_FLOP
   --
   -- Process Description:
   -- Registers m_valid handshake signals per Skid Buffer 
   -- Option 2 scheme
   --
   -------------------------------------------------------------
   M_VALID_FLOP : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST                  = '1' or
               sig_spcl_s_ready_set  = '1' or    -- Fix from AXI DMA
               sig_mvalid_stop       = '1' or
               sig_mvalid_stop_set   = '1') then -- Special stop condition

             sig_m_valid_out  <= '0';
             sig_m_valid_dup  <= '0';
            
           else

             sig_m_valid_out  <= sig_m_valid_comb;
             sig_m_valid_dup  <= sig_m_valid_comb;
            
           end if; 
        end if;       
      end process M_VALID_FLOP; 
   
   
   
   
            
            
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: SKID_REG
   --
   -- Process Description:
   -- This process implements the output registers for the 
   -- Skid Buffer Data signals
   --
   -------------------------------------------------------------
   SKID_REG : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST = '1') then
             
             sig_data_skid_reg <= (others => '0');
             sig_strb_skid_reg <= (others => '0');
             sig_last_skid_reg <= '0';
             
           elsif (sig_skid_reg_en = '1') then
             
             sig_data_skid_reg <= s_data;
             sig_strb_skid_reg <= sig_sstrb_with_stop;
             sig_last_skid_reg <= sig_slast_with_stop;
             
           else
             null;  -- hold current state
           end if; 
        end if;       
      end process SKID_REG; 
            
            
   
            
            
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: OUTPUT_REG
   --
   -- Process Description:
   -- This process implements the output registers for the 
   -- Skid Buffer Data signals
   --
   -------------------------------------------------------------
   OUTPUT_REG : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST                = '1' or
               sig_mvalid_stop_reg = '1') then
             
             sig_data_reg_out <= (others => '0');
             sig_strb_reg_out <= (others => '0');
             sig_last_reg_out <= '0';
             
           elsif (sig_data_reg_out_en = '1') then
             
             sig_data_reg_out <= sig_data_skid_mux_out;
             sig_strb_reg_out <= sig_strb_skid_mux_out;
             sig_last_reg_out <= sig_last_skid_mux_out;
             
           else
             null;  -- hold current state
           end if; 
        end if;       
      end process OUTPUT_REG; 
            
            
  
  
  
  
  
  
  
  
  
  
  
  
   -------- Special Stop Logic --------------------------------------
   
   
   sig_sready_stop        <= sig_sready_stop_reg; 
  
  
   sig_sready_early_stop  <= skid_stop; -- deassert S_READY immediately
   
  
   sig_sready_stop_set    <= sig_sready_early_stop;
   
                                     
   sig_mvalid_stop        <=  sig_mvalid_stop_reg;
                                     
                                     
   sig_mvalid_early_stop  <= sig_m_valid_dup and
                            m_ready and
                            skid_stop;
  
    
   sig_mvalid_stop_set    <=  sig_mvalid_early_stop or
                              (sig_stop_request and 
                               not(sig_m_valid_dup)) or
                              (sig_m_valid_dup and
                               m_ready         and
                               sig_stop_request);
   
   
   
                             
    
    
                                     
                                     
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: IMP_STOP_REQ_FLOP
   --
   -- Process Description:
   -- This process implements the Stop request flop. It is a 
   -- sample and hold register that can only be cleared by reset.
   --
   -------------------------------------------------------------
   IMP_STOP_REQ_FLOP : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST = '1') then
             
             sig_stop_request    <= '0';
             sig_sstrb_stop_mask <= (others => '0');
             
           elsif (skid_stop = '1') then
             
             sig_stop_request    <= '1';
             sig_sstrb_stop_mask <= (others => '1');
             
           else
             null;  -- hold current state
           end if; 
        end if;       
      end process IMP_STOP_REQ_FLOP; 
            
            
  
  
  
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: IMP_CLR_SREADY_FLOP
   --
   -- Process Description:
   -- This process implements the flag to clear the s_ready 
   -- flop at a stop condition.
   --
   -------------------------------------------------------------
   IMP_CLR_SREADY_FLOP : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST = '1') then
             
             sig_sready_stop_reg <= '0';
             
           elsif (sig_sready_stop_set  = '1') then
             
             sig_sready_stop_reg <= '1';
             
           else
             null;  -- hold current state
           end if; 
        end if;       
      end process IMP_CLR_SREADY_FLOP; 
            
            
  
  
  
   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: IMP_CLR_MVALID_FLOP
   --
   -- Process Description:
   -- This process implements the flag to clear the m_valid 
   -- flop at a stop condition.
   --
   -------------------------------------------------------------
   IMP_CLR_MVALID_FLOP : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST = '1') then
             
             sig_mvalid_stop_reg <= '0';
             
           elsif (sig_mvalid_stop_set  = '1') then
             
             sig_mvalid_stop_reg <= '1';
             
           else
             null;  -- hold current state
           end if; 
        end if;       
      end process IMP_CLR_MVALID_FLOP; 
            
            


end implementation;


-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_sg_mm2s_basic_wrap.vhd
  --
  -- Description:     
  --    This file implements the DataMover MM2S Basic Wrapper.                 
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  
  
  -- axi_sg Library Modules
  library axi_sg_v4_1_6; 
  use axi_sg_v4_1_6.axi_sg_reset;  
  use axi_sg_v4_1_6.axi_sg_cmd_status;
  use axi_sg_v4_1_6.axi_sg_scc;
  use axi_sg_v4_1_6.axi_sg_addr_cntl;
  use axi_sg_v4_1_6.axi_sg_rddata_cntl;
  use axi_sg_v4_1_6.axi_sg_rd_status_cntl;
  use axi_sg_v4_1_6.axi_sg_skid_buf;
  
  
  -------------------------------------------------------------------------------
  
  entity axi_sg_mm2s_basic_wrap is
    generic (
      
      C_INCLUDE_MM2S     : Integer range 0 to  2 :=  2;
         -- Specifies the type of MM2S function to include
         -- 0 = Omit MM2S functionality
         -- 1 = Full MM2S Functionality
         -- 2 = Basic MM2S functionality
         
      C_MM2S_ARID        : Integer range 0 to  255 :=  8;
         -- Specifies the constant value to output on 
         -- the ARID output port
         
      C_MM2S_ID_WIDTH    : Integer range 1 to   8 :=  4;
         -- Specifies the width of the MM2S ID port 
         
      C_MM2S_ADDR_WIDTH  : Integer range 32 to  64 :=  32;
         -- Specifies the width of the MMap Read Address Channel 
         -- Address bus
         
      C_MM2S_MDATA_WIDTH : Integer range 32 to  64 :=  32;
         -- Specifies the width of the MMap Read Data Channel
         -- data bus
      
      C_MM2S_SDATA_WIDTH : Integer range 8 to 64 :=  32;
         -- Specifies the width of the MM2S Master Stream Data 
         -- Channel data bus
      
      C_INCLUDE_MM2S_STSFIFO    : Integer range 0 to  1 :=  1;
         -- Specifies if a Status FIFO is to be implemented
         -- 0 = Omit MM2S Status FIFO
         -- 1 = Include MM2S Status FIFO
         
      C_MM2S_STSCMD_FIFO_DEPTH    : Integer range 1 to 16 :=  1;
         -- Specifies the depth of the MM2S Command FIFO and the 
         -- optional Status FIFO
         -- Valid values are 1,4,8,16
         
      C_MM2S_STSCMD_IS_ASYNC    : Integer range 0 to  1 :=  0;
         -- Specifies if the Status and Command interfaces need to
         -- be asynchronous to the primary data path clocking
         -- 0 = Use same clocking as data path
         -- 1 = Use special Status/Command clock for the interfaces
         
      C_INCLUDE_MM2S_DRE : Integer range 0 to  1 :=  0;
         -- Specifies if DRE is to be included in the MM2S function 
         -- 0 = Omit DRE
         -- 1 = Include DRE
      
      C_MM2S_BURST_SIZE  : Integer range 16 to  64 :=  16;
         -- Specifies the max number of databeats to use for MMap
         -- burst transfers by the MM2S function 
      
      C_MM2S_BTT_USED            : Integer range 8 to  23 :=  16;
        -- Specifies the number of bits used from the BTT field
        -- of the input Command Word of the MM2S Command Interface 
    
      C_MM2S_ADDR_PIPE_DEPTH     : Integer range 1 to 30 := 1;
        -- This parameter specifies the depth of the MM2S internal 
        -- child command queues in the Read Address Controller and 
        -- the Read Data Controller. Increasing this value will 
        -- allow more Read Addresses to be issued to the AXI4 Read 
        -- Address Channel before receipt of the associated read 
        -- data on the Read Data Channel.
      C_ENABLE_MULTI_CHANNEL    : Integer range 0 to 1 := 1;
      C_ENABLE_EXTRA_FIELD      : integer range 0 to 1 := 0;
      C_TAG_WIDTH        : Integer range 1 to 8 :=  4 ;
         -- Width of the TAG field
         
      C_FAMILY           : String := "virtex7"
         -- Specifies the target FPGA family type
      
      );
    port (
      
      
      -- MM2S Primary Clock and Reset inputs -----------------------
      mm2s_aclk          : in  std_logic;                         --
         -- Primary synchronization clock for the Master side     --
         -- interface and internal logic. It is also used         --
         -- for the User interface synchronization when           --
         -- C_STSCMD_IS_ASYNC = 0.                                --
                                                                  --
      -- MM2S Primary Reset input                                 --
      mm2s_aresetn       : in  std_logic;                         --
         -- Reset used for the internal master logic              --
      --------------------------------------------------------------
      sg_ctl             : in  std_logic_vector (7 downto 0);     
 
 
     
      -- MM2S Halt request input control ---------------------------
      mm2s_halt          : in  std_logic;                         --
         -- Active high soft shutdown request                     --
                                                                  --
      -- MM2S Halt Complete status flag                           --
      mm2s_halt_cmplt    : Out  std_logic;                        --
         -- Active high soft shutdown complete status             --
      --------------------------------------------------------------
      
      
      
      -- Error discrete output -------------------------------------
      mm2s_err           : Out std_logic;                         --
         -- Composite Error indication                            --
      --------------------------------------------------------------
     
     
     
      -- Optional MM2S Command and Status Clock and Reset ----------
      -- These are used when C_MM2S_STSCMD_IS_ASYNC = 1           --
      mm2s_cmdsts_awclk   : in  std_logic;                        --
      -- Secondary Clock input for async CMD/Status interface     --
                                                                  --
      mm2s_cmdsts_aresetn : in  std_logic;                        --
        -- Secondary Reset input for async CMD/Status interface   --
      --------------------------------------------------------------
      
      
      
      -- User Command Interface Ports (AXI Stream) -------------------------------------------------
      mm2s_cmd_wvalid     : in  std_logic;                                                        --
      mm2s_cmd_wready     : out std_logic;                                                        --
      mm2s_cmd_wdata      : in  std_logic_vector((C_TAG_WIDTH+(1+C_ENABLE_MULTI_CHANNEL)*C_MM2S_ADDR_WIDTH+36)-1 downto 0);  --
      ----------------------------------------------------------------------------------------------
     
      
      -- User Status Interface Ports (AXI Stream) -----------------
      mm2s_sts_wvalid     : out std_logic;                       --
      mm2s_sts_wready     : in  std_logic;                       --
      mm2s_sts_wdata      : out std_logic_vector(7 downto 0);    --
      mm2s_sts_wstrb      : out std_logic_vector(0 downto 0);    --
      mm2s_sts_wlast      : out std_logic;                       --
      -------------------------------------------------------------
      
      
      -- Address Posting contols ----------------------------------
      mm2s_allow_addr_req  : in  std_logic;                      --
      mm2s_addr_req_posted : out std_logic;                      --
      mm2s_rd_xfer_cmplt   : out std_logic;                      --
      -------------------------------------------------------------
      
                                                              
                                                              
      
      -- MM2S AXI Address Channel I/O  --------------------------------------
      mm2s_arid     : out std_logic_vector(C_MM2S_ID_WIDTH-1 downto 0);    --
         -- AXI Address Channel ID output                                  --
                                                                           --
      mm2s_araddr   : out std_logic_vector(C_MM2S_ADDR_WIDTH-1 downto 0);  --
         -- AXI Address Channel Address output                             --
                                                                           --
      mm2s_arlen    : out std_logic_vector(7 downto 0);                    --
         -- AXI Address Channel LEN output                                 --
         -- Sized to support 256 data beat bursts                          --
                                                                           --
      mm2s_arsize   : out std_logic_vector(2 downto 0);                    --
         -- AXI Address Channel SIZE output                                --
                                                                           --
      mm2s_arburst  : out std_logic_vector(1 downto 0);                    --
         -- AXI Address Channel BURST output                               --
                                                                           --
      mm2s_arprot   : out std_logic_vector(2 downto 0);                    --
         -- AXI Address Channel PROT output                                --
                                                                           --
      mm2s_arcache  : out std_logic_vector(3 downto 0);                    --
         -- AXI Address Channel CACHE output                               --

      mm2s_aruser  : out std_logic_vector(3 downto 0);                    --
         -- AXI Address Channel USER output                               --
                                                                           --
      mm2s_arvalid  : out std_logic;                                       --
         -- AXI Address Channel VALID output                               --
                                                                           --
      mm2s_arready  : in  std_logic;                                       --
         -- AXI Address Channel READY input                                --
      -----------------------------------------------------------------------
      
      
        
      -- Currently unsupported AXI Address Channel output signals -------
        -- addr2axi_alock   : out std_logic_vector(2 downto 0);        --
        -- addr2axi_acache  : out std_logic_vector(4 downto 0);        --
        -- addr2axi_aqos    : out std_logic_vector(3 downto 0);        --
        -- addr2axi_aregion : out std_logic_vector(3 downto 0);        --
      -------------------------------------------------------------------
  
  
      
      -- MM2S AXI MMap Read Data Channel I/O  ------------------------------------------
      mm2s_rdata              : In  std_logic_vector(C_MM2S_MDATA_WIDTH-1 downto 0);  --
      mm2s_rresp              : In  std_logic_vector(1 downto 0);                     --
      mm2s_rlast              : In  std_logic;                                        --
      mm2s_rvalid             : In  std_logic;                                        --
      mm2s_rready             : Out std_logic;                                        --
      ----------------------------------------------------------------------------------
      
      
      
      -- MM2S AXI Master Stream Channel I/O  -----------------------------------------------
      mm2s_strm_wdata         : Out  std_logic_vector(C_MM2S_SDATA_WIDTH-1 downto 0);     --
      mm2s_strm_wstrb         : Out  std_logic_vector((C_MM2S_SDATA_WIDTH/8)-1 downto 0); --
      mm2s_strm_wlast         : Out  std_logic;                                           --
      mm2s_strm_wvalid        : Out  std_logic;                                           --
      mm2s_strm_wready        : In   std_logic;                                           --
      --------------------------------------------------------------------------------------
  
      
      -- Testing Support I/O --------------------------------------------
      mm2s_dbg_sel            : in  std_logic_vector( 3 downto 0);     --
      mm2s_dbg_data           : out std_logic_vector(31 downto 0)      --
      -------------------------------------------------------------------
      
      
      );
  
  end entity axi_sg_mm2s_basic_wrap;
  
  
  architecture implementation of axi_sg_mm2s_basic_wrap is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    
    
    -- Function Declarations   ----------------------------------------
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: func_calc_rdmux_sel_bits
    --
    -- Function Description:
    --  This function calculates the number of address bits needed for  
    -- the Read data mux select control. 
    --
    -------------------------------------------------------------------
    function func_calc_rdmux_sel_bits (mmap_dwidth_value : integer) return integer is
    
      Variable num_addr_bits_needed : Integer range 1 to 5 := 1;
    
    begin
    
      case mmap_dwidth_value is
        when 32 =>
          num_addr_bits_needed := 2;
       -- coverage off
        when 64 =>
          num_addr_bits_needed := 3;
        when 128 =>
          num_addr_bits_needed := 4;
        when others => -- 256 bits
          num_addr_bits_needed := 5;
       -- coverage on
      end case;
      
      Return (num_addr_bits_needed);
       
    end function func_calc_rdmux_sel_bits;
    
     
     
     
    -- Constant Declarations   ----------------------------------------
    
     Constant LOGIC_LOW               : std_logic := '0';
     Constant LOGIC_HIGH              : std_logic := '1';
     Constant INCLUDE_MM2S            : integer range  0 to   2 :=  2;
     Constant MM2S_ARID_VALUE         : integer range  0 to 255 := C_MM2S_ARID;
     Constant MM2S_ARID_WIDTH         : integer range  1 to  8  := C_MM2S_ID_WIDTH;
     Constant MM2S_ADDR_WIDTH         : integer range 32 to  64 := C_MM2S_ADDR_WIDTH;
     Constant MM2S_MDATA_WIDTH        : integer range 32 to 256 := C_MM2S_MDATA_WIDTH;
     Constant MM2S_SDATA_WIDTH        : integer range  8 to 256 := C_MM2S_SDATA_WIDTH;
     Constant MM2S_CMD_WIDTH          : integer                 := (C_TAG_WIDTH+C_MM2S_ADDR_WIDTH+32);
     Constant MM2S_STS_WIDTH          : integer                 := 8; -- always 8 for MM2S
     Constant INCLUDE_MM2S_STSFIFO    : integer range  0 to   1 :=  1;
     Constant MM2S_STSCMD_FIFO_DEPTH  : integer range  1 to  64 :=  1;
     Constant MM2S_STSCMD_IS_ASYNC    : integer range  0 to   1 :=  0;
     Constant INCLUDE_MM2S_DRE        : integer range  0 to   1 :=  0;
     Constant DRE_ALIGN_WIDTH         : integer range  1 to   3 :=  2;
     Constant MM2S_BURST_SIZE         : integer range 16 to 256 := 16;
     Constant RD_ADDR_CNTL_FIFO_DEPTH : integer range  1 to  30 := C_MM2S_ADDR_PIPE_DEPTH;  
     Constant RD_DATA_CNTL_FIFO_DEPTH : integer range  1 to  30 := C_MM2S_ADDR_PIPE_DEPTH; 
     Constant SEL_ADDR_WIDTH          : integer := func_calc_rdmux_sel_bits(MM2S_MDATA_WIDTH);
     Constant DRE_ALIGN_ZEROS         : std_logic_vector(DRE_ALIGN_WIDTH-1 downto 0) := (others => '0');
     -- obsoleted   Constant DISABLE_WAIT_FOR_DATA   : integer := 0;
     
        
    
    -- Signal Declarations  ------------------------------------------
    
     signal sig_cmd_stat_rst_user        : std_logic := '0';
     signal sig_cmd_stat_rst_int         : std_logic := '0';
     signal sig_mmap_rst                 : std_logic := '0';
     signal sig_stream_rst               : std_logic := '0';
     signal sig_mm2s_cmd_wdata           : std_logic_vector(MM2S_CMD_WIDTH-1 downto 0);        
     signal sig_mm2s_cache_data          : std_logic_vector(7 downto 0);        
     signal sig_cmd2mstr_command         : std_logic_vector(MM2S_CMD_WIDTH-1 downto 0) := (others => '0');        
     signal sig_cmd2mstr_cmd_valid       : std_logic := '0';                                             
     signal sig_mst2cmd_cmd_ready        : std_logic := '0';                                             
     signal sig_mstr2addr_addr           : std_logic_vector(MM2S_ADDR_WIDTH-1 downto 0) := (others => '0');             
     signal sig_mstr2addr_len            : std_logic_vector(7 downto 0) := (others => '0');                          
     signal sig_mstr2addr_size           : std_logic_vector(2 downto 0) := (others => '0');                          
     signal sig_mstr2addr_burst          : std_logic_vector(1 downto 0) := (others => '0'); 
     signal sig_mstr2addr_cache          : std_logic_vector(3 downto 0) := (others => '0'); 
     signal sig_mstr2addr_user          : std_logic_vector(3 downto 0) := (others => '0'); 
     signal sig_mstr2addr_cmd_cmplt      : std_logic := '0';
     signal sig_mstr2addr_calc_error     : std_logic := '0';
     signal sig_mstr2addr_cmd_valid      : std_logic := '0';                                             
     signal sig_addr2mstr_cmd_ready      : std_logic := '0';                                              
     signal sig_mstr2data_saddr_lsb      : std_logic_vector(SEL_ADDR_WIDTH-1 downto 0) := (others => '0');
     signal sig_mstr2data_len            : std_logic_vector(7 downto 0) := (others => '0');
     signal sig_mstr2data_strt_strb      : std_logic_vector((MM2S_SDATA_WIDTH/8)-1 downto 0) := (others => '0');      
     signal sig_mstr2data_last_strb      : std_logic_vector((MM2S_SDATA_WIDTH/8)-1 downto 0) := (others => '0');
     signal sig_mstr2data_drr            : std_logic := '0';
     signal sig_mstr2data_eof            : std_logic := '0';
     signal sig_mstr2data_sequential     : std_logic := '0';
     signal sig_mstr2data_calc_error     : std_logic := '0';
     signal sig_mstr2data_cmd_cmplt      : std_logic := '0';
     signal sig_mstr2data_cmd_valid      : std_logic := '0';                                             
     signal sig_data2mstr_cmd_ready      : std_logic := '0';                                               
     signal sig_addr2data_addr_posted    : std_logic := '0';
     signal sig_data2all_dcntlr_halted   : std_logic := '0';
     signal sig_addr2rsc_calc_error      : std_logic := '0';
     signal sig_addr2rsc_cmd_fifo_empty  : std_logic := '0';
     signal sig_data2rsc_tag             : std_logic_vector(C_TAG_WIDTH-1 downto 0) := (others => '0');
     signal sig_data2rsc_calc_err        : std_logic := '0';                     
     signal sig_data2rsc_okay            : std_logic := '0';
     signal sig_data2rsc_decerr          : std_logic := '0';
     signal sig_data2rsc_slverr          : std_logic := '0';
     signal sig_data2rsc_cmd_cmplt       : std_logic := '0';
     signal sig_rsc2data_ready           : std_logic := '0';
     signal sig_data2rsc_valid           : std_logic := '0';
     signal sig_calc2dm_calc_err         : std_logic := '0';
     signal sig_data2skid_wvalid         : std_logic := '0';                                     
     signal sig_data2skid_wready         : std_logic := '0';                                     
     signal sig_data2skid_wdata          : std_logic_vector(MM2S_SDATA_WIDTH-1 downto 0) := (others => '0');     
     signal sig_data2skid_wstrb          : std_logic_vector((MM2S_SDATA_WIDTH/8)-1 downto 0) := (others => '0'); 
     signal sig_data2skid_wlast          : std_logic := '0';                                     
     signal sig_rsc2stat_status          : std_logic_vector(MM2S_STS_WIDTH-1 downto 0) := (others => '0');
     signal sig_stat2rsc_status_ready    : std_logic := '0';   
     signal sig_rsc2stat_status_valid    : std_logic := '0';  
     signal sig_rsc2mstr_halt_pipe       : std_logic := '0';  
     signal sig_mstr2data_tag            : std_logic_vector(C_TAG_WIDTH-1 downto 0) := (others => '0');
     signal sig_mstr2addr_tag            : std_logic_vector(C_TAG_WIDTH-1 downto 0) := (others => '0');
     signal sig_dbg_data_mux_out         : std_logic_vector(31 downto 0) := (others => '0');
     signal sig_dbg_data_0               : std_logic_vector(31 downto 0) := (others => '0');
     signal sig_dbg_data_1               : std_logic_vector(31 downto 0) := (others => '0');
     signal sig_rst2all_stop_request     : std_logic := '0';
     signal sig_data2rst_stop_cmplt      : std_logic := '0';
     signal sig_addr2rst_stop_cmplt      : std_logic := '0';
     signal sig_data2addr_stop_req       : std_logic := '0';
     signal sig_data2skid_halt           : std_logic := '0';

    signal sig_cache2mstr_command       : std_logic_vector (7 downto 0) := (others => '0');
     signal mm2s_arcache_int             : std_logic_vector (3 downto 0);

    
    
    
  begin --(architecture implementation)
 
 
  
    -- Debug Support ------------------------------------------
    
    mm2s_dbg_data  <= sig_dbg_data_mux_out;
    
    
    -- Note that only the mm2s_dbg_sel(0) is used at this time
    sig_dbg_data_mux_out <= sig_dbg_data_1
      When (mm2s_dbg_sel(0) = '1')
      else sig_dbg_data_0 ;
    
    
    sig_dbg_data_0              <=  X"BEEF2222"             ;    -- 32 bit Constant indicating MM2S Basic type
    
    sig_dbg_data_1(0)           <= sig_cmd_stat_rst_user    ;
    sig_dbg_data_1(1)           <= sig_cmd_stat_rst_int     ;
    sig_dbg_data_1(2)           <= sig_mmap_rst             ;
    sig_dbg_data_1(3)           <= sig_stream_rst           ;
    sig_dbg_data_1(4)           <= sig_cmd2mstr_cmd_valid   ;
    sig_dbg_data_1(5)           <= sig_mst2cmd_cmd_ready    ;
    sig_dbg_data_1(6)           <= sig_stat2rsc_status_ready;
    sig_dbg_data_1(7)           <= sig_rsc2stat_status_valid;
    sig_dbg_data_1(11 downto 8) <= sig_data2rsc_tag         ; -- Current TAG of active data transfer
                        

    sig_dbg_data_1(15 downto 12) <= sig_rsc2stat_status(3 downto 0); -- Internal status tag field
    sig_dbg_data_1(16)           <= sig_rsc2stat_status(4)         ; -- Internal error
    sig_dbg_data_1(17)           <= sig_rsc2stat_status(5)         ; -- Decode Error
    sig_dbg_data_1(18)           <= sig_rsc2stat_status(6)         ; -- Slave Error
    sig_dbg_data_1(19)           <= sig_rsc2stat_status(7)         ; -- OKAY
    sig_dbg_data_1(20)           <= sig_stat2rsc_status_ready      ; -- Status Ready Handshake
    sig_dbg_data_1(21)           <= sig_rsc2stat_status_valid      ; -- Status Valid Handshake
                        
    
    -- Spare bits in debug1
    sig_dbg_data_1(31 downto 22) <= (others => '0')                ; -- spare bits
                        
                        
                        
                        
                        
    
    GEN_CACHE : if (C_ENABLE_MULTI_CHANNEL = 0) generate

       begin
 
    -- Cache signal tie-off
    mm2s_arcache       <= "0011";  -- Per Interface-X guidelines for Masters
    mm2s_aruser        <= "0000";  -- Per Interface-X guidelines for Masters
    sig_mm2s_cache_data <= (others => '0'); --mm2s_cmd_wdata(103 downto 96);
     
       end generate GEN_CACHE;                 


    GEN_CACHE2 : if (C_ENABLE_MULTI_CHANNEL = 1) generate

       begin
 
    -- Cache signal tie-off
    mm2s_arcache       <= sg_ctl (3 downto 0);  -- SG Cache from register
    mm2s_aruser        <= sg_ctl (7 downto 4);  -- Per Interface-X guidelines for Masters
    sig_mm2s_cache_data <= mm2s_cmd_wdata(103 downto 96);
     
       end generate GEN_CACHE2;                 
    
    -- Cache signal tie-off
     
                        
                       
    -- Internal error output discrete ------------------------------
    mm2s_err                <=  sig_calc2dm_calc_err;
     
     
    -- Rip the used portion of the Command Interface Command Data
    -- and throw away the padding
    sig_mm2s_cmd_wdata <= mm2s_cmd_wdata(MM2S_CMD_WIDTH-1 downto 0);
     
     
          
   ------------------------------------------------------------
   -- Instance: I_RESET 
   --
   -- Description:
   --   Reset Block  
   --
   ------------------------------------------------------------
    I_RESET : entity axi_sg_v4_1_6.axi_sg_reset
    generic map (
  
      C_STSCMD_IS_ASYNC    =>  MM2S_STSCMD_IS_ASYNC       
  
      )
    port map (
  
      primary_aclk         =>  mm2s_aclk                , 
      primary_aresetn      =>  mm2s_aresetn             , 
      secondary_awclk      =>  mm2s_cmdsts_awclk        , 
      secondary_aresetn    =>  mm2s_cmdsts_aresetn      , 
      halt_req             =>  mm2s_halt                , 
      halt_cmplt           =>  mm2s_halt_cmplt          , 
      flush_stop_request   =>  sig_rst2all_stop_request , 
      data_cntlr_stopped   =>  sig_data2rst_stop_cmplt  , 
      addr_cntlr_stopped   =>  sig_addr2rst_stop_cmplt  , 
      aux1_stopped         =>  LOGIC_HIGH               , 
      aux2_stopped         =>  LOGIC_HIGH               , 
      cmd_stat_rst_user    =>  sig_cmd_stat_rst_user    , 
      cmd_stat_rst_int     =>  sig_cmd_stat_rst_int     , 
      mmap_rst             =>  sig_mmap_rst             , 
      stream_rst           =>  sig_stream_rst             
  
      );
  
  
  


        
   ------------------------------------------------------------
   -- Instance: I_CMD_STATUS 
   --
   -- Description:
   --   Command and Status Interface Block  
   --
   ------------------------------------------------------------
    I_CMD_STATUS : entity axi_sg_v4_1_6.axi_sg_cmd_status
    generic map (
  
      C_ADDR_WIDTH           =>  MM2S_ADDR_WIDTH           ,    
      C_INCLUDE_STSFIFO      =>  INCLUDE_MM2S_STSFIFO      ,    
      C_STSCMD_FIFO_DEPTH    =>  MM2S_STSCMD_FIFO_DEPTH    ,    
      C_STSCMD_IS_ASYNC      =>  MM2S_STSCMD_IS_ASYNC      ,    
      C_CMD_WIDTH            =>  MM2S_CMD_WIDTH            ,    
      C_STS_WIDTH            =>  MM2S_STS_WIDTH            ,    
      C_FAMILY               =>  C_FAMILY                      

      )
    port map (
  
      primary_aclk           =>  mm2s_aclk                 ,   
      secondary_awclk        =>  mm2s_cmdsts_awclk         ,   
      user_reset             =>  sig_cmd_stat_rst_user     ,   
      internal_reset         =>  sig_cmd_stat_rst_int      ,   
      cmd_wvalid             =>  mm2s_cmd_wvalid           ,   
      cmd_wready             =>  mm2s_cmd_wready           ,   
      cmd_wdata              =>  sig_mm2s_cmd_wdata        ,   
      cache_data             =>  sig_mm2s_cache_data       ,
      sts_wvalid             =>  mm2s_sts_wvalid           ,   
      sts_wready             =>  mm2s_sts_wready           ,   
      sts_wdata              =>  mm2s_sts_wdata            ,   
      sts_wstrb              =>  mm2s_sts_wstrb            ,   
      sts_wlast              =>  mm2s_sts_wlast            ,   
      cmd2mstr_command       =>  sig_cmd2mstr_command      ,   
      mst2cmd_cmd_valid      =>  sig_cmd2mstr_cmd_valid    ,   
      cmd2mstr_cmd_ready     =>  sig_mst2cmd_cmd_ready     ,   
      mstr2stat_status       =>  sig_rsc2stat_status       ,   
      stat2mstr_status_ready =>  sig_stat2rsc_status_ready ,   
      mst2stst_status_valid  =>  sig_rsc2stat_status_valid     
  
      );
  
  
  


        
   ------------------------------------------------------------
   -- Instance: I_RD_STATUS_CNTLR 
   --
   -- Description:
   -- Read Status Controller Block    
   --
   ------------------------------------------------------------
    I_RD_STATUS_CNTLR : entity axi_sg_v4_1_6.axi_sg_rd_status_cntl
    generic map (
  
      C_STS_WIDTH            =>  MM2S_STS_WIDTH              , 
      C_TAG_WIDTH            =>  C_TAG_WIDTH                   

      )
    port map (
  
      primary_aclk           =>  mm2s_aclk                   , 
      mmap_reset             =>  sig_mmap_rst                , 
      calc2rsc_calc_error    =>  sig_calc2dm_calc_err        , 
      addr2rsc_calc_error    =>  sig_addr2rsc_calc_error     , 
      addr2rsc_fifo_empty    =>  sig_addr2rsc_cmd_fifo_empty , 
      data2rsc_tag           =>  sig_data2rsc_tag            , 
      data2rsc_calc_error    =>  sig_data2rsc_calc_err       , 
      data2rsc_okay          =>  sig_data2rsc_okay           , 
      data2rsc_decerr        =>  sig_data2rsc_decerr         , 
      data2rsc_slverr        =>  sig_data2rsc_slverr         , 
      data2rsc_cmd_cmplt     =>  sig_data2rsc_cmd_cmplt      , 
      rsc2data_ready         =>  sig_rsc2data_ready          , 
      data2rsc_valid         =>  sig_data2rsc_valid          , 
      rsc2stat_status        =>  sig_rsc2stat_status         , 
      stat2rsc_status_ready  =>  sig_stat2rsc_status_ready   , 
      rsc2stat_status_valid  =>  sig_rsc2stat_status_valid   , 
      rsc2mstr_halt_pipe     =>  sig_rsc2mstr_halt_pipe        
    
      );
  
  
  


        
  ------------------------------------------------------------
  -- Instance: I_MSTR_SCC 
  --
  -- Description:
  -- Simple Command Calculator Block   
  --
  ------------------------------------------------------------
   I_MSTR_SCC : entity axi_sg_v4_1_6.axi_sg_scc
   generic map (
 
     C_SEL_ADDR_WIDTH     =>  SEL_ADDR_WIDTH           , 
     C_ADDR_WIDTH         =>  MM2S_ADDR_WIDTH          , 
     C_STREAM_DWIDTH      =>  MM2S_SDATA_WIDTH         , 
     C_MAX_BURST_LEN      =>  C_MM2S_BURST_SIZE        , 
     C_CMD_WIDTH          =>  MM2S_CMD_WIDTH           , 
     C_ENABLE_EXTRA_FIELD =>  C_ENABLE_EXTRA_FIELD     ,
     C_TAG_WIDTH          =>  C_TAG_WIDTH                
 
     )
   port map (
 
     -- Clock input
     primary_aclk         =>  mm2s_aclk                , 
     mmap_reset           =>  sig_mmap_rst             , 
     cmd2mstr_command     =>  sig_cmd2mstr_command     , 
     cache2mstr_command     =>  sig_cache2mstr_command     , 
     cmd2mstr_cmd_valid   =>  sig_cmd2mstr_cmd_valid   , 
     mst2cmd_cmd_ready    =>  sig_mst2cmd_cmd_ready    , 
     mstr2addr_tag        =>  sig_mstr2addr_tag        , 
     mstr2addr_addr       =>  sig_mstr2addr_addr       , 
     mstr2addr_len        =>  sig_mstr2addr_len        , 
     mstr2addr_size       =>  sig_mstr2addr_size       , 
     mstr2addr_burst      =>  sig_mstr2addr_burst      , 
     mstr2addr_calc_error =>  sig_mstr2addr_calc_error , 
     mstr2addr_cmd_cmplt  =>  sig_mstr2addr_cmd_cmplt  , 
     mstr2addr_cmd_valid  =>  sig_mstr2addr_cmd_valid  , 
     addr2mstr_cmd_ready  =>  sig_addr2mstr_cmd_ready  , 
     mstr2data_tag        =>  sig_mstr2data_tag        , 
     mstr2data_saddr_lsb  =>  sig_mstr2data_saddr_lsb  , 
     mstr2data_len        =>  sig_mstr2data_len        , 
     mstr2data_strt_strb  =>  sig_mstr2data_strt_strb  , 
     mstr2data_last_strb  =>  sig_mstr2data_last_strb  , 
     mstr2data_sof        =>  sig_mstr2data_drr        , 
     mstr2data_eof        =>  sig_mstr2data_eof        , 
     mstr2data_calc_error =>  sig_mstr2data_calc_error , 
     mstr2data_cmd_cmplt  =>  sig_mstr2data_cmd_cmplt  , 
     mstr2data_cmd_valid  =>  sig_mstr2data_cmd_valid  , 
     data2mstr_cmd_ready  =>  sig_data2mstr_cmd_ready  , 
     calc_error           =>  sig_calc2dm_calc_err       
     
     );
  
  
  


        
   ------------------------------------------------------------
   -- Instance: I_ADDR_CNTL 
   --
   -- Description:
   --   Address Controller Block  
   --
   ------------------------------------------------------------
    I_ADDR_CNTL : entity axi_sg_v4_1_6.axi_sg_addr_cntl
    generic map (
  
      -- obsoleted   C_ENABlE_WAIT_FOR_DATA   =>  DISABLE_WAIT_FOR_DATA       ,  
      --C_ADDR_FIFO_DEPTH        =>  MM2S_STSCMD_FIFO_DEPTH      ,  
      C_ADDR_FIFO_DEPTH        =>  RD_ADDR_CNTL_FIFO_DEPTH     ,
      C_ADDR_WIDTH             =>  MM2S_ADDR_WIDTH             ,  
      C_ADDR_ID                =>  MM2S_ARID_VALUE             ,  
      C_ADDR_ID_WIDTH          =>  MM2S_ARID_WIDTH             ,  
      C_TAG_WIDTH              =>  C_TAG_WIDTH                    
  
      )
    port map (
  
      primary_aclk             =>  mm2s_aclk                   ,  
      mmap_reset               =>  sig_mmap_rst                ,  
      addr2axi_aid             =>  mm2s_arid                   ,  
      addr2axi_aaddr           =>  mm2s_araddr                 ,  
      addr2axi_alen            =>  mm2s_arlen                  ,  
      addr2axi_asize           =>  mm2s_arsize                 ,  
      addr2axi_aburst          =>  mm2s_arburst                ,  
      addr2axi_aprot           =>  mm2s_arprot                 ,  
      addr2axi_avalid          =>  mm2s_arvalid                ,  
      addr2axi_acache           =>  open                        , 
      addr2axi_auser            =>  open                        , 
      axi2addr_aready          =>  mm2s_arready                ,  
      
      mstr2addr_tag            =>  sig_mstr2addr_tag           ,  
      mstr2addr_addr           =>  sig_mstr2addr_addr          ,  
      mstr2addr_len            =>  sig_mstr2addr_len           ,  
      mstr2addr_size           =>  sig_mstr2addr_size          ,  
      mstr2addr_burst          =>  sig_mstr2addr_burst         ,  
      mstr2addr_cache          =>  sig_mstr2addr_cache         ,  
      mstr2addr_user           =>  sig_mstr2addr_user          ,  
      mstr2addr_cmd_cmplt      =>  sig_mstr2addr_cmd_cmplt     ,  
      mstr2addr_calc_error     =>  sig_mstr2addr_calc_error    ,  
      mstr2addr_cmd_valid      =>  sig_mstr2addr_cmd_valid     ,  
      addr2mstr_cmd_ready      =>  sig_addr2mstr_cmd_ready     ,  
      
      addr2rst_stop_cmplt      =>  sig_addr2rst_stop_cmplt     ,  
       
      allow_addr_req           =>  mm2s_allow_addr_req         ,
      addr_req_posted          =>  mm2s_addr_req_posted        ,
       
      addr2data_addr_posted    =>  sig_addr2data_addr_posted   ,  
      data2addr_data_rdy       =>  LOGIC_LOW                   ,  
      data2addr_stop_req       =>  sig_data2addr_stop_req      ,  
      
      addr2stat_calc_error     =>  sig_addr2rsc_calc_error     ,  
      addr2stat_cmd_fifo_empty =>  sig_addr2rsc_cmd_fifo_empty    
      );
  
  
  


        
    ------------------------------------------------------------
    -- Instance: I_RD_DATA_CNTL 
    --
    -- Description:
    --     Read Data Controller Block
    --
    ------------------------------------------------------------
     I_RD_DATA_CNTL : entity axi_sg_v4_1_6.axi_sg_rddata_cntl
     generic map (
   
       C_INCLUDE_DRE           =>  INCLUDE_MM2S_DRE         ,  
       C_ALIGN_WIDTH           =>  DRE_ALIGN_WIDTH          ,  
       C_SEL_ADDR_WIDTH        =>  SEL_ADDR_WIDTH           ,  
       C_DATA_CNTL_FIFO_DEPTH  =>  RD_DATA_CNTL_FIFO_DEPTH  ,  
       C_MMAP_DWIDTH           =>  MM2S_MDATA_WIDTH         ,  
       C_STREAM_DWIDTH         =>  MM2S_SDATA_WIDTH         ,  
       C_TAG_WIDTH             =>  C_TAG_WIDTH              ,  
       C_FAMILY                =>  C_FAMILY                    
   
       )
     port map (
   
       -- Clock and Reset  -----------------------------------
       primary_aclk           =>  mm2s_aclk                 ,  
       mmap_reset             =>  sig_mmap_rst              ,  
     
       -- Soft Shutdown Interface -----------------------------
       rst2data_stop_request  =>  sig_rst2all_stop_request  ,  
       data2addr_stop_req     =>  sig_data2addr_stop_req    ,  
       data2rst_stop_cmplt    =>  sig_data2rst_stop_cmplt   ,  
        
       -- External Address Pipelining Contol support
       mm2s_rd_xfer_cmplt     =>  mm2s_rd_xfer_cmplt        ,
        
        
       -- AXI Read Data Channel I/O  -------------------------------
       mm2s_rdata             =>  mm2s_rdata                ,   
       mm2s_rresp             =>  mm2s_rresp                ,   
       mm2s_rlast             =>  mm2s_rlast                ,   
       mm2s_rvalid            =>  mm2s_rvalid               ,   
       mm2s_rready            =>  mm2s_rready               ,   
       
       -- MM2S DRE Control  -----------------------------------
       mm2s_dre_new_align     =>  open                      ,   
       mm2s_dre_use_autodest  =>  open                      ,   
       mm2s_dre_src_align     =>  open                      ,   
       mm2s_dre_dest_align    =>  open                      ,   
       mm2s_dre_flush         =>  open                      ,   
       
       -- AXI Master Stream  -----------------------------------
       mm2s_strm_wvalid       =>  mm2s_strm_wvalid      ,  
       mm2s_strm_wready       =>  mm2s_strm_wready      ,  
       mm2s_strm_wdata        =>  mm2s_strm_wdata       ,       
       mm2s_strm_wstrb        =>  mm2s_strm_wstrb       ,           
       mm2s_strm_wlast        =>  mm2s_strm_wlast       ,  
  
      -- MM2S Store and Forward Supplimental Control -----------
       mm2s_data2sf_cmd_cmplt  => open                      ,                               
                                                                                     
  
       
       -- Command Calculator Interface --------------------------
       mstr2data_tag          =>  sig_mstr2data_tag         ,  
       mstr2data_saddr_lsb    =>  sig_mstr2data_saddr_lsb   ,  
       mstr2data_len          =>  sig_mstr2data_len         ,  
       mstr2data_strt_strb    =>  sig_mstr2data_strt_strb   ,  
       mstr2data_last_strb    =>  sig_mstr2data_last_strb   ,  
       mstr2data_drr          =>  sig_mstr2data_drr         ,  
       mstr2data_eof          =>  sig_mstr2data_eof         ,  
       mstr2data_sequential   =>  LOGIC_LOW                 ,  
       mstr2data_calc_error   =>  sig_mstr2data_calc_error  ,  
       mstr2data_cmd_cmplt    =>  sig_mstr2data_cmd_cmplt   ,  
       mstr2data_cmd_valid    =>  sig_mstr2data_cmd_valid   ,  
       data2mstr_cmd_ready    =>  sig_data2mstr_cmd_ready   ,  
       mstr2data_dre_src_align  => DRE_ALIGN_ZEROS          ,  
       mstr2data_dre_dest_align => DRE_ALIGN_ZEROS          ,  
       
       -- Address Controller Interface --------------------------
       addr2data_addr_posted  =>  sig_addr2data_addr_posted , 
       
       -- Data Controller Halted Status
       data2all_dcntlr_halted =>  sig_data2all_dcntlr_halted, 
        
       -- Output Stream Skid Buffer Halt control
       data2skid_halt         =>  sig_data2skid_halt        , 
        
        
       -- Read Status Controller Interface --------------------------
       data2rsc_tag           =>  sig_data2rsc_tag          , 
       data2rsc_calc_err      =>  sig_data2rsc_calc_err     , 
       data2rsc_okay          =>  sig_data2rsc_okay         , 
       data2rsc_decerr        =>  sig_data2rsc_decerr       , 
       data2rsc_slverr        =>  sig_data2rsc_slverr       , 
       data2rsc_cmd_cmplt     =>  sig_data2rsc_cmd_cmplt    , 
       rsc2data_ready         =>  sig_rsc2data_ready        , 
       data2rsc_valid         =>  sig_data2rsc_valid        , 
       rsc2mstr_halt_pipe     =>  sig_rsc2mstr_halt_pipe      
        
       );
  
  
  


        
    ------------------------------------------------------------
    -- Instance: I_MM2S_SKID_BUF 
    --
    -- Description:
    --   Instance for the MM2S Skid Buffer which provides for
    -- registerd Master Stream outputs and supports bi-dir
    -- throttling.  
    --
    ------------------------------------------------------------
--     I_MM2S_SKID_BUF : entity axi_sg_v4_1_6.axi_sg_skid_buf
--     generic map (
--        
--       C_WDATA_WIDTH  =>  MM2S_SDATA_WIDTH        
--   
--       )
--     port map (
--   
--       -- System Ports
--       aclk           =>  mm2s_aclk             ,  
--       arst           =>  sig_stream_rst        ,  
--     
--        -- Shutdown control (assert for 1 clk pulse)
--       skid_stop      =>  sig_data2skid_halt    ,  
--     
--       -- Slave Side (Stream Data Input) 
--       s_valid        =>  sig_data2skid_wvalid  ,  
--       s_ready        =>  sig_data2skid_wready  ,  
--       s_data         =>  sig_data2skid_wdata   ,  
--       s_strb         =>  sig_data2skid_wstrb   ,  
--       s_last         =>  sig_data2skid_wlast   ,  
--
--       -- Master Side (Stream Data Output 
--       m_valid        =>  mm2s_strm_wvalid      ,  
--       m_ready        =>  mm2s_strm_wready      ,  
--       m_data         =>  mm2s_strm_wdata       ,  
--       m_strb         =>  mm2s_strm_wstrb       ,  
--       m_last         =>  mm2s_strm_wlast          
--   
--       );
--   

  
    
    
    
  end implementation;


-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
  --
  -- *************************************************************************
  --
  -------------------------------------------------------------------------------
  -- Filename:        axi_sg_s2mm_basic_wrap.vhd
  --
  -- Description:     
  --    This file implements the DataMover S2MM Basic Wrapper.                 
  --                  
  --                  
  --                  
  --                  
  -- VHDL-Standard:   VHDL'93
  -------------------------------------------------------------------------------
  library IEEE;
  use IEEE.std_logic_1164.all;
  use IEEE.numeric_std.all;
  
  
  
  -- axi_sg Library Modules
  library axi_sg_v4_1_6; 
  use axi_sg_v4_1_6.axi_sg_reset;  
  use axi_sg_v4_1_6.axi_sg_cmd_status;
  use axi_sg_v4_1_6.axi_sg_scc_wr;
  use axi_sg_v4_1_6.axi_sg_addr_cntl;
  use axi_sg_v4_1_6.axi_sg_wrdata_cntl;
  use axi_sg_v4_1_6.axi_sg_wr_status_cntl;
  Use axi_sg_v4_1_6.axi_sg_skid2mm_buf;
  Use axi_sg_v4_1_6.axi_sg_skid_buf;
  
  
  -------------------------------------------------------------------------------
  
  entity axi_sg_s2mm_basic_wrap is
    generic (
      
      C_INCLUDE_S2MM    : Integer range 0 to  2 :=  2;
         -- Specifies the type of S2MM function to include
         -- 0 = Omit S2MM functionality
         -- 1 = Full S2MM Functionality
         -- 2 = Basic S2MM functionality
         
      C_S2MM_AWID  : Integer range 0 to 255 :=  9;
         -- Specifies the constant value to output on 
         -- the ARID output port
         
      C_S2MM_ID_WIDTH    : Integer range 1 to  8 :=  4;
         -- Specifies the width of the S2MM ID port 
         
      C_S2MM_ADDR_WIDTH  : Integer range 32 to  64 :=  32;
         -- Specifies the width of the MMap Read Address Channel 
         -- Address bus
         
      C_S2MM_MDATA_WIDTH : Integer range 32 to 64 :=  32;
         -- Specifies the width of the MMap Read Data Channel
         -- data bus
      
      C_S2MM_SDATA_WIDTH : Integer range 8 to 64 :=  32;
         -- Specifies the width of the S2MM Master Stream Data 
         -- Channel data bus
      
      C_INCLUDE_S2MM_STSFIFO    : Integer range 0 to  1 :=  1;
         -- Specifies if a Status FIFO is to be implemented
         -- 0 = Omit S2MM Status FIFO
         -- 1 = Include S2MM Status FIFO
         
      C_S2MM_STSCMD_FIFO_DEPTH    : Integer range 1 to 16 :=  1;
         -- Specifies the depth of the S2MM Command FIFO and the 
         -- optional Status FIFO
         -- Valid values are 1,4,8,16
         
      C_S2MM_STSCMD_IS_ASYNC    : Integer range 0 to  1 :=  0;
         -- Specifies if the Status and Command interfaces need to
         -- be asynchronous to the primary data path clocking
         -- 0 = Use same clocking as data path
         -- 1 = Use special Status/Command clock for the interfaces
         
      C_INCLUDE_S2MM_DRE    : Integer range 0 to  1 :=  0;
         -- Specifies if DRE is to be included in the S2MM function 
         -- 0 = Omit DRE
         -- 1 = Include DRE
      
      C_S2MM_BURST_SIZE    : Integer range 16 to  64 :=  16;
         -- Specifies the max number of databeats to use for MMap
         -- burst transfers by the S2MM function 

      C_S2MM_ADDR_PIPE_DEPTH    : Integer range 1 to 30 := 1;
          -- This parameter specifies the depth of the S2MM internal 
          -- address pipeline queues in the Write Address Controller 
          -- and the Write Data Controller. Increasing this value will 
          -- allow more Write Addresses to be issued to the AXI4 Write 
          -- Address Channel before transmission of the associated  
          -- write data on the Write Data Channel.

      C_ENABLE_MULTI_CHANNEL           : Integer range 0 to 1 := 1; 
      C_ENABLE_EXTRA_FIELD             : integer range 0 to 1 := 0;
      C_TAG_WIDTH        : Integer range 1 to 8 :=  4 ;
         -- Width of the TAG field
         
      C_FAMILY : String := "virtex7"
         -- Specifies the target FPGA family type
      
      );
    port (
      
      
      -- S2MM Primary Clock and reset inputs -----------------------------
      s2mm_aclk         : in  std_logic;                                --
         -- Primary synchronization clock for the Master side           --
         -- interface and internal logic. It is also used               --
         -- for the User interface synchronization when                 --
         -- C_STSCMD_IS_ASYNC = 0.                                      --
                                                                        --
      -- S2MM Primary Reset input                                       --
      s2mm_aresetn      : in  std_logic;                                --
         -- Reset used for the internal master logic                    --
      --------------------------------------------------------------------
      sg_ctl            : in  std_logic_vector (7 downto 0);            -- 
 
     
      -- S2MM Halt request input control ---------------------------------
      s2mm_halt               : in  std_logic;                          --
         -- Active high soft shutdown request                           --
                                                                        --
      -- S2MM Halt Complete status flag                                 --
      s2mm_halt_cmplt         : Out  std_logic;                         --
         -- Active high soft shutdown complete status                   --
      --------------------------------------------------------------------
      
      
      
      -- S2MM Error discrete output --------------------------------------
      s2mm_err          : Out std_logic;                                --
         -- Composite Error indication                                  --
      --------------------------------------------------------------------
      
     
     
     
      -- Optional Command/Status Interface Clock and Reset Inputs  -------
      -- Only used when C_S2MM_STSCMD_IS_ASYNC = 1                      --
                                                                        --
      s2mm_cmdsts_awclk       : in  std_logic;                          --
      -- Secondary Clock input for async CMD/Status interface           --
                                                                        --
      s2mm_cmdsts_aresetn     : in  std_logic;                          --
        -- Secondary Reset input for async CMD/Status interface         --
      --------------------------------------------------------------------
      
      
      -- User Command Interface Ports (AXI Stream) ------------------------------------------------------
      s2mm_cmd_wvalid         : in  std_logic;                                                         --
      s2mm_cmd_wready         : out std_logic;                                                         --
      s2mm_cmd_wdata          : in  std_logic_vector((C_TAG_WIDTH+(1+C_ENABLE_MULTI_CHANNEL)*C_S2MM_ADDR_WIDTH+36)-1 downto 0);   --
      ---------------------------------------------------------------------------------------------------
      
      
      -- User Status Interface Ports (AXI Stream) ------------------------
      s2mm_sts_wvalid         : out std_logic;                          --
      s2mm_sts_wready         : in  std_logic;                          --
      s2mm_sts_wdata          : out std_logic_vector(7 downto 0);       --
      s2mm_sts_wstrb          : out std_logic_vector(0 downto 0);       --
      s2mm_sts_wlast          : out std_logic;                          --
      --------------------------------------------------------------------
      
      
      -- Address posting controls ----------------------------------------
      s2mm_allow_addr_req     : in  std_logic;                          --
      s2mm_addr_req_posted    : out std_logic;                          --
      s2mm_wr_xfer_cmplt      : out std_logic;                          --
      s2mm_ld_nxt_len         : out std_logic;                          --
      s2mm_wr_len             : out std_logic_vector(7 downto 0);       --
      --------------------------------------------------------------------
      
     
      
      -- S2MM AXI Address Channel I/O  --------------------------------------
      s2mm_awid     : out std_logic_vector(C_S2MM_ID_WIDTH-1 downto 0);    --
         -- AXI Address Channel ID output                                  --
                                                                           --
      s2mm_awaddr   : out std_logic_vector(C_S2MM_ADDR_WIDTH-1 downto 0);  --
         -- AXI Address Channel Address output                             --
                                                                           --
      s2mm_awlen    : out std_logic_vector(7 downto 0);                    --
         -- AXI Address Channel LEN output                                 --
         -- Sized to support 256 data beat bursts                          --
                                                                           --
      s2mm_awsize   : out std_logic_vector(2 downto 0);                    --
         -- AXI Address Channel SIZE output                                --
                                                                           --
      s2mm_awburst  : out std_logic_vector(1 downto 0);                    --
         -- AXI Address Channel BURST output                               --
                                                                           --
      s2mm_awprot   : out std_logic_vector(2 downto 0);                    --
         -- AXI Address Channel PROT output                                --
                                                                           --
      s2mm_awcache  : out std_logic_vector(3 downto 0);                    --
         -- AXI Address Channel PROT output                                --

      s2mm_awuser  : out std_logic_vector(3 downto 0);                    --
         -- AXI Address Channel PROT output                                --
                                                                           --
      s2mm_awvalid  : out std_logic;                                       --
         -- AXI Address Channel VALID output                               --
                                                                           --
      s2mm_awready  : in  std_logic;                                       --
         -- AXI Address Channel READY input                                --
      -----------------------------------------------------------------------
      
        
      -- Currently unsupported AXI Address Channel output signals -----------
        -- s2mm__awlock   : out std_logic_vector(2 downto 0);              --
        -- s2mm__awcache  : out std_logic_vector(4 downto 0);              --
        -- s2mm__awqos    : out std_logic_vector(3 downto 0);              --
        -- s2mm__awregion : out std_logic_vector(3 downto 0);              --
      -----------------------------------------------------------------------
  
  
  
  
      
      -- S2MM AXI MMap Write Data Channel I/O  ---------------------------------------------
      s2mm_wdata              : Out  std_logic_vector(C_S2MM_MDATA_WIDTH-1 downto 0);     --
      s2mm_wstrb              : Out  std_logic_vector((C_S2MM_MDATA_WIDTH/8)-1 downto 0); --
      s2mm_wlast              : Out  std_logic;                                           --
      s2mm_wvalid             : Out  std_logic;                                           --
      s2mm_wready             : In   std_logic;                                           --
      --------------------------------------------------------------------------------------
      
      
      -- S2MM AXI MMap Write response Channel I/O  -----------------------------------------
      s2mm_bresp              : In   std_logic_vector(1 downto 0);                        --
      s2mm_bvalid             : In   std_logic;                                           --
      s2mm_bready             : Out  std_logic;                                           --
      --------------------------------------------------------------------------------------
      
      
      
      -- S2MM AXI Master Stream Channel I/O  -----------------------------------------------
      s2mm_strm_wdata         : In  std_logic_vector(C_S2MM_SDATA_WIDTH-1 downto 0);      --
      s2mm_strm_wstrb         : In  std_logic_vector((C_S2MM_SDATA_WIDTH/8)-1 downto 0);  --
      s2mm_strm_wlast         : In  std_logic;                                            --
      s2mm_strm_wvalid        : In  std_logic;                                            --
      s2mm_strm_wready        : Out std_logic;                                            --
      --------------------------------------------------------------------------------------
      
      -- Testing Support I/O ------------------------------------------
      s2mm_dbg_sel            : in  std_logic_vector( 3 downto 0);   --
      s2mm_dbg_data           : out std_logic_vector(31 downto 0)    --
      -----------------------------------------------------------------
      
      
      );                            
  
  end entity axi_sg_s2mm_basic_wrap;
  
  
  architecture implementation of axi_sg_s2mm_basic_wrap is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  
    
    
    -- Function Declarations   ----------------------------------------
    
    -------------------------------------------------------------------
    -- Function
    --
    -- Function Name: func_calc_wdemux_sel_bits
    --
    -- Function Description:
    --  This function calculates the number of address bits needed for  
    -- the Write Strobe demux select control. 
    --
    -------------------------------------------------------------------
    function func_calc_wdemux_sel_bits (mmap_dwidth_value : integer) return integer is
    
      Variable num_addr_bits_needed : Integer range 1 to 5 := 1;
    
    begin
    
      case mmap_dwidth_value is
        when 32 =>
          num_addr_bits_needed := 2;
       -- coverage off
        when 64 =>
          num_addr_bits_needed := 3;
        when 128 =>
          num_addr_bits_needed := 4;
        when others => -- 256 bits
          num_addr_bits_needed := 5;
       -- coverage on
      end case;
      
      Return (num_addr_bits_needed);
       
    end function func_calc_wdemux_sel_bits;
    

   


    -- Constant Declarations   ----------------------------------------
    
     Constant LOGIC_LOW                 : std_logic := '0';
     Constant LOGIC_HIGH                : std_logic := '1';
     Constant S2MM_AWID_VALUE           : integer range  0 to 255 := C_S2MM_AWID;
     Constant S2MM_AWID_WIDTH           : integer range  1 to   8 := C_S2MM_ID_WIDTH;
     Constant S2MM_ADDR_WIDTH           : integer range 32 to  64 := C_S2MM_ADDR_WIDTH;
     Constant S2MM_MDATA_WIDTH          : integer range 32 to 256 := C_S2MM_MDATA_WIDTH;
     Constant S2MM_SDATA_WIDTH          : integer range  8 to 256 := C_S2MM_SDATA_WIDTH;
     Constant S2MM_CMD_WIDTH            : integer                 := (C_TAG_WIDTH+C_S2MM_ADDR_WIDTH+32);
     Constant S2MM_STS_WIDTH            : integer                 :=  8; -- always 8 for S2MM Basic Version
     Constant INCLUDE_S2MM_STSFIFO      : integer range  0 to   1 :=  1;
     Constant S2MM_STSCMD_FIFO_DEPTH    : integer range  1 to  16 :=  1;
     Constant S2MM_STSCMD_IS_ASYNC      : integer range  0 to   1 :=  0;
     Constant S2MM_BURST_SIZE           : integer range 16 to 256 := 16;
     Constant WR_ADDR_CNTL_FIFO_DEPTH   : integer range  1 to  30 := C_S2MM_ADDR_PIPE_DEPTH;
     Constant WR_DATA_CNTL_FIFO_DEPTH   : integer range  1 to  30 := C_S2MM_ADDR_PIPE_DEPTH;
     
     Constant WR_STATUS_CNTL_FIFO_DEPTH : integer range  1 to  32 := WR_DATA_CNTL_FIFO_DEPTH+2;-- 2 added for going 
                                                                                               -- full thresholding
                                                                                               -- in WSC           
     
     
     Constant SEL_ADDR_WIDTH            : integer := func_calc_wdemux_sel_bits(S2MM_MDATA_WIDTH);
     Constant INCLUDE_S2MM_DRE          : integer range  0 to   1 :=  1;
     Constant OMIT_S2MM_DRE             : integer range  0 to   1 :=  0;
     Constant OMIT_INDET_BTT            : integer := 0;
     Constant SF_BYTES_RCVD_WIDTH       : integer := 1;
     Constant ZEROS_8_BIT               : std_logic_vector(7 downto 0) := (others => '0');
     
     
        
    
    -- Signal Declarations  ------------------------------------------
    
     signal sig_cmd_stat_rst_user        : std_logic := '0';
     signal sig_cmd_stat_rst_int         : std_logic := '0';
     signal sig_mmap_rst                 : std_logic := '0';
     signal sig_stream_rst               : std_logic := '0';
     signal sig_s2mm_cmd_wdata           : std_logic_vector(S2MM_CMD_WIDTH-1 downto 0) := (others => '0');
     signal sig_s2mm_cache_data          : std_logic_vector(7 downto 0) := (others => '0');
     signal sig_cmd2mstr_command         : std_logic_vector(S2MM_CMD_WIDTH-1 downto 0) := (others => '0');        
     signal sig_cmd2mstr_cmd_valid       : std_logic := '0';                                             
     signal sig_mst2cmd_cmd_ready        : std_logic := '0';                                             
     signal sig_mstr2addr_addr           : std_logic_vector(S2MM_ADDR_WIDTH-1 downto 0) := (others => '0');             
     signal sig_mstr2addr_len            : std_logic_vector(7 downto 0) := (others => '0');                          
     signal sig_mstr2addr_size           : std_logic_vector(2 downto 0) := (others => '0');                          
     signal sig_mstr2addr_burst          : std_logic_vector(1 downto 0) := (others => '0'); 
     signal sig_mstr2addr_cache          : std_logic_vector(3 downto 0) := (others => '0'); 
     signal sig_mstr2addr_user           : std_logic_vector(3 downto 0) := (others => '0'); 
     signal sig_mstr2addr_cmd_cmplt      : std_logic := '0';
     signal sig_mstr2addr_calc_error     : std_logic := '0';
     signal sig_mstr2addr_cmd_valid      : std_logic := '0';                                             
     signal sig_addr2mstr_cmd_ready      : std_logic := '0';                                              
     signal sig_mstr2data_saddr_lsb      : std_logic_vector(SEL_ADDR_WIDTH-1 downto 0) := (others => '0');
     signal sig_mstr2data_len            : std_logic_vector(7 downto 0) := (others => '0');
     signal sig_mstr2data_strt_strb      : std_logic_vector((S2MM_SDATA_WIDTH/8)-1 downto 0) := (others => '0');      
     signal sig_mstr2data_last_strb      : std_logic_vector((S2MM_SDATA_WIDTH/8)-1 downto 0) := (others => '0');
     signal sig_mstr2data_drr            : std_logic := '0';
     signal sig_mstr2data_eof            : std_logic := '0';
     signal sig_mstr2data_calc_error     : std_logic := '0';
     signal sig_mstr2data_cmd_last       : std_logic := '0';
     signal sig_mstr2data_cmd_valid      : std_logic := '0';                                             
     signal sig_data2mstr_cmd_ready      : std_logic := '0';                                               
     signal sig_addr2data_addr_posted    : std_logic := '0';
     signal sig_data2addr_data_rdy       : std_logic := '0';
     signal sig_data2all_tlast_error     : std_logic := '0';
     signal sig_data2all_dcntlr_halted   : std_logic := '0';
     signal sig_addr2wsc_calc_error      : std_logic := '0';
     signal sig_addr2wsc_cmd_fifo_empty  : std_logic := '0';
     signal sig_data2wsc_rresp           : std_logic_vector(1 downto 0) := (others => '0'); 
     signal sig_data2wsc_cmd_empty       : std_logic := '0';                   
     signal sig_data2wsc_calc_err        : std_logic := '0'; 
     signal sig_data2wsc_cmd_cmplt       : std_logic := '0';
     signal sig_data2wsc_last_err        : std_logic := '0';
     signal sig_calc2dm_calc_err         : std_logic := '0';
     signal sig_wsc2stat_status          : std_logic_vector(7 downto 0) := (others => '0');
     signal sig_stat2wsc_status_ready    : std_logic := '0';   
     signal sig_wsc2stat_status_valid    : std_logic := '0';  
     signal sig_wsc2mstr_halt_pipe       : std_logic := '0';  
     signal sig_data2wsc_tag             : std_logic_vector(C_TAG_WIDTH-1 downto 0) := (others => '0');
     signal sig_mstr2data_tag            : std_logic_vector(C_TAG_WIDTH-1 downto 0) := (others => '0');
     signal sig_mstr2addr_tag            : std_logic_vector(C_TAG_WIDTH-1 downto 0) := (others => '0');
     signal sig_data2skid_addr_lsb       : std_logic_vector(SEL_ADDR_WIDTH-1 downto 0) := (others => '0');
     signal sig_data2skid_wvalid         : std_logic := '0';
     signal sig_skid2data_wready         : std_logic := '0';
     signal sig_data2skid_wdata          : std_logic_vector(C_S2MM_SDATA_WIDTH-1 downto 0) := (others => '0');
     signal sig_data2skid_wstrb          : std_logic_vector((C_S2MM_SDATA_WIDTH/8)-1 downto 0) := (others => '0');
     signal sig_data2skid_wlast          : std_logic := '0';
     signal sig_skid2axi_wvalid          : std_logic := '0';
     signal sig_axi2skid_wready          : std_logic := '0';
     signal sig_skid2axi_wdata           : std_logic_vector(C_S2MM_MDATA_WIDTH-1 downto 0) := (others => '0');     
     signal sig_skid2axi_wstrb           : std_logic_vector((C_S2MM_MDATA_WIDTH/8)-1 downto 0) := (others => '0'); 
     signal sig_skid2axi_wlast           : std_logic := '0';
     signal sig_data2wsc_sof             : std_logic := '0';
     signal sig_data2wsc_eof             : std_logic := '0';
     signal sig_data2wsc_valid           : std_logic := '0';
     signal sig_wsc2data_ready           : std_logic := '0';
     signal sig_data2wsc_eop             : std_logic := '0';
     signal sig_data2wsc_bytes_rcvd      : std_logic_vector(SF_BYTES_RCVD_WIDTH-1 downto 0) := (others => '0');
     signal sig_dbg_data_mux_out         : std_logic_vector(31 downto 0) := (others => '0');
     signal sig_dbg_data_0               : std_logic_vector(31 downto 0) := (others => '0');
     signal sig_dbg_data_1               : std_logic_vector(31 downto 0) := (others => '0');
     signal sig_rst2all_stop_request     : std_logic := '0';
     signal sig_data2rst_stop_cmplt      : std_logic := '0';
     signal sig_addr2rst_stop_cmplt      : std_logic := '0';
     signal sig_data2addr_stop_req       : std_logic := '0';
     signal sig_wsc2rst_stop_cmplt       : std_logic := '0';
     signal sig_data2skid_halt           : std_logic := '0';
     signal sig_realign2wdc_eop_error    : std_logic := '0';
     signal skid2wdc_wvalid              : std_logic := '0';
     signal wdc2skid_wready              : std_logic := '0';
     signal skid2wdc_wdata               : std_logic_vector(C_S2MM_SDATA_WIDTH-1 downto 0) := (others => '0');
     signal skid2wdc_wstrb               : std_logic_vector((C_S2MM_SDATA_WIDTH/8)-1 downto 0) := (others => '0');
     signal skid2wdc_wlast               : std_logic := '0';
     signal s2mm_awcache_int             : std_logic_vector (3 downto 0);
     signal sig_cache2mstr_command       : std_logic_vector (7 downto 0);
    
    
  begin --(architecture implementation)
  
    
    
    -- Debug Port Assignments
    
    s2mm_dbg_data        <= sig_dbg_data_mux_out;
    
    -- Note that only the s2mm_dbg_sel(0) is used at this time
    sig_dbg_data_mux_out <= sig_dbg_data_1
      When (s2mm_dbg_sel(0) = '1')
      else sig_dbg_data_0 ;
    
    
    sig_dbg_data_0              <=  X"CAFE2222"             ;    -- 32 bit Constant indicating S2MM Basic type
    
    sig_dbg_data_1(0)           <= sig_cmd_stat_rst_user    ;
    sig_dbg_data_1(1)           <= sig_cmd_stat_rst_int     ;
    sig_dbg_data_1(2)           <= sig_mmap_rst             ;
    sig_dbg_data_1(3)           <= sig_stream_rst           ;
    sig_dbg_data_1(4)           <= sig_cmd2mstr_cmd_valid   ;
    sig_dbg_data_1(5)           <= sig_mst2cmd_cmd_ready    ;
    sig_dbg_data_1(6)           <= sig_stat2wsc_status_ready;
    sig_dbg_data_1(7)           <= sig_wsc2stat_status_valid;
    sig_dbg_data_1(11 downto 8) <= sig_data2wsc_tag         ; -- Current TAG of active data transfer
                         
                        
    sig_dbg_data_1(15 downto 12) <= sig_wsc2stat_status(3 downto 0); -- Internal status tag field
    sig_dbg_data_1(16)           <= sig_wsc2stat_status(4)         ; -- Internal error
    sig_dbg_data_1(17)           <= sig_wsc2stat_status(5)         ; -- Decode Error
    sig_dbg_data_1(18)           <= sig_wsc2stat_status(6)         ; -- Slave Error
    --sig_dbg_data_1(19)           <= sig_wsc2stat_status(7)         ; -- OKAY
    sig_dbg_data_1(19)           <= '0'                            ; -- OKAY not used by TB
    sig_dbg_data_1(20)           <= sig_stat2wsc_status_ready      ; -- Status Ready Handshake
    sig_dbg_data_1(21)           <= sig_wsc2stat_status_valid      ; -- Status Valid Handshake
                        
    
    sig_dbg_data_1(29 downto 22) <= sig_mstr2data_len              ; -- WDC Cmd FIFO LEN input
    sig_dbg_data_1(30)           <= sig_mstr2data_cmd_valid        ; -- WDC Cmd FIFO Valid Inpute
    sig_dbg_data_1(31)           <= sig_data2mstr_cmd_ready        ; -- WDC Cmd FIFO Ready Output
    
    
                        
    
    
    -- Write Data Channel I/O
     s2mm_wvalid         <= sig_skid2axi_wvalid; 
     sig_axi2skid_wready <= s2mm_wready        ;
     s2mm_wdata          <= sig_skid2axi_wdata ; 
     s2mm_wstrb          <= sig_skid2axi_wstrb ; 
     s2mm_wlast          <= sig_skid2axi_wlast ; 
     
     
    GEN_CACHE : if (C_ENABLE_MULTI_CHANNEL = 0) generate
      begin
     -- Cache signal tie-off
     s2mm_awcache <= "0011";  -- pre Interface-X guidelines for Masters
     s2mm_awuser <= "0000";  -- pre Interface-X guidelines for Masters
     sig_s2mm_cache_data <= (others => '0'); --s2mm_cmd_wdata(103 downto 96);
    end generate GEN_CACHE;
     
                        
    GEN_CACHE2 : if (C_ENABLE_MULTI_CHANNEL = 1) generate
      begin
     -- Cache signal tie-off
     s2mm_awcache <= sg_ctl (3 downto 0);  -- SG Cache from register
     s2mm_awuser <= sg_ctl (7 downto 4);  -- SG Cache from register
     sig_s2mm_cache_data <= s2mm_cmd_wdata(103 downto 96);
    end generate GEN_CACHE2;
                       
     -- Internal error output discrete
     s2mm_err            <=  sig_calc2dm_calc_err or sig_data2all_tlast_error;
     
     
     -- Rip the used portion of the Command Interface Command Data
     -- and throw away the padding
     sig_s2mm_cmd_wdata <= s2mm_cmd_wdata(S2MM_CMD_WIDTH-1 downto 0);
     
     
     
 
     -- No Realigner in S2MM Basic
     sig_realign2wdc_eop_error <= '0';

     
     
     
     
     
     
     
     
          
          
     ------------------------------------------------------------
     -- Instance: I_RESET 
     --
     -- Description:
     --   Reset Block  
     --
     ------------------------------------------------------------
      I_RESET : entity axi_sg_v4_1_6.axi_sg_reset
      generic map (
    
        C_STSCMD_IS_ASYNC    =>  S2MM_STSCMD_IS_ASYNC      
    
        )
      port map (
    
        primary_aclk         =>  s2mm_aclk               , 
        primary_aresetn      =>  s2mm_aresetn            , 
        secondary_awclk      =>  s2mm_cmdsts_awclk       , 
        secondary_aresetn    =>  s2mm_cmdsts_aresetn     , 
        halt_req             =>  s2mm_halt               , 
        halt_cmplt           =>  s2mm_halt_cmplt         , 
        flush_stop_request   =>  sig_rst2all_stop_request, 
        data_cntlr_stopped   =>  sig_data2rst_stop_cmplt , 
        addr_cntlr_stopped   =>  sig_addr2rst_stop_cmplt , 
        aux1_stopped         =>  sig_wsc2rst_stop_cmplt  , 
        aux2_stopped         =>  LOGIC_HIGH              , 
        cmd_stat_rst_user    =>  sig_cmd_stat_rst_user   , 
        cmd_stat_rst_int     =>  sig_cmd_stat_rst_int    , 
        mmap_rst             =>  sig_mmap_rst            , 
        stream_rst           =>  sig_stream_rst            
    
        );
    
    
    
  
  
          
     ------------------------------------------------------------
     -- Instance: I_CMD_STATUS 
     --
     -- Description:
     --   Command and Status Interface Block  
     --
     ------------------------------------------------------------
      I_CMD_STATUS : entity axi_sg_v4_1_6.axi_sg_cmd_status
      generic map (
    
        C_ADDR_WIDTH           =>  S2MM_ADDR_WIDTH           ,     
        C_INCLUDE_STSFIFO      =>  INCLUDE_S2MM_STSFIFO      ,     
        C_STSCMD_FIFO_DEPTH    =>  S2MM_STSCMD_FIFO_DEPTH    ,     
        C_STSCMD_IS_ASYNC      =>  S2MM_STSCMD_IS_ASYNC      ,     
        C_CMD_WIDTH            =>  S2MM_CMD_WIDTH            ,     
        C_STS_WIDTH            =>  S2MM_STS_WIDTH            ,     
        C_FAMILY               =>  C_FAMILY                        

        )
      port map (
    
        primary_aclk           =>  s2mm_aclk                 ,     
        secondary_awclk        =>  s2mm_cmdsts_awclk         ,     
        user_reset             =>  sig_cmd_stat_rst_user     ,     
        internal_reset         =>  sig_cmd_stat_rst_int      ,     
        cmd_wvalid             =>  s2mm_cmd_wvalid           ,     
        cmd_wready             =>  s2mm_cmd_wready           ,     
        cmd_wdata              =>  sig_s2mm_cmd_wdata        ,     
        cache_data             =>  sig_s2mm_cache_data        ,     
        sts_wvalid             =>  s2mm_sts_wvalid           ,     
        sts_wready             =>  s2mm_sts_wready           ,     
        sts_wdata              =>  s2mm_sts_wdata            ,     
        sts_wstrb              =>  s2mm_sts_wstrb            ,     
        sts_wlast              =>  s2mm_sts_wlast            ,     
        cmd2mstr_command       =>  sig_cmd2mstr_command      ,     
        cache2mstr_command     =>  sig_cache2mstr_command      ,
        mst2cmd_cmd_valid      =>  sig_cmd2mstr_cmd_valid    ,     
        cmd2mstr_cmd_ready     =>  sig_mst2cmd_cmd_ready     ,     
        mstr2stat_status       =>  sig_wsc2stat_status       ,     
        stat2mstr_status_ready =>  sig_stat2wsc_status_ready ,     
        mst2stst_status_valid  =>  sig_wsc2stat_status_valid       
    
        );
    
    
    
  
  
          
     ------------------------------------------------------------
     -- Instance: I_RD_STATUS_CNTLR 
     --
     -- Description:
     -- Write Status Controller Block    
     --
     ------------------------------------------------------------
      I_WR_STATUS_CNTLR : entity axi_sg_v4_1_6.axi_sg_wr_status_cntl
      generic map (
    
        C_ENABLE_INDET_BTT     =>  OMIT_INDET_BTT              ,  
        C_SF_BYTES_RCVD_WIDTH  =>  SF_BYTES_RCVD_WIDTH         ,  
        C_STS_FIFO_DEPTH       =>  WR_STATUS_CNTL_FIFO_DEPTH   ,
        C_STS_WIDTH            =>  S2MM_STS_WIDTH              ,  
        C_TAG_WIDTH            =>  C_TAG_WIDTH                 ,  
        C_FAMILY               =>  C_FAMILY                       

        )
      port map (
    
        primary_aclk           =>  s2mm_aclk                   ,  
        mmap_reset             =>  sig_mmap_rst                ,  
        rst2wsc_stop_request   =>  sig_rst2all_stop_request    ,  
        wsc2rst_stop_cmplt     =>  sig_wsc2rst_stop_cmplt      ,  
        addr2wsc_addr_posted   =>  sig_addr2data_addr_posted   ,  
        s2mm_bresp             =>  s2mm_bresp                  ,  
        s2mm_bvalid            =>  s2mm_bvalid                 ,  
        s2mm_bready            =>  s2mm_bready                 ,  
        calc2wsc_calc_error    =>  sig_calc2dm_calc_err        ,  
        addr2wsc_calc_error    =>  sig_addr2wsc_calc_error     ,  
        addr2wsc_fifo_empty    =>  sig_addr2wsc_cmd_fifo_empty ,  
        data2wsc_tag           =>  sig_data2wsc_tag            ,  
        data2wsc_calc_error    =>  sig_data2wsc_calc_err       ,  
        data2wsc_last_error    =>  sig_data2wsc_last_err       ,  
        data2wsc_cmd_cmplt     =>  sig_data2wsc_cmd_cmplt      ,  
        data2wsc_valid         =>  sig_data2wsc_valid          ,  
        wsc2data_ready         =>  sig_wsc2data_ready          ,  
        data2wsc_eop           =>  sig_data2wsc_eop            ,  
        data2wsc_bytes_rcvd    =>  sig_data2wsc_bytes_rcvd     ,  
        wsc2stat_status        =>  sig_wsc2stat_status         ,  
        stat2wsc_status_ready  =>  sig_stat2wsc_status_ready   ,  
        wsc2stat_status_valid  =>  sig_wsc2stat_status_valid   ,  
        wsc2mstr_halt_pipe     =>  sig_wsc2mstr_halt_pipe         
      
        );
    
    
    
  
  
          
    ------------------------------------------------------------
    -- Instance: I_MSTR_SCC 
    --
    -- Description:
    -- Simple Command Calculator Block   
    --
    ------------------------------------------------------------
     I_MSTR_SCC : entity axi_sg_v4_1_6.axi_sg_scc_wr
     generic map (
   
       C_SEL_ADDR_WIDTH     =>  SEL_ADDR_WIDTH           ,   
       C_ADDR_WIDTH         =>  S2MM_ADDR_WIDTH          ,   
       C_STREAM_DWIDTH      =>  S2MM_SDATA_WIDTH         ,   
       C_MAX_BURST_LEN      =>  C_S2MM_BURST_SIZE        ,   
       C_CMD_WIDTH          =>  S2MM_CMD_WIDTH           ,   
       C_ENABLE_EXTRA_FIELD => C_ENABLE_EXTRA_FIELD,
       C_TAG_WIDTH          =>  C_TAG_WIDTH                  
   
       )
     port map (
   
       -- Clock input
       primary_aclk         =>  s2mm_aclk                ,   
       mmap_reset           =>  sig_mmap_rst             ,   
       cmd2mstr_command     =>  sig_cmd2mstr_command     ,   
       cache2mstr_command   =>  sig_cache2mstr_command     ,   
       cmd2mstr_cmd_valid   =>  sig_cmd2mstr_cmd_valid   ,   
       mst2cmd_cmd_ready    =>  sig_mst2cmd_cmd_ready    ,   
       mstr2addr_tag        =>  sig_mstr2addr_tag        ,   
       mstr2addr_addr       =>  sig_mstr2addr_addr       ,   
       mstr2addr_len        =>  sig_mstr2addr_len        ,   
       mstr2addr_size       =>  sig_mstr2addr_size       ,   
       mstr2addr_burst      =>  sig_mstr2addr_burst      ,   
       mstr2addr_cache      =>  sig_mstr2addr_cache      ,   
       mstr2addr_user       =>  sig_mstr2addr_user       ,   
       mstr2addr_calc_error =>  sig_mstr2addr_calc_error ,   
       mstr2addr_cmd_cmplt  =>  sig_mstr2addr_cmd_cmplt  ,   
       mstr2addr_cmd_valid  =>  sig_mstr2addr_cmd_valid  ,   
       addr2mstr_cmd_ready  =>  sig_addr2mstr_cmd_ready  ,   
       mstr2data_tag        =>  sig_mstr2data_tag        ,   
       mstr2data_saddr_lsb  =>  sig_mstr2data_saddr_lsb  ,   
       mstr2data_len        =>  sig_mstr2data_len        ,   
       mstr2data_strt_strb  =>  sig_mstr2data_strt_strb  ,   
       mstr2data_last_strb  =>  sig_mstr2data_last_strb  ,   
       mstr2data_sof        =>  sig_mstr2data_drr        ,   
       mstr2data_eof        =>  sig_mstr2data_eof        ,   
       mstr2data_calc_error =>  sig_mstr2data_calc_error ,   
       mstr2data_cmd_cmplt  =>  sig_mstr2data_cmd_last   ,   
       mstr2data_cmd_valid  =>  sig_mstr2data_cmd_valid  ,   
       data2mstr_cmd_ready  =>  sig_data2mstr_cmd_ready  ,   
       calc_error           =>  sig_calc2dm_calc_err         
       
       );
    
    
    
  
  
          
     ------------------------------------------------------------
     -- Instance: I_ADDR_CNTL 
     --
     -- Description:
     --   Address Controller Block  
     --
     ------------------------------------------------------------
      I_ADDR_CNTL : entity axi_sg_v4_1_6.axi_sg_addr_cntl
      generic map (
    
        -- obsoleted   C_ENABlE_WAIT_FOR_DATA       =>  ENABLE_WAIT_FOR_DATA        ,     
        C_ADDR_FIFO_DEPTH            =>  WR_ADDR_CNTL_FIFO_DEPTH     ,
        --C_ADDR_FIFO_DEPTH            =>  S2MM_STSCMD_FIFO_DEPTH      ,     
        C_ADDR_WIDTH                 =>  S2MM_ADDR_WIDTH             ,     
        C_ADDR_ID                    =>  S2MM_AWID_VALUE             ,     
        C_ADDR_ID_WIDTH              =>  S2MM_AWID_WIDTH             ,     
        C_TAG_WIDTH                  =>  C_TAG_WIDTH                      
    
        )
      port map (
    
        primary_aclk                 =>  s2mm_aclk                   ,    
        mmap_reset                   =>  sig_mmap_rst                ,    
        addr2axi_aid                 =>  s2mm_awid                   ,    
        addr2axi_aaddr               =>  s2mm_awaddr                 ,    
        addr2axi_alen                =>  s2mm_awlen                  ,    
        addr2axi_asize               =>  s2mm_awsize                 ,    
        addr2axi_aburst              =>  s2mm_awburst                ,    
        addr2axi_aprot               =>  s2mm_awprot                 ,    
        addr2axi_avalid              =>  s2mm_awvalid                ,    
        addr2axi_acache               =>  open            ,
        addr2axi_auser                =>  open                 ,

        axi2addr_aready              =>  s2mm_awready                ,    
        
        mstr2addr_tag                =>  sig_mstr2addr_tag           ,    
        mstr2addr_addr               =>  sig_mstr2addr_addr          ,    
        mstr2addr_len                =>  sig_mstr2addr_len           ,    
        mstr2addr_size               =>  sig_mstr2addr_size          ,    
        mstr2addr_burst              =>  sig_mstr2addr_burst         ,    
        mstr2addr_cache              =>  sig_mstr2addr_cache         ,    
        mstr2addr_user               =>  sig_mstr2addr_user         ,    
        mstr2addr_cmd_cmplt          =>  sig_mstr2addr_cmd_cmplt     ,    
        mstr2addr_calc_error         =>  sig_mstr2addr_calc_error    ,    
        mstr2addr_cmd_valid          =>  sig_mstr2addr_cmd_valid     ,    
        addr2mstr_cmd_ready          =>  sig_addr2mstr_cmd_ready     ,    
        
        addr2rst_stop_cmplt          =>  sig_addr2rst_stop_cmplt     ,    
 
        allow_addr_req               =>  s2mm_allow_addr_req         ,
        addr_req_posted              =>  s2mm_addr_req_posted        ,
        
        addr2data_addr_posted        =>  sig_addr2data_addr_posted   ,    
        data2addr_data_rdy           =>  sig_data2addr_data_rdy      ,    
        data2addr_stop_req           =>  sig_data2addr_stop_req      ,    
        
        addr2stat_calc_error         =>  sig_addr2wsc_calc_error     ,    
        addr2stat_cmd_fifo_empty     =>  sig_addr2wsc_cmd_fifo_empty      
        );
    
    
    




   
  


      ------------------------------------------------------------
      -- Instance: I_S2MM_STRM_SKID_BUF 
      --
      -- Description:
      --   Instance for the S2MM Skid Buffer which provides for
      -- registerd Slave Stream inputs and supports bi-dir
      -- throttling.  
      --
      ------------------------------------------------------------
--      I_S2MM_STRM_SKID_BUF : entity axi_sg_v4_1_6.axi_sg_skid_buf
--      generic map (
--         
--        C_WDATA_WIDTH  =>  S2MM_SDATA_WIDTH        
--    
--        )
--      port map (
--    
--        -- System Ports
--        aclk           =>  s2mm_aclk             ,  
--        arst           =>  sig_mmap_rst          ,  
--     
--        -- Shutdown control (assert for 1 clk pulse)
--        skid_stop      =>  sig_data2skid_halt    ,  
--     
--        -- Slave Side (Stream Data Input) 
--        s_valid        =>  s2mm_strm_wvalid      ,  
--        s_ready        =>  s2mm_strm_wready      ,  
--        s_data         =>  s2mm_strm_wdata       ,  
--        s_strb         =>  s2mm_strm_wstrb       ,  
--        s_last         =>  s2mm_strm_wlast       ,  
--
--        -- Master Side (Stream Data Output 
--        m_valid        =>  skid2wdc_wvalid       ,  
--        m_ready        =>  wdc2skid_wready       ,  
--        m_data         =>  skid2wdc_wdata        ,  
--        m_strb         =>  skid2wdc_wstrb        ,  
--        m_last         =>  skid2wdc_wlast           
--    
--        );
--    
       
       
       
        
          
 
 
 
 
 
 
 
 
 
 


      ------------------------------------------------------------
      -- Instance: I_WR_DATA_CNTL 
      --
      -- Description:
      --     Write Data Controller Block
      --
      ------------------------------------------------------------
      I_WR_DATA_CNTL : entity axi_sg_v4_1_6.axi_sg_wrdata_cntl
      generic map (
    
        -- obsoleted   C_ENABlE_WAIT_FOR_DATA =>  ENABLE_WAIT_FOR_DATA       , 
        C_REALIGNER_INCLUDED   =>  OMIT_S2MM_DRE              , 
        C_ENABLE_INDET_BTT     =>  OMIT_INDET_BTT             , 
        C_SF_BYTES_RCVD_WIDTH  =>  SF_BYTES_RCVD_WIDTH        , 
        C_SEL_ADDR_WIDTH       =>  SEL_ADDR_WIDTH             , 
        C_DATA_CNTL_FIFO_DEPTH =>  WR_DATA_CNTL_FIFO_DEPTH    , 
        C_MMAP_DWIDTH          =>  S2MM_MDATA_WIDTH           , 
        C_STREAM_DWIDTH        =>  S2MM_SDATA_WIDTH           , 
        C_TAG_WIDTH            =>  C_TAG_WIDTH                , 
        C_FAMILY               =>  C_FAMILY                     
    
        )
      port map (
    
        primary_aclk           =>  s2mm_aclk                  , 
        mmap_reset             =>  sig_mmap_rst               , 
        rst2data_stop_request  =>  sig_rst2all_stop_request   , 
        data2addr_stop_req     =>  sig_data2addr_stop_req     , 
        data2rst_stop_cmplt    =>  sig_data2rst_stop_cmplt    , 
        wr_xfer_cmplt          =>  s2mm_wr_xfer_cmplt         ,
        s2mm_ld_nxt_len        =>  s2mm_ld_nxt_len            ,
        s2mm_wr_len            =>  s2mm_wr_len                ,
        data2skid_saddr_lsb    =>  sig_data2skid_addr_lsb     , 
        data2skid_wdata        =>  sig_skid2axi_wdata,-- sig_data2skid_wdata        , 
        data2skid_wstrb        =>  sig_skid2axi_wstrb,-- sig_data2skid_wstrb        , 
        data2skid_wlast        =>  sig_skid2axi_wlast,--  sig_data2skid_wlast        , 
        data2skid_wvalid       =>  sig_skid2axi_wvalid,--  sig_data2skid_wvalid       , 
        skid2data_wready       =>  sig_axi2skid_wready,--  sig_skid2data_wready       , 
        s2mm_strm_wvalid       =>  s2mm_strm_wvalid, --skid2wdc_wvalid            , 
        s2mm_strm_wready       =>  s2mm_strm_wready, --wdc2skid_wready            , 
        s2mm_strm_wdata        =>  s2mm_strm_wdata, --skid2wdc_wdata             ,     
        s2mm_strm_wstrb        =>  s2mm_strm_wstrb, --skid2wdc_wstrb             ,         
        s2mm_strm_wlast        =>  s2mm_strm_wlast, --skid2wdc_wlast             , 
        s2mm_strm_eop          =>  s2mm_strm_wlast, --skid2wdc_wlast             , 
        s2mm_stbs_asserted     =>  ZEROS_8_BIT                , 
        realign2wdc_eop_error  =>  sig_realign2wdc_eop_error  , 
        mstr2data_tag          =>  sig_mstr2data_tag          , 
        mstr2data_saddr_lsb    =>  sig_mstr2data_saddr_lsb    , 
        mstr2data_len          =>  sig_mstr2data_len          , 
        mstr2data_strt_strb    =>  sig_mstr2data_strt_strb    , 
        mstr2data_last_strb    =>  sig_mstr2data_last_strb    , 
        mstr2data_drr          =>  sig_mstr2data_drr          , 
        mstr2data_eof          =>  sig_mstr2data_eof          , 
        mstr2data_sequential   =>  LOGIC_LOW                  , 
        mstr2data_calc_error   =>  sig_mstr2data_calc_error   , 
        mstr2data_cmd_cmplt    =>  sig_mstr2data_cmd_last     , 
        mstr2data_cmd_valid    =>  sig_mstr2data_cmd_valid    , 
        data2mstr_cmd_ready    =>  sig_data2mstr_cmd_ready    , 
        addr2data_addr_posted  =>  sig_addr2data_addr_posted  , 
        data2addr_data_rdy     =>  sig_data2addr_data_rdy     , 
        data2all_tlast_error   =>  sig_data2all_tlast_error   , 
        data2all_dcntlr_halted =>  sig_data2all_dcntlr_halted , 
        data2skid_halt         =>  sig_data2skid_halt         , 
        data2wsc_tag           =>  sig_data2wsc_tag           , 
        data2wsc_calc_err      =>  sig_data2wsc_calc_err      , 
        data2wsc_last_err      =>  sig_data2wsc_last_err      , 
        data2wsc_cmd_cmplt     =>  sig_data2wsc_cmd_cmplt     , 
        wsc2data_ready         =>  sig_wsc2data_ready         , 
        data2wsc_valid         =>  sig_data2wsc_valid         , 
        data2wsc_eop           =>  sig_data2wsc_eop           , 
        data2wsc_bytes_rcvd    =>  sig_data2wsc_bytes_rcvd    , 
        wsc2mstr_halt_pipe     =>  sig_wsc2mstr_halt_pipe       
       
        );
   
   
    
  
  
          
      ------------------------------------------------------------
      -- Instance: I_S2MM_MMAP_SKID_BUF 
      --
      -- Description:
      --   Instance for the S2MM Skid Buffer which provides for
      -- registered outputs and supports bi-dir throttling. 
      -- 
      -- This Module also provides Write Data Bus Mirroring and WSTRB
      -- Demuxing to match a narrow Stream to a wider MMap Write 
      -- Channel. By doing this in the skid buffer, the resource 
      -- utilization of the skid buffer can be minimized by only
      -- having to buffer/mux the Stream data width, not the MMap
      -- Data width.   
      --
      ------------------------------------------------------------
--       I_S2MM_MMAP_SKID_BUF : entity axi_sg_v4_1_6.axi_sg_skid2mm_buf
--       generic map (
--          
--         C_MDATA_WIDTH    =>  S2MM_MDATA_WIDTH       ,  
--         C_SDATA_WIDTH    =>  S2MM_SDATA_WIDTH       ,  
--         C_ADDR_LSB_WIDTH =>  SEL_ADDR_WIDTH            
--         
--         )
--       port map (
--     
--         -- System Ports
--         ACLK             =>   s2mm_aclk             ,  
--         ARST             =>   sig_stream_rst        ,  
--         
--         -- Slave Side (Wr Data Controller Input Side ) 
--         S_ADDR_LSB       =>   sig_data2skid_addr_lsb,   
--         S_VALID          =>   sig_data2skid_wvalid  ,  
--         S_READY          =>   sig_skid2data_wready  ,  
--         S_Data           =>   sig_data2skid_wdata   ,  
--         S_STRB           =>   sig_data2skid_wstrb   ,  
--         S_Last           =>   sig_data2skid_wlast   ,  
--
--         -- Master Side (MMap Write Data Output Side) 
--         M_VALID          =>   sig_skid2axi_wvalid   ,  
--         M_READY          =>   sig_axi2skid_wready   ,  
--         M_Data           =>   sig_skid2axi_wdata    ,  
--         M_STRB           =>   sig_skid2axi_wstrb    ,  
--         M_Last           =>   sig_skid2axi_wlast       
--     
--         );
--                              
                              
                              
                              
                           
  end implementation;


-- *************************************************************************
--                                                                      
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:        axi_sg.vhd
--
-- Description:     
--  Top level VHDL wrapper for the AXI DataMover                
--                  
--                  
--                  
--                  
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;



library axi_sg_v4_1_6;
use axi_sg_v4_1_6.axi_sg_mm2s_basic_wrap;
use axi_sg_v4_1_6.axi_sg_s2mm_basic_wrap;


-------------------------------------------------------------------------------

entity axi_sg_datamover is
  generic (
    C_INCLUDE_MM2S              : Integer range 0 to  2 :=  2;
       -- Specifies the type of MM2S function to include
       -- 0 = Omit MM2S functionality
       -- 1 = Full MM2S Functionality
       -- 2 = Basic MM2S functionality
       
    C_M_AXI_MM2S_ARID           : Integer range 0 to  255 :=  0;
       -- Specifies the constant value to output on 
       -- the ARID output port
       
    C_M_AXI_MM2S_ID_WIDTH       : Integer range 1 to  8 :=  4;
       -- Specifies the width of the MM2S ID port 
       
    C_M_AXI_MM2S_ADDR_WIDTH     : Integer range 32 to  64 :=  32;
       -- Specifies the width of the MMap Read Address Channel 
       -- Address bus
       
    C_M_AXI_MM2S_DATA_WIDTH     : Integer range 32 to 1024 :=  32;
       -- Specifies the width of the MMap Read Data Channel
       -- data bus
    
    C_M_AXIS_MM2S_TDATA_WIDTH   : Integer range 8 to 1024 :=  32;
       -- Specifies the width of the MM2S Master Stream Data 
       -- Channel data bus
    
    C_INCLUDE_MM2S_STSFIFO      : Integer range 0 to  1 :=  1;
       -- Specifies if a Status FIFO is to be implemented
       -- 0 = Omit MM2S Status FIFO
       -- 1 = Include MM2S Status FIFO
       
    C_MM2S_STSCMD_FIFO_DEPTH    : Integer range 1 to 16 :=  4;
       -- Specifies the depth of the MM2S Command FIFO and the 
       -- optional Status FIFO
       -- Valid values are 1,4,8,16
       
    C_MM2S_STSCMD_IS_ASYNC      : Integer range 0 to  1 :=  0;
       -- Specifies if the Status and Command interfaces need to
       -- be asynchronous to the primary data path clocking
       -- 0 = Use same clocking as data path
       -- 1 = Use special Status/Command clock for the interfaces
       
    C_INCLUDE_MM2S_DRE          : Integer range 0 to  1 :=  1;
       -- Specifies if DRE is to be included in the MM2S function 
       -- 0 = Omit DRE
       -- 1 = Include DRE
    
    C_MM2S_BURST_SIZE           : Integer range 16 to  256 :=  16;
       -- Specifies the max number of databeats to use for MMap
       -- burst transfers by the MM2S function 
    
    C_MM2S_BTT_USED             : Integer range 8 to  23 :=  16;
      -- Specifies the number of bits used from the BTT field
      -- of the input Command Word of the MM2S Command Interface 
   
    C_MM2S_ADDR_PIPE_DEPTH      : Integer range 1 to 30 := 3;
      -- This parameter specifies the depth of the MM2S internal 
      -- child command queues in the Read Address Controller and 
      -- the Read Data Controller. Increasing this value will 
      -- allow more Read Addresses to be issued to the AXI4 Read 
      -- Address Channel before receipt of the associated read 
      -- data on the Read Data Channel.

    C_MM2S_INCLUDE_SF           : Integer range 0 to 1 := 1 ;
      -- This parameter specifies the inclusion/omission of the
      -- MM2S (Read) Store and Forward function
      -- 0 = Omit MM2S Store and Forward
      -- 1 = Include MM2S Store and Forward
    
  
   
     
    
    C_INCLUDE_S2MM              : Integer range 0 to  4 :=  2;
       -- Specifies the type of S2MM function to include
       -- 0 = Omit S2MM functionality
       -- 1 = Full S2MM Functionality
       -- 2 = Basic S2MM functionality
       
    C_M_AXI_S2MM_AWID           : Integer range 0 to  255 :=  1;
       -- Specifies the constant value to output on 
       -- the ARID output port
       
    C_M_AXI_S2MM_ID_WIDTH       : Integer range 1 to  8 :=  4;
       -- Specifies the width of the S2MM ID port 
       
    C_M_AXI_S2MM_ADDR_WIDTH     : Integer range 32 to  64 :=  32;
       -- Specifies the width of the MMap Read Address Channel 
       -- Address bus
       
    C_M_AXI_S2MM_DATA_WIDTH     : Integer range 32 to 1024 :=  32;
       -- Specifies the width of the MMap Read Data Channel
       -- data bus
    
    C_S_AXIS_S2MM_TDATA_WIDTH   : Integer range 8 to 1024 :=  32;
       -- Specifies the width of the S2MM Master Stream Data 
       -- Channel data bus
    
    C_INCLUDE_S2MM_STSFIFO      : Integer range 0 to  1 :=  1;
       -- Specifies if a Status FIFO is to be implemented
       -- 0 = Omit S2MM Status FIFO
       -- 1 = Include S2MM Status FIFO
       
    C_S2MM_STSCMD_FIFO_DEPTH    : Integer range 1 to 16 :=  4;
       -- Specifies the depth of the S2MM Command FIFO and the 
       -- optional Status FIFO
       -- Valid values are 1,4,8,16
       
    C_S2MM_STSCMD_IS_ASYNC      : Integer range 0 to  1 :=  0;
       -- Specifies if the Status and Command interfaces need to
       -- be asynchronous to the primary data path clocking
       -- 0 = Use same clocking as data path
       -- 1 = Use special Status/Command clock for the interfaces
       
    C_INCLUDE_S2MM_DRE          : Integer range 0 to  1 :=  1;
       -- Specifies if DRE is to be included in the S2MM function 
       -- 0 = Omit DRE
       -- 1 = Include DRE
    
    C_S2MM_BURST_SIZE           : Integer range 16 to  256 :=  16;
       -- Specifies the max number of databeats to use for MMap
       -- burst transfers by the S2MM function 
    
    C_S2MM_BTT_USED             : Integer range 8 to  23 :=  16;
      -- Specifies the number of bits used from the BTT field
      -- of the input Command Word of the S2MM Command Interface 
   
    C_S2MM_SUPPORT_INDET_BTT    : Integer range 0 to  1 :=  0;
       -- Specifies if support for indeterminate packet lengths
       -- are to be received on the input Stream interface 
       -- 0 = Omit support (User MUST transfer the exact number of  
       --     bytes on the Stream interface as specified in the BTT 
       --     field of the Corresponding DataMover Command)
       -- 1 = Include support for indeterminate packet lengths
       --     This causes FIFOs to be added and "Store and Forward" 
       --     behavior of the S2MM function

    C_S2MM_ADDR_PIPE_DEPTH      : Integer range 1 to 30 := 3;
        -- This parameter specifies the depth of the S2MM internal 
        -- address pipeline queues in the Write Address Controller 
        -- and the Write Data Controller. Increasing this value will 
        -- allow more Write Addresses to be issued to the AXI4 Write 
        -- Address Channel before transmission of the associated  
        -- write data on the Write Data Channel.

    C_S2MM_INCLUDE_SF           : Integer range 0 to 1 := 1 ;
      -- This parameter specifies the inclusion/omission of the
      -- S2MM (Write) Store and Forward function
      -- 0 = Omit S2MM Store and Forward
      -- 1 = Include S2MM Store and Forward
 
    C_ENABLE_MULTI_CHANNEL             : integer range 0 to 1 := 1; 

    C_ENABLE_EXTRA_FIELD               : integer range 0 to 1 := 0;
     
    C_FAMILY                    : String := "virtex7"
       -- Specifies the target FPGA family type
    
    
    );
  port (

      -- MM2S Primary Clock input ----------------------------------
      m_axi_mm2s_aclk           : in  std_logic;                  --
         -- Primary synchronization clock for the Master side     --
         -- interface and internal logic. It is also used         --
         -- for the User interface synchronization when           --
         -- C_STSCMD_IS_ASYNC = 0.                                --
                                                                  --
      -- MM2S Primary Reset input                                 --
      m_axi_mm2s_aresetn        : in  std_logic;                  --
         -- Reset used for the internal master logic              --
      --------------------------------------------------------------

      sg_ctl                    : in std_logic_vector (7 downto 0) ; 
     
      -- MM2S Halt request input control --------------------
      mm2s_halt                 : in  std_logic;           --
         -- Active high soft shutdown request              --
                                                           --
      -- MM2S Halt Complete status flag                    --
      mm2s_halt_cmplt           : Out std_logic;           --
         -- Active high soft shutdown complete status      --
      -------------------------------------------------------
      
      
      -- Error discrete output -------------------------
      mm2s_err                  : Out std_logic;      --
         -- Composite Error indication                --
      --------------------------------------------------
     
     
      -- Memory Map to Stream Command FIFO and Status FIFO I/O ---------
      m_axis_mm2s_cmdsts_aclk    : in  std_logic;                     --
      -- Secondary Clock input for async CMD/Status interface         --
                                                                      --
      m_axis_mm2s_cmdsts_aresetn : in  std_logic;                     --
        -- Secondary Reset input for async CMD/Status interface       --
      ------------------------------------------------------------------
      
      
      
      -- User Command Interface Ports (AXI Stream) -------------------------------------------------
      s_axis_mm2s_cmd_tvalid     : in  std_logic;                                                 -- 
      s_axis_mm2s_cmd_tready     : out std_logic;                                                 -- 
      s_axis_mm2s_cmd_tdata      : in  std_logic_vector(((1+C_ENABLE_MULTI_CHANNEL)*C_M_AXI_MM2S_ADDR_WIDTH+40)-1 downto 0); --
      ---------------------------------------------------------------------------------------------- 
      
      -- User Status Interface Ports (AXI Stream) ------------------------
      m_axis_mm2s_sts_tvalid     : out std_logic;                       --
      m_axis_mm2s_sts_tready     : in  std_logic;                       --
      m_axis_mm2s_sts_tdata      : out std_logic_vector(7 downto 0);    --
      m_axis_mm2s_sts_tkeep      : out std_logic_vector(0 downto 0);    --
      m_axis_mm2s_sts_tlast      : out std_logic;                       --
      --------------------------------------------------------------------
      
      
      -- Address Posting contols -----------------------
      mm2s_allow_addr_req       : in  std_logic;      --                 
      mm2s_addr_req_posted      : out std_logic;      --                 
      mm2s_rd_xfer_cmplt        : out std_logic;      --
      --------------------------------------------------                 
      
      
      -- MM2S AXI Address Channel I/O  --------------------------------------------------
      m_axi_mm2s_arid     : out std_logic_vector(C_M_AXI_MM2S_ID_WIDTH-1 downto 0);    --
         -- AXI Address Channel ID output                                              --
                                                                                       --
      m_axi_mm2s_araddr   : out std_logic_vector(C_M_AXI_MM2S_ADDR_WIDTH-1 downto 0);  --
         -- AXI Address Channel Address output                                         --
                                                                                       --
      m_axi_mm2s_arlen    : out std_logic_vector(7 downto 0);                          --
         -- AXI Address Channel LEN output                                             --
         -- Sized to support 256 data beat bursts                                      --
                                                                                       --
      m_axi_mm2s_arsize   : out std_logic_vector(2 downto 0);                          --
         -- AXI Address Channel SIZE output                                            --
                                                                                       --
      m_axi_mm2s_arburst  : out std_logic_vector(1 downto 0);                          --
         -- AXI Address Channel BURST output                                           --
                                                                                       --
      m_axi_mm2s_arprot   : out std_logic_vector(2 downto 0);                          --
         -- AXI Address Channel PROT output                                            --
                                                                                       --
      m_axi_mm2s_arcache  : out std_logic_vector(3 downto 0);                          --
         -- AXI Address Channel CACHE output                                           --

      m_axi_mm2s_aruser   : out std_logic_vector(3 downto 0);                          --
         -- AXI Address Channel USER output                                           --
                                                                                       --
      m_axi_mm2s_arvalid  : out std_logic;                                             --
         -- AXI Address Channel VALID output                                           --
                                                                                       --
      m_axi_mm2s_arready  : in  std_logic;                                             --
         -- AXI Address Channel READY input                                            --
      -----------------------------------------------------------------------------------  
                                              
                                              
                                              
                                              
      -- Currently unsupported AXI Address Channel output signals -------
        -- m_axi_mm2s_alock   : out std_logic_vector(2 downto 0);      --
        -- m_axi_mm2s_acache  : out std_logic_vector(4 downto 0);      --
        -- m_axi_mm2s_aqos    : out std_logic_vector(3 downto 0);      --
        -- m_axi_mm2s_aregion : out std_logic_vector(3 downto 0);      --
      -------------------------------------------------------------------
  
  
  
  
      
      -- MM2S AXI MMap Read Data Channel I/O  ------------------------------------------------
      m_axi_mm2s_rdata        : In  std_logic_vector(C_M_AXI_MM2S_DATA_WIDTH-1 downto 0);   -- 
      m_axi_mm2s_rresp        : In  std_logic_vector(1 downto 0);                           -- 
      m_axi_mm2s_rlast        : In  std_logic;                                              -- 
      m_axi_mm2s_rvalid       : In  std_logic;                                              -- 
      m_axi_mm2s_rready       : Out std_logic;                                              -- 
      ----------------------------------------------------------------------------------------
      
      
      -- MM2S AXI Master Stream Channel I/O  -------------------------------------------------------
      m_axis_mm2s_tdata       : Out  std_logic_vector(C_M_AXIS_MM2S_TDATA_WIDTH-1 downto 0);      -- 
      m_axis_mm2s_tkeep       : Out  std_logic_vector((C_M_AXIS_MM2S_TDATA_WIDTH/8)-1 downto 0);  -- 
      m_axis_mm2s_tlast       : Out  std_logic;                                                   -- 
      m_axis_mm2s_tvalid      : Out  std_logic;                                                   -- 
      m_axis_mm2s_tready      : In   std_logic;                                                   -- 

----------------------------------------------------------------------------------------------
      
      -- Testing Support I/O --------------------------------------------------------
      mm2s_dbg_sel            : in  std_logic_vector( 3 downto 0);                 -- 
      mm2s_dbg_data           : out std_logic_vector(31 downto 0) ;                -- 
      -------------------------------------------------------------------------------
      

 
      
      
      -- S2MM Primary Clock input ---------------------------------
      m_axi_s2mm_aclk         : in  std_logic;                   --
         -- Primary synchronization clock for the Master side    --
         -- interface and internal logic. It is also used        --
         -- for the User interface synchronization when          --
         -- C_STSCMD_IS_ASYNC = 0.                               --
                                                                 --
      -- S2MM Primary Reset input                                --
      m_axi_s2mm_aresetn      : in  std_logic;                   --
         -- Reset used for the internal master logic             --
      -------------------------------------------------------------                                                           
 
     
      -- S2MM Halt request input control ------------------
      s2mm_halt               : in  std_logic;           --
         -- Active high soft shutdown request            --
                                                         --
      -- S2MM Halt Complete status flag                  --
      s2mm_halt_cmplt         : out std_logic;           --
         -- Active high soft shutdown complete status    --
      -----------------------------------------------------
      
      
      -- S2MM Error discrete output ------------------
      s2mm_err                : Out std_logic;      --
         -- Composite Error indication              --
      ------------------------------------------------
      
      
     
      -- Memory Map to Stream Command FIFO and Status FIFO I/O -----------------
      m_axis_s2mm_cmdsts_awclk   : in  std_logic;                             --
      -- Secondary Clock input for async CMD/Status interface                 --
                                                                              --
      m_axis_s2mm_cmdsts_aresetn : in  std_logic;                             --
        -- Secondary Reset input for async CMD/Status interface               --
      --------------------------------------------------------------------------  
      
      -- User Command Interface Ports (AXI Stream) --------------------------------------------------
      s_axis_s2mm_cmd_tvalid     : in  std_logic;                                                  --
      s_axis_s2mm_cmd_tready     : out std_logic;                                                  --
      s_axis_s2mm_cmd_tdata      : in  std_logic_vector(((1+C_ENABLE_MULTI_CHANNEL)*C_M_AXI_S2MM_ADDR_WIDTH+40)-1 downto 0);  --
      -----------------------------------------------------------------------------------------------
                                                                                                     
                                                                                                     
      -- User Status Interface Ports (AXI Stream) -----------------------------------------------------------
      m_axis_s2mm_sts_tvalid     : out std_logic;                                                          --
      m_axis_s2mm_sts_tready     : in  std_logic;                                                          --
      m_axis_s2mm_sts_tdata      : out std_logic_vector(((C_S2MM_SUPPORT_INDET_BTT*24)+8)-1 downto 0);     --
      m_axis_s2mm_sts_tkeep      : out std_logic_vector((((C_S2MM_SUPPORT_INDET_BTT*24)+8)/8)-1 downto 0); --
      m_axis_s2mm_sts_tlast      : out std_logic;                                                          --
      -------------------------------------------------------------------------------------------------------
      
      -- Address posting controls -----------------------------------------
      s2mm_allow_addr_req    : in  std_logic;                            --
      s2mm_addr_req_posted   : out std_logic;                            --
      s2mm_wr_xfer_cmplt     : out std_logic;                            --
      s2mm_ld_nxt_len        : out std_logic;                            --
      s2mm_wr_len            : out std_logic_vector(7 downto 0);         --
      ---------------------------------------------------------------------
      
     
      -- S2MM AXI Address Channel I/O  ----------------------------------------------------
      m_axi_s2mm_awid       : out std_logic_vector(C_M_AXI_S2MM_ID_WIDTH-1 downto 0);    --
         -- AXI Address Channel ID output                                                --
                                                                                         --
      m_axi_s2mm_awaddr     : out std_logic_vector(C_M_AXI_S2MM_ADDR_WIDTH-1 downto 0);  --
         -- AXI Address Channel Address output                                           --
                                                                                         --
      m_axi_s2mm_awlen      : out std_logic_vector(7 downto 0);                          --
         -- AXI Address Channel LEN output                                               --
         -- Sized to support 256 data beat bursts                                        --
                                                                                         --
      m_axi_s2mm_awsize     : out std_logic_vector(2 downto 0);                          --
         -- AXI Address Channel SIZE output                                              --
                                                                                         --
      m_axi_s2mm_awburst    : out std_logic_vector(1 downto 0);                          --
         -- AXI Address Channel BURST output                                             --
                                                                                         --
      m_axi_s2mm_awprot     : out std_logic_vector(2 downto 0);                          --
         -- AXI Address Channel PROT output                                              --
                                                                                         --
      m_axi_s2mm_awcache    : out std_logic_vector(3 downto 0);                          --
         -- AXI Address Channel CACHE output                                             --

      m_axi_s2mm_awuser    : out std_logic_vector(3 downto 0);                          --
         -- AXI Address Channel USER output                                             --
                                                                                         --
      m_axi_s2mm_awvalid    : out std_logic;                                             --
         -- AXI Address Channel VALID output                                             --
                                                                                         --
      m_axi_s2mm_awready    : in  std_logic;                                             --
         -- AXI Address Channel READY input                                              --
      -------------------------------------------------------------------------------------   
      
        
      -- Currently unsupported AXI Address Channel output signals -------
        -- m_axi_s2mm__awlock   : out std_logic_vector(2 downto 0);    --
        -- m_axi_s2mm__awcache  : out std_logic_vector(4 downto 0);    --
        -- m_axi_s2mm__awqos    : out std_logic_vector(3 downto 0);    --
        -- m_axi_s2mm__awregion : out std_logic_vector(3 downto 0);    --
      -------------------------------------------------------------------
  
  
  
  
      
      -- S2MM AXI MMap Write Data Channel I/O  --------------------------------------------------
      m_axi_s2mm_wdata        : Out  std_logic_vector(C_M_AXI_S2MM_DATA_WIDTH-1 downto 0);     --
      m_axi_s2mm_wstrb        : Out  std_logic_vector((C_M_AXI_S2MM_DATA_WIDTH/8)-1 downto 0); --
      m_axi_s2mm_wlast        : Out  std_logic;                                                --
      m_axi_s2mm_wvalid       : Out  std_logic;                                                --
      m_axi_s2mm_wready       : In   std_logic;                                                --
      -------------------------------------------------------------------------------------------
      
      
      -- S2MM AXI MMap Write response Channel I/O  -------------------------
      m_axi_s2mm_bresp        : In   std_logic_vector(1 downto 0);        --
      m_axi_s2mm_bvalid       : In   std_logic;                           --
      m_axi_s2mm_bready       : Out  std_logic;                           --
      ----------------------------------------------------------------------
                                                                          
                                                                          
      -- S2MM AXI Slave Stream Channel I/O  -------------------------------------------------------
      s_axis_s2mm_tdata       : In  std_logic_vector(C_S_AXIS_S2MM_TDATA_WIDTH-1 downto 0);      --
      s_axis_s2mm_tkeep       : In  std_logic_vector((C_S_AXIS_S2MM_TDATA_WIDTH/8)-1 downto 0);  --
      s_axis_s2mm_tlast       : In  std_logic;                                                   --
      s_axis_s2mm_tvalid      : In  std_logic;                                                   --
      s_axis_s2mm_tready      : Out std_logic;                                                   --
      ---------------------------------------------------------------------------------------------
      
      -- Testing Support I/O  ------------------------------------------------
      s2mm_dbg_sel            : in  std_logic_vector( 3 downto 0);          --
      s2mm_dbg_data           : out std_logic_vector(31 downto 0)           --
      ------------------------------------------------------------------------
      
 
    );

end entity axi_sg_datamover;


architecture implementation of axi_sg_datamover is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


 
   
   -- Function Declarations 
   
   -------------------------------------------------------------------
   -- Function
   --
   -- Function Name: funct_clip_brst_len
   --
   -- Function Description:
   -- This function is used to limit the parameterized max burst 
   -- databeats when the tranfer data width is 256 bits or greater.
   -- This is required to keep from crossing the 4K byte xfer
   -- boundary required by AXI. This process is further complicated
   -- by the inclusion/omission of upsizers or downsizers in the 
   -- data path.
   --
   -------------------------------------------------------------------
   function funct_clip_brst_len (param_burst_beats         : integer;
                                 mmap_transfer_bit_width   : integer;
                                 stream_transfer_bit_width : integer;
                                 down_up_sizers_enabled    : integer) return integer is
   
     constant FCONST_SIZERS_ENABLED : boolean := (down_up_sizers_enabled > 0);
     Variable fvar_max_burst_dbeats : Integer;
     
     
     
   begin
      
      -- coverage off 
      if (FCONST_SIZERS_ENABLED) then -- use MMap dwidth for calc
      
        If (mmap_transfer_bit_width <= 128) Then -- allowed
      
          fvar_max_burst_dbeats := param_burst_beats;
        
        Elsif (mmap_transfer_bit_width <= 256) Then
        
           If (param_burst_beats <= 128) Then
    
             fvar_max_burst_dbeats := param_burst_beats;
           
           Else 
           
             fvar_max_burst_dbeats := 128;
           
           End if;
        
        Elsif (mmap_transfer_bit_width <= 512) Then
        
           If (param_burst_beats <= 64) Then
    
             fvar_max_burst_dbeats := param_burst_beats;
           
           Else 
           
             fvar_max_burst_dbeats := 64;
           
           End if;
        
        Else -- 1024 bit mmap width case
        
           If (param_burst_beats <= 32) Then
    
             fvar_max_burst_dbeats := param_burst_beats;
           
           Else 
           
             fvar_max_burst_dbeats := 32;
           
           End if;
        
        
        End if;
      
      else                            -- use stream dwidth for calc
      
        If (stream_transfer_bit_width <= 128) Then -- allowed
      
          fvar_max_burst_dbeats := param_burst_beats;
        
        Elsif (stream_transfer_bit_width <= 256) Then
        
           If (param_burst_beats <= 128) Then
    
             fvar_max_burst_dbeats := param_burst_beats;
           
           Else 
           
             fvar_max_burst_dbeats := 128;
           
           End if;
        
        Elsif (stream_transfer_bit_width <= 512) Then
        
           If (param_burst_beats <= 64) Then
    
             fvar_max_burst_dbeats := param_burst_beats;
           
           Else 
           
             fvar_max_burst_dbeats := 64;
           
           End if;
        
        Else -- 1024 bit stream width case
        
           If (param_burst_beats <= 32) Then
    
             fvar_max_burst_dbeats := param_burst_beats;
           
           Else 
           
             fvar_max_burst_dbeats := 32;
           
           End if;
       -- coverage on 
        
        End if;
      
      end if;
      
      
      
      Return (fvar_max_burst_dbeats);
        
        
   end function funct_clip_brst_len;
   
   
   
    
   -------------------------------------------------------------------
   -- Function
   --
   -- Function Name: funct_fix_depth_16
   --
   -- Function Description:
   -- This function is used to fix the Command and Status FIFO depths to
   -- 16 entries when Async clocking mode is enabled. This is required
   -- due to the way the async_fifo_fg.vhd design in proc_common is 
   -- implemented.
   -------------------------------------------------------------------
   function funct_fix_depth_16 (async_clocking_mode : integer;
                                requested_depth     : integer) return integer is
   
     Variable fvar_depth_2_use : Integer;
   
   begin
       -- coverage off 
      If (async_clocking_mode = 1) Then -- async mode so fix at 16
      
        fvar_depth_2_use := 16;
      
      Elsif (requested_depth > 16) Then -- limit at 16
      
        fvar_depth_2_use := 16;

       -- coverage on 

      Else -- use requested depth
        
        fvar_depth_2_use := requested_depth;
      
      End if;
      
      Return (fvar_depth_2_use);
        
        
   end function funct_fix_depth_16;
   
   
   
    
   -------------------------------------------------------------------
   -- Function
   --
   -- Function Name: funct_get_min_btt_width
   --
   -- Function Description:
   --   This function calculates the minimum required value  
   -- for the used width of the command BTT field. 
   --
   -------------------------------------------------------------------
   function funct_get_min_btt_width (max_burst_beats : integer;
                                     bytes_per_beat  : integer ) return integer is
   
     Variable var_min_btt_needed      : Integer;
     Variable var_max_bytes_per_burst : Integer;
   
   
   begin
   
     var_max_bytes_per_burst := max_burst_beats*bytes_per_beat;
     
     -- coverage off 
     if (var_max_bytes_per_burst <= 16) then
     
        var_min_btt_needed := 5;
     
     elsif (var_max_bytes_per_burst <= 32) then
     
        var_min_btt_needed := 6;
     -- coverage on
     elsif (var_max_bytes_per_burst <= 64) then
     
        var_min_btt_needed := 7;
     -- coverage off
     elsif (var_max_bytes_per_burst <= 128) then
     
        var_min_btt_needed := 8;
     
     elsif (var_max_bytes_per_burst <= 256) then
     
        var_min_btt_needed := 9;
     
     elsif (var_max_bytes_per_burst <= 512) then
     
        var_min_btt_needed := 10;
     
     elsif (var_max_bytes_per_burst <= 1024) then
     
        var_min_btt_needed := 11;
     
     elsif (var_max_bytes_per_burst <= 2048) then
     
        var_min_btt_needed := 12;
     
     elsif (var_max_bytes_per_burst <= 4096) then
     
        var_min_btt_needed := 13;
     
     else   -- 8K byte range
     
        var_min_btt_needed := 14;
          
     end if;
     
     -- coverage on
     
     Return (var_min_btt_needed);
   
     
   end function funct_get_min_btt_width;
   
   
   -------------------------------------------------------------------
   -- Function
   --
   -- Function Name: funct_get_xfer_bytes_per_dbeat
   --
   -- Function Description:
   --  Calculates the nuber of bytes that will transfered per databeat
   -- on the AXI4 MMap Bus.
   --
   -------------------------------------------------------------------
   function funct_get_xfer_bytes_per_dbeat (mmap_transfer_bit_width   : integer;
                                            stream_transfer_bit_width : integer;
                                            down_up_sizers_enabled    : integer) return integer is
     
     Variable temp_bytes_per_dbeat : Integer := 4;
   
   begin
     -- coverage off 
     if (down_up_sizers_enabled > 0) then  -- down/up sizers are in use, use full mmap dwidth
     
        temp_bytes_per_dbeat := mmap_transfer_bit_width/8;
     -- coverage on
     else                                  -- No down/up sizers so use Stream data width
     
        temp_bytes_per_dbeat := stream_transfer_bit_width/8;
     
     end if;
     
   
     Return (temp_bytes_per_dbeat);
   
   
   
   end function funct_get_xfer_bytes_per_dbeat;
   
   
   -------------------------------------------------------------------
   -- Function
   --
   -- Function Name: funct_fix_btt_used
   --
   -- Function Description:
   --  THis function makes sure the BTT width used is at least the
   -- minimum needed.
   --
   -------------------------------------------------------------------
   function funct_fix_btt_used (requested_btt_width : integer;
                                min_btt_width       : integer) return integer is
   
     Variable var_corrected_btt_width : Integer;
   
   begin
   
     -- coverage off 
     If (requested_btt_width < min_btt_width) Then
         
       var_corrected_btt_width :=  min_btt_width;
     -- coverage on
     else
         
       var_corrected_btt_width :=  requested_btt_width;
     
     End if;
     
     
     Return (var_corrected_btt_width);
   
     
   end function funct_fix_btt_used;
   
   -------------------------------------------------------------------
   -- Constant Declarations 
   -------------------------------------------------------------------
    
   Constant MM2S_TAG_WIDTH         : integer := 4;
   Constant S2MM_TAG_WIDTH         : integer := 4;
   
   Constant MM2S_DOWNSIZER_ENABLED : integer := C_MM2S_INCLUDE_SF;
   Constant S2MM_UPSIZER_ENABLED   : integer := C_S2MM_INCLUDE_SF + C_S2MM_SUPPORT_INDET_BTT;
   
   
   
   Constant MM2S_MAX_BURST_BEATS    : integer   := funct_clip_brst_len(C_MM2S_BURST_SIZE,
                                                                       C_M_AXI_MM2S_DATA_WIDTH,
                                                                       C_M_AXIS_MM2S_TDATA_WIDTH,
                                                                       MM2S_DOWNSIZER_ENABLED);
   
   Constant S2MM_MAX_BURST_BEATS    : integer   := funct_clip_brst_len(C_S2MM_BURST_SIZE,
                                                                       C_M_AXI_S2MM_DATA_WIDTH,
                                                                       C_S_AXIS_S2MM_TDATA_WIDTH,
                                                                       S2MM_UPSIZER_ENABLED);
   
   
   Constant MM2S_CMDSTS_FIFO_DEPTH  : integer := funct_fix_depth_16(C_MM2S_STSCMD_IS_ASYNC,
                                                                    C_MM2S_STSCMD_FIFO_DEPTH);
   
   Constant S2MM_CMDSTS_FIFO_DEPTH  : integer := funct_fix_depth_16(C_S2MM_STSCMD_IS_ASYNC,
                                                                    C_S2MM_STSCMD_FIFO_DEPTH);
   
 
   
   
   Constant MM2S_BYTES_PER_BEAT     : integer := funct_get_xfer_bytes_per_dbeat(C_M_AXI_MM2S_DATA_WIDTH,
                                                                                C_M_AXIS_MM2S_TDATA_WIDTH,
                                                                                MM2S_DOWNSIZER_ENABLED);
   
   Constant MM2S_MIN_BTT_NEEDED     : integer := funct_get_min_btt_width(MM2S_MAX_BURST_BEATS,
                                                                         MM2S_BYTES_PER_BEAT);
   
   Constant MM2S_CORRECTED_BTT_USED : integer := funct_fix_btt_used(C_MM2S_BTT_USED,
                                                                    MM2S_MIN_BTT_NEEDED);
   
   
   Constant S2MM_BYTES_PER_BEAT     : integer := funct_get_xfer_bytes_per_dbeat(C_M_AXI_S2MM_DATA_WIDTH,
                                                                                C_S_AXIS_S2MM_TDATA_WIDTH,
                                                                                S2MM_UPSIZER_ENABLED);
   
   Constant S2MM_MIN_BTT_NEEDED     : integer := funct_get_min_btt_width(S2MM_MAX_BURST_BEATS,
                                                                         S2MM_BYTES_PER_BEAT);
   
   Constant S2MM_CORRECTED_BTT_USED : integer := funct_fix_btt_used(C_S2MM_BTT_USED,
                                                                    S2MM_MIN_BTT_NEEDED);
 
 
  -- Signals
   signal sig_mm2s_tstrb     : std_logic_vector((C_M_AXIS_MM2S_TDATA_WIDTH/8)-1 downto 0) := (others => '0');
   signal sig_mm2s_sts_tstrb : std_logic_vector(0 downto 0) := (others => '0');
   signal sig_s2mm_tstrb     : std_logic_vector((C_S_AXIS_S2MM_TDATA_WIDTH/8)-1 downto 0) := (others => '0');
   signal sig_s2mm_sts_tstrb : std_logic_vector((((C_S2MM_SUPPORT_INDET_BTT*24)+8)/8)-1 downto 0) := (others => '0');
 
 
 

begin --(architecture implementation)


  
  
  -------------------------------------------------------------
  -- Conversion to tkeep for external stream connnections
  -------------------------------------------------------------
  
  -- MM2S Stream Output
  m_axis_mm2s_tkeep     <= sig_mm2s_tstrb     ;
  
  -- MM2S Status Stream Output    
  m_axis_mm2s_sts_tkeep <= sig_mm2s_sts_tstrb ;
  
  
  -- S2MM Stream Input
  sig_s2mm_tstrb        <= s_axis_s2mm_tkeep  ;
  
  -- S2MM Status Stream Output    
  m_axis_s2mm_sts_tkeep <= sig_s2mm_sts_tstrb ;
  
  


  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: GEN_MM2S_BASIC
  --
  -- If Generate Description:
  --  Instantiate the MM2S Basic Wrapper 
  --
  --
  ------------------------------------------------------------
  GEN_MM2S_BASIC : if (C_INCLUDE_MM2S = 2) generate
  
  
     begin
  
       ------------------------------------------------------------
       -- Instance: I_MM2S_BASIC_WRAPPER 
       --
       -- Description:
       -- Read Basic Wrapper Instance    
       --
       ------------------------------------------------------------
        I_MM2S_BASIC_WRAPPER : entity axi_sg_v4_1_6.axi_sg_mm2s_basic_wrap
        generic map (
      
          C_INCLUDE_MM2S           =>  C_INCLUDE_MM2S             ,
          C_MM2S_ARID              =>  C_M_AXI_MM2S_ARID          ,   
          C_MM2S_ID_WIDTH          =>  C_M_AXI_MM2S_ID_WIDTH      ,
          C_MM2S_ADDR_WIDTH        =>  C_M_AXI_MM2S_ADDR_WIDTH    ,  
          C_MM2S_MDATA_WIDTH       =>  C_M_AXI_MM2S_DATA_WIDTH    ,  
          C_MM2S_SDATA_WIDTH       =>  C_M_AXIS_MM2S_TDATA_WIDTH  ,  
          C_INCLUDE_MM2S_STSFIFO   =>  C_INCLUDE_MM2S_STSFIFO     ,
          C_MM2S_STSCMD_FIFO_DEPTH =>  MM2S_CMDSTS_FIFO_DEPTH     ,
          C_MM2S_STSCMD_IS_ASYNC   =>  C_MM2S_STSCMD_IS_ASYNC     ,
          C_INCLUDE_MM2S_DRE       =>  C_INCLUDE_MM2S_DRE         ,
          C_MM2S_BURST_SIZE        =>  MM2S_MAX_BURST_BEATS       ,
          C_MM2S_BTT_USED          =>  MM2S_CORRECTED_BTT_USED    ,
          C_MM2S_ADDR_PIPE_DEPTH   =>  C_MM2S_ADDR_PIPE_DEPTH     ,
          C_TAG_WIDTH              =>  MM2S_TAG_WIDTH             ,
          C_ENABLE_MULTI_CHANNEL          =>  C_ENABLE_MULTI_CHANNEL            ,
          C_ENABLE_EXTRA_FIELD     =>  C_ENABLE_EXTRA_FIELD,
          C_FAMILY                 =>  C_FAMILY
          )
        port map (
      
          mm2s_aclk                =>  m_axi_mm2s_aclk            ,  
          mm2s_aresetn             =>  m_axi_mm2s_aresetn         ,  
          sg_ctl                   =>  sg_ctl                     ,          
 
          mm2s_halt                =>  mm2s_halt                  ,
          mm2s_halt_cmplt          =>  mm2s_halt_cmplt            ,          
          mm2s_err                 =>  mm2s_err                   ,  
          mm2s_cmdsts_awclk        =>  m_axis_mm2s_cmdsts_aclk    ,  
          mm2s_cmdsts_aresetn      =>  m_axis_mm2s_cmdsts_aresetn ,  
                                   
          mm2s_cmd_wvalid          =>  s_axis_mm2s_cmd_tvalid     ,  
          mm2s_cmd_wready          =>  s_axis_mm2s_cmd_tready     ,  
          mm2s_cmd_wdata           =>  s_axis_mm2s_cmd_tdata      ,  
                                                  
          mm2s_sts_wvalid          =>  m_axis_mm2s_sts_tvalid     ,  
          mm2s_sts_wready          =>  m_axis_mm2s_sts_tready     ,  
          mm2s_sts_wdata           =>  m_axis_mm2s_sts_tdata      ,  
          mm2s_sts_wstrb           =>  sig_mm2s_sts_tstrb         ,  
          mm2s_sts_wlast           =>  m_axis_mm2s_sts_tlast      ,  
                                                  
          mm2s_allow_addr_req      =>  mm2s_allow_addr_req        ,   
          mm2s_addr_req_posted     =>  mm2s_addr_req_posted       ,
          mm2s_rd_xfer_cmplt       =>  mm2s_rd_xfer_cmplt         ,
          
          mm2s_arid                =>  m_axi_mm2s_arid            ,  
          mm2s_araddr              =>  m_axi_mm2s_araddr          ,  
          mm2s_arlen               =>  m_axi_mm2s_arlen           ,  
          mm2s_arsize              =>  m_axi_mm2s_arsize          ,  
          mm2s_arburst             =>  m_axi_mm2s_arburst         ,  
          mm2s_arprot              =>  m_axi_mm2s_arprot          ,  
          mm2s_arcache             =>  m_axi_mm2s_arcache         ,
          mm2s_aruser              =>  m_axi_mm2s_aruser          ,
          mm2s_arvalid             =>  m_axi_mm2s_arvalid         ,  
          mm2s_arready             =>  m_axi_mm2s_arready         ,  
                                                  
          mm2s_rdata               =>  m_axi_mm2s_rdata           ,  
          mm2s_rresp               =>  m_axi_mm2s_rresp           ,  
          mm2s_rlast               =>  m_axi_mm2s_rlast           ,  
          mm2s_rvalid              =>  m_axi_mm2s_rvalid          ,  
          mm2s_rready              =>  m_axi_mm2s_rready          ,  
                                                  
          mm2s_strm_wdata          =>  m_axis_mm2s_tdata          ,  
          mm2s_strm_wstrb          =>  sig_mm2s_tstrb             ,  
          mm2s_strm_wlast          =>  m_axis_mm2s_tlast          ,  
          mm2s_strm_wvalid         =>  m_axis_mm2s_tvalid         ,  
          mm2s_strm_wready         =>  m_axis_mm2s_tready         ,   

     
          mm2s_dbg_sel             =>  mm2s_dbg_sel               ,
          mm2s_dbg_data            =>  mm2s_dbg_data 
          );
      
 
  
  
     end generate GEN_MM2S_BASIC;


  
  ------------------------------------------------------------
  -- If Generate
  --
  -- Label: GEN_S2MM_BASIC
  --
  -- If Generate Description:
  --  Instantiate the S2MM Basic Wrapper 
  --
  --
  ------------------------------------------------------------
  GEN_S2MM_BASIC : if (C_INCLUDE_S2MM = 2) generate
  
  
     begin
  
       ------------------------------------------------------------
       -- Instance: I_S2MM_BASIC_WRAPPER 
       --
       -- Description:
       -- Write Basic Wrapper Instance    
       --
       ------------------------------------------------------------
        I_S2MM_BASIC_WRAPPER : entity axi_sg_v4_1_6.axi_sg_s2mm_basic_wrap
        generic map (
      
          C_INCLUDE_S2MM            =>  C_INCLUDE_S2MM           ,
          C_S2MM_AWID               =>  C_M_AXI_S2MM_AWID        ,
          C_S2MM_ID_WIDTH           =>  C_M_AXI_S2MM_ID_WIDTH    ,
          C_S2MM_ADDR_WIDTH         =>  C_M_AXI_S2MM_ADDR_WIDTH  ,  
          C_S2MM_MDATA_WIDTH        =>  C_M_AXI_S2MM_DATA_WIDTH  ,  
          C_S2MM_SDATA_WIDTH        =>  C_S_AXIS_S2MM_TDATA_WIDTH , 
          C_INCLUDE_S2MM_STSFIFO    =>  C_INCLUDE_S2MM_STSFIFO   ,
          C_S2MM_STSCMD_FIFO_DEPTH  =>  S2MM_CMDSTS_FIFO_DEPTH   ,
          C_S2MM_STSCMD_IS_ASYNC    =>  C_S2MM_STSCMD_IS_ASYNC   ,
          C_INCLUDE_S2MM_DRE        =>  C_INCLUDE_S2MM_DRE       ,
          C_S2MM_BURST_SIZE         =>  S2MM_MAX_BURST_BEATS     ,
          C_S2MM_ADDR_PIPE_DEPTH    =>  C_S2MM_ADDR_PIPE_DEPTH   ,
          C_TAG_WIDTH               =>  S2MM_TAG_WIDTH           ,
          C_ENABLE_MULTI_CHANNEL           =>  C_ENABLE_MULTI_CHANNEL          ,
          C_ENABLE_EXTRA_FIELD      =>  C_ENABLE_EXTRA_FIELD,
          C_FAMILY                  =>  C_FAMILY
                  
          )
        port map (
      
          s2mm_aclk            =>  m_axi_s2mm_aclk               , 
          s2mm_aresetn         =>  m_axi_s2mm_aresetn            , 
          sg_ctl               =>  sg_ctl                        ,

          s2mm_halt            =>  s2mm_halt                     ,
          s2mm_halt_cmplt      =>  s2mm_halt_cmplt               ,
          s2mm_err             =>  s2mm_err                      , 
          s2mm_cmdsts_awclk    =>  m_axis_s2mm_cmdsts_awclk      , 
          s2mm_cmdsts_aresetn  =>  m_axis_s2mm_cmdsts_aresetn    , 
          
          s2mm_cmd_wvalid      =>  s_axis_s2mm_cmd_tvalid        , 
          s2mm_cmd_wready      =>  s_axis_s2mm_cmd_tready        , 
          s2mm_cmd_wdata       =>  s_axis_s2mm_cmd_tdata         , 
                                              
          s2mm_sts_wvalid      =>  m_axis_s2mm_sts_tvalid        , 
          s2mm_sts_wready      =>  m_axis_s2mm_sts_tready        , 
          s2mm_sts_wdata       =>  m_axis_s2mm_sts_tdata         , 
          s2mm_sts_wstrb       =>  sig_s2mm_sts_tstrb            , 
          s2mm_sts_wlast       =>  m_axis_s2mm_sts_tlast         , 
       
          s2mm_allow_addr_req  =>  s2mm_allow_addr_req           ,
          s2mm_addr_req_posted =>  s2mm_addr_req_posted          ,
          s2mm_wr_xfer_cmplt   =>  s2mm_wr_xfer_cmplt            ,
          s2mm_ld_nxt_len      =>  s2mm_ld_nxt_len               ,
          s2mm_wr_len          =>  s2mm_wr_len                   ,
                                                        
          s2mm_awid            =>  m_axi_s2mm_awid               , 
          s2mm_awaddr          =>  m_axi_s2mm_awaddr             , 
          s2mm_awlen           =>  m_axi_s2mm_awlen              , 
          s2mm_awsize          =>  m_axi_s2mm_awsize             , 
          s2mm_awburst         =>  m_axi_s2mm_awburst            , 
          s2mm_awprot          =>  m_axi_s2mm_awprot             , 
          s2mm_awcache         =>  m_axi_s2mm_awcache            ,
          s2mm_awuser          =>  m_axi_s2mm_awuser             ,
          s2mm_awvalid         =>  m_axi_s2mm_awvalid            , 
          s2mm_awready         =>  m_axi_s2mm_awready            , 
                                              
          s2mm_wdata           =>  m_axi_s2mm_wdata              , 
          s2mm_wstrb           =>  m_axi_s2mm_wstrb              , 
          s2mm_wlast           =>  m_axi_s2mm_wlast              , 
          s2mm_wvalid          =>  m_axi_s2mm_wvalid             , 
          s2mm_wready          =>  m_axi_s2mm_wready             , 
    
          s2mm_bresp           =>  m_axi_s2mm_bresp              , 
          s2mm_bvalid          =>  m_axi_s2mm_bvalid             , 
          s2mm_bready          =>  m_axi_s2mm_bready             , 
    
                                              
          s2mm_strm_wdata      =>  s_axis_s2mm_tdata             , 
          s2mm_strm_wstrb      =>  sig_s2mm_tstrb                , 
          s2mm_strm_wlast      =>  s_axis_s2mm_tlast             , 
          s2mm_strm_wvalid     =>  s_axis_s2mm_tvalid            , 
          s2mm_strm_wready     =>  s_axis_s2mm_tready            ,  
     
          s2mm_dbg_sel         =>  s2mm_dbg_sel                  ,
          s2mm_dbg_data        =>  s2mm_dbg_data      
     
          );
  
  
  
     end generate GEN_S2MM_BASIC;


end implementation;


-- *************************************************************************
--
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:          axi_sg_ftch_sm.vhd
-- Description: This entity manages fetching of descriptors.
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_sg_v4_1_6;
use axi_sg_v4_1_6.axi_sg_pkg.all;

library lib_pkg_v1_0_2;
use lib_pkg_v1_0_2.lib_pkg.clog2;

-------------------------------------------------------------------------------
entity  axi_sg_ftch_sm is
    generic (
        C_M_AXI_SG_ADDR_WIDTH       : integer range 32 to 64    := 32;
            -- Master AXI Memory Map Address Width for Scatter Gather R/W Port
        C_ENABLE_MULTI_CHANNEL             : integer range 0 to 1      := 0;

        C_INCLUDE_CH1               : integer range 0 to 1      := 1;
            -- Include or Exclude channel 1 scatter gather engine
            -- 0 = Exclude Channel 1 SG Engine
            -- 1 = Include Channel 1 SG Engine

        C_INCLUDE_CH2               : integer range 0 to 1       := 1;
            -- Include or Exclude channel 2 scatter gather engine
            -- 0 = Exclude Channel 2 SG Engine
            -- 1 = Include Channel 2 SG Engine

        C_SG_CH1_WORDS_TO_FETCH     : integer range 4 to 16      := 8;
            -- Number of words to fetch

        C_SG_CH2_WORDS_TO_FETCH     : integer range 4 to 16      := 8;
            -- Number of words to fetch

        C_SG_FTCH_DESC2QUEUE     : integer range 0 to 8          := 0;
            -- Number of descriptors to fetch and queue for each channel.
            -- A value of zero excludes the fetch queues.


        C_SG_CH1_ENBL_STALE_ERROR   : integer range 0 to 1      := 1;
            -- Enable or disable stale descriptor check
            -- 0 = Disable stale descriptor error check
            -- 1 = Enable stale descriptor error check

        C_SG_CH2_ENBL_STALE_ERROR   : integer range 0 to 1      := 1
            -- Enable or disable stale descriptor check
            -- 0 = Disable stale descriptor error check
            -- 1 = Enable stale descriptor error check

    );
    port (
        -----------------------------------------------------------------------
        -- AXI Scatter Gather Interface
        -----------------------------------------------------------------------
        m_axi_sg_aclk               : in  std_logic                         ;                   --
        m_axi_sg_aresetn            : in  std_logic                         ;                   --
                                                                                                --
        updt_error                  : in  std_logic                         ;                   --
                                                                                                --
        -- Channel 1 Control and Status                                                         --
        ch1_run_stop                : in  std_logic                         ;                   --
        ch1_desc_flush              : in  std_logic                         ;                   --
        ch1_updt_done               : in  std_logic                         ;                   --
        ch1_sg_idle                 : in  std_logic                         ;                   --
        ch1_tailpntr_enabled        : in  std_logic                         ;                   --
        ch1_ftch_queue_full         : in  std_logic                         ;                   --
        ch1_ftch_queue_empty        : in  std_logic                         ;                   --
        ch1_ftch_pause              : in  std_logic                         ;                   --
        ch1_fetch_address           : in std_logic_vector                                       --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                   --
        ch1_ftch_active             : out std_logic                         ;                   --
        ch1_ftch_idle               : out std_logic                         ;                   --
        ch1_ftch_interr_set         : out std_logic                         ;                   --
        ch1_ftch_slverr_set         : out std_logic                         ;                   --
        ch1_ftch_decerr_set         : out std_logic                         ;                   --
        ch1_ftch_err_early          : out std_logic                         ;                   --
        ch1_ftch_stale_desc         : out std_logic                         ;                   --
                                                                                                --
        -- Channel 2 Control and Status                                                         --
        ch2_run_stop                : in  std_logic                         ;                   --
        ch2_desc_flush              : in  std_logic                         ;                   --
        ch2_updt_done               : in  std_logic                         ;                   --
        ch2_sg_idle                 : in  std_logic                         ;                   --
        ch2_tailpntr_enabled        : in  std_logic                         ;                   --
        ch2_ftch_queue_full         : in  std_logic                         ;                   --
        ch2_ftch_queue_empty        : in  std_logic                         ;                   --
        ch2_ftch_pause              : in  std_logic                         ;                   --
        ch2_fetch_address           : in std_logic_vector                                       --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                   --
        ch2_ftch_active             : out std_logic                         ;                   --
        ch2_ftch_idle               : out std_logic                         ;                   --
        ch2_ftch_interr_set         : out std_logic                         ;                   --
        ch2_ftch_slverr_set         : out std_logic                         ;                   --
        ch2_ftch_decerr_set         : out std_logic                         ;                   --
        ch2_ftch_err_early          : out std_logic                         ;                   --
        ch2_ftch_stale_desc         : out std_logic                         ;                   --
        no_fetch_window             : in std_logic                          ;
                                                                                                --
        -- DataMover Command                                                                    --
        ftch_cmnd_wr                : out std_logic                         ;                   --
        ftch_cmnd_data              : out std_logic_vector                                      --
                                        ((C_M_AXI_SG_ADDR_WIDTH+CMD_BASE_WIDTH)-1 downto 0);    --
        -- DataMover Status                                                                     --
        ftch_done                   : in  std_logic                         ;                   --
        ftch_error                  : in  std_logic                         ;                   --
        ftch_interr                 : in  std_logic                         ;                   --
        ftch_slverr                 : in  std_logic                         ;                   --
        ftch_decerr                 : in  std_logic                         ;                   --
        ftch_stale_desc             : in  std_logic                         ;                   --
        ftch_error_early            : in  std_logic                         ;                   --
        ftch_error_addr             : out std_logic_vector                                      --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)                      --


    );

end axi_sg_ftch_sm;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_sg_ftch_sm is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------
-- DataMover Command Type
constant FETCH_CMD_TYPE     : std_logic := '1';
-- DataMover Cmnd Reserved Bits
constant FETCH_MSB_IGNORED  : std_logic_vector(7 downto 0)  := (others => '0');
-- DataMover Cmnd Reserved Bits
constant FETCH_LSB_IGNORED  : std_logic_vector(15 downto 0) := (others => '0');
-- DataMover Cmnd Bytes to Xfer for Channel 1
constant FETCH_CH1_CMD_BTT  : std_logic_vector(SG_BTT_WIDTH-1 downto 0)
                                := std_logic_vector(to_unsigned(
                                (C_SG_CH1_WORDS_TO_FETCH*4),SG_BTT_WIDTH));
-- DataMover Cmnd Bytes to Xfer for Channel 2
constant FETCH_CH2_CMD_BTT  : std_logic_vector(SG_BTT_WIDTH-1 downto 0)
                                := std_logic_vector(to_unsigned(
                                (C_SG_CH2_WORDS_TO_FETCH*4),SG_BTT_WIDTH));
-- DataMover Cmnd Reserved Bits
constant FETCH_CMD_RSVD     : std_logic_vector(
                                DATAMOVER_CMD_RSVMSB_BOFST + C_M_AXI_SG_ADDR_WIDTH downto
                                DATAMOVER_CMD_RSVLSB_BOFST + C_M_AXI_SG_ADDR_WIDTH)
                                := (others => '0');

-- CR585958 Constant declaration in axi_sg_ftch_sm needs to move under associated generate
-- Required width in bits for C_SG_FTCH_DESC2QUEUE
--constant SG_FTCH_DESC2QUEUE_WIDTH : integer := clog2(C_SG_FTCH_DESC2QUEUE+1);
--
---- Vector version of C_SG_FTCH_DESC2QUEUE
--constant SG_FTCH_DESC2QUEUE_VEC   : std_logic_vector(SG_FTCH_DESC2QUEUE_WIDTH-1 downto 0)
--                                    := std_logic_vector(to_unsigned
--                                    (C_SG_FTCH_DESC2QUEUE,SG_FTCH_DESC2QUEUE_WIDTH));


-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
-- DataMover Commmand TAG
signal fetch_tag      : std_logic_vector(3 downto 0)  := (others => '0');
type SG_FTCH_STATE_TYPE      is (
                                IDLE,
                                FETCH_DESCRIPTOR,
                                FETCH_STATUS,
                                FETCH_ERROR
                                );

signal ftch_cs                  : SG_FTCH_STATE_TYPE;
signal ftch_ns                  : SG_FTCH_STATE_TYPE;


-- State Machine Signals
signal ch1_active_set           : std_logic := '0';
signal ch2_active_set           : std_logic := '0';
signal write_cmnd_cmb           : std_logic := '0';
signal ch1_ftch_sm_idle         : std_logic := '0';
signal ch2_ftch_sm_idle         : std_logic := '0';
signal ch1_pause_fetch          : std_logic := '0';
signal ch2_pause_fetch          : std_logic := '0';
signal ch2_pause_fetch1          : std_logic := '0';
signal ch2_pause_fetch2          : std_logic := '0';
signal ch2_pause_fetch3          : std_logic := '0';
signal ch2_updt_done1          : std_logic := '0';
signal ch2_updt_done2          : std_logic := '0';


-- Misc Signals
signal fetch_cmd_addr           : std_logic_vector
                                    (C_M_AXI_SG_ADDR_WIDTH-1 downto 0) := (others => '0');
signal ch1_active_i             : std_logic := '0';
signal service_ch1              : std_logic := '0';

signal ch2_active_i             : std_logic := '0';
signal service_ch2              : std_logic := '0';
signal fetch_cmd_btt            : std_logic_vector
                                    (SG_BTT_WIDTH-1 downto 0) := (others => '0');
signal ch1_stale_descriptor     : std_logic := '0';
signal ch2_stale_descriptor     : std_logic := '0';

signal ch1_ftch_interr_set_i    : std_logic := '0';
signal ch2_ftch_interr_set_i    : std_logic := '0';

-- CR585958 Constant declaration in axi_sg_ftch_sm needs to move under associated generate
-- counts for keeping track of queue descriptors to prevent
-- fifo fill
--signal ch1_desc_ftched_count    : std_logic_vector
--                                    (SG_FTCH_DESC2QUEUE_WIDTH-1 downto 0) := (others => '0');
--signal ch2_desc_ftched_count    : std_logic_vector
--                                    (SG_FTCH_DESC2QUEUE_WIDTH-1 downto 0) := (others => '0');

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin
ch1_ftch_active  <= ch1_active_i;
ch2_ftch_active  <= ch2_active_i;


-------------------------------------------------------------------------------
-- Scatter Gather Fetch State Machine
-------------------------------------------------------------------------------
SG_FTCH_MACHINE : process(ftch_cs,
                            ch1_active_i,
                            ch2_active_i,
                            service_ch1,
                            service_ch2,
                            ftch_error,
                            ftch_done)

    begin
        -- Default signal assignment
        ch1_active_set          <= '0';
        ch2_active_set          <= '0';
        write_cmnd_cmb          <= '0';
        ch1_ftch_sm_idle        <= '0';
        ch2_ftch_sm_idle        <= '0';
        ftch_ns                 <= ftch_cs;

        case ftch_cs is

            -------------------------------------------------------------------
            when IDLE =>
                ch1_ftch_sm_idle       <=  not service_ch1;
                ch2_ftch_sm_idle       <=  not service_ch2;
                -- sg error during fetch - shut down
                if(ftch_error = '1')then
                    ftch_ns     <= FETCH_ERROR;
                -- If channel 1 is running and not idle and queue is not full
                -- then fetch descriptor for channel 1
                elsif(service_ch1 = '1')then
                    ch1_active_set  <= '1';
                    ftch_ns <= FETCH_DESCRIPTOR;
                -- If channel 2 is running and not idle and queue is not full
                -- then fetch descriptor for channel 2
                elsif(service_ch2 = '1')then
                    ch2_active_set  <= '1';
                    ftch_ns <= FETCH_DESCRIPTOR;

                else
                    ftch_ns                <= IDLE;
                end if;

            -------------------------------------------------------------------
            when FETCH_DESCRIPTOR =>
                -- sg error during fetch - shut down
                if(ftch_error = '1')then
                    ftch_ns     <= FETCH_ERROR;
                else
                    ch1_ftch_sm_idle    <= not ch1_active_i and not service_ch1;
                    ch2_ftch_sm_idle    <= not ch2_active_i and not service_ch2;
                    write_cmnd_cmb      <= '1';
                    ftch_ns             <= FETCH_STATUS;
                end if;

            -------------------------------------------------------------------
            when FETCH_STATUS =>

                ch1_ftch_sm_idle    <= not ch1_active_i and not service_ch1;
                ch2_ftch_sm_idle    <= not ch2_active_i and not service_ch2;

                -- sg error during fetch - shut down
                if(ftch_error = '1')then
                    ftch_ns     <= FETCH_ERROR;

                elsif(ftch_done = '1')then
                    -- If just finished fethcing for channel 2 then...
                    if(ch2_active_i = '1')then
                        -- If ready, fetch descriptor for channel 1
                        if(service_ch1 = '1')then
                            ch1_active_set <= '1';
                            ftch_ns <= FETCH_DESCRIPTOR;
                        -- Else if channel 2 still ready then fetch
                        -- another descriptor for channel 2
                        elsif(service_ch2 = '1')then
                            ch1_ftch_sm_idle    <= '1';
                            ftch_ns             <= FETCH_DESCRIPTOR;
                        -- Otherwise return to IDLE
                        else
                            ftch_ns <= IDLE;
                        end if;

                    -- If just finished fethcing for channel 1 then...
                    elsif(ch1_active_i = '1')then
                        -- If ready, fetch descriptor for channel 2
                        if(service_ch2 = '1')then
                            ch2_active_set <= '1';
                            ftch_ns <= FETCH_DESCRIPTOR;
                        -- Else if channel 1 still ready then fetch
                        -- another descriptor for channel 1
                        elsif(service_ch1 = '1')then
                            ch2_ftch_sm_idle    <= '1';
                            ftch_ns             <= FETCH_DESCRIPTOR;
                        -- Otherwise return to IDLE
                        else
                            ftch_ns <= IDLE;
                        end if;
                    else
                        ftch_ns <= IDLE;
                    end if;
                else
                    ftch_ns <= FETCH_STATUS;

                end if;

            -------------------------------------------------------------------
            when FETCH_ERROR =>
                ch1_ftch_sm_idle    <= '1';
                ch2_ftch_sm_idle    <= '1';
                ftch_ns             <= FETCH_ERROR;

            -------------------------------------------------------------------
            -- coverage off
            when others =>
                ftch_ns <= IDLE;

            -- coverage on
        end case;
    end process SG_FTCH_MACHINE;

-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
REGISTER_STATE : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                ftch_cs     <= IDLE;
            else
                ftch_cs     <= ftch_ns;
            end if;
        end if;
    end process REGISTER_STATE;


-------------------------------------------------------------------------------
-- Channel included therefore generate fetch logic
-------------------------------------------------------------------------------
GEN_CH1_FETCH : if C_INCLUDE_CH1 = 1 generate
begin

    -------------------------------------------------------------------------------
    -- Active channel flag.  Indicates which channel is active.
    -- 0 = channel active
    -- 1 = channel active
    -------------------------------------------------------------------------------
    CH1_ACTIVE_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0' or ch2_active_set = '1')then
                    ch1_active_i <= '0';
                elsif(ch1_active_set = '1')then
                    ch1_active_i <= '1';
                end if;
            end if;
        end process CH1_ACTIVE_PROCESS;

    -------------------------------------------------------------------------------
    -- Channel 1 IDLE process. Indicates channel 1 fetch process is IDLE
    -- This is 1 part of determining IDLE for a channel
    -------------------------------------------------------------------------------
    CH1_IDLE_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                -- If reset or stopped then clear idle bit
                if(m_axi_sg_aresetn = '0')then
                    ch1_ftch_idle   <= '1';

                -- SG Error therefore force IDLE
                -- CR564855 - fetch idle asserted too soon when update error occured.
                -- fetch idle does not need to be concerned with updt_error.  This is
                -- because on going fetch is guarentteed to complete regardless of dma
                -- controller or sg update engine.
                --elsif(updt_error = '1' or ftch_error = '1'
                elsif(ftch_error = '1'
                or ch1_ftch_interr_set_i = '1')then
                    ch1_ftch_idle   <= '1';

                -- When SG Fetch no longer idle then clear fetch idle
                elsif(ch1_sg_idle = '0')then
                    ch1_ftch_idle   <= '0';

                -- If tail = cur and fetch queue is empty then
                elsif(ch1_sg_idle = '1' and ch1_ftch_queue_empty = '1' and ch1_ftch_sm_idle = '1')then
                    ch1_ftch_idle   <= '1';

                end if;
            end if;
        end process CH1_IDLE_PROCESS;

    -------------------------------------------------------------------------------
    -- For No Fetch Queue, generate pause logic to prevent partial descriptor from
    -- being fetched and then endless throttle on AXI read bus
    -------------------------------------------------------------------------------
    GEN_CH1_FETCH_PAUSE : if C_SG_FTCH_DESC2QUEUE = 0 generate
    begin
        REG_PAUSE_FETCH : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    -- On descriptor update done clear pause
                    if(m_axi_sg_aresetn = '0' or ch1_updt_done = '1')then
                        ch1_pause_fetch <= '0';
                    -- If channel active and command written then pause
                    elsif(ch1_active_i='1' and write_cmnd_cmb = '1')then
                        ch1_pause_fetch <= '1';
                    end if;
                 end if;
             end process REG_PAUSE_FETCH;
    end generate GEN_CH1_FETCH_PAUSE;

    -- Fetch queues so do not need to pause
    GEN_CH1_NO_FETCH_PAUSE : if C_SG_FTCH_DESC2QUEUE /= 0 generate
--    -- CR585958
--    -- Required width in bits for C_SG_FTCH_DESC2QUEUE
--    constant SG_FTCH_DESC2QUEUE_WIDTH : integer := clog2(C_SG_FTCH_DESC2QUEUE+1);
--    -- Vector version of C_SG_FTCH_DESC2QUEUE
--    constant SG_FTCH_DESC2QUEUE_VEC   : std_logic_vector(SG_FTCH_DESC2QUEUE_WIDTH-1 downto 0)
--                                        := std_logic_vector(to_unsigned
--                                        (C_SG_FTCH_DESC2QUEUE,SG_FTCH_DESC2QUEUE_WIDTH));
--    signal desc_queued_incr     : std_logic := '0';
--    signal desc_queued_decr     : std_logic := '0';
--
--    -- CR585958
--    signal ch1_desc_ftched_count: std_logic_vector
--                                    (SG_FTCH_DESC2QUEUE_WIDTH-1 downto 0) := (others => '0');
--    begin
--
--        desc_queued_incr <= '1' when ch1_active_i = '1'
--                                 and write_cmnd_cmb = '1'
--                                 and ch1_ftch_descpulled = '0'
--                       else '0';
--
--        desc_queued_decr <= '1' when ch1_ftch_descpulled = '1'
--                                and not (ch1_active_i = '1' and write_cmnd_cmb = '1')
--                       else '0';
--
--        -- Keep track of descriptors queued version descriptors updated
--        DESC_FETCHED_CNTR : process(m_axi_sg_aclk)
--            begin
--                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
--                    if(m_axi_sg_aresetn = '0')then
--                        ch1_desc_ftched_count <= (others => '0');
--                    elsif(desc_queued_incr = '1')then
--                        ch1_desc_ftched_count <= std_logic_vector(unsigned(ch1_desc_ftched_count) + 1);
--                    elsif(desc_queued_decr = '1')then
--                        ch1_desc_ftched_count <= std_logic_vector(unsigned(ch1_desc_ftched_count) - 1);
--                    end if;
--                end if;
--            end process DESC_FETCHED_CNTR;
--
--        REG_PAUSE_FETCH : process(m_axi_sg_aclk)
--            begin
--                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
--                    if(m_axi_sg_aresetn = '0')then
--                        ch1_pause_fetch <= '0';
--                    elsif(ch1_desc_ftched_count >= SG_FTCH_DESC2QUEUE_VEC)then
--                        ch1_pause_fetch <= '1';
--                    else
--                        ch1_pause_fetch <= '0';
--                    end if;
--                end if;
--            end process REG_PAUSE_FETCH;
--
--
--
            ch1_pause_fetch <= ch1_ftch_pause;

    end generate GEN_CH1_NO_FETCH_PAUSE;


    -------------------------------------------------------------------------------
    -- Channel 1 ready to be serviced?
    -------------------------------------------------------------------------------
    service_ch1 <= '1' when ch1_run_stop = '1'          -- Channel running
                        and ch1_sg_idle = '0'           -- SG Engine running
                        and ch1_ftch_queue_full = '0'   -- Queue not full
                        and updt_error = '0'            -- No SG Update error
                        and ch1_stale_descriptor = '0'  -- No Stale Descriptors
                        and ch1_desc_flush = '0'        -- Not flushing desc
                        and ch1_pause_fetch = '0'       -- Not pausing


              else '0';

    -------------------------------------------------------------------------------
    -- Log Fetch Errors
    -------------------------------------------------------------------------------
    INT_ERROR_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                -- If reset or stopped then clear idle bit
                if(m_axi_sg_aresetn = '0')then
                    ch1_ftch_interr_set_i  <= '0';
                -- Channel active and datamover int error or fetch done and descriptor stale
                elsif((ch1_active_i = '1' and ftch_interr = '1')
                   or ((ftch_done = '1' or ftch_error = '1') and ch1_stale_descriptor = '1'))then
                    ch1_ftch_interr_set_i  <= '1';
                end if;
            end if;
        end process INT_ERROR_PROCESS;

    ch1_ftch_interr_set <= ch1_ftch_interr_set_i;


    SLV_ERROR_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                -- If reset or stopped then clear idle bit
                if(m_axi_sg_aresetn = '0')then
                    ch1_ftch_slverr_set  <= '0';
                elsif(ch1_active_i = '1' and ftch_slverr = '1')then
                    ch1_ftch_slverr_set  <= '1';
                end if;
            end if;
        end process SLV_ERROR_PROCESS;

    DEC_ERROR_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                -- If reset or stopped then clear idle bit
                if(m_axi_sg_aresetn = '0')then
                    ch1_ftch_decerr_set  <= '0';
                elsif(ch1_active_i = '1' and ftch_decerr = '1')then
                    ch1_ftch_decerr_set  <= '1';
                end if;
            end if;
        end process DEC_ERROR_PROCESS;


    -- Early detection of SlvErr or DecErr, used to prevent error'ed descriptor
    -- from being used by dma controller
    ch1_ftch_err_early  <= '1' when ftch_error_early = '1' and ch1_active_i = '1'
                      else '0';

    -- Enable stale descriptor check
    GEN_CH1_STALE_CHECK : if C_SG_CH1_ENBL_STALE_ERROR = 1 generate
    begin
        -----------------------------------------------------------------------
        -- Stale Descriptor Error
        -----------------------------------------------------------------------
        CH1_STALE_DESC : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    -- If reset then clear flag
                    if(m_axi_sg_aresetn = '0')then
                        ch1_stale_descriptor <= '0';
                    elsif(ftch_stale_desc = '1' and ch1_active_i = '1' )then
                        ch1_stale_descriptor <= '1';
                    end if;
                end if;
            end process CH1_STALE_DESC;

    end generate GEN_CH1_STALE_CHECK;

    -- Disable stale descriptor check
    GEN_CH1_NO_STALE_CHECK : if C_SG_CH1_ENBL_STALE_ERROR = 0 generate
    begin
        ch1_stale_descriptor <= '0';
    end generate GEN_CH1_NO_STALE_CHECK;


    -- Early detection of Stale Descriptor (valid only in tailpntr mode) used
    -- to prevent error'ed descriptor from being used.
    ch1_ftch_stale_desc <= ch1_stale_descriptor;

end generate GEN_CH1_FETCH;

-------------------------------------------------------------------------------
-- Channel excluded therefore do not generate fetch logic
-------------------------------------------------------------------------------
GEN_NO_CH1_FETCH : if C_INCLUDE_CH1 = 0 generate
begin
    service_ch1         <= '0';
    ch1_active_i        <= '0';
    ch1_ftch_idle       <= '0';
    ch1_ftch_interr_set <= '0';
    ch1_ftch_slverr_set <= '0';
    ch1_ftch_decerr_set <= '0';
    ch1_ftch_err_early  <= '0';
    ch1_ftch_stale_desc <= '0';
end generate GEN_NO_CH1_FETCH;


-------------------------------------------------------------------------------
-- Channel included therefore generate fetch logic
-------------------------------------------------------------------------------
GEN_CH2_FETCH : if C_INCLUDE_CH2 = 1 generate
begin
    -------------------------------------------------------------------------------
    -- Active channel flag.  Indicates which channel is active.
    -- 0 = channel active
    -- 1 = channel active
    -------------------------------------------------------------------------------
    CH2_ACTIVE_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0' or ch1_active_set = '1')then
                    ch2_active_i <= '0';
                elsif(ch2_active_set = '1')then
                    ch2_active_i <= '1';
                end if;
            end if;
        end process CH2_ACTIVE_PROCESS;

    -------------------------------------------------------------------------------
    -- Channel 2 IDLE process. Indicates channel 2 fetch process is IDLE
    -- This is 1 part of determining IDLE for a channel
    -------------------------------------------------------------------------------
    CH2_IDLE_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                -- If reset or stopped then clear idle bit
                if(m_axi_sg_aresetn = '0')then
                    ch2_ftch_idle   <= '1';

                -- SG Error therefore force IDLE
                -- CR564855 - fetch idle asserted too soon when update error occured.
                -- fetch idle does not need to be concerned with updt_error.  This is
                -- because on going fetch is guarentteed to complete regardless of dma
                -- controller or sg update engine.
--                elsif(updt_error = '1' or ftch_error = '1'
                elsif(ftch_error = '1'
                or ch2_ftch_interr_set_i = '1')then

                    ch2_ftch_idle   <= '1';

                -- When SG Fetch no longer idle then clear fetch idle
                elsif(ch2_sg_idle = '0')then
                    ch2_ftch_idle   <= '0';

                -- If tail = cur and fetch queue is empty then
                elsif(ch2_sg_idle = '1' and ch2_ftch_queue_empty = '1' and ch2_ftch_sm_idle = '1')then
                    ch2_ftch_idle   <= '1';
                end if;
            end if;
        end process CH2_IDLE_PROCESS;

    -------------------------------------------------------------------------------
    -- For No Fetch Queue, generate pause logic to prevent partial descriptor from
    -- being fetched and then endless throttle on AXI read bus
    -------------------------------------------------------------------------------
    GEN_CH2_FETCH_PAUSE : if C_SG_FTCH_DESC2QUEUE = 0 generate
    begin
        REG_PAUSE_FETCH : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    -- On descriptor update done clear pause
                    if(m_axi_sg_aresetn = '0' or ch2_updt_done = '1')then
                        ch2_pause_fetch <= '0';
                    -- If channel active and command written then pause
                    elsif(ch2_active_i='1' and write_cmnd_cmb = '1')then
                        ch2_pause_fetch <= '1';
                    end if;
                        ch2_pause_fetch1 <= ch2_pause_fetch;
                        ch2_pause_fetch2 <= ch2_pause_fetch1;
                        ch2_pause_fetch3 <= ch2_pause_fetch2;
                 end if;
             end process REG_PAUSE_FETCH;
    end generate GEN_CH2_FETCH_PAUSE;

    -- Fetch queues so do not need to pause
    GEN_CH2_NO_FETCH_PAUSE : if C_SG_FTCH_DESC2QUEUE /= 0 generate
--    -- CR585958
--    -- Required width in bits for C_SG_FTCH_DESC2QUEUE
--    constant SG_FTCH_DESC2QUEUE_WIDTH : integer := clog2(C_SG_FTCH_DESC2QUEUE+1);
--    -- Vector version of C_SG_FTCH_DESC2QUEUE
--    constant SG_FTCH_DESC2QUEUE_VEC   : std_logic_vector(SG_FTCH_DESC2QUEUE_WIDTH-1 downto 0)
--                                        := std_logic_vector(to_unsigned
--                                        (C_SG_FTCH_DESC2QUEUE,SG_FTCH_DESC2QUEUE_WIDTH));
--    signal desc_queued_incr     : std_logic := '0';
--    signal desc_queued_decr     : std_logic := '0';
--
--    -- CR585958
--    signal ch2_desc_ftched_count: std_logic_vector
--                                    (SG_FTCH_DESC2QUEUE_WIDTH-1 downto 0) := (others => '0');
--
--    begin
--
--        desc_queued_incr <= '1' when ch2_active_i = '1'
--                                 and write_cmnd_cmb = '1'
--                                 and ch2_ftch_descpulled = '0'
--                       else '0';
--
--        desc_queued_decr <= '1' when ch2_ftch_descpulled = '1'
--                                and not (ch2_active_i = '1' and write_cmnd_cmb = '1')
--                       else '0';
--
--        -- Keep track of descriptors queued version descriptors updated
--        DESC_FETCHED_CNTR : process(m_axi_sg_aclk)
--            begin
--                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
--                    if(m_axi_sg_aresetn = '0')then
--                        ch2_desc_ftched_count <= (others => '0');
--                    elsif(desc_queued_incr = '1')then
--                        ch2_desc_ftched_count <= std_logic_vector(unsigned(ch2_desc_ftched_count) + 1);
--                    elsif(desc_queued_decr = '1')then
--                        ch2_desc_ftched_count <= std_logic_vector(unsigned(ch2_desc_ftched_count) - 1);
--                    end if;
--                end if;
--            end process DESC_FETCHED_CNTR;
--
--        REG_PAUSE_FETCH : process(m_axi_sg_aclk)
--            begin
--                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
--                    if(m_axi_sg_aresetn = '0')then
--                        ch2_pause_fetch <= '0';
--                    elsif(ch2_desc_ftched_count >= SG_FTCH_DESC2QUEUE_VEC)then
--                        ch2_pause_fetch <= '1';
--                    else
--                        ch2_pause_fetch <= '0';
--                    end if;
--                end if;
--            end process REG_PAUSE_FETCH;
--
            ch2_pause_fetch <= ch2_ftch_pause;
    end generate GEN_CH2_NO_FETCH_PAUSE;

    -------------------------------------------------------------------------------
    -- Channel 2 ready to be serviced?
    -------------------------------------------------------------------------------

MCDMA : if (C_ENABLE_MULTI_CHANNEL = 1) generate

NOQUEUE : if (C_SG_FTCH_DESC2QUEUE = 0) generate
    service_ch2 <= '1' when ch2_run_stop = '1'          -- Channel running
                        and ch2_sg_idle = '0'           -- SG Engine running
                        and ch2_ftch_queue_full = '0'   -- Queue not full
                        and updt_error = '0'            -- No SG Update error
                        and ch2_stale_descriptor = '0'  -- No Stale Descriptors
                        and ch2_desc_flush = '0'        -- Not flushing desc
                        and ch2_pause_fetch3 = '0'       -- No fetch pause
                        and no_fetch_window = '0'
              else '0';
end generate NOQUEUE;

QUEUE : if (C_SG_FTCH_DESC2QUEUE /= 0) generate
    service_ch2 <= '1' when ch2_run_stop = '1'          -- Channel running
                        and ch2_sg_idle = '0'           -- SG Engine running
                        and ch2_ftch_queue_full = '0'   -- Queue not full
                        and updt_error = '0'            -- No SG Update error
                        and ch2_stale_descriptor = '0'  -- No Stale Descriptors
                        and ch2_desc_flush = '0'        -- Not flushing desc
                        and ch2_pause_fetch = '0'       -- No fetch pause
              else '0';
end generate QUEUE;

end generate MCDMA;


NO_MCDMA : if (C_ENABLE_MULTI_CHANNEL = 0) generate
    service_ch2 <= '1' when ch2_run_stop = '1'          -- Channel running
                        and ch2_sg_idle = '0'           -- SG Engine running
                        and ch2_ftch_queue_full = '0'   -- Queue not full
                        and updt_error = '0'            -- No SG Update error
                        and ch2_stale_descriptor = '0'  -- No Stale Descriptors
                        and ch2_desc_flush = '0'        -- Not flushing desc
                        and ch2_pause_fetch = '0'       -- No fetch pause
              else '0';
end generate NO_MCDMA;

    -------------------------------------------------------------------------------
    -- Log Fetch Errors
    -------------------------------------------------------------------------------
    INT_ERROR_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                -- If reset or stopped then clear idle bit
                if(m_axi_sg_aresetn = '0')then
                    ch2_ftch_interr_set_i  <= '0';
                -- Channel active and datamover int error or fetch done and descriptor stale
                elsif((ch2_active_i = '1' and ftch_interr = '1')
                   or ((ftch_done = '1' or ftch_error = '1') and ch2_stale_descriptor = '1'))then
                    ch2_ftch_interr_set_i  <= '1';
                end if;
            end if;
        end process INT_ERROR_PROCESS;

    ch2_ftch_interr_set <= ch2_ftch_interr_set_i;

    SLV_ERROR_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                -- If reset or stopped then clear idle bit
                if(m_axi_sg_aresetn = '0')then
                    ch2_ftch_slverr_set  <= '0';
                elsif(ch2_active_i = '1' and ftch_slverr = '1')then
                    ch2_ftch_slverr_set  <= '1';
                end if;
            end if;
        end process SLV_ERROR_PROCESS;

    DEC_ERROR_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                -- If reset or stopped then clear idle bit
                if(m_axi_sg_aresetn = '0')then
                    ch2_ftch_decerr_set  <= '0';
                elsif(ch2_active_i = '1' and ftch_decerr = '1')then
                    ch2_ftch_decerr_set  <= '1';
                end if;
            end if;
        end process DEC_ERROR_PROCESS;

    -- Early detection of SlvErr or DecErr, used to prevent error'ed descriptor
    -- from being used by dma controller
    ch2_ftch_err_early  <= '1' when ftch_error_early = '1' and ch2_active_i = '1'
                      else '0';


    -- Enable stale descriptor check
    GEN_CH2_STALE_CHECK : if C_SG_CH2_ENBL_STALE_ERROR = 1 generate
    begin
        -----------------------------------------------------------------------
        -- Stale Descriptor Error
        -----------------------------------------------------------------------
        CH2_STALE_DESC : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    -- If reset then clear flag
                    if(m_axi_sg_aresetn = '0')then
                        ch2_stale_descriptor <= '0';
                    elsif(ftch_stale_desc = '1' and ch2_active_i = '1' )then
                        ch2_stale_descriptor <= '1';
                    end if;
                end if;
            end process CH2_STALE_DESC;

    end generate GEN_CH2_STALE_CHECK;

    -- Disable stale descriptor check
    GEN_CH2_NO_STALE_CHECK : if C_SG_CH2_ENBL_STALE_ERROR = 0 generate
    begin
        ch2_stale_descriptor <= '0';
    end generate GEN_CH2_NO_STALE_CHECK;

    -- Early detection of Stale Descriptor (valid only in tailpntr mode) used
    -- to prevent error'ed descriptor from being used.
    ch2_ftch_stale_desc <= ch2_stale_descriptor;

end generate GEN_CH2_FETCH;

-------------------------------------------------------------------------------
-- Channel excluded therefore do not generate fetch logic
-------------------------------------------------------------------------------
GEN_NO_CH2_FETCH : if C_INCLUDE_CH2 = 0 generate
begin
    service_ch2         <= '0';
    ch2_active_i        <= '0';
    ch2_ftch_idle       <= '0';
    ch2_ftch_interr_set <= '0';
    ch2_ftch_slverr_set <= '0';
    ch2_ftch_decerr_set <= '0';
    ch2_ftch_err_early  <= '0';
    ch2_ftch_stale_desc <= '0';
end generate GEN_NO_CH2_FETCH;

-------------------------------------------------------------------------------
-- Build DataMover command
-------------------------------------------------------------------------------
-- Assign fetch address
fetch_cmd_addr  <= ch1_fetch_address when ch1_active_i = '1'
             else  ch2_fetch_address;

-- Assign bytes to transfer (BTT)
fetch_cmd_btt   <= FETCH_CH1_CMD_BTT when ch1_active_i = '1'
             else  FETCH_CH2_CMD_BTT;

fetch_tag <= "0001" when ch1_active_i = '1'
              else "0000";

-- When command by sm, drive command to ftch_cmdsts_if
--GEN_DATAMOVER_CMND : process(m_axi_sg_aclk)
--    begin
--        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
--            if(m_axi_sg_aresetn = '0')then
--                ftch_cmnd_wr    <= '0';
--                ftch_cmnd_data  <= (others => '0');

--            -- Fetch SM issued a command write
--            elsif(write_cmnd_cmb = '1')then
--                ftch_cmnd_wr    <= '1';
--                ftch_cmnd_data  <=  FETCH_CMD_RSVD
--                                    & fetch_tag
--                                    & fetch_cmd_addr
--                                    & FETCH_MSB_IGNORED
--                                    & FETCH_CMD_TYPE
--                                    & FETCH_LSB_IGNORED
--                                    & fetch_cmd_btt;
--            else
--                ftch_cmnd_wr    <= '0';

--            end if;
--        end if;
--    end process GEN_DATAMOVER_CMND;


                ftch_cmnd_wr    <= write_cmnd_cmb;
                ftch_cmnd_data  <=  FETCH_CMD_RSVD
                                    & fetch_tag
                                    & fetch_cmd_addr
                                    & FETCH_MSB_IGNORED
                                    & FETCH_CMD_TYPE
                                    & FETCH_LSB_IGNORED
                                    & fetch_cmd_btt;

-------------------------------------------------------------------------------
-- Capture and hold fetch address in case an error occurs
-------------------------------------------------------------------------------
LOG_ERROR_ADDR : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                ftch_error_addr (C_M_AXI_SG_ADDR_WIDTH-1 downto SG_ADDR_LSB)    <= (others => '0');
            elsif(write_cmnd_cmb = '1')then
              ftch_error_addr (C_M_AXI_SG_ADDR_WIDTH-1 downto SG_ADDR_LSB)   <= fetch_cmd_addr (C_M_AXI_SG_ADDR_WIDTH-1 downto SG_ADDR_LSB);
            end if;
        end if;
    end process LOG_ERROR_ADDR;

     ftch_error_addr (5 downto 0) <= "000000";

end implementation;


-- *************************************************************************
--
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:    axi_sg_ftch_pntr.vhd
-- Description: This entity manages descriptor pointers and determine scatter
--              gather idle mode.
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_sg_v4_1_6;
use axi_sg_v4_1_6.axi_sg_pkg.all;

-------------------------------------------------------------------------------
entity  axi_sg_ftch_pntr is
    generic (
        C_M_AXI_SG_ADDR_WIDTH       : integer range 32 to 64        := 32   ;
            -- Master AXI Memory Map Address Width for Scatter Gather R/W Port

        C_INCLUDE_CH1               : integer range 0 to 1          := 1    ;
            -- Include or Exclude channel 1 scatter gather engine
            -- 0 = Exclude Channel 1 SG Engine
            -- 1 = Include Channel 1 SG Engine

        C_INCLUDE_CH2               : integer range 0 to 1          := 1           
            -- Include or Exclude channel 2 scatter gather engine
            -- 0 = Exclude Channel 2 SG Engine
            -- 1 = Include Channel 2 SG Engine
    );
    port (
        -----------------------------------------------------------------------
        -- AXI Scatter Gather Interface
        -----------------------------------------------------------------------
        m_axi_sg_aclk               : in  std_logic                         ;                  --
        m_axi_sg_aresetn            : in  std_logic                         ;                  --
                                                                                               --
        nxtdesc                     : in  std_logic_vector                                     --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                  --
                                                                                               --
        -------------------------------                                                        --
        -- CHANNEL 1                                                                           --
        -------------------------------                                                        --
        ch1_run_stop                : in  std_logic                         ;                  --
        ch1_desc_flush              : in  std_logic                         ; --CR568950       --
                                                                                               --
        -- CURDESC update to fetch pointer on run/stop assertion                               --
        ch1_curdesc                 : in  std_logic_vector                                     --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                  --
                                                                                               --
        -- TAILDESC update on CPU write (from axi_dma_reg_module)                              --
        ch1_tailpntr_enabled        : in  std_logic                         ;                  --
        ch1_taildesc_wren           : in  std_logic                         ;                  --
        ch1_taildesc                : in  std_logic_vector                                     --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                  --
                                                                                               --
        -- NXTDESC update on descriptor fetch (from axi_sg_ftchq_if)                           --
        ch1_nxtdesc_wren            : in  std_logic                         ;                  --
                                                                                               --
        -- Current address of descriptor to fetch                                              --
        ch1_fetch_address           : out std_logic_vector                                     --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                  --
        ch1_sg_idle                 : out std_logic                         ;                  --
                                                                                               --
        -------------------------------                                                        --
        -- CHANNEL 2                                                                           --
        -------------------------------                                                        --
        ch2_run_stop                : in  std_logic                         ;                  --
        ch2_desc_flush              : in  std_logic                         ;--CR568950        --
        ch2_eof_detected            : in  std_logic                         ;                  --
                                                                                               --
        -- CURDESC update to fetch pointer on run/stop assertion                               --
        ch2_curdesc                 : in  std_logic_vector                                     --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                  --
                                                                                               --
        -- TAILDESC update on CPU write (from axi_dma_reg_module)                              --
        ch2_tailpntr_enabled        : in  std_logic                         ;                  --
        ch2_taildesc_wren           : in  std_logic                         ;                  --
        ch2_taildesc                : in  std_logic_vector                                     --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                  --
        tail_updt                   : in std_logic;
        tail_updt_latch             : out std_logic;
        ch2_updt_done               : in std_logic;
                                                                                               --
        -- NXTDESC update on descriptor fetch (from axi_sg_ftchq_if)                           --
        ch2_nxtdesc_wren            : in  std_logic                         ;                  --
                                                                                               --
        -- Current address of descriptor to fetch                                              --
        ch2_fetch_address           : out std_logic_vector                                     --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                  --
        ch2_sg_idle                 : out std_logic                         ;                   --
        bd_eq                       : out std_logic
    );

end axi_sg_ftch_pntr;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_sg_ftch_pntr is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";



-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------

-- No Constants Declared

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
signal ch1_run_stop_d1              : std_logic := '0';
signal ch1_run_stop_re              : std_logic := '0';
signal ch1_use_crntdesc             : std_logic := '0';
signal ch1_fetch_address_i          : std_logic_vector
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)
                                        := (others => '0');

signal ch2_run_stop_d1              : std_logic := '0';
signal ch2_run_stop_re              : std_logic := '0';
signal ch2_use_crntdesc             : std_logic := '0';
signal ch2_fetch_address_i          : std_logic_vector
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)
                                        := (others => '0');
signal first : std_logic;
signal eof_latch : std_logic;

signal ch2_sg_idle_int : std_logic;
-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin

-- Channel 1 is included therefore generate pointer logic
GEN_PNTR_FOR_CH1 : if C_INCLUDE_CH1 = 1 generate
begin


    GEN_RUNSTOP_RE : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    ch1_run_stop_d1 <= '0';
                else
                    ch1_run_stop_d1 <= ch1_run_stop;
                end if;
            end if;
        end process GEN_RUNSTOP_RE;

    ch1_run_stop_re <= ch1_run_stop and not ch1_run_stop_d1;


    ---------------------------------------------------------------------------
    -- At setting of run/stop need to use current descriptor pointer therefor
    -- flag for use
    ---------------------------------------------------------------------------
    GEN_INIT_PNTR : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0' or ch1_nxtdesc_wren = '1')then
                    ch1_use_crntdesc <= '0';
                elsif(ch1_run_stop_re = '1')then
                    ch1_use_crntdesc <= '1';
                end if;
            end if;
        end process GEN_INIT_PNTR;

    ---------------------------------------------------------------------------
    -- Register Current Fetch Address.  During start (run/stop asserts) reg
    -- curdesc pointer from register module.  Once running use nxtdesc pointer.
    ---------------------------------------------------------------------------
    REG_FETCH_ADDRESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    ch1_fetch_address_i <= (others => '0');
                -- On initial tail pointer write use current desc pointer
                elsif(ch1_use_crntdesc = '1' and ch1_nxtdesc_wren = '0')then
                    ch1_fetch_address_i <= ch1_curdesc;
                -- On desriptor fetch capture next pointer
                elsif(ch1_nxtdesc_wren = '1')then
                    ch1_fetch_address_i <= nxtdesc;
                end if;
            end if;
        end process REG_FETCH_ADDRESS;

    -- Pass address out of module
    -- Addresses are always 16 word 32-bit aligned
    ch1_fetch_address <= ch1_fetch_address_i (C_M_AXI_SG_ADDR_WIDTH-1 downto 6) & "000000";

    ---------------------------------------------------------------------------
    -- Compair tail descriptor pointer to scatter gather engine current
    -- descriptor pointer.  Set idle if matched.  Only check if DMA engine
    -- is running and current descriptor is in process of being fetched.  This
    -- forces at least 1 descriptor fetch before checking for IDLE condition.
    ---------------------------------------------------------------------------
    COMPARE_ADDRESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                -- SG is IDLE on reset and on stop.
                --CR568950 - reset idlag on descriptor flush
                --if(m_axi_sg_aresetn = '0' or ch1_run_stop = '0')then
                if(m_axi_sg_aresetn = '0' or ch1_run_stop = '0' or ch1_desc_flush = '1')then
                    ch1_sg_idle <= '1';

                -- taildesc_wren must be in this 'if' to force a minimum
                -- of 1 clock of sg_idle = '0'.
                elsif(ch1_taildesc_wren = '1' or ch1_tailpntr_enabled = '0')then
                    ch1_sg_idle <= '0';

                -- Descriptor at fetch_address is being fetched (wren=1)
                -- therefore safe to check if tail matches the fetch address
                elsif(ch1_nxtdesc_wren = '1'
                and ch1_taildesc = ch1_fetch_address_i)then
                    ch1_sg_idle <= '1';
                end if;
            end if;
        end process COMPARE_ADDRESS;

end generate GEN_PNTR_FOR_CH1;


-- Channel 1 is NOT included therefore tie off pointer logic
GEN_NO_PNTR_FOR_CH1 : if C_INCLUDE_CH1 = 0 generate
begin
    ch1_fetch_address   <= (others =>'0');
    ch1_sg_idle         <= '0';
end generate GEN_NO_PNTR_FOR_CH1;

-- Channel 2 is included therefore generate pointer logic
GEN_PNTR_FOR_CH2 : if C_INCLUDE_CH2 = 1 generate
begin

    ---------------------------------------------------------------------------
    -- Create clock delay of run_stop in order to generate a rising edge pulse
    ---------------------------------------------------------------------------
    GEN_RUNSTOP_RE : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    ch2_run_stop_d1 <= '0';
                else
                    ch2_run_stop_d1 <= ch2_run_stop;
                end if;
            end if;
        end process GEN_RUNSTOP_RE;

    ch2_run_stop_re <= ch2_run_stop and not ch2_run_stop_d1;

    ---------------------------------------------------------------------------
    -- At setting of run/stop need to use current descriptor pointer therefor
    -- flag for use
    ---------------------------------------------------------------------------
    GEN_INIT_PNTR : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0' or ch2_nxtdesc_wren = '1')then
                    ch2_use_crntdesc <= '0';
                elsif(ch2_run_stop_re = '1')then
                    ch2_use_crntdesc <= '1';
                end if;
            end if;
        end process GEN_INIT_PNTR;

    ---------------------------------------------------------------------------
    -- Register Current Fetch Address.  During start (run/stop asserts) reg
    -- curdesc pointer from register module.  Once running use nxtdesc pointer.
    ---------------------------------------------------------------------------
    REG_FETCH_ADDRESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    ch2_fetch_address_i <= (others => '0');
                -- On initial tail pointer write use current desc pointer
                elsif((ch2_use_crntdesc = '1' and ch2_nxtdesc_wren = '0'))then
                    ch2_fetch_address_i <= ch2_curdesc;
                -- On descirptor fetch capture next pointer
                elsif(ch2_nxtdesc_wren = '1')then
                    ch2_fetch_address_i <= nxtdesc;
                end if;
            end if;
        end process REG_FETCH_ADDRESS;

    -- Pass address out of module
    -- Addresses are always 16 word 32-bit aligned
    ch2_fetch_address <= ch2_fetch_address_i (C_M_AXI_SG_ADDR_WIDTH-1 downto 6) & "000000";

    ---------------------------------------------------------------------------
    -- Compair tail descriptor pointer to scatter gather engine current
    -- descriptor pointer.  Set idle if matched.  Only check if DMA engine
    -- is running and current descriptor is in process of being fetched.  This
    -- forces at least 1 descriptor fetch before checking for IDLE condition.
    ---------------------------------------------------------------------------
    COMPARE_ADDRESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                -- SG is IDLE on reset and on stop.
                --CR568950 - reset idlag on descriptor flush
                --if(m_axi_sg_aresetn = '0' or ch2_run_stop = '0')then
                if(m_axi_sg_aresetn = '0' or ch2_run_stop = '0' or ch2_desc_flush = '1' or ch2_eof_detected = '1')then
                    ch2_sg_idle <= '1';
                    ch2_sg_idle_int <= '1';

                -- taildesc_wren must be in this 'if' to force a minimum
                -- of 1 clock of sg_idle = '0'.
                elsif(ch2_taildesc_wren = '1' or ch2_tailpntr_enabled = '0')then
                    ch2_sg_idle <= '0';
                    ch2_sg_idle_int <= '0';

                -- Descriptor at fetch_address is being fetched (wren=1)
                -- therefore safe to check if tail matches the fetch address
                elsif(ch2_nxtdesc_wren = '1'
                and ch2_taildesc = ch2_fetch_address_i)then
                    ch2_sg_idle <= '1';
                    ch2_sg_idle_int <= '1';
                end if;
            end if;
        end process COMPARE_ADDRESS;

-- Needed for multi channel

    EOF_LATCH_PROC : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0' or ch2_taildesc_wren = '1' or eof_latch = '1')then  -- nned to have some reset condition here
                    eof_latch <= '0';
                elsif (ch2_sg_idle_int = '1' and ch2_updt_done = '1') then
                    eof_latch <= '1';
                end if;
            end if;
        end process EOF_LATCH_PROC;



    TAILUPDT_LATCH : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0' or eof_latch = '1')then  -- nned to have some reset condition here
                    tail_updt_latch <= '0';
                    first <= '0';
                elsif (tail_updt = '1') then
                    tail_updt_latch <= '0';
                elsif(ch2_taildesc_wren = '1' and first = '0')then
                    first <= '1';
                elsif(ch2_taildesc_wren = '1' and first = '1')then
                    tail_updt_latch <= '1';
                end if;
            end if;
        end process TAILUPDT_LATCH;

    EQUAL_BD : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0' or ch2_run_stop = '0' or ch2_desc_flush = '1')then
                    bd_eq <= '0';

                elsif(ch2_taildesc_wren = '1' or ch2_tailpntr_enabled = '0')then
                    bd_eq <= '0';

                elsif(ch2_nxtdesc_wren = '1'
                and ch2_taildesc = ch2_fetch_address_i)then
                    bd_eq <= '1';
                end if;
            end if;
        end process EQUAL_BD;


end generate GEN_PNTR_FOR_CH2;


-- Channel 2 is NOT included therefore tie off pointer logic
GEN_NO_PNTR_FOR_CH2 : if C_INCLUDE_CH2 = 0 generate
begin
    ch2_fetch_address   <= (others =>'0');
    ch2_sg_idle         <= '0';
    tail_updt_latch <= '0';
    bd_eq <= '0';
end generate GEN_NO_PNTR_FOR_CH2;

end implementation;


-- *************************************************************************
--
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:    axi_sg_ftch_cmdsts_if.vhd
-- Description: This entity is the descriptor fetch command and status inteface
--              for the Scatter Gather Engine AXI DataMover.
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_sg_v4_1_6;
use axi_sg_v4_1_6.axi_sg_pkg.all;

-------------------------------------------------------------------------------
entity  axi_sg_ftch_cmdsts_if is
    generic (
        C_M_AXI_SG_ADDR_WIDTH       : integer range 32 to 64        := 32
            -- Master AXI Memory Map Address Width for Scatter Gather R/W Port

    );
    port (
        -----------------------------------------------------------------------
        -- AXI Scatter Gather Interface
        -----------------------------------------------------------------------
        m_axi_sg_aclk               : in  std_logic                         ;                   --
        m_axi_sg_aresetn            : in  std_logic                         ;                   --
                                                                                                --
        -- Fetch command write interface from fetch sm                                          --
        ftch_cmnd_wr                : in  std_logic                         ;                   --
        ftch_cmnd_data              : in  std_logic_vector                                      --
                                        ((C_M_AXI_SG_ADDR_WIDTH+CMD_BASE_WIDTH)-1 downto 0);    --
                                                                                                --
        -- User Command Interface Ports (AXI Stream)                                            --
        s_axis_ftch_cmd_tvalid      : out std_logic                         ;                   --
        s_axis_ftch_cmd_tready      : in  std_logic                         ;                   --
        s_axis_ftch_cmd_tdata       : out std_logic_vector                                      --
                                        ((C_M_AXI_SG_ADDR_WIDTH+CMD_BASE_WIDTH)-1 downto 0);    --
                                                                                                --
        -- Read response for detecting slverr, decerr early                                     --
        m_axi_sg_rresp              : in  std_logic_vector(1 downto 0)      ;                   --
        m_axi_sg_rvalid             : in  std_logic                         ;                   --
                                                                                                --
        -- User Status Interface Ports (AXI Stream)                                             --
        m_axis_ftch_sts_tvalid      : in  std_logic                         ;                   --
        m_axis_ftch_sts_tready      : out std_logic                         ;                   --
        m_axis_ftch_sts_tdata       : in  std_logic_vector(7 downto 0)      ;                   --
        m_axis_ftch_sts_tkeep       : in  std_logic_vector(0 downto 0)      ;                   --
                                                                                                --
        -- Scatter Gather Fetch Status                                                          --
        mm2s_err                    : in  std_logic                         ;                   --
        ftch_done                   : out std_logic                         ;                   --
        ftch_error                  : out std_logic                         ;                   --
        ftch_interr                 : out std_logic                         ;                   --
        ftch_slverr                 : out std_logic                         ;                   --
        ftch_decerr                 : out std_logic                         ;                   --
        ftch_error_early            : out std_logic                                             --

    );

end axi_sg_ftch_cmdsts_if;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_sg_ftch_cmdsts_if is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------

-- No Constants Declared

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
signal ftch_slverr_i    : std_logic := '0';
signal ftch_decerr_i    : std_logic := '0';
signal ftch_interr_i    : std_logic := '0';
signal mm2s_error       : std_logic := '0';

signal sg_rresp         : std_logic_vector(1 downto 0) := (others => '0');
signal sg_rvalid        : std_logic := '0';

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin

ftch_slverr <= ftch_slverr_i;
ftch_decerr <= ftch_decerr_i;
ftch_interr <= ftch_interr_i;

-------------------------------------------------------------------------------
-- DataMover Command Interface
-------------------------------------------------------------------------------


-------------------------------------------------------------------------------
-- When command by fetch sm, drive descriptor fetch command to data mover.
-- Hold until data mover indicates ready.
-------------------------------------------------------------------------------
GEN_DATAMOVER_CMND : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                s_axis_ftch_cmd_tvalid  <= '0';
             --   s_axis_ftch_cmd_tdata   <= (others => '0');

            elsif(ftch_cmnd_wr = '1')then
                s_axis_ftch_cmd_tvalid  <= '1';
             --   s_axis_ftch_cmd_tdata   <= ftch_cmnd_data;

            elsif(s_axis_ftch_cmd_tready = '1')then
                s_axis_ftch_cmd_tvalid  <= '0';
             --   s_axis_ftch_cmd_tdata   <= (others => '0');

            end if;
        end if;
    end process GEN_DATAMOVER_CMND;

                s_axis_ftch_cmd_tdata   <= ftch_cmnd_data;

-------------------------------------------------------------------------------
-- DataMover Status Interface
-------------------------------------------------------------------------------
-- Drive ready low during reset to indicate not ready
REG_STS_READY : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                m_axis_ftch_sts_tready <= '0';
            else
                m_axis_ftch_sts_tready <= '1';
            end if;
        end if;
    end process REG_STS_READY;

-------------------------------------------------------------------------------
-- Log status bits out of data mover.
-------------------------------------------------------------------------------
DATAMOVER_STS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                ftch_done      <= '0';
                ftch_slverr_i  <= '0';
                ftch_decerr_i  <= '0';
                ftch_interr_i  <= '0';
            -- Status valid, therefore capture status
            elsif(m_axis_ftch_sts_tvalid = '1')then
                ftch_done      <= m_axis_ftch_sts_tdata(DATAMOVER_STS_CMDDONE_BIT);
                ftch_slverr_i  <= m_axis_ftch_sts_tdata(DATAMOVER_STS_SLVERR_BIT);
                ftch_decerr_i  <= m_axis_ftch_sts_tdata(DATAMOVER_STS_DECERR_BIT);
                ftch_interr_i  <= m_axis_ftch_sts_tdata(DATAMOVER_STS_INTERR_BIT);
            -- Only assert when valid
            else
                ftch_done      <= '0';
                ftch_slverr_i  <= '0';
                ftch_decerr_i  <= '0';
                ftch_interr_i  <= '0';
            end if;
        end if;
    end process DATAMOVER_STS;


-------------------------------------------------------------------------------
-- Early SlvErr and DecErr detections
-- Early detection primarily required for non-queue mode because fetched desc
-- is immediatle fed to DMA controller.  Status from SG Datamover arrives
-- too late to stop the insuing transfer on fetch error
-------------------------------------------------------------------------------
REG_MM_RD_SIGNALS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                sg_rresp  <= (others => '0');
                sg_rvalid <= '0';
            else
                sg_rresp  <= m_axi_sg_rresp;
                sg_rvalid <= m_axi_sg_rvalid;
            end if;
        end if;
    end process REG_MM_RD_SIGNALS;


REG_ERLY_FTCH_ERROR : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                ftch_error_early     <= '0';
            elsif(sg_rvalid = '1' and (sg_rresp = SLVERR_RESP
                                    or sg_rresp = DECERR_RESP))then
                ftch_error_early     <= '1';
            end if;
        end if;
    end process REG_ERLY_FTCH_ERROR;


-------------------------------------------------------------------------------
-- Register global error from data mover.
-------------------------------------------------------------------------------
mm2s_error <= ftch_slverr_i or ftch_decerr_i or ftch_interr_i;

-- Log errors into a global error output
FETCH_ERROR_PROCESS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                ftch_error <= '0';
            elsif(mm2s_error = '1')then
                ftch_error <= '1';
            end if;
        end if;
    end process FETCH_ERROR_PROCESS;

end implementation;


-- *************************************************************************
--
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:          axi_sg_ftch_mngr.vhd
-- Description: This entity manages fetching of descriptors.
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_sg_v4_1_6;
use axi_sg_v4_1_6.axi_sg_pkg.all;


-------------------------------------------------------------------------------
entity  axi_sg_ftch_mngr is
    generic (
        C_M_AXI_SG_ADDR_WIDTH       : integer range 32 to 64    := 32;
            -- Master AXI Memory Map Address Width for Scatter Gather R/W Port
        C_ENABLE_MULTI_CHANNEL         : integer range 0 to 1          := 0;

        C_INCLUDE_CH1           : integer range 0 to 1          := 1;
            -- Include or Exclude channel 1 scatter gather engine
            -- 0 = Exclude Channel 1 SG Engine
            -- 1 = Include Channel 1 SG Engine

        C_INCLUDE_CH2           : integer range 0 to 1          := 1;
            -- Include or Exclude channel 2 scatter gather engine
            -- 0 = Exclude Channel 2 SG Engine
            -- 1 = Include Channel 2 SG Engine

        C_SG_CH1_WORDS_TO_FETCH : integer range 4 to 16         := 8;
            -- Number of words to fetch for channel 1

        C_SG_CH2_WORDS_TO_FETCH : integer range 4 to 16         := 8;
            -- Number of words to fetch for channel 1

        C_SG_FTCH_DESC2QUEUE     : integer range 0 to 8         := 0;
            -- Number of descriptors to fetch and queue for each channel.
            -- A value of zero excludes the fetch queues.

        C_SG_CH1_ENBL_STALE_ERROR   : integer range 0 to 1      := 1;
            -- Enable or disable stale descriptor check
            -- 0 = Disable stale descriptor error check
            -- 1 = Enable stale descriptor error check

        C_SG_CH2_ENBL_STALE_ERROR   : integer range 0 to 1      := 1
            -- Enable or disable stale descriptor check
            -- 0 = Disable stale descriptor error check
            -- 1 = Enable stale descriptor error check



    );
    port (
        -----------------------------------------------------------------------
        -- AXI Scatter Gather Interface
        -----------------------------------------------------------------------
        m_axi_sg_aclk               : in  std_logic                         ;                    --
        m_axi_sg_aresetn            : in  std_logic                         ;                    --
                                                                                                 --
        -- Channel 1 Control and Status                                                          --
        ch1_run_stop                : in  std_logic                         ;                    --
        ch1_desc_flush              : in  std_logic                         ;                    --
        ch1_updt_done               : in  std_logic                         ;                    --
        ch1_ftch_idle               : out std_logic                         ;                    --
        ch1_ftch_active             : out std_logic                         ;                    --
        ch1_ftch_interr_set         : out std_logic                         ;                    --
        ch1_ftch_slverr_set         : out std_logic                         ;                    --
        ch1_ftch_decerr_set         : out std_logic                         ;                    --
        ch1_ftch_err_early          : out std_logic                         ;                    --
        ch1_ftch_stale_desc         : out std_logic                         ;                    --
        ch1_tailpntr_enabled        : in  std_logic                         ;                    --
        ch1_taildesc_wren           : in  std_logic                         ;                    --
        ch1_taildesc                : in  std_logic_vector                                       --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                    --
        ch1_nxtdesc_wren            : in  std_logic                         ;                    --
        ch1_curdesc                 : in  std_logic_vector                                       --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                    --
        ch1_ftch_queue_empty        : in  std_logic                         ;                    --
        ch1_ftch_queue_full         : in  std_logic                         ;                    --
        ch1_ftch_pause              : in  std_logic                         ;                    --
                                                                                                 --
        -- Channel 2 Control and Status                                                          --
        ch2_run_stop                : in  std_logic                         ;                    --
        ch2_updt_done               : in  std_logic                         ;                    --
        ch2_desc_flush              : in  std_logic                         ;                    --
        ch2_ftch_idle               : out std_logic                         ;                    --
        ch2_ftch_active             : out std_logic                         ;                    --
        ch2_ftch_interr_set         : out std_logic                         ;                    --
        ch2_ftch_slverr_set         : out std_logic                         ;                    --
        ch2_ftch_decerr_set         : out std_logic                         ;                    --
        ch2_ftch_err_early          : out std_logic                         ;                    --
        ch2_ftch_stale_desc         : out std_logic                         ;                    --
        ch2_tailpntr_enabled        : in  std_logic                         ;                    --
        ch2_taildesc_wren           : in  std_logic                         ;                    --
        ch2_taildesc                : in  std_logic_vector                                       --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                    --
        ch2_nxtdesc_wren            : in  std_logic                         ;                    --
        ch2_curdesc                 : in  std_logic_vector                                       --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                    --
        ch2_ftch_queue_empty        : in  std_logic                         ;                    --
        ch2_ftch_queue_full         : in  std_logic                         ;                    --
        ch2_ftch_pause              : in  std_logic                         ;                    --
        ch2_eof_detected            : in  std_logic                         ;
        tail_updt                   : in std_logic                         ; 
        tail_updt_latch             : out std_logic                         ; 
        ch2_sg_idle                 : out std_logic                         ;
        no_fetch_window             : in std_logic                          ;
                                                                                                 --
        nxtdesc                     : in  std_logic_vector                                       --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                    --
                                                                                                 --
        -- Read response for detecting slverr, decerr early                                      --
        m_axi_sg_rresp              : in  std_logic_vector(1 downto 0)      ;                    --
        m_axi_sg_rvalid             : in  std_logic                         ;                    --
                                                                                                 --
        -- User Command Interface Ports (AXI Stream)                                             --
        s_axis_ftch_cmd_tvalid      : out std_logic                         ;                    --
        s_axis_ftch_cmd_tready      : in  std_logic                         ;                    --
        s_axis_ftch_cmd_tdata       : out std_logic_vector                                       --
                                        ((C_M_AXI_SG_ADDR_WIDTH+CMD_BASE_WIDTH)-1 downto 0);     --
                                                                                                 --
        -- User Status Interface Ports (AXI Stream)                                              --
        m_axis_ftch_sts_tvalid      : in  std_logic                         ;                    --
        m_axis_ftch_sts_tready      : out std_logic                         ;                    --
        m_axis_ftch_sts_tdata       : in  std_logic_vector(7 downto 0)      ;                    --
        m_axis_ftch_sts_tkeep       : in  std_logic_vector(0 downto 0)      ;                    --
        mm2s_err                    : in  std_logic                         ;                    --
                                                                                                 --
                                                                                                 --
        ftch_cmnd_wr                : out std_logic                         ;                    --
        ftch_cmnd_data              : out std_logic_vector                                       --
                                        ((C_M_AXI_SG_ADDR_WIDTH+CMD_BASE_WIDTH)-1 downto 0);     --
        ftch_stale_desc             : in  std_logic                         ;                    --
        updt_error                  : in  std_logic                         ;                    --
        ftch_error                  : out std_logic                         ;                    --
        ftch_error_addr             : out std_logic_vector                                       --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                     --
        bd_eq                       : out std_logic

    );

end axi_sg_ftch_mngr;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_sg_ftch_mngr is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------

-- No Constants Declared

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
signal ftch_cmnd_wr_i               : std_logic := '0';
signal ftch_cmnd_data_i             : std_logic_vector
                                        ((C_M_AXI_SG_ADDR_WIDTH+CMD_BASE_WIDTH)-1 downto 0)
                                        := (others => '0');

signal ch1_sg_idle                  : std_logic := '0';
signal ch1_fetch_address            : std_logic_vector
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)
                                        := (others => '0');

signal ch2_sg_idle_int                  : std_logic := '0';
signal ch2_fetch_address            : std_logic_vector
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)
                                        := (others => '0');

signal ftch_done                    : std_logic := '0';
signal ftch_error_i                 : std_logic := '0';
signal ftch_interr                  : std_logic := '0';
signal ftch_slverr                  : std_logic := '0';
signal ftch_decerr                  : std_logic := '0';
signal ftch_error_early             : std_logic := '0';

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin
ftch_cmnd_wr        <= ftch_cmnd_wr_i;
ftch_cmnd_data      <= ftch_cmnd_data_i;
ftch_error          <= ftch_error_i;
ch2_sg_idle <= ch2_sg_idle_int; 
-------------------------------------------------------------------------------
--  Scatter Gather Fetch State Machine
-------------------------------------------------------------------------------
I_FTCH_SG : entity  axi_sg_v4_1_6.axi_sg_ftch_sm
    generic map(
        C_M_AXI_SG_ADDR_WIDTH       => C_M_AXI_SG_ADDR_WIDTH                ,
        C_ENABLE_MULTI_CHANNEL             => C_ENABLE_MULTI_CHANNEL                      ,
        C_INCLUDE_CH1               => C_INCLUDE_CH1                        ,
        C_INCLUDE_CH2               => C_INCLUDE_CH2                        ,
        C_SG_CH1_WORDS_TO_FETCH     => C_SG_CH1_WORDS_TO_FETCH              ,
        C_SG_CH2_WORDS_TO_FETCH     => C_SG_CH2_WORDS_TO_FETCH              ,
        C_SG_FTCH_DESC2QUEUE        => C_SG_FTCH_DESC2QUEUE                 ,
        C_SG_CH1_ENBL_STALE_ERROR   => C_SG_CH1_ENBL_STALE_ERROR            ,
        C_SG_CH2_ENBL_STALE_ERROR   => C_SG_CH2_ENBL_STALE_ERROR
    )
    port map(
        -----------------------------------------------------------------------
        -- AXI Scatter Gather Interface
        -----------------------------------------------------------------------
        m_axi_sg_aclk               => m_axi_sg_aclk                        ,
        m_axi_sg_aresetn            => m_axi_sg_aresetn                     ,
        updt_error                  => updt_error                           ,

        -- Channel 1 Control and Status
        ch1_run_stop                => ch1_run_stop                         ,
        ch1_updt_done               => ch1_updt_done                        ,
        ch1_desc_flush              => ch1_desc_flush                       ,
        ch1_sg_idle                 => ch1_sg_idle                          ,
        ch1_tailpntr_enabled        => ch1_tailpntr_enabled                 ,
        ch1_ftch_queue_empty        => ch1_ftch_queue_empty                 ,
        ch1_ftch_queue_full         => ch1_ftch_queue_full                  ,
        ch1_fetch_address           => ch1_fetch_address                    ,
        ch1_ftch_active             => ch1_ftch_active                      ,
        ch1_ftch_idle               => ch1_ftch_idle                        ,
        ch1_ftch_interr_set         => ch1_ftch_interr_set                  ,
        ch1_ftch_slverr_set         => ch1_ftch_slverr_set                  ,
        ch1_ftch_decerr_set         => ch1_ftch_decerr_set                  ,
        ch1_ftch_err_early          => ch1_ftch_err_early                   ,
        ch1_ftch_stale_desc         => ch1_ftch_stale_desc                  ,
        ch1_ftch_pause              => ch1_ftch_pause                       ,

        -- Channel 2 Control and Status
        ch2_run_stop                => ch2_run_stop                         ,
        ch2_updt_done               => ch2_updt_done                        ,
        ch2_desc_flush              => ch2_desc_flush                       ,
        ch2_sg_idle                 => ch2_sg_idle_int                          ,
        ch2_tailpntr_enabled        => ch2_tailpntr_enabled                 ,
        ch2_ftch_queue_empty        => ch2_ftch_queue_empty                 ,
        ch2_ftch_queue_full         => ch2_ftch_queue_full                  ,
        ch2_fetch_address           => ch2_fetch_address                    ,
        ch2_ftch_active             => ch2_ftch_active                      ,
        ch2_ftch_idle               => ch2_ftch_idle                        ,
        ch2_ftch_interr_set         => ch2_ftch_interr_set                  ,
        ch2_ftch_slverr_set         => ch2_ftch_slverr_set                  ,
        ch2_ftch_decerr_set         => ch2_ftch_decerr_set                  ,
        ch2_ftch_err_early          => ch2_ftch_err_early                   ,
        ch2_ftch_stale_desc         => ch2_ftch_stale_desc                  ,
        ch2_ftch_pause              => ch2_ftch_pause                       ,
        no_fetch_window             => no_fetch_window                      ,

        -- Transfer Request
        ftch_cmnd_wr                => ftch_cmnd_wr_i                       ,
        ftch_cmnd_data              => ftch_cmnd_data_i                     ,

        -- Transfer Status
        ftch_done                   => ftch_done                            ,
        ftch_error                  => ftch_error_i                         ,
        ftch_interr                 => ftch_interr                          ,
        ftch_slverr                 => ftch_slverr                          ,
        ftch_decerr                 => ftch_decerr                          ,
        ftch_stale_desc             => ftch_stale_desc                      ,
        ftch_error_addr             => ftch_error_addr                      ,
        ftch_error_early            => ftch_error_early
    );

-------------------------------------------------------------------------------
--  Scatter Gather Fetch Pointer Manager
-------------------------------------------------------------------------------
I_FTCH_PNTR_MNGR : entity  axi_sg_v4_1_6.axi_sg_ftch_pntr
    generic map(
        C_M_AXI_SG_ADDR_WIDTH       => C_M_AXI_SG_ADDR_WIDTH                ,
        C_INCLUDE_CH1               => C_INCLUDE_CH1                        ,
        C_INCLUDE_CH2               => C_INCLUDE_CH2
    )
    port map(
        -----------------------------------------------------------------------
        -- AXI Scatter Gather Interface
        -----------------------------------------------------------------------
        m_axi_sg_aclk               => m_axi_sg_aclk                        ,
        m_axi_sg_aresetn            => m_axi_sg_aresetn                     ,


        nxtdesc                     => nxtdesc                              ,

        -------------------------------
        -- CHANNEL 1
        -------------------------------
        ch1_run_stop                => ch1_run_stop                         ,
        ch1_desc_flush              => ch1_desc_flush                       ,--CR568950

        -- CURDESC update on run/stop assertion (from ftch_sm)
        ch1_curdesc                 => ch1_curdesc                          ,

        -- TAILDESC update on CPU write (from axi_dma_reg_module)
        ch1_tailpntr_enabled        => ch1_tailpntr_enabled                 ,
        ch1_taildesc_wren           => ch1_taildesc_wren                    ,
        ch1_taildesc                => ch1_taildesc                         ,

        -- NXTDESC update on descriptor fetch (from axi_sg_ftchq_if)
        ch1_nxtdesc_wren            => ch1_nxtdesc_wren                     ,

        -- Current address of descriptor to fetch
        ch1_fetch_address           => ch1_fetch_address                    ,
        ch1_sg_idle                 => ch1_sg_idle                          ,

        -------------------------------
        -- CHANNEL 2
        -------------------------------
        ch2_run_stop                => ch2_run_stop                         ,
        ch2_desc_flush              => ch2_desc_flush                       ,--CR568950
        ch2_eof_detected            => ch2_eof_detected                     ,

        -- CURDESC update on run/stop assertion (from ftch_sm)
        ch2_curdesc                 => ch2_curdesc                          ,

        -- TAILDESC update on CPU write (from axi_dma_reg_module)
        ch2_tailpntr_enabled        => ch2_tailpntr_enabled                 ,
        ch2_taildesc_wren           => ch2_taildesc_wren                    ,
        ch2_taildesc                => ch2_taildesc                         ,
        tail_updt_latch             => tail_updt_latch                      ,   
        tail_updt                   => tail_updt                            ,   
        ch2_updt_done               => ch2_updt_done                        ,  
        -- NXTDESC update on descriptor fetch (from axi_sg_ftchq_if)
        ch2_nxtdesc_wren            => ch2_nxtdesc_wren                     ,

        -- Current address of descriptor to fetch
        ch2_fetch_address           => ch2_fetch_address                    ,
        ch2_sg_idle                 => ch2_sg_idle_int                      ,
        bd_eq                       => bd_eq
    );

-------------------------------------------------------------------------------
--  Scatter Gather Fetch Command / Status Interface
-------------------------------------------------------------------------------
I_FTCH_CMDSTS_IF : entity  axi_sg_v4_1_6.axi_sg_ftch_cmdsts_if
    generic map(
        C_M_AXI_SG_ADDR_WIDTH       => C_M_AXI_SG_ADDR_WIDTH
    )
    port map(
        -----------------------------------------------------------------------
        -- AXI Scatter Gather Interface
        -----------------------------------------------------------------------
        m_axi_sg_aclk               => m_axi_sg_aclk                        ,
        m_axi_sg_aresetn            => m_axi_sg_aresetn                     ,

        -- Fetch command write interface from fetch sm
        ftch_cmnd_wr                => ftch_cmnd_wr_i                       ,
        ftch_cmnd_data              => ftch_cmnd_data_i                     ,


        -- Read response for detecting slverr, decerr early
        m_axi_sg_rresp              => m_axi_sg_rresp                       ,
        m_axi_sg_rvalid             => m_axi_sg_rvalid                      ,

        -- User Command Interface Ports (AXI Stream)
        s_axis_ftch_cmd_tvalid      => s_axis_ftch_cmd_tvalid               ,
        s_axis_ftch_cmd_tready      => s_axis_ftch_cmd_tready               ,
        s_axis_ftch_cmd_tdata       => s_axis_ftch_cmd_tdata                ,

        -- User Status Interface Ports (AXI Stream)
        m_axis_ftch_sts_tvalid      => m_axis_ftch_sts_tvalid               ,
        m_axis_ftch_sts_tready      => m_axis_ftch_sts_tready               ,
        m_axis_ftch_sts_tdata       => m_axis_ftch_sts_tdata                ,
        m_axis_ftch_sts_tkeep       => m_axis_ftch_sts_tkeep                ,

        -- Scatter Gather Fetch Status
        mm2s_err                    => mm2s_err                             ,

        ftch_done                   => ftch_done                            ,
        ftch_error                  => ftch_error_i                         ,
        ftch_interr                 => ftch_interr                          ,
        ftch_slverr                 => ftch_slverr                          ,
        ftch_decerr                 => ftch_decerr                          ,
        ftch_error_early            => ftch_error_early
    );




end implementation;


-- *************************************************************************
--
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:          axi_sg_cntrl_strm.vhd
-- Description: This entity is MM2S control stream logic
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_sg_v4_1_6;
use axi_sg_v4_1_6.axi_sg_pkg.all;

library lib_fifo_v1_0_8;
library lib_cdc_v1_0_2;

library lib_pkg_v1_0_2;
use lib_pkg_v1_0_2.lib_pkg.clog2;
use lib_pkg_v1_0_2.lib_pkg.max2;

-------------------------------------------------------------------------------
entity  axi_sg_cntrl_strm is
    generic(
        C_PRMRY_IS_ACLK_ASYNC           : integer range 0 to 1      := 0;
            -- Primary MM2S/S2MM sync/async mode
            -- 0 = synchronous mode     - all clocks are synchronous
            -- 1 = asynchronous mode    - Primary data path channels (MM2S and S2MM)
            --                            run asynchronous to AXI Lite, DMA Control,
            --                            and SG.

        C_PRMY_CMDFIFO_DEPTH        : integer range 1 to 16         := 1;
            -- Depth of DataMover command FIFO

        C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH : integer range 32 to 32    := 32;
            -- Master AXI Control Stream Data Width

        C_FAMILY                        : string                    := "virtex7"
            -- Target FPGA Device Family
    );
    port (
        -- Secondary clock / reset
        m_axi_sg_aclk               : in  std_logic                         ;           --
        m_axi_sg_aresetn            : in  std_logic                         ;           --
                                                                                        --
        -- Primary clock / reset                                                        --
        axi_prmry_aclk              : in  std_logic                         ;           --
        p_reset_n                   : in  std_logic                         ;           --
                                                                                        --
        -- MM2S Error                                                                   --
        mm2s_stop                   : in  std_logic                         ;           --
                                                                                        --
        -- Control Stream FIFO write signals (from axi_dma_mm2s_sg_if)                  --
        cntrlstrm_fifo_wren         : in  std_logic                         ;           --
        cntrlstrm_fifo_din          : in  std_logic_vector                              --
                                        (C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH downto 0);     --
        cntrlstrm_fifo_full         : out std_logic                         ;           --
                                                                                        --
                                                                                        --
        -- Memory Map to Stream Control Stream Interface                                --
        m_axis_mm2s_cntrl_tdata     : out std_logic_vector                              --
                                        (C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH-1 downto 0);   --
        m_axis_mm2s_cntrl_tkeep     : out std_logic_vector                              --
                                        ((C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH/8)-1 downto 0);--
        m_axis_mm2s_cntrl_tvalid    : out std_logic                         ;           --
        m_axis_mm2s_cntrl_tready    : in  std_logic                         ;           --
        m_axis_mm2s_cntrl_tlast     : out std_logic                                     --



    );
end axi_sg_cntrl_strm;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_sg_cntrl_strm is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------

-- Number of words deep fifo needs to be
-- Only 5 app fields, but set to 8 so depth is a power of 2
constant CNTRL_FIFO_DEPTH       : integer := max2(16,8 * C_PRMY_CMDFIFO_DEPTH);


-- Width of fifo rd and wr counts - only used for proper fifo operation
constant CNTRL_FIFO_CNT_WIDTH   : integer   := clog2(CNTRL_FIFO_DEPTH+1);

constant USE_LOGIC_FIFOS        : integer   := 0; -- Use Logic FIFOs
constant USE_BRAM_FIFOS         : integer   := 1; -- Use BRAM FIFOs

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
-- FIFO signals
signal cntrl_fifo_rden  : std_logic := '0';
signal cntrl_fifo_empty : std_logic := '0';
signal cntrl_fifo_dout, follower_reg_mm2s  : std_logic_vector
                            (C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH downto 0) := (others => '0');
signal cntrl_fifo_dvalid: std_logic := '0';

signal cntrl_tdata      : std_logic_vector
                            (C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH-1 downto 0) := (others => '0');
signal cntrl_tkeep      : std_logic_vector
                            ((C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH/8)-1 downto 0) := (others => '0');
signal follower_full_mm2s, follower_empty_mm2s : std_logic := '0';
signal cntrl_tvalid     : std_logic := '0';
signal cntrl_tready     : std_logic := '0';
signal cntrl_tlast      : std_logic := '0';
signal sinit            : std_logic := '0';

signal m_valid          : std_logic := '0';
signal m_ready          : std_logic := '0';
signal m_data           : std_logic_vector(C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH-1 downto 0) := (others => '0');
signal m_strb           : std_logic_vector((C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH/8)-1 downto 0) := (others => '0');
signal m_last           : std_logic := '0';

signal skid_rst         : std_logic := '0';
-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin
-- All bytes always valid
cntrl_tkeep <= (others => '1');


-- Primary Clock is synchronous to Secondary Clock therfore
-- instantiate a sync fifo.
GEN_SYNC_FIFO : if C_PRMRY_IS_ACLK_ASYNC = 0 generate
signal mm2s_stop_d1     : std_logic := '0';
signal mm2s_stop_re     : std_logic := '0';
signal xfer_in_progress : std_logic := '0';
begin
    -- reset on hard reset or mm2s stop
    sinit   <= not m_axi_sg_aresetn or mm2s_stop;

    -- Generate Synchronous FIFO
    I_CNTRL_FIFO : entity lib_fifo_v1_0_8.sync_fifo_fg
    generic map (
        C_FAMILY                =>  C_FAMILY                ,
        C_MEMORY_TYPE           =>  USE_LOGIC_FIFOS,
        C_WRITE_DATA_WIDTH      =>  C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH + 1,
        C_WRITE_DEPTH           =>  CNTRL_FIFO_DEPTH       ,
        C_READ_DATA_WIDTH       =>  C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH + 1,
        C_READ_DEPTH            =>  CNTRL_FIFO_DEPTH       ,
        C_PORTS_DIFFER          =>  0,
        C_HAS_DCOUNT            =>  0, --req for proper fifo operation
        C_HAS_ALMOST_FULL       =>  0,
        C_HAS_RD_ACK            =>  0,
        C_HAS_RD_ERR            =>  0,
        C_HAS_WR_ACK            =>  0,
        C_HAS_WR_ERR            =>  0,
        C_RD_ACK_LOW            =>  0,
        C_RD_ERR_LOW            =>  0,
        C_WR_ACK_LOW            =>  0,
        C_WR_ERR_LOW            =>  0,
        C_PRELOAD_REGS          =>  1,-- 1 = first word fall through
        C_PRELOAD_LATENCY       =>  0 -- 0 = first word fall through
 --       C_USE_EMBEDDED_REG      =>  1 -- 0 ;
    )
    port map (

        Clk             =>  m_axi_sg_aclk       ,
        Sinit           =>  sinit               ,
        Din             =>  cntrlstrm_fifo_din  ,
        Wr_en           =>  cntrlstrm_fifo_wren ,
        Rd_en           =>  cntrl_fifo_rden     ,
        Dout            =>  cntrl_fifo_dout     ,
        Full            =>  cntrlstrm_fifo_full ,
        Empty           =>  cntrl_fifo_empty    ,
        Almost_full     =>  open                ,
        Data_count      =>  open                ,
        Rd_ack          =>  open                ,
        Rd_err          =>  open                ,
        Wr_ack          =>  open                ,
        Wr_err          =>  open

    );


--       I_UPDT_DATA_FIFO : entity proc_common_srl_fifo_v5_0.srl_fifo_f
--       generic map (
--         C_DWIDTH            =>  33   ,
--         C_DEPTH             =>  24   ,
--         C_FAMILY            =>  C_FAMILY
--         )
--       port map (
--         Clk           =>  m_axi_sg_aclk  ,
--         Reset         =>  sinit          ,
--         FIFO_Write    =>  cntrlstrm_fifo_wren  ,
--         Data_In       =>  cntrlstrm_fifo_din ,
--         FIFO_Read     =>  cntrl_fifo_rden ,
--         Data_Out      =>  cntrl_fifo_dout ,
--         FIFO_Empty    =>  cntrl_fifo_empty ,
--         FIFO_Full     =>  cntrlstrm_fifo_full,
--         Addr          =>  open
--         );


    cntrl_fifo_rden <= follower_empty_mm2s and (not cntrl_fifo_empty);

    VALID_REG_MM2S_ACTIVE : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0' or (cntrl_tready = '1' and follower_full_mm2s = '1'))then
                   --    follower_reg_mm2s <= (others => '0');
                       follower_full_mm2s <= '0';
                       follower_empty_mm2s <= '1'; 
                else
                    if (cntrl_fifo_rden = '1') then
                   --    follower_reg_mm2s <= sts_queue_dout; 
                       follower_full_mm2s <= '1';
                       follower_empty_mm2s <= '0'; 
                    end if;
                end if;
            end if;
        end process VALID_REG_MM2S_ACTIVE;


    VALID_REG_MM2S_ACTIVE1 : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                       follower_reg_mm2s <= (others => '0');
                else
                   if (cntrl_fifo_rden = '1') then
                       follower_reg_mm2s <= cntrl_fifo_dout;
                   end if;
                end if;
            end if;
        end process VALID_REG_MM2S_ACTIVE1;


    -----------------------------------------------------------------------
    -- Control Stream OUT Side
    -----------------------------------------------------------------------
    -- Read if fifo is not empty and target is ready
--    cntrl_fifo_rden  <= not cntrl_fifo_empty
--                        and cntrl_tready;

    -- Drive valid if fifo is not empty or in the middle
    -- of transfer and stop issued.
    cntrl_tvalid  <= follower_full_mm2s --not cntrl_fifo_empty
                    or (xfer_in_progress and mm2s_stop_re);

    -- Pass data out to control channel with MSB driving tlast
    cntrl_tlast   <= (cntrl_tvalid and follower_reg_mm2s(C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH))
                    or (xfer_in_progress and mm2s_stop_re);

    cntrl_tdata   <= follower_reg_mm2s(C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH-1 downto 0);

    -- Register stop to create re pulse for cleaning shutting down
    -- stream out during soft reset.
    REG_STOP : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    mm2s_stop_d1    <= '0';
                else
                    mm2s_stop_d1    <= mm2s_stop;
                end if;
            end if;
        end process REG_STOP;

    mm2s_stop_re <= mm2s_stop and not mm2s_stop_d1;

    -------------------------------------------------------------
    -- Flag transfer in progress. If xfer in progress then
    -- a fake tlast and tvalid need to be asserted during soft
    -- reset else no need of tlast.
    -------------------------------------------------------------
    TRANSFER_IN_PROGRESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(cntrl_tlast = '1' and cntrl_tvalid = '1' and cntrl_tready = '1')then
                    xfer_in_progress <= '0';
                elsif(xfer_in_progress = '0' and cntrl_tvalid = '1')then
                    xfer_in_progress <= '1';
                end if;
            end if;
        end process TRANSFER_IN_PROGRESS;

    skid_rst   <= not m_axi_sg_aresetn;



   ---------------------------------------------------------------------------
    -- Buffer AXI Signals
    ---------------------------------------------------------------------------
--    CNTRL_SKID_BUF_I : entity axi_sg_v4_1_6.axi_sg_skid_buf
--        generic map(
--            C_WDATA_WIDTH           => C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH
--        )
--        port map(
--            -- System Ports
--            ACLK            => m_axi_sg_aclk                          ,
--            ARST            => skid_rst                                 ,

--            skid_stop       => mm2s_stop_re                             ,

--            -- Slave Side (Stream Data Input)
--            S_VALID         => cntrl_tvalid                             ,
--            S_READY         => cntrl_tready                             ,
--            S_Data          => cntrl_tdata                              ,
--            S_STRB          => cntrl_tkeep                              ,
--            S_Last          => cntrl_tlast                              ,

--            -- Master Side (Stream Data Output
--            M_VALID         => m_axis_mm2s_cntrl_tvalid                 ,
--            M_READY         => m_axis_mm2s_cntrl_tready                 ,
--            M_Data          => m_axis_mm2s_cntrl_tdata                  ,
--            M_STRB          => m_axis_mm2s_cntrl_tkeep                  ,
--            M_Last          => m_axis_mm2s_cntrl_tlast
--        );

m_axis_mm2s_cntrl_tvalid <=  cntrl_tvalid;
 cntrl_tready <= m_axis_mm2s_cntrl_tready;   
m_axis_mm2s_cntrl_tdata <=   cntrl_tdata; 
m_axis_mm2s_cntrl_tkeep <=   cntrl_tkeep; 
m_axis_mm2s_cntrl_tlast <=   cntrl_tlast; 

end generate GEN_SYNC_FIFO;

-- Primary Clock is asynchronous to Secondary Clock therfore
-- instantiate an async fifo.
GEN_ASYNC_FIFO : if C_PRMRY_IS_ACLK_ASYNC = 1 generate
  ATTRIBUTE async_reg                      : STRING;

signal mm2s_stop_reg        : std_logic := '0'; -- CR605883
signal p_mm2s_stop_d1_cdc_tig       : std_logic := '0';
signal p_mm2s_stop_d2       : std_logic := '0';
signal p_mm2s_stop_d3       : std_logic := '0';
signal p_mm2s_stop_re       : std_logic := '0';
signal xfer_in_progress     : std_logic := '0';
--  ATTRIBUTE async_reg OF p_mm2s_stop_d1_cdc_tig  : SIGNAL IS "true";
--  ATTRIBUTE async_reg OF p_mm2s_stop_d2  : SIGNAL IS "true";

begin

    -- reset on hard reset, soft reset, or mm2s error
    sinit   <= not p_reset_n or p_mm2s_stop_d2;

    -- Generate Asynchronous FIFO
    I_CNTRL_STRM_FIFO : entity axi_sg_v4_1_6.axi_sg_afifo_autord
      generic map(
         C_DWIDTH        => C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH + 1  ,
-- Temp work around for issue in async fifo model
         C_DEPTH         => CNTRL_FIFO_DEPTH-1                    ,
         C_CNT_WIDTH     => CNTRL_FIFO_CNT_WIDTH                ,
--         C_DEPTH         => 31                                  ,
--         C_CNT_WIDTH     => 5                                   ,
         C_USE_BLKMEM    => USE_LOGIC_FIFOS                     ,
         C_FAMILY        => C_FAMILY
        )
      port map(
        -- Inputs
         AFIFO_Ainit                => sinit                    ,
         AFIFO_Wr_clk               => m_axi_sg_aclk            ,
         AFIFO_Wr_en                => cntrlstrm_fifo_wren      ,
         AFIFO_Din                  => cntrlstrm_fifo_din       ,
         AFIFO_Rd_clk               => axi_prmry_aclk           ,
         AFIFO_Rd_en                => cntrl_fifo_rden          ,
         AFIFO_Clr_Rd_Data_Valid    => '0'                      ,

        -- Outputs
         AFIFO_DValid               => cntrl_fifo_dvalid        ,
         AFIFO_Dout                 => cntrl_fifo_dout          ,
         AFIFO_Full                 => cntrlstrm_fifo_full      ,
         AFIFO_Empty                => cntrl_fifo_empty         ,
         AFIFO_Almost_full          => open                     ,
         AFIFO_Almost_empty         => open                     ,
         AFIFO_Wr_count             => open                     ,
         AFIFO_Rd_count             => open                     ,
         AFIFO_Corr_Rd_count        => open                     ,
         AFIFO_Corr_Rd_count_minus1 => open                     ,
         AFIFO_Rd_ack               => open
        );


    -----------------------------------------------------------------------
    -- Control Stream OUT Side
    -----------------------------------------------------------------------
    -- Read if fifo is not empty and target is ready
    cntrl_fifo_rden <= not cntrl_fifo_empty        -- fifo has data
                       and cntrl_tready;           -- target ready


    -- Drive valid if fifo is not empty or in the middle
    -- of transfer and stop issued.
    cntrl_tvalid  <= cntrl_fifo_dvalid
                    or (xfer_in_progress and p_mm2s_stop_re);

    -- Pass data out to control channel with MSB driving tlast
    cntrl_tlast   <= cntrl_tvalid and cntrl_fifo_dout(C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH);

--    cntrl_tlast   <= (cntrl_tvalid and cntrl_fifo_dout(C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH))
--                    or (xfer_in_progress and p_mm2s_stop_re);

    cntrl_tdata   <= cntrl_fifo_dout(C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH-1 downto 0);

    -- CR605883
    -- Register stop to provide pure FF output for synchronizer
    REG_STOP : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    mm2s_stop_reg <= '0';
                else
                    mm2s_stop_reg <= mm2s_stop;
                end if;
            end if;
        end process REG_STOP;


    -- Double/triple register mm2s error into primary clock domain
    -- Triple register to give two versions with min double reg for use
    -- in rising edge detection.

IMP_SYNC_FLOP : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => 2
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => mm2s_stop_reg,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => axi_prmry_aclk,
        scndry_resetn              => '0',
        scndry_out                 => p_mm2s_stop_d2,
        scndry_vect_out            => open
    );


    REG_ERR2PRMRY : process(axi_prmry_aclk)
        begin
            if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
                if(p_reset_n = '0')then
--                    p_mm2s_stop_d1_cdc_tig <= '0';
--                    p_mm2s_stop_d2 <= '0';
                    p_mm2s_stop_d3 <= '0';
                else
                    --p_mm2s_stop_d1_cdc_tig <= mm2s_stop;
--                    p_mm2s_stop_d1_cdc_tig <= mm2s_stop_reg;
--                    p_mm2s_stop_d2 <= p_mm2s_stop_d1_cdc_tig;
                    p_mm2s_stop_d3 <= p_mm2s_stop_d2;
                end if;
            end if;
        end process REG_ERR2PRMRY;

    -- Rising edge pulse for use in shutting down stream output
    p_mm2s_stop_re <= p_mm2s_stop_d2 and not p_mm2s_stop_d3;

    -------------------------------------------------------------
    -- Flag transfer in progress. If xfer in progress then
    -- a fake tlast needs to be asserted during soft reset.
    -- else no need of tlast.
    -------------------------------------------------------------
    TRANSFER_IN_PROGRESS : process(axi_prmry_aclk)
        begin
            if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
                if(cntrl_tlast = '1' and cntrl_tvalid = '1' and cntrl_tready = '1')then
                    xfer_in_progress <= '0';
                elsif(xfer_in_progress = '0' and cntrl_tvalid = '1')then
                    xfer_in_progress <= '1';
                end if;
            end if;
        end process TRANSFER_IN_PROGRESS;


    skid_rst   <= not p_reset_n;


    CNTRL_SKID_BUF_I : entity axi_sg_v4_1_6.axi_sg_skid_buf
        generic map(
            C_WDATA_WIDTH           => C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH
        )
        port map(
            -- System Ports
            ACLK            => axi_prmry_aclk                           ,
            ARST            => skid_rst                                 ,

            skid_stop       => p_mm2s_stop_re                           ,

            -- Slave Side (Stream Data Input)
            S_VALID         => cntrl_tvalid                             ,
            S_READY         => cntrl_tready                             ,
            S_Data          => cntrl_tdata                              ,
            S_STRB          => cntrl_tkeep                              ,
            S_Last          => cntrl_tlast                              ,

            -- Master Side (Stream Data Output
            M_VALID         => m_axis_mm2s_cntrl_tvalid                 ,
            M_READY         => m_axis_mm2s_cntrl_tready                 ,
            M_Data          => m_axis_mm2s_cntrl_tdata                  ,
            M_STRB          => m_axis_mm2s_cntrl_tkeep                  ,
            M_Last          => m_axis_mm2s_cntrl_tlast
        );



end generate GEN_ASYNC_FIFO;


end implementation;


-- *************************************************************************
--
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:          axi_sg_ftch_queue.vhd
-- Description: This entity is the descriptor fetch queue interface
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library axi_sg_v4_1_6;
use axi_sg_v4_1_6.axi_sg_pkg.all;
--use axi_sg_v4_1_6.axi_sg_afifo_autord.all;

library lib_fifo_v1_0_8;
use lib_fifo_v1_0_8.sync_fifo_fg;
library lib_pkg_v1_0_2;
use lib_pkg_v1_0_2.lib_pkg.all;

-------------------------------------------------------------------------------
entity  axi_sg_ftch_queue is
    generic (
        C_M_AXI_SG_ADDR_WIDTH       : integer range 32 to 64    := 32;
            -- Master AXI Memory Map Address Width

        C_M_AXIS_SG_TDATA_WIDTH     : integer range 32 to 32    := 32;
            -- Master AXI Stream Data width

        C_SG_FTCH_DESC2QUEUE        : integer range 0 to 8      := 0;
            -- Number of descriptors to fetch and queue for each channel.
            -- A value of zero excludes the fetch queues.

        C_SG_WORDS_TO_FETCH         : integer range 4 to 16     := 8;
            -- Number of words to fetch for channel 1
        C_SG2_WORDS_TO_FETCH         : integer range 4 to 16     := 8;
            -- Number of words to fetch for channel 1

        C_ENABLE_MULTI_CHANNEL      : integer range 0 to 1      := 0; 

        C_INCLUDE_MM2S              : integer range 0 to 1      := 0;
        C_INCLUDE_S2MM              : integer range 0 to 1      := 0;
        C_ENABLE_CDMA               : integer range 0 to 1      := 0;

        C_AXIS_IS_ASYNC             : integer range 0 to 1      := 0;
        C_ASYNC             : integer range 0 to 1      := 0;
            -- Channel 1 is async to sg_aclk
            -- 0 = Synchronous to SG ACLK
            -- 1 = Asynchronous to SG ACLK

        C_FAMILY                    : string            := "virtex7"
            -- Device family used for proper BRAM selection
    );
    port (
        -----------------------------------------------------------------------
        -- AXI Scatter Gather Interface
        -----------------------------------------------------------------------
        m_axi_sg_aclk               : in  std_logic                         ;                   --
        m_axi_primary_aclk          : in  std_logic                         ;
        m_axi_sg_aresetn            : in  std_logic                         ;                   --
        p_reset_n                   : in  std_logic                         ;
 
        ch2_sg_idle                 : in std_logic                          ;              

        -- Channel Control                                                                    --
        desc1_flush                  : in  std_logic                         ;                   --
        ch1_cntrl_strm_stop          : in  std_logic                         ;
        desc2_flush                  : in  std_logic                         ;                   --
        ftch1_active                 : in  std_logic                         ;                   --
        ftch2_active                 : in  std_logic                         ;                   --
        ftch1_queue_empty            : out std_logic                         ;                   --
        ftch2_queue_empty            : out std_logic                         ;                   --
        ftch1_queue_full             : out std_logic                         ;                   --
        ftch2_queue_full             : out std_logic                         ;                   --
        ftch1_pause                  : out std_logic                         ;                   --
        ftch2_pause                  : out std_logic                         ;                   --
                                                                                                --
        writing_nxtdesc_in          : in  std_logic                         ;                   --
        writing1_curdesc_out         : out std_logic                         ;                   --
        writing2_curdesc_out         : out std_logic                         ;                   --
                                                                                                --
        -- DataMover Command                                                                    --
        ftch_cmnd_wr                : in  std_logic                         ;                   --
        ftch_cmnd_data              : in  std_logic_vector                                      --
                                        ((C_M_AXI_SG_ADDR_WIDTH+CMD_BASE_WIDTH)-1 downto 0);    --
                                                                                                --
        -- MM2S Stream In from DataMover                                                        --
        m_axis_mm2s_tdata           : in  std_logic_vector                                      --
                                        (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0) ;                  --
        m_axis_mm2s_tlast           : in  std_logic                         ;                   --
        m_axis_mm2s_tvalid          : in  std_logic                         ;                   --
        sof_ftch_desc               : in  std_logic                         ;
        m_axis1_mm2s_tready          : out std_logic                         ;                   --
        m_axis2_mm2s_tready          : out std_logic                         ;                   --
                                                                                                --
        data_concat_64           : in  std_logic_vector                                      --
                                        (31 downto 0) ;                  --
        data_concat_64_cdma           : in  std_logic_vector                                      --
                                        (31 downto 0) ;                  --
        data_concat           : in  std_logic_vector                                      --
                                        (95 downto 0) ;                  --
        data_concat_mcdma           : in  std_logic_vector                                      --
                                        (63 downto 0) ;                  --
        data_concat_tlast           : in  std_logic                         ;                   --
        next_bd                     : in std_logic_vector (C_M_AXI_SG_ADDR_WIDTH-1 downto 0);
        data_concat_valid          : in  std_logic                         ;                   --
                                                                                                --
        -- Channel 1 AXI Fetch Stream Out                                                       --
        m_axis_ftch_aclk            : in  std_logic                         ;                   --
        m_axis_ftch1_tdata           : out std_logic_vector                                      --
                                        (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0);                   --
        m_axis_ftch1_tvalid          : out std_logic                         ;                   --
        m_axis_ftch1_tready          : in  std_logic                         ;                   --
        m_axis_ftch1_tlast           : out std_logic                         ;                    --

        m_axis_ftch1_tdata_new           : out std_logic_vector                                      --
                                        (96+31*C_ENABLE_CDMA+(2+C_ENABLE_CDMA)*(C_M_AXI_SG_ADDR_WIDTH-32) downto 0);                   --
        m_axis_ftch1_tdata_mcdma_new           : out std_logic_vector                                      --
                                        (63 downto 0);                   --
        m_axis_ftch1_tvalid_new          : out std_logic                         ;                   --
        m_axis_ftch1_desc_available            : out std_logic                     ;

        m_axis_ftch2_tdata           : out std_logic_vector                                      --
                                        (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0);                   --
        m_axis_ftch2_tvalid          : out std_logic                         ;                   --

        m_axis_ftch2_tdata_new           : out std_logic_vector                                      --
                                        (96+31*C_ENABLE_CDMA+(2+C_ENABLE_CDMA)*(C_M_AXI_SG_ADDR_WIDTH-32) downto 0);                   --
        m_axis_ftch2_tdata_mcdma_new           : out std_logic_vector                                      --
                                        (63 downto 0);                   --
        m_axis_ftch2_tvalid_new          : out std_logic                         ;                   --
        m_axis_ftch2_desc_available            : out std_logic                     ;
        m_axis_ftch2_tready          : in  std_logic                         ;                   --
        m_axis_ftch2_tlast           : out std_logic                         ;                    --

        m_axis_mm2s_cntrl_tdata     : out std_logic_vector                                 --
                                        (31 downto 0);      --
        m_axis_mm2s_cntrl_tkeep     : out std_logic_vector                                 --
                                        (3 downto 0);  --
        m_axis_mm2s_cntrl_tvalid    : out std_logic                         ;              --
        m_axis_mm2s_cntrl_tready    : in  std_logic                         := '0';              --
        m_axis_mm2s_cntrl_tlast     : out std_logic                                       --

    );

end axi_sg_ftch_queue;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_sg_ftch_queue is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";



-- Number of words deep fifo needs to be
-- 6 is subtracted as BD address are always 16 word aligned

constant FIFO_WIDTH : integer := (128*C_ENABLE_CDMA + 97*(1-C_ENABLE_CDMA) -6);
constant C_SG_WORDS_TO_FETCH1 : integer := C_SG_WORDS_TO_FETCH + 2*C_ENABLE_MULTI_CHANNEL;
--constant    FETCH_QUEUE_DEPTH       : integer := max2(16,pad_power2(C_SG_FTCH_DESC2QUEUE
--                                                                  * C_SG_WORDS_TO_FETCH1));
constant    FETCH_QUEUE_DEPTH       : integer := 16;

-- Select between BRAM or Logic Memory Type
constant    MEMORY_TYPE : integer := bo2int(C_SG_FTCH_DESC2QUEUE
                                    * C_SG_WORDS_TO_FETCH1 > 16);

constant    FETCH_QUEUE_CNT_WIDTH   : integer   := clog2(FETCH_QUEUE_DEPTH+1);

constant DCNT_LO_INDEX              : integer :=  max2(1,clog2(C_SG_WORDS_TO_FETCH1)) - 1;

constant DCNT_HI_INDEX              : integer :=  FETCH_QUEUE_CNT_WIDTH-1;                          --  CR616461


constant C_SG2_WORDS_TO_FETCH1 : integer := C_SG2_WORDS_TO_FETCH;

constant    FETCH2_QUEUE_DEPTH       : integer := max2(16,pad_power2(C_SG_FTCH_DESC2QUEUE
                                                                  * C_SG2_WORDS_TO_FETCH1));
-- Select between BRAM or Logic Memory Type
constant    MEMORY2_TYPE : integer := bo2int(C_SG_FTCH_DESC2QUEUE
                                    * C_SG2_WORDS_TO_FETCH1 > 16);
constant    FETCH2_QUEUE_CNT_WIDTH   : integer   := clog2(FETCH2_QUEUE_DEPTH+1);
constant DCNT2_LO_INDEX              : integer :=  max2(1,clog2(C_SG2_WORDS_TO_FETCH1)) - 1;
constant DCNT2_HI_INDEX              : integer :=  FETCH2_QUEUE_CNT_WIDTH-1;                          --  CR616461


-- Width of fifo rd and wr counts - only used for proper fifo operation


constant DESC2QUEUE_VECT_WIDTH      : integer := 4;
--constant SG_FTCH_DESC2QUEUE_VECT    : std_logic_vector(DESC2QUEUE_VECT_WIDTH-1 downto 0)
--                                        := std_logic_vector(to_unsigned(C_SG_FTCH_DESC2QUEUE,DESC2QUEUE_VECT_WIDTH)); --  CR616461
constant SG_FTCH_DESC2QUEUE_VECT    : std_logic_vector(DESC2QUEUE_VECT_WIDTH-1 downto 0)
                                        := std_logic_vector(to_unsigned(C_SG_FTCH_DESC2QUEUE,DESC2QUEUE_VECT_WIDTH));   --  CR616461

--constant DCNT_HI_INDEX              : integer :=  (DCNT_LO_INDEX + DESC2QUEUE_VECT_WIDTH) - 1;    --  CR616461


constant ZERO_COUNT                 : std_logic_vector(FETCH_QUEUE_CNT_WIDTH-1 downto 0) := (others => '0');
constant ZERO_COUNT1                 : std_logic_vector(FETCH2_QUEUE_CNT_WIDTH-1 downto 0) := (others => '0');

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
-- Internal signals
signal curdesc_tdata            : std_logic_vector
                                    (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc_tvalid           : std_logic := '0';
signal ftch_tvalid              : std_logic := '0';
signal ftch_tvalid_new         : std_logic := '0';
signal ftch_tdata               : std_logic_vector
                                    (31 downto 0) := (others => '0');
signal ftch_tdata_new, reg1, reg2               : std_logic_vector
                                    (FIFO_WIDTH-1 downto 0) := (others => '0');

signal ftch_tdata_new_64, reg1_64, reg2_64 : std_logic_vector ((1+C_ENABLE_CDMA)*(C_M_AXI_SG_ADDR_WIDTH-32) -1 downto 0) := (others => '0');
signal ftch_tdata_new_bd, reg2_bd_64, reg1_bd_64 : std_logic_vector (31 downto 0) := (others => '0');

signal ftch_tlast               : std_logic := '0';
signal ftch_tlast_new               : std_logic := '0';
signal ftch_tready              : std_logic := '0';
signal ftch_tready_ch1              : std_logic := '0';
signal ftch_tready_ch2              : std_logic := '0';

-- Misc Signals
signal writing_curdesc          : std_logic := '0';
signal writing_nxtdesc          : std_logic := '0';

signal msb_curdesc              : std_logic_vector(31 downto 0) := (others => '0');
signal writing_lsb              : std_logic := '0';
signal writing_msb              : std_logic := '0';

-- FIFO signals
signal queue_rden2               : std_logic := '0';
signal queue_rden2_new               : std_logic := '0';
signal queue_wren2               : std_logic := '0';
signal queue_wren2_new               : std_logic := '0';
signal queue_empty2              : std_logic := '0';
signal queue_empty2_new              : std_logic := '0';
signal queue_rden               : std_logic := '0';
signal queue_rden_new               : std_logic := '0';
signal queue_wren               : std_logic := '0';
signal queue_wren_new               : std_logic := '0';
signal queue_empty              : std_logic := '0';
signal queue_empty_new              : std_logic := '0';
signal queue_dout_valid              : std_logic := '0';
signal queue_dout2_valid              : std_logic := '0';
signal queue_full_new               : std_logic := '0';
signal queue_full2_new               : std_logic := '0';
signal queue_full, queue_full2               : std_logic := '0';
signal queue_din_new                : std_logic_vector
                                    (127 downto 0) := (others => '0');
signal queue_dout_new_64            : std_logic_vector ((1+C_ENABLE_CDMA)*(C_M_AXI_SG_ADDR_WIDTH-32) -1 downto 0) := (others => '0');
signal queue_dout_new_bd            : std_logic_vector (31 downto 0) := (others => '0');
signal queue_dout_new               : std_logic_vector
                                    (96+31*C_ENABLE_CDMA-6 downto 0) := (others => '0');
signal queue_dout_mcdma_new               : std_logic_vector
                                    (63 downto 0) := (others => '0');
signal queue_dout2_new_64            : std_logic_vector ((1+C_ENABLE_CDMA)*(C_M_AXI_SG_ADDR_WIDTH-32) -1 downto 0) := (others => '0');
signal queue_dout2_new_bd            : std_logic_vector (31 downto 0) := (others => '0');
signal queue_dout2_new               : std_logic_vector
                                    (96+31*C_ENABLE_CDMA-6 downto 0) := (others => '0');
signal queue_dout2_mcdma_new               : std_logic_vector
                                    (63 downto 0) := (others => '0');
signal queue_din                : std_logic_vector
                                    (C_M_AXIS_SG_TDATA_WIDTH downto 0) := (others => '0');
signal queue_dout               : std_logic_vector
                                    (C_M_AXIS_SG_TDATA_WIDTH downto 0) := (others => '0');
signal queue_dout2               : std_logic_vector
                                    (C_M_AXIS_SG_TDATA_WIDTH downto 0) := (others => '0');
signal queue_sinit              : std_logic := '0';
signal queue_sinit2              : std_logic := '0';
signal queue_dcount_new        : std_logic_vector(FETCH_QUEUE_CNT_WIDTH-1 downto 0) := (others => '0');
signal queue_dcount2_new       : std_logic_vector(FETCH_QUEUE_CNT_WIDTH-1 downto 0) := (others => '0');
signal ftch_no_room             : std_logic;

signal ftch_active : std_logic := '0';

signal ftch_tvalid_mult              : std_logic := '0';
signal ftch_tdata_mult               : std_logic_vector
                                    (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0) := (others => '0');
signal ftch_tlast_mult               : std_logic := '0';

signal counter : std_logic_vector (3 downto 0) := (others => '0');
signal wr_cntl : std_logic := '0';

signal sof_ftch_desc_del : std_logic;
signal sof_ftch_desc_del1 : std_logic;
signal sof_ftch_desc_pulse : std_logic;
signal current_bd : std_logic_vector (C_M_AXI_SG_ADDR_WIDTH-1 downto 0) := (others => '0');

signal xfer_in_progress : std_logic := '0';

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin


SOF_DEL_PROCESS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                sof_ftch_desc_del <= '0';
            else
                sof_ftch_desc_del <= sof_ftch_desc;
            end if;
        end if;
   end process SOF_DEL_PROCESS;

SOF_DEL1_PROCESS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0' or (m_axis_mm2s_tlast = '1' and m_axis_mm2s_tvalid = '1'))then
                sof_ftch_desc_del1 <= '0';
            elsif (m_axis_mm2s_tvalid = '1') then
                sof_ftch_desc_del1 <= sof_ftch_desc;
            end if;
        end if;
   end process SOF_DEL1_PROCESS;

sof_ftch_desc_pulse <= sof_ftch_desc and (not sof_ftch_desc_del1);

ftch_active <= ftch1_active or ftch2_active;
---------------------------------------------------------------------------
-- Write current descriptor to FIFO or out channel port
---------------------------------------------------------------------------

CURRENT_BD_64 : if C_M_AXI_SG_ADDR_WIDTH > 32 generate
begin

CMDDATA_PROCESS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                  current_bd <= (others => '0');
            elsif (ftch2_active = '1' and C_ENABLE_MULTI_CHANNEL = 1) then 
                  current_bd <= next_bd;
            elsif (ftch_cmnd_wr = '1' and ftch_active = '1') then
                current_bd       <= ftch_cmnd_data(32+DATAMOVER_CMD_ADDRMSB_BOFST
                                                        + DATAMOVER_CMD_ADDRLSB_BIT
                                                        downto DATAMOVER_CMD_ADDRLSB_BIT);
            end if;
        end if;
   end process CMDDATA_PROCESS;
end generate CURRENT_BD_64;


CURRENT_BD_32 : if C_M_AXI_SG_ADDR_WIDTH = 32 generate
begin

CMDDATA_PROCESS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                  current_bd <= (others => '0');
            elsif (ftch2_active = '1' and C_ENABLE_MULTI_CHANNEL = 1) then 
                  current_bd <= next_bd;
            elsif (ftch_cmnd_wr = '1' and ftch_active = '1') then
                current_bd       <= ftch_cmnd_data(DATAMOVER_CMD_ADDRMSB_BOFST
                                                        + DATAMOVER_CMD_ADDRLSB_BIT
                                                        downto DATAMOVER_CMD_ADDRLSB_BIT);
            end if;
        end if;
   end process CMDDATA_PROCESS;
end generate CURRENT_BD_32;



GEN_MULT_CHANNEL : if C_ENABLE_MULTI_CHANNEL = 1 generate
begin
            ftch_tvalid_mult  <= m_axis_mm2s_tvalid;
            ftch_tdata_mult   <= m_axis_mm2s_tdata;
            ftch_tlast_mult   <= m_axis_mm2s_tlast;
            wr_cntl <= m_axis_mm2s_tvalid;

end generate GEN_MULT_CHANNEL;


GEN_NOMULT_CHANNEL : if C_ENABLE_MULTI_CHANNEL = 0 generate
begin
            ftch_tvalid_mult  <= '0'; --m_axis_mm2s_tvalid;
            ftch_tdata_mult   <= (others => '0'); --m_axis_mm2s_tdata;
            ftch_tlast_mult   <= '0'; --m_axis_mm2s_tlast;
            m_axis_ftch1_tdata_mcdma_new <= (others => '0');
            m_axis_ftch2_tdata_mcdma_new <= (others => '0');

COUNTER_PROCESS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0' or m_axis_mm2s_tlast = '1')then
                counter <= (others => '0');
            elsif (m_axis_mm2s_tvalid = '1') then
                counter <= std_logic_vector(unsigned(counter) + 1);
            end if;
        end if;
   end process COUNTER_PROCESS;


end generate GEN_NOMULT_CHANNEL;

---------------------------------------------------------------------------
-- TVALID MUX
-- MUX tvalid out channel port
---------------------------------------------------------------------------
CDMA_FIELDS : if C_ENABLE_CDMA = 1 generate
begin

CDMA_FIELDS_64 : if C_M_AXI_SG_ADDR_WIDTH > 32 generate
begin

ftch_tdata_new_64 (63 downto 0) <= data_concat_64_cdma & data_concat_64;
ftch_tdata_new_bd (31 downto 0) <= current_bd (C_M_AXI_SG_ADDR_WIDTH-1 downto 32);

end generate CDMA_FIELDS_64;

ftch_tdata_new (95 downto 0) <= data_concat;
-- BD is always 16 word aligned
ftch_tdata_new (121 downto 96) <= current_bd (31 downto 6);

end generate CDMA_FIELDS;

DMA_FIELDS : if C_ENABLE_CDMA = 0 generate
begin

DMA_FIELDS_64 : if C_M_AXI_SG_ADDR_WIDTH > 32 generate
begin

ftch_tdata_new_64 (31 downto 0) <= data_concat_64;
ftch_tdata_new_bd (31 downto 0) <= current_bd (C_M_AXI_SG_ADDR_WIDTH-1 downto 32);

end generate DMA_FIELDS_64;

ftch_tdata_new (64 downto 0) <= data_concat (95) & data_concat (63 downto 0);-- when (ftch_active = '1') else (others =>'0');
-- BD is always 16 word aligned
ftch_tdata_new (90 downto 65) <= current_bd (31 downto 6);

end generate DMA_FIELDS;


ftch_tvalid_new  <= data_concat_valid and ftch_active;
ftch_tlast_new  <= data_concat_tlast and ftch_active;


GEN_MM2S : if C_INCLUDE_MM2S = 1 generate
begin

process (m_axi_sg_aclk)
begin
    if (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
        if (queue_sinit = '1' or queue_rden_new = '1') then
           queue_empty_new <= '1';
           queue_full_new <= '0';
        elsif (queue_wren_new = '1') then
           queue_empty_new <= '0';
           queue_full_new <= '1';
        end if;
    end if;
end process;

process (m_axi_sg_aclk)
begin
    if (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
        if (queue_sinit = '1') then
           reg1 <= (others => '0');
           reg1_64 <= (others => '0');
           reg1_bd_64 <= (others => '0');
        elsif (queue_wren_new = '1') then
           reg1 <= ftch_tdata_new;
           reg1_64 <= ftch_tdata_new_64;
           reg1_bd_64 <= ftch_tdata_new_bd;
        end if;
    end if;
end process;


process (m_axi_sg_aclk)
begin
    if (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
        if (queue_sinit = '1') then
           queue_dout_new <= (others => '0');
           queue_dout_new_64 <= (others => '0');
           queue_dout_new_bd <= (others => '0');
        elsif (queue_rden_new = '1') then
           queue_dout_new <= reg1;
           queue_dout_new_64 <= reg1_64;
           queue_dout_new_bd <= reg1_bd_64;
        end if;
    end if;
end process;


process (m_axi_sg_aclk)
begin
    if (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
        if (queue_sinit = '1' or queue_dout_valid = '1') then
           queue_dout_valid <= '0';
        elsif (queue_rden_new = '1') then
           queue_dout_valid <= '1';
        end if;
    end if;
end process;


MCDMA_MM2S : if C_ENABLE_MULTI_CHANNEL = 1 generate
begin

    -- Generate Synchronous FIFO
    I_CH1_FTCH_MCDMA_FIFO_NEW : entity lib_fifo_v1_0_8.sync_fifo_fg
    generic map (
        C_FAMILY                =>  C_FAMILY                ,
        C_MEMORY_TYPE           =>  0, --MEMORY_TYPE             ,
        C_WRITE_DATA_WIDTH      =>  64,
        C_WRITE_DEPTH           =>  FETCH_QUEUE_DEPTH       ,
        C_READ_DATA_WIDTH       =>  64,
        C_READ_DEPTH            =>  FETCH_QUEUE_DEPTH       ,
        C_PORTS_DIFFER          =>  0,
        C_HAS_DCOUNT            =>  0,
        C_DCOUNT_WIDTH          =>  FETCH_QUEUE_CNT_WIDTH,
        C_HAS_ALMOST_FULL       =>  0,
        C_HAS_RD_ACK            =>  0,
        C_HAS_RD_ERR            =>  0,
        C_HAS_WR_ACK            =>  0,
        C_HAS_WR_ERR            =>  0,
        C_RD_ACK_LOW            =>  0,
        C_RD_ERR_LOW            =>  0,
        C_WR_ACK_LOW            =>  0,
        C_WR_ERR_LOW            =>  0,
        C_PRELOAD_REGS          =>  0,-- 1 = first word fall through
        C_PRELOAD_LATENCY       =>  1 -- 0 = first word fall through

    )
    port map (

        Clk             =>  m_axi_sg_aclk       ,
        Sinit           =>  queue_sinit         ,
        Din             =>  data_concat_mcdma, --ftch_tdata_new, --queue_din           ,
        Wr_en           =>  queue_wren_new          ,
        Rd_en           =>  queue_rden_new          ,
        Dout            =>  queue_dout_mcdma_new          ,
        Full            =>  open, --queue_full_new          ,
        Empty           =>  open, --queue_empty_new         ,
        Almost_full     =>  open                ,
        Data_count      =>  open, --queue_dcount_new    ,
        Rd_ack          =>  open, --queue_dout_valid, --open                ,
        Rd_err          =>  open                ,
        Wr_ack          =>  open                ,
        Wr_err          =>  open

    );

m_axis_ftch1_tdata_mcdma_new <= queue_dout_mcdma_new;

end generate MCDMA_MM2S;



CONTROL_STREAM : if C_SG_WORDS_TO_FETCH = 13 generate
begin


        I_MM2S_CNTRL_STREAM : entity axi_sg_v4_1_6.axi_sg_cntrl_strm
            generic map(
                C_PRMRY_IS_ACLK_ASYNC           => C_ASYNC           ,
                C_PRMY_CMDFIFO_DEPTH            => FETCH_QUEUE_DEPTH             ,
                C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH => C_M_AXIS_SG_TDATA_WIDTH  ,
                C_FAMILY                        => C_FAMILY
            )
            port map(
                -- Secondary clock / reset
                m_axi_sg_aclk               => m_axi_sg_aclk                ,
                m_axi_sg_aresetn            => m_axi_sg_aresetn             ,

                -- Primary clock / reset
                axi_prmry_aclk              => m_axi_primary_aclk           ,
                p_reset_n                   => p_reset_n                    ,

                -- MM2S Error
                mm2s_stop                   => ch1_cntrl_strm_stop              ,

                -- Control Stream input
                cntrlstrm_fifo_wren         => queue_wren          ,
                cntrlstrm_fifo_full         => queue_full          ,
                cntrlstrm_fifo_din          => queue_din           ,

                -- Memory Map to Stream Control Stream Interface
                m_axis_mm2s_cntrl_tdata     => m_axis_mm2s_cntrl_tdata      ,
                m_axis_mm2s_cntrl_tkeep     => m_axis_mm2s_cntrl_tkeep      ,
                m_axis_mm2s_cntrl_tvalid    => m_axis_mm2s_cntrl_tvalid     ,
                m_axis_mm2s_cntrl_tready    => m_axis_mm2s_cntrl_tready     ,
                m_axis_mm2s_cntrl_tlast     => m_axis_mm2s_cntrl_tlast

            );


end generate CONTROL_STREAM;


end generate GEN_MM2S;

GEN_S2MM : if C_INCLUDE_S2MM = 1 generate
begin

process (m_axi_sg_aclk)
begin
    if (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
        if (queue_sinit2 = '1' or queue_rden2_new = '1') then
           queue_empty2_new <= '1';
           queue_full2_new <= '0';
        elsif (queue_wren2_new = '1') then
           queue_empty2_new <= '0';
           queue_full2_new <= '1';
        end if;
    end if;
end process;

process (m_axi_sg_aclk)
begin
    if (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
        if (queue_sinit2 = '1') then
           reg2 <= (others => '0');
           reg2_64 <= (others => '0');
           reg2_bd_64 <= (others => '0');
        elsif (queue_wren2_new = '1') then
           reg2 <= ftch_tdata_new;
           reg2_64 <= ftch_tdata_new_64;
           reg2_bd_64 <= ftch_tdata_new_bd;
        end if;
    end if;
end process;


process (m_axi_sg_aclk)
begin
    if (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
        if (queue_sinit2 = '1') then
           queue_dout2_new <= (others => '0');
           queue_dout2_new_64 <= (others => '0');
           queue_dout2_new_bd <= (others => '0');
        elsif (queue_rden2_new = '1') then
           queue_dout2_new <= reg2;
           queue_dout2_new_64 <= reg2_64;
           queue_dout2_new_bd <= reg2_bd_64;
        end if;
    end if;
end process;


process (m_axi_sg_aclk)
begin
    if (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
        if (queue_sinit2 = '1' or queue_dout2_valid = '1') then
           queue_dout2_valid <= '0';
        elsif (queue_rden2_new = '1') then
           queue_dout2_valid <= '1';
        end if;
    end if;
end process;



MCDMA_S2MM : if C_ENABLE_MULTI_CHANNEL = 1 generate
begin

    -- Generate Synchronous FIFO
    I_CH2_FTCH_MCDMA_FIFO_NEW : entity lib_fifo_v1_0_8.sync_fifo_fg
    generic map (
        C_FAMILY                =>  C_FAMILY                ,
        C_MEMORY_TYPE           =>  0, --MEMORY_TYPE             ,
        C_WRITE_DATA_WIDTH      =>  64,
        C_WRITE_DEPTH           =>  FETCH_QUEUE_DEPTH       ,
        C_READ_DATA_WIDTH       =>  64,
        C_READ_DEPTH            =>  FETCH_QUEUE_DEPTH       ,
        C_PORTS_DIFFER          =>  0,
        C_HAS_DCOUNT            =>  0,
        C_DCOUNT_WIDTH          =>  FETCH_QUEUE_CNT_WIDTH,
        C_HAS_ALMOST_FULL       =>  0,
        C_HAS_RD_ACK            =>  0,
        C_HAS_RD_ERR            =>  0,
        C_HAS_WR_ACK            =>  0,
        C_HAS_WR_ERR            =>  0,
        C_RD_ACK_LOW            =>  0,
        C_RD_ERR_LOW            =>  0,
        C_WR_ACK_LOW            =>  0,
        C_WR_ERR_LOW            =>  0,
        C_PRELOAD_REGS          =>  0,-- 1 = first word fall through
        C_PRELOAD_LATENCY       =>  1 -- 0 = first word fall through

    )
    port map (

        Clk             =>  m_axi_sg_aclk       ,
        Sinit           =>  queue_sinit2         ,
        Din             =>  data_concat_mcdma, --ftch_tdata_new, --queue_din           ,
        Wr_en           =>  queue_wren2_new          ,
        Rd_en           =>  queue_rden2_new          ,
        Dout            =>  queue_dout2_new          ,
        Full            =>  open, --queue_full2_new          ,
        Empty           =>  open, --queue_empty2_new         ,
        Almost_full     =>  open                ,
        Data_count      =>  queue_dcount2_new        ,
        Rd_ack          =>  open, --queue_dout2_valid                ,
        Rd_err          =>  open                ,
        Wr_ack          =>  open                ,
        Wr_err          =>  open

    );

m_axis_ftch2_tdata_mcdma_new <= queue_dcount2_new;

end generate MCDMA_S2MM;

end generate GEN_S2MM;


-----------------------------------------------------------------------
-- Internal Side
-----------------------------------------------------------------------

-- Drive tready with fifo not full
ftch_tready <= ftch_tready_ch1 or ftch_tready_ch2;
              

-- Following is the APP data that goes into APP FIFO
queue_din(C_M_AXIS_SG_TDATA_WIDTH)               <= m_axis_mm2s_tlast;
queue_din(C_M_AXIS_SG_TDATA_WIDTH-1 downto 0)    <= x"A0000000" when (sof_ftch_desc_pulse = '1') else m_axis_mm2s_tdata;

GEN_CH1_CTRL : if C_INCLUDE_MM2S =1 generate
begin

--queue_full_new <= '1' when (queue_dcount_new = "00100") else '0';

queue_sinit <= desc1_flush or not m_axi_sg_aresetn;

ftch_tready_ch1 <= (not queue_full and ftch1_active); 
m_axis1_mm2s_tready <= ftch_tready_ch1;

-- Wr_en to APP FIFO. Data is written only when BD with SOF is fetched.

queue_wren  <= not queue_full
               and sof_ftch_desc
               and m_axis_mm2s_tvalid
               and ftch1_active;

-- Wr_en of BD FIFO
queue_wren_new  <= not queue_full_new
                   and ftch_tvalid_new
                   and ftch1_active;


ftch1_queue_empty <= queue_empty_new;

ftch1_queue_full     <= queue_full_new;

ftch1_pause <= queue_full_new;

-- RD_en of APP FIFO based on empty and tready

-- RD_EN of BD FIFO based on empty and tready
queue_rden_new  <= not queue_empty_new
               and m_axis_ftch1_tready;

-- drive valid if fifo is not empty

m_axis_ftch1_tvalid  <= '0';
m_axis_ftch1_tvalid_new  <= queue_dout_valid; --not queue_empty_new and (not ch2_sg_idle);

-- below signal triggers the fetch of BD in MM2S Mngr
m_axis_ftch1_desc_available <= not queue_empty_new and (not ch2_sg_idle);

-- Pass data out to port channel with MSB driving tlast
m_axis_ftch1_tlast   <= '0'; 
m_axis_ftch1_tdata   <= (others => '0'); 


FTCH_FIELDS_64 : if C_M_AXI_SG_ADDR_WIDTH > 32 generate
begin

m_axis_ftch1_tdata_new   <= queue_dout_new_bd & queue_dout_new_64 & queue_dout_new (FIFO_WIDTH-1 downto FIFO_WIDTH-26) & "000000" & queue_dout_new (FIFO_WIDTH-27 downto 0);

end generate FTCH_FIELDS_64;

FTCH_FIELDS_32 : if C_M_AXI_SG_ADDR_WIDTH = 32 generate
begin

m_axis_ftch1_tdata_new   <= queue_dout_new (FIFO_WIDTH-1 downto FIFO_WIDTH-26) & "000000" & queue_dout_new (FIFO_WIDTH-27 downto 0);

end generate FTCH_FIELDS_32;

writing1_curdesc_out <= writing_curdesc and ftch1_active;


NOCONTROL_STREAM_ASST : if C_SG_WORDS_TO_FETCH = 8 generate
begin

        m_axis_mm2s_cntrl_tdata  <= (others => '0');
        m_axis_mm2s_cntrl_tkeep  <= (others => '0');
        m_axis_mm2s_cntrl_tvalid <= '0';
        m_axis_mm2s_cntrl_tlast  <= '0';

end generate NOCONTROL_STREAM_ASST;

end generate GEN_CH1_CTRL;


GEN_NO_CH1_CTRL : if C_INCLUDE_MM2S =0 generate
begin

        m_axis_mm2s_cntrl_tdata  <= (others => '0');
        m_axis_mm2s_cntrl_tkeep  <= "0000";   
        m_axis_mm2s_cntrl_tvalid <= '0';   
        m_axis_mm2s_cntrl_tlast  <= '0';   

ftch_tready_ch1 <= '0';
m_axis1_mm2s_tready <= '0';

-- Write to fifo if it is not full and data is valid
queue_wren  <= '0';

ftch1_queue_empty <= '0';
ftch1_queue_full     <= '0';
ftch1_pause <= '0';
queue_rden  <= '0';

-- drive valid if fifo is not empty
m_axis_ftch1_tvalid  <= '0';

-- Pass data out to port channel with MSB driving tlast
m_axis_ftch1_tlast   <= '0';
m_axis_ftch1_tdata   <= (others => '0');
writing1_curdesc_out <= '0';
m_axis_ftch1_tdata_new   <= (others => '0');
m_axis_ftch1_tvalid_new  <= '0';
m_axis_ftch1_desc_available <= '0';

end generate GEN_NO_CH1_CTRL;



GEN_CH2_CTRL : if C_INCLUDE_S2MM =1 generate
begin
queue_sinit2 <= desc2_flush or not m_axi_sg_aresetn;

ftch_tready_ch2 <= (not queue_full2_new and ftch2_active); 
m_axis2_mm2s_tready <= ftch_tready_ch2;


queue_wren2  <= '0';

-- Wr_en for S2MM BD FIFO
queue_wren2_new  <= not queue_full2_new
                    and ftch_tvalid_new
                    and ftch2_active;

--queue_full2_new <= '1' when (queue_dcount2_new = "00100") else '0';

-- Pass fifo status back to fetch sm for channel IDLE determination

ftch2_queue_empty <= queue_empty2_new;

ftch2_queue_full     <= queue_full2_new;


ftch2_pause <= queue_full2_new;

queue_rden2  <= '0';

-- Rd_en for S2MM BD FIFO
queue_rden2_new  <= not queue_empty2_new
                    and m_axis_ftch2_tready;

m_axis_ftch2_tvalid  <= '0';
m_axis_ftch2_tvalid_new  <= queue_dout2_valid; -- not queue_empty2_new and (not ch2_sg_idle);
m_axis_ftch2_desc_available <= not queue_empty2_new and (not ch2_sg_idle);

-- Pass data out to port channel with MSB driving tlast
m_axis_ftch2_tlast   <= '0';
m_axis_ftch2_tdata   <= (others => '0');

FTCH_FIELDS_64_2 : if C_M_AXI_SG_ADDR_WIDTH > 32 generate

m_axis_ftch2_tdata_new   <= queue_dout2_new_bd & queue_dout2_new_64 & queue_dout2_new (FIFO_WIDTH-1 downto FIFO_WIDTH-26) & "000000" & queue_dout2_new (FIFO_WIDTH-27 downto 0);

end generate FTCH_FIELDS_64_2;


FTCH_FIELDS_32_2 : if C_M_AXI_SG_ADDR_WIDTH = 32 generate

m_axis_ftch2_tdata_new   <= queue_dout2_new (FIFO_WIDTH-1 downto FIFO_WIDTH-26) & "000000" & queue_dout2_new (FIFO_WIDTH-27 downto 0);

end generate FTCH_FIELDS_32_2;

writing2_curdesc_out <= writing_curdesc and ftch2_active;

end generate GEN_CH2_CTRL;

GEN_NO_CH2_CTRL : if C_INCLUDE_S2MM =0 generate
begin

ftch_tready_ch2 <= '0';
m_axis2_mm2s_tready <= '0';
queue_wren2  <= '0';

-- Pass fifo status back to fetch sm for channel IDLE determination
--ftch_queue_empty    <= queue_empty; CR 621600

ftch2_queue_empty <= '0';
ftch2_queue_full  <= '0';
ftch2_pause <= '0';
queue_rden2 <= '0';

m_axis_ftch2_tvalid  <= '0';

-- Pass data out to port channel with MSB driving tlast
m_axis_ftch2_tlast   <= '0';
m_axis_ftch2_tdata   <= (others => '0');
m_axis_ftch2_tdata_new   <= (others => '0');
m_axis_ftch2_tvalid_new  <= '0';
writing2_curdesc_out <= '0';
m_axis_ftch2_desc_available <= '0';

end generate GEN_NO_CH2_CTRL;


-- If writing curdesc out then flag for proper mux selection
writing_curdesc     <= curdesc_tvalid;
-- Map intnal signal to port
-- Map port to internal signal
writing_nxtdesc     <= writing_nxtdesc_in;


end implementation;


-- *************************************************************************
--
--  (c) Copyright 2010-2011 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:          axi_sg_ftch_noqueue.vhd
-- Description: This entity is the no queue version
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library axi_sg_v4_1_6;
use axi_sg_v4_1_6.axi_sg_pkg.all;

library lib_pkg_v1_0_2;
use lib_pkg_v1_0_2.lib_pkg.all;

-------------------------------------------------------------------------------
entity  axi_sg_ftch_noqueue is
    generic (
        C_M_AXI_SG_ADDR_WIDTH       : integer range 32 to 64        := 32;
            -- Master AXI Memory Map Address Width

        C_M_AXIS_SG_TDATA_WIDTH     : integer range 32 to 32        := 32;
            -- Master AXI Stream Data Width

        C_ENABLE_MULTI_CHANNEL      : integer range 0 to 1          := 0;

        C_AXIS_IS_ASYNC             : integer range 0 to 1      := 0;
        C_ASYNC             : integer range 0 to 1      := 0;

        C_SG_WORDS_TO_FETCH : integer range 8 to 13                 := 8;

        C_ENABLE_CDMA  : integer range 0 to 1                       := 0;

        C_ENABLE_CH1        : integer range 0 to 1                  := 0;

        C_FAMILY                    : string            := "virtex7"
            -- Device family used for proper BRAM selection


    );
    port (
        -----------------------------------------------------------------------
        -- AXI Scatter Gather Interface
        -----------------------------------------------------------------------
        m_axi_sg_aclk               : in  std_logic                         ;                   --
        m_axi_primary_aclk          : in  std_logic                         ;
        m_axi_sg_aresetn            : in  std_logic                         ;                   --
        p_reset_n                   : in  std_logic                         ;
                                                                                                --
        -- Channel Control                                                                    --
        desc_flush                  : in  std_logic                         ;                   --
        ch1_cntrl_strm_stop         : in  std_logic                         ;
        ftch_active                 : in  std_logic                         ;                   --
        ftch_queue_empty            : out std_logic                         ;                   --
        ftch_queue_full             : out std_logic                         ;                   --

        sof_ftch_desc               : in  std_logic                         ;

        desc2_flush                  : in  std_logic                         ;                   --
        ftch2_active                 : in  std_logic                         ;                   --
        ftch2_queue_empty            : out std_logic                         ;                   --
        ftch2_queue_full             : out std_logic                         ;                   --
                                                                                                --
        writing_nxtdesc_in          : in  std_logic                         ;                   --
        writing_curdesc_out         : out std_logic                         ;                   --
        writing2_curdesc_out         : out std_logic                         ;                   --

        -- DataMover Command                                                                    --
        ftch_cmnd_wr                : in  std_logic                         ;                   --
        ftch_cmnd_data              : in  std_logic_vector                                      --
                                        ((C_M_AXI_SG_ADDR_WIDTH+CMD_BASE_WIDTH)-1 downto 0);    --
                                                                                                --
        -- MM2S Stream In from DataMover                                                        --
        m_axis_mm2s_tdata           : in  std_logic_vector                                      --
                                        (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0) ;                  --
        m_axis_mm2s_tlast           : in  std_logic                         ;                   --
        m_axis_mm2s_tvalid          : in  std_logic                         ;                   --
        m_axis_mm2s_tready          : out std_logic                         ;                   --
        m_axis2_mm2s_tready          : out std_logic                         ;                   --

        data_concat           : in  std_logic_vector                                      --
                                        (95 downto 0) ;                  --
        data_concat_64           : in  std_logic_vector                                      --
                                        (31 downto 0) ;                  --
        data_concat_mcdma           : in  std_logic_vector                                      --
                                        (63 downto 0) ;                  --
        next_bd                     : in std_logic_vector (C_M_AXI_SG_ADDR_WIDTH-1 downto 0);
        data_concat_tlast           : in  std_logic                         ;                   --
        data_concat_valid          : in  std_logic                         ;                   --
                                                                                                --
        -- Channel 1 AXI Fetch Stream Out                                                       --
        m_axis_ftch_tdata           : out std_logic_vector                                      --
                                            (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0) ;              --
        m_axis_ftch_tvalid          : out std_logic                         ;                   --
        m_axis_ftch_tready          : in  std_logic                         ;                   --
        m_axis_ftch_tlast           : out std_logic                         ;                    --

        m_axis_ftch_tdata_new           : out std_logic_vector                                      --
                                        (96+31*C_ENABLE_CDMA+(2+C_ENABLE_CDMA)*(C_M_AXI_SG_ADDR_WIDTH-32) downto 0);                   --
        m_axis_ftch_tdata_mcdma_new           : out std_logic_vector                                      --
                                        (63 downto 0);                   --
        m_axis_ftch_tvalid_new          : out std_logic                         ;                   --
        m_axis_ftch_desc_available            : out std_logic                     ;

        m_axis2_ftch_tdata           : out std_logic_vector                                      --
                                            (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0) ;              --
        m_axis2_ftch_tvalid          : out std_logic                         ;                   --
        m_axis2_ftch_tready          : in  std_logic                         ;                   --
        m_axis2_ftch_tlast           : out std_logic                         ;                    --
        m_axis2_ftch_tdata_new           : out std_logic_vector                                      --
                                        (96+31*C_ENABLE_CDMA+(2+C_ENABLE_CDMA)*(C_M_AXI_SG_ADDR_WIDTH-32) downto 0);                   --
        m_axis2_ftch_tdata_mcdma_new           : out std_logic_vector                                      --
                                        (63 downto 0);                   --
        m_axis2_ftch_tdata_mcdma_nxt           : out std_logic_vector                                      --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0);                   --
        m_axis2_ftch_tvalid_new          : out std_logic                         ;                   --
        m_axis2_ftch_desc_available            : out std_logic                   ;

        m_axis_mm2s_cntrl_tdata     : out std_logic_vector                                 --
                                        (31 downto 0);      --
        m_axis_mm2s_cntrl_tkeep     : out std_logic_vector                                 --
                                        (3 downto 0);  --
        m_axis_mm2s_cntrl_tvalid    : out std_logic                         ;              --
        m_axis_mm2s_cntrl_tready    : in  std_logic                         := '0';              --
        m_axis_mm2s_cntrl_tlast     : out std_logic                                       --
  


    );

end axi_sg_ftch_noqueue;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_sg_ftch_noqueue is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------

-- No Constants Declared

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
-- Channel 1 internal signals
signal curdesc_tdata            : std_logic_vector
                                    (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc_tvalid           : std_logic := '0';
signal ftch_tvalid              : std_logic := '0';
signal ftch_tdata               : std_logic_vector
                                    (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0) := (others => '0');
signal ftch_tlast               : std_logic := '0';
signal ftch_tready              : std_logic := '0';

-- Misc Signals
signal writing_curdesc          : std_logic := '0';
signal writing_nxtdesc          : std_logic := '0';
signal msb_curdesc              : std_logic_vector(31 downto 0) := (others => '0');
signal ftch_tdata_new_64        : std_logic_vector (C_M_AXI_SG_ADDR_WIDTH-1 downto 0);

signal writing_lsb              : std_logic := '0';
signal writing_msb              : std_logic := '0';

signal ftch_active_int : std_logic := '0';

signal ftch_tvalid_mult              : std_logic := '0';
signal ftch_tdata_mult               : std_logic_vector
                                    (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0) := (others => '0');
signal ftch_tlast_mult               : std_logic := '0';

signal counter : std_logic_vector (3 downto 0) := (others => '0');
signal wr_cntl : std_logic := '0';

signal ftch_tdata_new : std_logic_vector (96+31*C_ENABLE_CDMA downto 0);

signal queue_wren, queue_rden : std_logic := '0';
signal queue_din : std_logic_vector (32 downto 0);
signal queue_dout : std_logic_vector (32 downto 0);
signal queue_empty, queue_full : std_logic := '0';

signal sof_ftch_desc_del, sof_ftch_desc_pulse : std_logic := '0';
signal sof_ftch_desc_del1 : std_logic := '0';
signal queue_sinit : std_logic := '0';
signal data_concat_mcdma_nxt : std_logic_vector (C_M_AXI_SG_ADDR_WIDTH-1 downto 0) := (others => '0');
signal current_bd : std_logic_vector (C_M_AXI_SG_ADDR_WIDTH-1 downto 0) := (others => '0');

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin

queue_sinit <= not m_axi_sg_aresetn;

ftch_active_int <= ftch_active or ftch2_active;



ftch_tdata_new (64 downto 0) <= data_concat (95) & data_concat (63 downto 0);-- when (ftch_active = '1') else (others =>'0');
ftch_tdata_new (96 downto 65) <= current_bd (31 downto 0);


ADDR641 : if C_M_AXI_SG_ADDR_WIDTH > 32 generate
begin

ftch_tdata_new_64 <= data_concat_64 & current_bd (C_M_AXI_SG_ADDR_WIDTH-1 downto 32);

end generate ADDR641;


-----------------------------------------------------------------------